/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details of a package that is associated with a domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PackageDetailsForAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<PackageDetailsForAssociation.Builder, PackageDetailsForAssociation> {
    private static final SdkField<String> PACKAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PackageID").getter(getter(PackageDetailsForAssociation::packageID)).setter(setter(Builder::packageID))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PackageID").build()).build();

    private static final SdkField<List<String>> PREREQUISITE_PACKAGE_ID_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PrerequisitePackageIDList")
            .getter(getter(PackageDetailsForAssociation::prerequisitePackageIDList))
            .setter(setter(Builder::prerequisitePackageIDList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrerequisitePackageIDList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<PackageAssociationConfiguration> ASSOCIATION_CONFIGURATION_FIELD = SdkField
            .<PackageAssociationConfiguration> builder(MarshallingType.SDK_POJO).memberName("AssociationConfiguration")
            .getter(getter(PackageDetailsForAssociation::associationConfiguration))
            .setter(setter(Builder::associationConfiguration)).constructor(PackageAssociationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PACKAGE_ID_FIELD,
            PREREQUISITE_PACKAGE_ID_LIST_FIELD, ASSOCIATION_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String packageID;

    private final List<String> prerequisitePackageIDList;

    private final PackageAssociationConfiguration associationConfiguration;

    private PackageDetailsForAssociation(BuilderImpl builder) {
        this.packageID = builder.packageID;
        this.prerequisitePackageIDList = builder.prerequisitePackageIDList;
        this.associationConfiguration = builder.associationConfiguration;
    }

    /**
     * <p>
     * Internal ID of the package that you want to associate with a domain.
     * </p>
     * 
     * @return Internal ID of the package that you want to associate with a domain.
     */
    public final String packageID() {
        return packageID;
    }

    /**
     * For responses, this returns true if the service returned a value for the PrerequisitePackageIDList property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasPrerequisitePackageIDList() {
        return prerequisitePackageIDList != null && !(prerequisitePackageIDList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of package IDs that must be linked to the domain before or simultaneously with the package association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPrerequisitePackageIDList} method.
     * </p>
     * 
     * @return List of package IDs that must be linked to the domain before or simultaneously with the package
     *         association.
     */
    public final List<String> prerequisitePackageIDList() {
        return prerequisitePackageIDList;
    }

    /**
     * <p>
     * The configuration parameters for associating the package with a domain.
     * </p>
     * 
     * @return The configuration parameters for associating the package with a domain.
     */
    public final PackageAssociationConfiguration associationConfiguration() {
        return associationConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(packageID());
        hashCode = 31 * hashCode + Objects.hashCode(hasPrerequisitePackageIDList() ? prerequisitePackageIDList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(associationConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PackageDetailsForAssociation)) {
            return false;
        }
        PackageDetailsForAssociation other = (PackageDetailsForAssociation) obj;
        return Objects.equals(packageID(), other.packageID())
                && hasPrerequisitePackageIDList() == other.hasPrerequisitePackageIDList()
                && Objects.equals(prerequisitePackageIDList(), other.prerequisitePackageIDList())
                && Objects.equals(associationConfiguration(), other.associationConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PackageDetailsForAssociation").add("PackageID", packageID())
                .add("PrerequisitePackageIDList", hasPrerequisitePackageIDList() ? prerequisitePackageIDList() : null)
                .add("AssociationConfiguration", associationConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PackageID":
            return Optional.ofNullable(clazz.cast(packageID()));
        case "PrerequisitePackageIDList":
            return Optional.ofNullable(clazz.cast(prerequisitePackageIDList()));
        case "AssociationConfiguration":
            return Optional.ofNullable(clazz.cast(associationConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PackageID", PACKAGE_ID_FIELD);
        map.put("PrerequisitePackageIDList", PREREQUISITE_PACKAGE_ID_LIST_FIELD);
        map.put("AssociationConfiguration", ASSOCIATION_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PackageDetailsForAssociation, T> g) {
        return obj -> g.apply((PackageDetailsForAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PackageDetailsForAssociation> {
        /**
         * <p>
         * Internal ID of the package that you want to associate with a domain.
         * </p>
         * 
         * @param packageID
         *        Internal ID of the package that you want to associate with a domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder packageID(String packageID);

        /**
         * <p>
         * List of package IDs that must be linked to the domain before or simultaneously with the package association.
         * </p>
         * 
         * @param prerequisitePackageIDList
         *        List of package IDs that must be linked to the domain before or simultaneously with the package
         *        association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prerequisitePackageIDList(Collection<String> prerequisitePackageIDList);

        /**
         * <p>
         * List of package IDs that must be linked to the domain before or simultaneously with the package association.
         * </p>
         * 
         * @param prerequisitePackageIDList
         *        List of package IDs that must be linked to the domain before or simultaneously with the package
         *        association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prerequisitePackageIDList(String... prerequisitePackageIDList);

        /**
         * <p>
         * The configuration parameters for associating the package with a domain.
         * </p>
         * 
         * @param associationConfiguration
         *        The configuration parameters for associating the package with a domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationConfiguration(PackageAssociationConfiguration associationConfiguration);

        /**
         * <p>
         * The configuration parameters for associating the package with a domain.
         * </p>
         * This is a convenience method that creates an instance of the {@link PackageAssociationConfiguration.Builder}
         * avoiding the need to create one manually via {@link PackageAssociationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PackageAssociationConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #associationConfiguration(PackageAssociationConfiguration)}.
         * 
         * @param associationConfiguration
         *        a consumer that will call methods on {@link PackageAssociationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associationConfiguration(PackageAssociationConfiguration)
         */
        default Builder associationConfiguration(Consumer<PackageAssociationConfiguration.Builder> associationConfiguration) {
            return associationConfiguration(PackageAssociationConfiguration.builder().applyMutation(associationConfiguration)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String packageID;

        private List<String> prerequisitePackageIDList = DefaultSdkAutoConstructList.getInstance();

        private PackageAssociationConfiguration associationConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(PackageDetailsForAssociation model) {
            packageID(model.packageID);
            prerequisitePackageIDList(model.prerequisitePackageIDList);
            associationConfiguration(model.associationConfiguration);
        }

        public final String getPackageID() {
            return packageID;
        }

        public final void setPackageID(String packageID) {
            this.packageID = packageID;
        }

        @Override
        public final Builder packageID(String packageID) {
            this.packageID = packageID;
            return this;
        }

        public final Collection<String> getPrerequisitePackageIDList() {
            if (prerequisitePackageIDList instanceof SdkAutoConstructList) {
                return null;
            }
            return prerequisitePackageIDList;
        }

        public final void setPrerequisitePackageIDList(Collection<String> prerequisitePackageIDList) {
            this.prerequisitePackageIDList = PackageIDListCopier.copy(prerequisitePackageIDList);
        }

        @Override
        public final Builder prerequisitePackageIDList(Collection<String> prerequisitePackageIDList) {
            this.prerequisitePackageIDList = PackageIDListCopier.copy(prerequisitePackageIDList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder prerequisitePackageIDList(String... prerequisitePackageIDList) {
            prerequisitePackageIDList(Arrays.asList(prerequisitePackageIDList));
            return this;
        }

        public final PackageAssociationConfiguration.Builder getAssociationConfiguration() {
            return associationConfiguration != null ? associationConfiguration.toBuilder() : null;
        }

        public final void setAssociationConfiguration(PackageAssociationConfiguration.BuilderImpl associationConfiguration) {
            this.associationConfiguration = associationConfiguration != null ? associationConfiguration.build() : null;
        }

        @Override
        public final Builder associationConfiguration(PackageAssociationConfiguration associationConfiguration) {
            this.associationConfiguration = associationConfiguration;
            return this;
        }

        @Override
        public PackageDetailsForAssociation build() {
            return new PackageDetailsForAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
