/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opensearchserverless;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.opensearchserverless.model.BatchGetCollectionRequest;
import software.amazon.awssdk.services.opensearchserverless.model.BatchGetCollectionResponse;
import software.amazon.awssdk.services.opensearchserverless.model.BatchGetVpcEndpointRequest;
import software.amazon.awssdk.services.opensearchserverless.model.BatchGetVpcEndpointResponse;
import software.amazon.awssdk.services.opensearchserverless.model.ConflictException;
import software.amazon.awssdk.services.opensearchserverless.model.CreateAccessPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.CreateAccessPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.CreateCollectionRequest;
import software.amazon.awssdk.services.opensearchserverless.model.CreateCollectionResponse;
import software.amazon.awssdk.services.opensearchserverless.model.CreateSecurityConfigRequest;
import software.amazon.awssdk.services.opensearchserverless.model.CreateSecurityConfigResponse;
import software.amazon.awssdk.services.opensearchserverless.model.CreateSecurityPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.CreateSecurityPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.CreateVpcEndpointRequest;
import software.amazon.awssdk.services.opensearchserverless.model.CreateVpcEndpointResponse;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteAccessPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteAccessPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteCollectionRequest;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteCollectionResponse;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteSecurityConfigRequest;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteSecurityConfigResponse;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteSecurityPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteSecurityPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteVpcEndpointRequest;
import software.amazon.awssdk.services.opensearchserverless.model.DeleteVpcEndpointResponse;
import software.amazon.awssdk.services.opensearchserverless.model.GetAccessPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.GetAccessPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.GetAccountSettingsRequest;
import software.amazon.awssdk.services.opensearchserverless.model.GetAccountSettingsResponse;
import software.amazon.awssdk.services.opensearchserverless.model.GetPoliciesStatsRequest;
import software.amazon.awssdk.services.opensearchserverless.model.GetPoliciesStatsResponse;
import software.amazon.awssdk.services.opensearchserverless.model.GetSecurityConfigRequest;
import software.amazon.awssdk.services.opensearchserverless.model.GetSecurityConfigResponse;
import software.amazon.awssdk.services.opensearchserverless.model.GetSecurityPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.GetSecurityPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.InternalServerException;
import software.amazon.awssdk.services.opensearchserverless.model.ListAccessPoliciesRequest;
import software.amazon.awssdk.services.opensearchserverless.model.ListAccessPoliciesResponse;
import software.amazon.awssdk.services.opensearchserverless.model.ListCollectionsRequest;
import software.amazon.awssdk.services.opensearchserverless.model.ListCollectionsResponse;
import software.amazon.awssdk.services.opensearchserverless.model.ListSecurityConfigsRequest;
import software.amazon.awssdk.services.opensearchserverless.model.ListSecurityConfigsResponse;
import software.amazon.awssdk.services.opensearchserverless.model.ListSecurityPoliciesRequest;
import software.amazon.awssdk.services.opensearchserverless.model.ListSecurityPoliciesResponse;
import software.amazon.awssdk.services.opensearchserverless.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.opensearchserverless.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.opensearchserverless.model.ListVpcEndpointsRequest;
import software.amazon.awssdk.services.opensearchserverless.model.ListVpcEndpointsResponse;
import software.amazon.awssdk.services.opensearchserverless.model.OcuLimitExceededException;
import software.amazon.awssdk.services.opensearchserverless.model.OpenSearchServerlessException;
import software.amazon.awssdk.services.opensearchserverless.model.ResourceNotFoundException;
import software.amazon.awssdk.services.opensearchserverless.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.opensearchserverless.model.TagResourceRequest;
import software.amazon.awssdk.services.opensearchserverless.model.TagResourceResponse;
import software.amazon.awssdk.services.opensearchserverless.model.UntagResourceRequest;
import software.amazon.awssdk.services.opensearchserverless.model.UntagResourceResponse;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateAccessPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateAccessPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateAccountSettingsRequest;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateAccountSettingsResponse;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateCollectionRequest;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateCollectionResponse;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateSecurityConfigRequest;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateSecurityConfigResponse;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateSecurityPolicyRequest;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateSecurityPolicyResponse;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateVpcEndpointRequest;
import software.amazon.awssdk.services.opensearchserverless.model.UpdateVpcEndpointResponse;
import software.amazon.awssdk.services.opensearchserverless.model.ValidationException;
import software.amazon.awssdk.services.opensearchserverless.transform.BatchGetCollectionRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.BatchGetVpcEndpointRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.CreateAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.CreateCollectionRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.CreateSecurityConfigRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.CreateSecurityPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.CreateVpcEndpointRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.DeleteAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.DeleteCollectionRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.DeleteSecurityConfigRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.DeleteSecurityPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.DeleteVpcEndpointRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.GetAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.GetAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.GetPoliciesStatsRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.GetSecurityConfigRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.GetSecurityPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.ListAccessPoliciesRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.ListCollectionsRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.ListSecurityConfigsRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.ListSecurityPoliciesRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.ListVpcEndpointsRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.UpdateAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.UpdateAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.UpdateCollectionRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.UpdateSecurityConfigRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.UpdateSecurityPolicyRequestMarshaller;
import software.amazon.awssdk.services.opensearchserverless.transform.UpdateVpcEndpointRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link OpenSearchServerlessAsyncClient}.
 *
 * @see OpenSearchServerlessAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOpenSearchServerlessAsyncClient implements OpenSearchServerlessAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultOpenSearchServerlessAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final OpenSearchServerlessServiceClientConfiguration serviceClientConfiguration;

    protected DefaultOpenSearchServerlessAsyncClient(OpenSearchServerlessServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns attributes for one or more collections, including the collection endpoint and the OpenSearch Dashboards
     * endpoint. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-manage.html">Creating and
     * managing Amazon OpenSearch Serverless collections</a>.
     * </p>
     *
     * @param batchGetCollectionRequest
     * @return A Java Future containing the result of the BatchGetCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.BatchGetCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/BatchGetCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetCollectionResponse> batchGetCollection(BatchGetCollectionRequest batchGetCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetCollectionRequest, BatchGetCollectionResponse>()
                            .withOperationName("BatchGetCollection")
                            .withMarshaller(new BatchGetCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGetCollectionRequest));
            CompletableFuture<BatchGetCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns attributes for one or more VPC endpoints associated with the current account. For more information, see
     * <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-vpc.html">Access Amazon
     * OpenSearch Serverless using an interface endpoint</a>.
     * </p>
     *
     * @param batchGetVpcEndpointRequest
     * @return A Java Future containing the result of the BatchGetVpcEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.BatchGetVpcEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/BatchGetVpcEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetVpcEndpointResponse> batchGetVpcEndpoint(
            BatchGetVpcEndpointRequest batchGetVpcEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetVpcEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetVpcEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetVpcEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetVpcEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetVpcEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetVpcEndpointRequest, BatchGetVpcEndpointResponse>()
                            .withOperationName("BatchGetVpcEndpoint")
                            .withMarshaller(new BatchGetVpcEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGetVpcEndpointRequest));
            CompletableFuture<BatchGetVpcEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a data access policy for OpenSearch Serverless. Access policies limit access to collections and the
     * resources within them, and allow a user to access that data irrespective of the access mechanism or network
     * source. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-data-access.html">Data
     * access control for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param createAccessPolicyRequest
     * @return A Java Future containing the result of the CreateAccessPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>ServiceQuotaExceededException Thrown when you attempt to create more resources than the service
     *         allows based on service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.CreateAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/CreateAccessPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessPolicyResponse> createAccessPolicy(CreateAccessPolicyRequest createAccessPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAccessPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAccessPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessPolicyRequest, CreateAccessPolicyResponse>()
                            .withOperationName("CreateAccessPolicy")
                            .withMarshaller(new CreateAccessPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAccessPolicyRequest));
            CompletableFuture<CreateAccessPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new OpenSearch Serverless collection. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-manage.html">Creating and
     * managing Amazon OpenSearch Serverless collections</a>.
     * </p>
     *
     * @param createCollectionRequest
     * @return A Java Future containing the result of the CreateCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OcuLimitExceededException Thrown when the collection you're attempting to create results in a number
     *         of search or indexing OCUs that exceeds the account limit.</li>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>ServiceQuotaExceededException Thrown when you attempt to create more resources than the service
     *         allows based on service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.CreateCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/CreateCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCollectionResponse> createCollection(CreateCollectionRequest createCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCollectionRequest, CreateCollectionResponse>()
                            .withOperationName("CreateCollection")
                            .withMarshaller(new CreateCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCollectionRequest));
            CompletableFuture<CreateCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Specifies a security configuration for OpenSearch Serverless. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-saml.html">SAML
     * authentication for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param createSecurityConfigRequest
     * @return A Java Future containing the result of the CreateSecurityConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>ServiceQuotaExceededException Thrown when you attempt to create more resources than the service
     *         allows based on service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.CreateSecurityConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/CreateSecurityConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSecurityConfigResponse> createSecurityConfig(
            CreateSecurityConfigRequest createSecurityConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSecurityConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSecurityConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSecurityConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSecurityConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSecurityConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSecurityConfigRequest, CreateSecurityConfigResponse>()
                            .withOperationName("CreateSecurityConfig")
                            .withMarshaller(new CreateSecurityConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSecurityConfigRequest));
            CompletableFuture<CreateSecurityConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a security policy to be used by one or more OpenSearch Serverless collections. Security policies provide
     * access to a collection and its OpenSearch Dashboards endpoint from public networks or specific VPC endpoints.
     * They also allow you to secure a collection with a KMS encryption key. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-network.html">Network
     * access for Amazon OpenSearch Serverless</a> and <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-encryption.html">Encryption
     * at rest for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param createSecurityPolicyRequest
     * @return A Java Future containing the result of the CreateSecurityPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>ServiceQuotaExceededException Thrown when you attempt to create more resources than the service
     *         allows based on service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.CreateSecurityPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/CreateSecurityPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSecurityPolicyResponse> createSecurityPolicy(
            CreateSecurityPolicyRequest createSecurityPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSecurityPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSecurityPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSecurityPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSecurityPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSecurityPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSecurityPolicyRequest, CreateSecurityPolicyResponse>()
                            .withOperationName("CreateSecurityPolicy")
                            .withMarshaller(new CreateSecurityPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSecurityPolicyRequest));
            CompletableFuture<CreateSecurityPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an OpenSearch Serverless-managed interface VPC endpoint. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-vpc.html">Access Amazon
     * OpenSearch Serverless using an interface endpoint</a>.
     * </p>
     *
     * @param createVpcEndpointRequest
     * @return A Java Future containing the result of the CreateVpcEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>ServiceQuotaExceededException Thrown when you attempt to create more resources than the service
     *         allows based on service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.CreateVpcEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/CreateVpcEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVpcEndpointResponse> createVpcEndpoint(CreateVpcEndpointRequest createVpcEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVpcEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVpcEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVpcEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVpcEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVpcEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVpcEndpointRequest, CreateVpcEndpointResponse>()
                            .withOperationName("CreateVpcEndpoint")
                            .withMarshaller(new CreateVpcEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVpcEndpointRequest));
            CompletableFuture<CreateVpcEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an OpenSearch Serverless access policy. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-data-access.html">Data
     * access control for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param deleteAccessPolicyRequest
     * @return A Java Future containing the result of the DeleteAccessPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.DeleteAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/DeleteAccessPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessPolicyResponse> deleteAccessPolicy(DeleteAccessPolicyRequest deleteAccessPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAccessPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAccessPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessPolicyRequest, DeleteAccessPolicyResponse>()
                            .withOperationName("DeleteAccessPolicy")
                            .withMarshaller(new DeleteAccessPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAccessPolicyRequest));
            CompletableFuture<DeleteAccessPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an OpenSearch Serverless collection. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-manage.html">Creating and
     * managing Amazon OpenSearch Serverless collections</a>.
     * </p>
     *
     * @param deleteCollectionRequest
     * @return A Java Future containing the result of the DeleteCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.DeleteCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/DeleteCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCollectionResponse> deleteCollection(DeleteCollectionRequest deleteCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCollectionRequest, DeleteCollectionResponse>()
                            .withOperationName("DeleteCollection")
                            .withMarshaller(new DeleteCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCollectionRequest));
            CompletableFuture<DeleteCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a security configuration for OpenSearch Serverless. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-saml.html">SAML
     * authentication for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param deleteSecurityConfigRequest
     * @return A Java Future containing the result of the DeleteSecurityConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.DeleteSecurityConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/DeleteSecurityConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSecurityConfigResponse> deleteSecurityConfig(
            DeleteSecurityConfigRequest deleteSecurityConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSecurityConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSecurityConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSecurityConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSecurityConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSecurityConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSecurityConfigRequest, DeleteSecurityConfigResponse>()
                            .withOperationName("DeleteSecurityConfig")
                            .withMarshaller(new DeleteSecurityConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSecurityConfigRequest));
            CompletableFuture<DeleteSecurityConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an OpenSearch Serverless security policy.
     * </p>
     *
     * @param deleteSecurityPolicyRequest
     * @return A Java Future containing the result of the DeleteSecurityPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.DeleteSecurityPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/DeleteSecurityPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSecurityPolicyResponse> deleteSecurityPolicy(
            DeleteSecurityPolicyRequest deleteSecurityPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSecurityPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSecurityPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSecurityPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSecurityPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSecurityPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSecurityPolicyRequest, DeleteSecurityPolicyResponse>()
                            .withOperationName("DeleteSecurityPolicy")
                            .withMarshaller(new DeleteSecurityPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSecurityPolicyRequest));
            CompletableFuture<DeleteSecurityPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an OpenSearch Serverless-managed interface endpoint. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-vpc.html">Access Amazon
     * OpenSearch Serverless using an interface endpoint</a>.
     * </p>
     *
     * @param deleteVpcEndpointRequest
     * @return A Java Future containing the result of the DeleteVpcEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.DeleteVpcEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/DeleteVpcEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVpcEndpointResponse> deleteVpcEndpoint(DeleteVpcEndpointRequest deleteVpcEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVpcEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVpcEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVpcEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVpcEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVpcEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVpcEndpointRequest, DeleteVpcEndpointResponse>()
                            .withOperationName("DeleteVpcEndpoint")
                            .withMarshaller(new DeleteVpcEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVpcEndpointRequest));
            CompletableFuture<DeleteVpcEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an OpenSearch Serverless access policy. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-data-access.html">Data
     * access control for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param getAccessPolicyRequest
     * @return A Java Future containing the result of the GetAccessPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.GetAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/GetAccessPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPolicyResponse> getAccessPolicy(GetAccessPolicyRequest getAccessPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccessPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccessPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPolicyRequest, GetAccessPolicyResponse>()
                            .withOperationName("GetAccessPolicy")
                            .withMarshaller(new GetAccessPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccessPolicyRequest));
            CompletableFuture<GetAccessPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns account-level settings related to OpenSearch Serverless.
     * </p>
     *
     * @param getAccountSettingsRequest
     * @return A Java Future containing the result of the GetAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/GetAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountSettingsResponse> getAccountSettings(GetAccountSettingsRequest getAccountSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccountSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccountSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountSettingsRequest, GetAccountSettingsResponse>()
                            .withOperationName("GetAccountSettings")
                            .withMarshaller(new GetAccountSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccountSettingsRequest));
            CompletableFuture<GetAccountSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns statistical information about your OpenSearch Serverless access policies, security configurations, and
     * security policies.
     * </p>
     *
     * @param getPoliciesStatsRequest
     * @return A Java Future containing the result of the GetPoliciesStats operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.GetPoliciesStats
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/GetPoliciesStats"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPoliciesStatsResponse> getPoliciesStats(GetPoliciesStatsRequest getPoliciesStatsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPoliciesStatsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPoliciesStats");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPoliciesStatsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPoliciesStatsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPoliciesStatsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPoliciesStatsRequest, GetPoliciesStatsResponse>()
                            .withOperationName("GetPoliciesStats")
                            .withMarshaller(new GetPoliciesStatsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPoliciesStatsRequest));
            CompletableFuture<GetPoliciesStatsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an OpenSearch Serverless security configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-saml.html">SAML
     * authentication for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param getSecurityConfigRequest
     * @return A Java Future containing the result of the GetSecurityConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.GetSecurityConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/GetSecurityConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSecurityConfigResponse> getSecurityConfig(GetSecurityConfigRequest getSecurityConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSecurityConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSecurityConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSecurityConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSecurityConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSecurityConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSecurityConfigRequest, GetSecurityConfigResponse>()
                            .withOperationName("GetSecurityConfig")
                            .withMarshaller(new GetSecurityConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSecurityConfigRequest));
            CompletableFuture<GetSecurityConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a configured OpenSearch Serverless security policy. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-network.html">Network
     * access for Amazon OpenSearch Serverless</a> and <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-encryption.html">Encryption
     * at rest for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param getSecurityPolicyRequest
     * @return A Java Future containing the result of the GetSecurityPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.GetSecurityPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/GetSecurityPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSecurityPolicyResponse> getSecurityPolicy(GetSecurityPolicyRequest getSecurityPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSecurityPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSecurityPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSecurityPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSecurityPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSecurityPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSecurityPolicyRequest, GetSecurityPolicyResponse>()
                            .withOperationName("GetSecurityPolicy")
                            .withMarshaller(new GetSecurityPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSecurityPolicyRequest));
            CompletableFuture<GetSecurityPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a list of OpenSearch Serverless access policies.
     * </p>
     *
     * @param listAccessPoliciesRequest
     * @return A Java Future containing the result of the ListAccessPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.ListAccessPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/ListAccessPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessPoliciesResponse> listAccessPolicies(ListAccessPoliciesRequest listAccessPoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccessPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAccessPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccessPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessPoliciesRequest, ListAccessPoliciesResponse>()
                            .withOperationName("ListAccessPolicies")
                            .withMarshaller(new ListAccessPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAccessPoliciesRequest));
            CompletableFuture<ListAccessPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all OpenSearch Serverless collections. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-manage.html">Creating and
     * managing Amazon OpenSearch Serverless collections</a>.
     * </p>
     * <note>
     * <p>
     * Make sure to include an empty request body {} if you don't include any collection filters in the request.
     * </p>
     * </note>
     *
     * @param listCollectionsRequest
     * @return A Java Future containing the result of the ListCollections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.ListCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/ListCollections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCollectionsResponse> listCollections(ListCollectionsRequest listCollectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCollectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCollections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCollectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCollectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCollectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCollectionsRequest, ListCollectionsResponse>()
                            .withOperationName("ListCollections")
                            .withMarshaller(new ListCollectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCollectionsRequest));
            CompletableFuture<ListCollectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about configured OpenSearch Serverless security configurations. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-saml.html">SAML
     * authentication for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param listSecurityConfigsRequest
     * @return A Java Future containing the result of the ListSecurityConfigs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.ListSecurityConfigs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/ListSecurityConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSecurityConfigsResponse> listSecurityConfigs(
            ListSecurityConfigsRequest listSecurityConfigsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecurityConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecurityConfigs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSecurityConfigsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSecurityConfigsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSecurityConfigsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSecurityConfigsRequest, ListSecurityConfigsResponse>()
                            .withOperationName("ListSecurityConfigs")
                            .withMarshaller(new ListSecurityConfigsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSecurityConfigsRequest));
            CompletableFuture<ListSecurityConfigsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about configured OpenSearch Serverless security policies.
     * </p>
     *
     * @param listSecurityPoliciesRequest
     * @return A Java Future containing the result of the ListSecurityPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.ListSecurityPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/ListSecurityPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSecurityPoliciesResponse> listSecurityPolicies(
            ListSecurityPoliciesRequest listSecurityPoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecurityPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecurityPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSecurityPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSecurityPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSecurityPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSecurityPoliciesRequest, ListSecurityPoliciesResponse>()
                            .withOperationName("ListSecurityPolicies")
                            .withMarshaller(new ListSecurityPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSecurityPoliciesRequest));
            CompletableFuture<ListSecurityPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the tags for an OpenSearch Serverless resource. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-collection.html">Tagging Amazon
     * OpenSearch Serverless collections</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the OpenSearch Serverless-managed interface VPC endpoints associated with the current account. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-vpc.html">Access Amazon
     * OpenSearch Serverless using an interface endpoint</a>.
     * </p>
     *
     * @param listVpcEndpointsRequest
     * @return A Java Future containing the result of the ListVpcEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.ListVpcEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/ListVpcEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVpcEndpointsResponse> listVpcEndpoints(ListVpcEndpointsRequest listVpcEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVpcEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVpcEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVpcEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVpcEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVpcEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVpcEndpointsRequest, ListVpcEndpointsResponse>()
                            .withOperationName("ListVpcEndpoints")
                            .withMarshaller(new ListVpcEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVpcEndpointsRequest));
            CompletableFuture<ListVpcEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates tags with an OpenSearch Serverless resource. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-collection.html">Tagging Amazon
     * OpenSearch Serverless collections</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>ServiceQuotaExceededException Thrown when you attempt to create more resources than the service
     *         allows based on service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag or set of tags from an OpenSearch Serverless resource. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-collection.html">Tagging Amazon
     * OpenSearch Serverless collections</a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an OpenSearch Serverless access policy. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-data-access.html">Data
     * access control for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param updateAccessPolicyRequest
     * @return A Java Future containing the result of the UpdateAccessPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.UpdateAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/UpdateAccessPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAccessPolicyResponse> updateAccessPolicy(UpdateAccessPolicyRequest updateAccessPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccessPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAccessPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAccessPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAccessPolicyRequest, UpdateAccessPolicyResponse>()
                            .withOperationName("UpdateAccessPolicy")
                            .withMarshaller(new UpdateAccessPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAccessPolicyRequest));
            CompletableFuture<UpdateAccessPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the OpenSearch Serverless settings for the current Amazon Web Services account. For more information, see
     * <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-scaling.html">Managing
     * capacity limits for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param updateAccountSettingsRequest
     * @return A Java Future containing the result of the UpdateAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.UpdateAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/UpdateAccountSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAccountSettingsResponse> updateAccountSettings(
            UpdateAccountSettingsRequest updateAccountSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccountSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAccountSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAccountSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAccountSettingsRequest, UpdateAccountSettingsResponse>()
                            .withOperationName("UpdateAccountSettings")
                            .withMarshaller(new UpdateAccountSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAccountSettingsRequest));
            CompletableFuture<UpdateAccountSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an OpenSearch Serverless collection.
     * </p>
     *
     * @param updateCollectionRequest
     * @return A Java Future containing the result of the UpdateCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.UpdateCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/UpdateCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCollectionResponse> updateCollection(UpdateCollectionRequest updateCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCollectionRequest, UpdateCollectionResponse>()
                            .withOperationName("UpdateCollection")
                            .withMarshaller(new UpdateCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateCollectionRequest));
            CompletableFuture<UpdateCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a security configuration for OpenSearch Serverless. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-saml.html">SAML
     * authentication for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param updateSecurityConfigRequest
     * @return A Java Future containing the result of the UpdateSecurityConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.UpdateSecurityConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/UpdateSecurityConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSecurityConfigResponse> updateSecurityConfig(
            UpdateSecurityConfigRequest updateSecurityConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSecurityConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSecurityConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSecurityConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSecurityConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSecurityConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSecurityConfigRequest, UpdateSecurityConfigResponse>()
                            .withOperationName("UpdateSecurityConfig")
                            .withMarshaller(new UpdateSecurityConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSecurityConfigRequest));
            CompletableFuture<UpdateSecurityConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an OpenSearch Serverless security policy. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-network.html">Network
     * access for Amazon OpenSearch Serverless</a> and <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-encryption.html">Encryption
     * at rest for Amazon OpenSearch Serverless</a>.
     * </p>
     *
     * @param updateSecurityPolicyRequest
     * @return A Java Future containing the result of the UpdateSecurityPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ResourceNotFoundException Thrown when accessing or deleting a resource that does not exist.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>ServiceQuotaExceededException Thrown when you attempt to create more resources than the service
     *         allows based on service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.UpdateSecurityPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/UpdateSecurityPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSecurityPolicyResponse> updateSecurityPolicy(
            UpdateSecurityPolicyRequest updateSecurityPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSecurityPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSecurityPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSecurityPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSecurityPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSecurityPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSecurityPolicyRequest, UpdateSecurityPolicyResponse>()
                            .withOperationName("UpdateSecurityPolicy")
                            .withMarshaller(new UpdateSecurityPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSecurityPolicyRequest));
            CompletableFuture<UpdateSecurityPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an OpenSearch Serverless-managed interface endpoint. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/serverless-vpc.html">Access Amazon
     * OpenSearch Serverless using an interface endpoint</a>.
     * </p>
     *
     * @param updateVpcEndpointRequest
     * @return A Java Future containing the result of the UpdateVpcEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Thrown when an error internal to the service occurs while processing a
     *         request.</li>
     *         <li>ConflictException When creating a resource, thrown when a resource with the same name already exists
     *         or is being created. When deleting a resource, thrown when the resource is not in the ACTIVE or FAILED
     *         state.</li>
     *         <li>ValidationException Thrown when the HTTP request contains invalid input or is missing required input.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OpenSearchServerlessException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample OpenSearchServerlessAsyncClient.UpdateVpcEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/opensearchserverless-2021-11-01/UpdateVpcEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVpcEndpointResponse> updateVpcEndpoint(UpdateVpcEndpointRequest updateVpcEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVpcEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OpenSearchServerless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVpcEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVpcEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVpcEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVpcEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVpcEndpointRequest, UpdateVpcEndpointResponse>()
                            .withOperationName("UpdateVpcEndpoint")
                            .withMarshaller(new UpdateVpcEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateVpcEndpointRequest));
            CompletableFuture<UpdateVpcEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final OpenSearchServerlessServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OpenSearchServerlessException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OcuLimitExceededException")
                                .exceptionBuilderSupplier(OcuLimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
