/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opsworks.model;

import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.opsworks.transform.EbsBlockDeviceMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon EBS volume. This data type maps directly to the Amazon EC2 <a
 * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a> data type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EbsBlockDevice implements StructuredPojo, ToCopyableBuilder<EbsBlockDevice.Builder, EbsBlockDevice> {
    private final String snapshotId;

    private final Integer iops;

    private final Integer volumeSize;

    private final String volumeType;

    private final Boolean deleteOnTermination;

    private EbsBlockDevice(BuilderImpl builder) {
        this.snapshotId = builder.snapshotId;
        this.iops = builder.iops;
        this.volumeSize = builder.volumeSize;
        this.volumeType = builder.volumeType;
        this.deleteOnTermination = builder.deleteOnTermination;
    }

    /**
     * <p>
     * The snapshot ID.
     * </p>
     * 
     * @return The snapshot ID.
     */
    public String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The number of I/O operations per second (IOPS) that the volume supports. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a>.
     * </p>
     * 
     * @return The number of I/O operations per second (IOPS) that the volume supports. For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a>.
     */
    public Integer iops() {
        return iops;
    }

    /**
     * <p>
     * The volume size, in GiB. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a>.
     * </p>
     * 
     * @return The volume size, in GiB. For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a>.
     */
    public Integer volumeSize() {
        return volumeSize;
    }

    /**
     * <p>
     * The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned IOPS (SSD)
     * volumes, and <code>standard</code> for Magnetic volumes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned
     *         IOPS (SSD) volumes, and <code>standard</code> for Magnetic volumes.
     * @see VolumeType
     */
    public VolumeType volumeType() {
        return VolumeType.fromValue(volumeType);
    }

    /**
     * <p>
     * The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned IOPS (SSD)
     * volumes, and <code>standard</code> for Magnetic volumes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned
     *         IOPS (SSD) volumes, and <code>standard</code> for Magnetic volumes.
     * @see VolumeType
     */
    public String volumeTypeAsString() {
        return volumeType;
    }

    /**
     * <p>
     * Whether the volume is deleted on instance termination.
     * </p>
     * 
     * @return Whether the volume is deleted on instance termination.
     */
    public Boolean deleteOnTermination() {
        return deleteOnTermination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSize());
        hashCode = 31 * hashCode + Objects.hashCode(volumeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(deleteOnTermination());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EbsBlockDevice)) {
            return false;
        }
        EbsBlockDevice other = (EbsBlockDevice) obj;
        return Objects.equals(snapshotId(), other.snapshotId()) && Objects.equals(iops(), other.iops())
                && Objects.equals(volumeSize(), other.volumeSize())
                && Objects.equals(volumeTypeAsString(), other.volumeTypeAsString())
                && Objects.equals(deleteOnTermination(), other.deleteOnTermination());
    }

    @Override
    public String toString() {
        return ToString.builder("EbsBlockDevice").add("SnapshotId", snapshotId()).add("Iops", iops())
                .add("VolumeSize", volumeSize()).add("VolumeType", volumeTypeAsString())
                .add("DeleteOnTermination", deleteOnTermination()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "VolumeSize":
            return Optional.ofNullable(clazz.cast(volumeSize()));
        case "VolumeType":
            return Optional.ofNullable(clazz.cast(volumeTypeAsString()));
        case "DeleteOnTermination":
            return Optional.ofNullable(clazz.cast(deleteOnTermination()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        EbsBlockDeviceMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, EbsBlockDevice> {
        /**
         * <p>
         * The snapshot ID.
         * </p>
         * 
         * @param snapshotId
         *        The snapshot ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The number of I/O operations per second (IOPS) that the volume supports. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a>.
         * </p>
         * 
         * @param iops
         *        The number of I/O operations per second (IOPS) that the volume supports. For more information, see <a
         *        href
         *        ="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * The volume size, in GiB. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html">EbsBlockDevice</a>.
         * </p>
         * 
         * @param volumeSize
         *        The volume size, in GiB. For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EbsBlockDevice.html"
         *        >EbsBlockDevice</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSize(Integer volumeSize);

        /**
         * <p>
         * The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned IOPS
         * (SSD) volumes, and <code>standard</code> for Magnetic volumes.
         * </p>
         * 
         * @param volumeType
         *        The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned
         *        IOPS (SSD) volumes, and <code>standard</code> for Magnetic volumes.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(String volumeType);

        /**
         * <p>
         * The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned IOPS
         * (SSD) volumes, and <code>standard</code> for Magnetic volumes.
         * </p>
         * 
         * @param volumeType
         *        The volume type. <code>gp2</code> for General Purpose (SSD) volumes, <code>io1</code> for Provisioned
         *        IOPS (SSD) volumes, and <code>standard</code> for Magnetic volumes.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(VolumeType volumeType);

        /**
         * <p>
         * Whether the volume is deleted on instance termination.
         * </p>
         * 
         * @param deleteOnTermination
         *        Whether the volume is deleted on instance termination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteOnTermination(Boolean deleteOnTermination);
    }

    static final class BuilderImpl implements Builder {
        private String snapshotId;

        private Integer iops;

        private Integer volumeSize;

        private String volumeType;

        private Boolean deleteOnTermination;

        private BuilderImpl() {
        }

        private BuilderImpl(EbsBlockDevice model) {
            snapshotId(model.snapshotId);
            iops(model.iops);
            volumeSize(model.volumeSize);
            volumeType(model.volumeType);
            deleteOnTermination(model.deleteOnTermination);
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        public final Integer getIops() {
            return iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        public final Integer getVolumeSize() {
            return volumeSize;
        }

        @Override
        public final Builder volumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
            return this;
        }

        public final void setVolumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
        }

        public final String getVolumeType() {
            return volumeType;
        }

        @Override
        public final Builder volumeType(String volumeType) {
            this.volumeType = volumeType;
            return this;
        }

        @Override
        public final Builder volumeType(VolumeType volumeType) {
            this.volumeType(volumeType.toString());
            return this;
        }

        public final void setVolumeType(String volumeType) {
            this.volumeType = volumeType;
        }

        public final Boolean getDeleteOnTermination() {
            return deleteOnTermination;
        }

        @Override
        public final Builder deleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
            return this;
        }

        public final void setDeleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
        }

        @Override
        public EbsBlockDevice build() {
            return new EbsBlockDevice(this);
        }
    }
}
