/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opsworks.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.opsworks.transform.AutoScalingThresholdsMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a load-based auto scaling upscaling or downscaling threshold configuration, which specifies when AWS
 * OpsWorks Stacks starts or stops load-based instances.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class AutoScalingThresholds implements StructuredPojo,
        ToCopyableBuilder<AutoScalingThresholds.Builder, AutoScalingThresholds> {
    private final Integer instanceCount;

    private final Integer thresholdsWaitTime;

    private final Integer ignoreMetricsTime;

    private final Double cpuThreshold;

    private final Double memoryThreshold;

    private final Double loadThreshold;

    private final List<String> alarms;

    private AutoScalingThresholds(BuilderImpl builder) {
        this.instanceCount = builder.instanceCount;
        this.thresholdsWaitTime = builder.thresholdsWaitTime;
        this.ignoreMetricsTime = builder.ignoreMetricsTime;
        this.cpuThreshold = builder.cpuThreshold;
        this.memoryThreshold = builder.memoryThreshold;
        this.loadThreshold = builder.loadThreshold;
        this.alarms = builder.alarms;
    }

    /**
     * <p>
     * The number of instances to add or remove when the load exceeds a threshold.
     * </p>
     * 
     * @return The number of instances to add or remove when the load exceeds a threshold.
     */
    public Integer instanceCount() {
        return instanceCount;
    }

    /**
     * <p>
     * The amount of time, in minutes, that the load must exceed a threshold before more instances are added or removed.
     * </p>
     * 
     * @return The amount of time, in minutes, that the load must exceed a threshold before more instances are added or
     *         removed.
     */
    public Integer thresholdsWaitTime() {
        return thresholdsWaitTime;
    }

    /**
     * <p>
     * The amount of time (in minutes) after a scaling event occurs that AWS OpsWorks Stacks should ignore metrics and
     * suppress additional scaling events. For example, AWS OpsWorks Stacks adds new instances following an upscaling
     * event but the instances won't start reducing the load until they have been booted and configured. There is no
     * point in raising additional scaling events during that operation, which typically takes several minutes.
     * <code>IgnoreMetricsTime</code> allows you to direct AWS OpsWorks Stacks to suppress scaling events long enough to
     * get the new instances online.
     * </p>
     * 
     * @return The amount of time (in minutes) after a scaling event occurs that AWS OpsWorks Stacks should ignore
     *         metrics and suppress additional scaling events. For example, AWS OpsWorks Stacks adds new instances
     *         following an upscaling event but the instances won't start reducing the load until they have been booted
     *         and configured. There is no point in raising additional scaling events during that operation, which
     *         typically takes several minutes. <code>IgnoreMetricsTime</code> allows you to direct AWS OpsWorks Stacks
     *         to suppress scaling events long enough to get the new instances online.
     */
    public Integer ignoreMetricsTime() {
        return ignoreMetricsTime;
    }

    /**
     * <p>
     * The CPU utilization threshold, as a percent of the available CPU. A value of -1 disables the threshold.
     * </p>
     * 
     * @return The CPU utilization threshold, as a percent of the available CPU. A value of -1 disables the threshold.
     */
    public Double cpuThreshold() {
        return cpuThreshold;
    }

    /**
     * <p>
     * The memory utilization threshold, as a percent of the available memory. A value of -1 disables the threshold.
     * </p>
     * 
     * @return The memory utilization threshold, as a percent of the available memory. A value of -1 disables the
     *         threshold.
     */
    public Double memoryThreshold() {
        return memoryThreshold;
    }

    /**
     * <p>
     * The load threshold. A value of -1 disables the threshold. For more information about how load is computed, see <a
     * href="http://en.wikipedia.org/wiki/Load_%28computing%29">Load (computing)</a>.
     * </p>
     * 
     * @return The load threshold. A value of -1 disables the threshold. For more information about how load is
     *         computed, see <a href="http://en.wikipedia.org/wiki/Load_%28computing%29">Load (computing)</a>.
     */
    public Double loadThreshold() {
        return loadThreshold;
    }

    /**
     * <p>
     * Custom Cloudwatch auto scaling alarms, to be used as thresholds. This parameter takes a list of up to five alarm
     * names, which are case sensitive and must be in the same region as the stack.
     * </p>
     * <note>
     * <p>
     * To use custom alarms, you must update your service role to allow <code>cloudwatch:DescribeAlarms</code>. You can
     * either have AWS OpsWorks Stacks update the role for you when you first use this feature or you can edit the role
     * manually. For more information, see <a
     * href="http://docs.aws.amazon.com/opsworks/latest/userguide/opsworks-security-servicerole.html">Allowing AWS
     * OpsWorks Stacks to Act on Your Behalf</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Custom Cloudwatch auto scaling alarms, to be used as thresholds. This parameter takes a list of up to
     *         five alarm names, which are case sensitive and must be in the same region as the stack.</p> <note>
     *         <p>
     *         To use custom alarms, you must update your service role to allow <code>cloudwatch:DescribeAlarms</code>.
     *         You can either have AWS OpsWorks Stacks update the role for you when you first use this feature or you
     *         can edit the role manually. For more information, see <a
     *         href="http://docs.aws.amazon.com/opsworks/latest/userguide/opsworks-security-servicerole.html">Allowing
     *         AWS OpsWorks Stacks to Act on Your Behalf</a>.
     *         </p>
     */
    public List<String> alarms() {
        return alarms;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(thresholdsWaitTime());
        hashCode = 31 * hashCode + Objects.hashCode(ignoreMetricsTime());
        hashCode = 31 * hashCode + Objects.hashCode(cpuThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(memoryThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(loadThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(alarms());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingThresholds)) {
            return false;
        }
        AutoScalingThresholds other = (AutoScalingThresholds) obj;
        return Objects.equals(instanceCount(), other.instanceCount())
                && Objects.equals(thresholdsWaitTime(), other.thresholdsWaitTime())
                && Objects.equals(ignoreMetricsTime(), other.ignoreMetricsTime())
                && Objects.equals(cpuThreshold(), other.cpuThreshold())
                && Objects.equals(memoryThreshold(), other.memoryThreshold())
                && Objects.equals(loadThreshold(), other.loadThreshold()) && Objects.equals(alarms(), other.alarms());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (instanceCount() != null) {
            sb.append("InstanceCount: ").append(instanceCount()).append(",");
        }
        if (thresholdsWaitTime() != null) {
            sb.append("ThresholdsWaitTime: ").append(thresholdsWaitTime()).append(",");
        }
        if (ignoreMetricsTime() != null) {
            sb.append("IgnoreMetricsTime: ").append(ignoreMetricsTime()).append(",");
        }
        if (cpuThreshold() != null) {
            sb.append("CpuThreshold: ").append(cpuThreshold()).append(",");
        }
        if (memoryThreshold() != null) {
            sb.append("MemoryThreshold: ").append(memoryThreshold()).append(",");
        }
        if (loadThreshold() != null) {
            sb.append("LoadThreshold: ").append(loadThreshold()).append(",");
        }
        if (alarms() != null) {
            sb.append("Alarms: ").append(alarms()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceCount":
            return Optional.of(clazz.cast(instanceCount()));
        case "ThresholdsWaitTime":
            return Optional.of(clazz.cast(thresholdsWaitTime()));
        case "IgnoreMetricsTime":
            return Optional.of(clazz.cast(ignoreMetricsTime()));
        case "CpuThreshold":
            return Optional.of(clazz.cast(cpuThreshold()));
        case "MemoryThreshold":
            return Optional.of(clazz.cast(memoryThreshold()));
        case "LoadThreshold":
            return Optional.of(clazz.cast(loadThreshold()));
        case "Alarms":
            return Optional.of(clazz.cast(alarms()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AutoScalingThresholdsMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AutoScalingThresholds> {
        /**
         * <p>
         * The number of instances to add or remove when the load exceeds a threshold.
         * </p>
         * 
         * @param instanceCount
         *        The number of instances to add or remove when the load exceeds a threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <p>
         * The amount of time, in minutes, that the load must exceed a threshold before more instances are added or
         * removed.
         * </p>
         * 
         * @param thresholdsWaitTime
         *        The amount of time, in minutes, that the load must exceed a threshold before more instances are added
         *        or removed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder thresholdsWaitTime(Integer thresholdsWaitTime);

        /**
         * <p>
         * The amount of time (in minutes) after a scaling event occurs that AWS OpsWorks Stacks should ignore metrics
         * and suppress additional scaling events. For example, AWS OpsWorks Stacks adds new instances following an
         * upscaling event but the instances won't start reducing the load until they have been booted and configured.
         * There is no point in raising additional scaling events during that operation, which typically takes several
         * minutes. <code>IgnoreMetricsTime</code> allows you to direct AWS OpsWorks Stacks to suppress scaling events
         * long enough to get the new instances online.
         * </p>
         * 
         * @param ignoreMetricsTime
         *        The amount of time (in minutes) after a scaling event occurs that AWS OpsWorks Stacks should ignore
         *        metrics and suppress additional scaling events. For example, AWS OpsWorks Stacks adds new instances
         *        following an upscaling event but the instances won't start reducing the load until they have been
         *        booted and configured. There is no point in raising additional scaling events during that operation,
         *        which typically takes several minutes. <code>IgnoreMetricsTime</code> allows you to direct AWS
         *        OpsWorks Stacks to suppress scaling events long enough to get the new instances online.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ignoreMetricsTime(Integer ignoreMetricsTime);

        /**
         * <p>
         * The CPU utilization threshold, as a percent of the available CPU. A value of -1 disables the threshold.
         * </p>
         * 
         * @param cpuThreshold
         *        The CPU utilization threshold, as a percent of the available CPU. A value of -1 disables the
         *        threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuThreshold(Double cpuThreshold);

        /**
         * <p>
         * The memory utilization threshold, as a percent of the available memory. A value of -1 disables the threshold.
         * </p>
         * 
         * @param memoryThreshold
         *        The memory utilization threshold, as a percent of the available memory. A value of -1 disables the
         *        threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryThreshold(Double memoryThreshold);

        /**
         * <p>
         * The load threshold. A value of -1 disables the threshold. For more information about how load is computed,
         * see <a href="http://en.wikipedia.org/wiki/Load_%28computing%29">Load (computing)</a>.
         * </p>
         * 
         * @param loadThreshold
         *        The load threshold. A value of -1 disables the threshold. For more information about how load is
         *        computed, see <a href="http://en.wikipedia.org/wiki/Load_%28computing%29">Load (computing)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadThreshold(Double loadThreshold);

        /**
         * <p>
         * Custom Cloudwatch auto scaling alarms, to be used as thresholds. This parameter takes a list of up to five
         * alarm names, which are case sensitive and must be in the same region as the stack.
         * </p>
         * <note>
         * <p>
         * To use custom alarms, you must update your service role to allow <code>cloudwatch:DescribeAlarms</code>. You
         * can either have AWS OpsWorks Stacks update the role for you when you first use this feature or you can edit
         * the role manually. For more information, see <a
         * href="http://docs.aws.amazon.com/opsworks/latest/userguide/opsworks-security-servicerole.html">Allowing AWS
         * OpsWorks Stacks to Act on Your Behalf</a>.
         * </p>
         * </note>
         * 
         * @param alarms
         *        Custom Cloudwatch auto scaling alarms, to be used as thresholds. This parameter takes a list of up to
         *        five alarm names, which are case sensitive and must be in the same region as the stack.</p> <note>
         *        <p>
         *        To use custom alarms, you must update your service role to allow
         *        <code>cloudwatch:DescribeAlarms</code>. You can either have AWS OpsWorks Stacks update the role for
         *        you when you first use this feature or you can edit the role manually. For more information, see <a
         *        href
         *        ="http://docs.aws.amazon.com/opsworks/latest/userguide/opsworks-security-servicerole.html">Allowing
         *        AWS OpsWorks Stacks to Act on Your Behalf</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarms(Collection<String> alarms);

        /**
         * <p>
         * Custom Cloudwatch auto scaling alarms, to be used as thresholds. This parameter takes a list of up to five
         * alarm names, which are case sensitive and must be in the same region as the stack.
         * </p>
         * <note>
         * <p>
         * To use custom alarms, you must update your service role to allow <code>cloudwatch:DescribeAlarms</code>. You
         * can either have AWS OpsWorks Stacks update the role for you when you first use this feature or you can edit
         * the role manually. For more information, see <a
         * href="http://docs.aws.amazon.com/opsworks/latest/userguide/opsworks-security-servicerole.html">Allowing AWS
         * OpsWorks Stacks to Act on Your Behalf</a>.
         * </p>
         * </note>
         * 
         * @param alarms
         *        Custom Cloudwatch auto scaling alarms, to be used as thresholds. This parameter takes a list of up to
         *        five alarm names, which are case sensitive and must be in the same region as the stack.</p> <note>
         *        <p>
         *        To use custom alarms, you must update your service role to allow
         *        <code>cloudwatch:DescribeAlarms</code>. You can either have AWS OpsWorks Stacks update the role for
         *        you when you first use this feature or you can edit the role manually. For more information, see <a
         *        href
         *        ="http://docs.aws.amazon.com/opsworks/latest/userguide/opsworks-security-servicerole.html">Allowing
         *        AWS OpsWorks Stacks to Act on Your Behalf</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarms(String... alarms);
    }

    static final class BuilderImpl implements Builder {
        private Integer instanceCount;

        private Integer thresholdsWaitTime;

        private Integer ignoreMetricsTime;

        private Double cpuThreshold;

        private Double memoryThreshold;

        private Double loadThreshold;

        private List<String> alarms;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingThresholds model) {
            instanceCount(model.instanceCount);
            thresholdsWaitTime(model.thresholdsWaitTime);
            ignoreMetricsTime(model.ignoreMetricsTime);
            cpuThreshold(model.cpuThreshold);
            memoryThreshold(model.memoryThreshold);
            loadThreshold(model.loadThreshold);
            alarms(model.alarms);
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        public final Integer getThresholdsWaitTime() {
            return thresholdsWaitTime;
        }

        @Override
        public final Builder thresholdsWaitTime(Integer thresholdsWaitTime) {
            this.thresholdsWaitTime = thresholdsWaitTime;
            return this;
        }

        public final void setThresholdsWaitTime(Integer thresholdsWaitTime) {
            this.thresholdsWaitTime = thresholdsWaitTime;
        }

        public final Integer getIgnoreMetricsTime() {
            return ignoreMetricsTime;
        }

        @Override
        public final Builder ignoreMetricsTime(Integer ignoreMetricsTime) {
            this.ignoreMetricsTime = ignoreMetricsTime;
            return this;
        }

        public final void setIgnoreMetricsTime(Integer ignoreMetricsTime) {
            this.ignoreMetricsTime = ignoreMetricsTime;
        }

        public final Double getCpuThreshold() {
            return cpuThreshold;
        }

        @Override
        public final Builder cpuThreshold(Double cpuThreshold) {
            this.cpuThreshold = cpuThreshold;
            return this;
        }

        public final void setCpuThreshold(Double cpuThreshold) {
            this.cpuThreshold = cpuThreshold;
        }

        public final Double getMemoryThreshold() {
            return memoryThreshold;
        }

        @Override
        public final Builder memoryThreshold(Double memoryThreshold) {
            this.memoryThreshold = memoryThreshold;
            return this;
        }

        public final void setMemoryThreshold(Double memoryThreshold) {
            this.memoryThreshold = memoryThreshold;
        }

        public final Double getLoadThreshold() {
            return loadThreshold;
        }

        @Override
        public final Builder loadThreshold(Double loadThreshold) {
            this.loadThreshold = loadThreshold;
            return this;
        }

        public final void setLoadThreshold(Double loadThreshold) {
            this.loadThreshold = loadThreshold;
        }

        public final Collection<String> getAlarms() {
            return alarms;
        }

        @Override
        public final Builder alarms(Collection<String> alarms) {
            this.alarms = StringsCopier.copy(alarms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alarms(String... alarms) {
            alarms(Arrays.asList(alarms));
            return this;
        }

        public final void setAlarms(Collection<String> alarms) {
            this.alarms = StringsCopier.copy(alarms);
        }

        @Override
        public AutoScalingThresholds build() {
            return new AutoScalingThresholds(this);
        }
    }
}
