/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opsworks.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.services.opsworks.OpsWorksAsyncClient;
import software.amazon.awssdk.services.opsworks.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.opsworks.model.DescribeAppsRequest;
import software.amazon.awssdk.services.opsworks.model.DescribeAppsResponse;
import software.amazon.awssdk.services.opsworks.model.DescribeDeploymentsRequest;
import software.amazon.awssdk.services.opsworks.model.DescribeDeploymentsResponse;
import software.amazon.awssdk.services.opsworks.model.DescribeInstancesRequest;
import software.amazon.awssdk.services.opsworks.model.DescribeInstancesResponse;
import software.amazon.awssdk.services.opsworks.model.OpsWorksRequest;
import software.amazon.awssdk.services.opsworks.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultOpsWorksAsyncWaiter implements OpsWorksAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final OpsWorksAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeAppsResponse> appExistsWaiter;

    private final AsyncWaiter<DescribeDeploymentsResponse> deploymentSuccessfulWaiter;

    private final AsyncWaiter<DescribeInstancesResponse> instanceOnlineWaiter;

    private final AsyncWaiter<DescribeInstancesResponse> instanceRegisteredWaiter;

    private final AsyncWaiter<DescribeInstancesResponse> instanceStoppedWaiter;

    private final AsyncWaiter<DescribeInstancesResponse> instanceTerminatedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultOpsWorksAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = OpsWorksAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.appExistsWaiter = AsyncWaiter.builder(DescribeAppsResponse.class).acceptors(appExistsWaiterAcceptors())
                .overrideConfiguration(appExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.deploymentSuccessfulWaiter = AsyncWaiter.builder(DescribeDeploymentsResponse.class)
                .acceptors(deploymentSuccessfulWaiterAcceptors())
                .overrideConfiguration(deploymentSuccessfulWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceOnlineWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceOnlineWaiterAcceptors())
                .overrideConfiguration(instanceOnlineWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceRegisteredWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceRegisteredWaiterAcceptors())
                .overrideConfiguration(instanceRegisteredWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceStoppedWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceStoppedWaiterAcceptors())
                .overrideConfiguration(instanceStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceTerminatedWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceTerminatedWaiterAcceptors())
                .overrideConfiguration(instanceTerminatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAppsResponse>> waitUntilAppExists(DescribeAppsRequest describeAppsRequest) {
        return appExistsWaiter.runAsync(() -> client.describeApps(applyWaitersUserAgent(describeAppsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAppsResponse>> waitUntilAppExists(DescribeAppsRequest describeAppsRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return appExistsWaiter.runAsync(() -> client.describeApps(applyWaitersUserAgent(describeAppsRequest)),
                appExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDeploymentsResponse>> waitUntilDeploymentSuccessful(
            DescribeDeploymentsRequest describeDeploymentsRequest) {
        return deploymentSuccessfulWaiter.runAsync(() -> client
                .describeDeployments(applyWaitersUserAgent(describeDeploymentsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeDeploymentsResponse>> waitUntilDeploymentSuccessful(
            DescribeDeploymentsRequest describeDeploymentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return deploymentSuccessfulWaiter.runAsync(
                () -> client.describeDeployments(applyWaitersUserAgent(describeDeploymentsRequest)),
                deploymentSuccessfulWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceOnline(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceOnlineWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceOnline(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceOnlineWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceOnlineWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceRegistered(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceRegisteredWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceRegistered(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceRegisteredWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceStopped(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceStoppedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceStopped(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceStoppedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceTerminated(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceTerminatedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceTerminated(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceTerminatedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceTerminatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAppsResponse>> appExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAppsResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(400, WaiterState.FAILURE));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDeploymentsResponse>> deploymentSuccessfulWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDeploymentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Deployments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "successful"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Deployments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceOnlineWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "online"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "shutting_down"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "start_failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopped"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminating"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminated"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stop_failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "registered"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "shutting_down"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopped"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminating"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminated"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stop_failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "stopped"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "booting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "pending"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "requested"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "running_setup"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "start_failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stop_failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceTerminatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "terminated"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "booting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "online"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "pending"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "requested"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "running_setup"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "setup_failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Instances").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "start_failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration appExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration deploymentSuccessfulWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceOnlineWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceRegisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceTerminatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static OpsWorksAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends OpsWorksRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements OpsWorksAsyncWaiter.Builder {
        private OpsWorksAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public OpsWorksAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public OpsWorksAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public OpsWorksAsyncWaiter.Builder client(OpsWorksAsyncClient client) {
            this.client = client;
            return this;
        }

        public OpsWorksAsyncWaiter build() {
            return new DefaultOpsWorksAsyncWaiter(this);
        }
    }
}
