/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.opsworkscm.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.opsworkscm.OpsWorksCmAsyncClient;
import software.amazon.awssdk.services.opsworkscm.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.opsworkscm.model.DescribeNodeAssociationStatusRequest;
import software.amazon.awssdk.services.opsworkscm.model.DescribeNodeAssociationStatusResponse;
import software.amazon.awssdk.services.opsworkscm.model.OpsWorksCmRequest;
import software.amazon.awssdk.services.opsworkscm.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultOpsWorksCmAsyncWaiter implements OpsWorksCmAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final OpsWorksCmAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeNodeAssociationStatusResponse> nodeAssociatedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultOpsWorksCmAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = OpsWorksCmAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.nodeAssociatedWaiter = AsyncWaiter.builder(DescribeNodeAssociationStatusResponse.class)
                .acceptors(nodeAssociatedWaiterAcceptors())
                .overrideConfiguration(nodeAssociatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodeAssociationStatusResponse>> waitUntilNodeAssociated(
            DescribeNodeAssociationStatusRequest describeNodeAssociationStatusRequest) {
        return nodeAssociatedWaiter.runAsync(() -> client
                .describeNodeAssociationStatus(applyWaitersUserAgent(describeNodeAssociationStatusRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNodeAssociationStatusResponse>> waitUntilNodeAssociated(
            DescribeNodeAssociationStatusRequest describeNodeAssociationStatusRequest, WaiterOverrideConfiguration overrideConfig) {
        return nodeAssociatedWaiter.runAsync(
                () -> client.describeNodeAssociationStatus(applyWaitersUserAgent(describeNodeAssociationStatusRequest)),
                nodeAssociatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeNodeAssociationStatusResponse>> nodeAssociatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNodeAssociationStatusResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("NodeAssociationStatus").value(), "SUCCESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("NodeAssociationStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (NodeAssociationStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration nodeAssociatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(15);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static OpsWorksCmAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends OpsWorksCmRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements OpsWorksCmAsyncWaiter.Builder {
        private OpsWorksCmAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public OpsWorksCmAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public OpsWorksCmAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public OpsWorksCmAsyncWaiter.Builder client(OpsWorksCmAsyncClient client) {
            this.client = client;
            return this;
        }

        public OpsWorksCmAsyncWaiter build() {
            return new DefaultOpsWorksCmAsyncWaiter(this);
        }
    }
}
