/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a root, OU, or account that a policy is attached to.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PolicyTargetSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<PolicyTargetSummary.Builder, PolicyTargetSummary> {
    private static final SdkField<String> TARGET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetId").getter(getter(PolicyTargetSummary::targetId)).setter(setter(Builder::targetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(PolicyTargetSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(PolicyTargetSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(PolicyTargetSummary::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_ID_FIELD, ARN_FIELD,
            NAME_FIELD, TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String targetId;

    private final String arn;

    private final String name;

    private final String type;

    private PolicyTargetSummary(BuilderImpl builder) {
        this.targetId = builder.targetId;
        this.arn = builder.arn;
        this.name = builder.name;
        this.type = builder.type;
    }

    /**
     * <p>
     * The unique identifier (ID) of the policy target.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a target ID string requires one of the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Root</b> - A string that begins with "r-" followed by from 4 to 32 lowercase letters or digits.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Account</b> - A string that consists of exactly 12 digits.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Organizational unit (OU)</b> - A string that begins with "ou-" followed by from 4 to 32 lowercase letters or
     * digits (the ID of the root that the OU is in). This string is followed by a second "-" dash and from 8 to 32
     * additional lowercase letters or digits.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The unique identifier (ID) of the policy target.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a target ID string requires one of
     *         the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Root</b> - A string that begins with "r-" followed by from 4 to 32 lowercase letters or digits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Account</b> - A string that consists of exactly 12 digits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Organizational unit (OU)</b> - A string that begins with "ou-" followed by from 4 to 32 lowercase
     *         letters or digits (the ID of the root that the OU is in). This string is followed by a second "-" dash
     *         and from 8 to 32 additional lowercase letters or digits.
     *         </p>
     *         </li>
     */
    public final String targetId() {
        return targetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the policy target.
     * </p>
     * <p>
     * For more information about ARNs in Organizations, see <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
     * >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the policy target.</p>
     *         <p>
     *         For more information about ARNs in Organizations, see <a href=
     *         "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
     *         >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The friendly name of the policy target.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter is a
     * string of any of the characters in the ASCII character range.
     * </p>
     * 
     * @return The friendly name of the policy target.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter
     *         is a string of any of the characters in the ASCII character range.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of the policy target.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TargetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the policy target.
     * @see TargetType
     */
    public final TargetType type() {
        return TargetType.fromValue(type);
    }

    /**
     * <p>
     * The type of the policy target.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TargetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the policy target.
     * @see TargetType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(targetId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PolicyTargetSummary)) {
            return false;
        }
        PolicyTargetSummary other = (PolicyTargetSummary) obj;
        return Objects.equals(targetId(), other.targetId()) && Objects.equals(arn(), other.arn())
                && Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PolicyTargetSummary").add("TargetId", targetId()).add("Arn", arn()).add("Name", name())
                .add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TargetId":
            return Optional.ofNullable(clazz.cast(targetId()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PolicyTargetSummary, T> g) {
        return obj -> g.apply((PolicyTargetSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PolicyTargetSummary> {
        /**
         * <p>
         * The unique identifier (ID) of the policy target.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a target ID string requires one of the
         * following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Root</b> - A string that begins with "r-" followed by from 4 to 32 lowercase letters or digits.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Account</b> - A string that consists of exactly 12 digits.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Organizational unit (OU)</b> - A string that begins with "ou-" followed by from 4 to 32 lowercase letters
         * or digits (the ID of the root that the OU is in). This string is followed by a second "-" dash and from 8 to
         * 32 additional lowercase letters or digits.
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetId
         *        The unique identifier (ID) of the policy target.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a target ID string requires one of
         *        the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Root</b> - A string that begins with "r-" followed by from 4 to 32 lowercase letters or digits.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Account</b> - A string that consists of exactly 12 digits.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Organizational unit (OU)</b> - A string that begins with "ou-" followed by from 4 to 32 lowercase
         *        letters or digits (the ID of the root that the OU is in). This string is followed by a second "-" dash
         *        and from 8 to 32 additional lowercase letters or digits.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetId(String targetId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the policy target.
         * </p>
         * <p>
         * For more information about ARNs in Organizations, see <a href=
         * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
         * >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the policy target.</p>
         *        <p>
         *        For more information about ARNs in Organizations, see <a href=
         *        "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
         *        >ARN Formats Supported by Organizations</a> in the <i>AWS Service Authorization Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The friendly name of the policy target.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this parameter is a
         * string of any of the characters in the ASCII character range.
         * </p>
         * 
         * @param name
         *        The friendly name of the policy target.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> that is used to validate this
         *        parameter is a string of any of the characters in the ASCII character range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the policy target.
         * </p>
         * 
         * @param type
         *        The type of the policy target.
         * @see TargetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the policy target.
         * </p>
         * 
         * @param type
         *        The type of the policy target.
         * @see TargetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetType
         */
        Builder type(TargetType type);
    }

    static final class BuilderImpl implements Builder {
        private String targetId;

        private String arn;

        private String name;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(PolicyTargetSummary model) {
            targetId(model.targetId);
            arn(model.arn);
            name(model.name);
            type(model.type);
        }

        public final String getTargetId() {
            return targetId;
        }

        public final void setTargetId(String targetId) {
            this.targetId = targetId;
        }

        @Override
        @Transient
        public final Builder targetId(String targetId) {
            this.targetId = targetId;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(TargetType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public PolicyTargetSummary build() {
            return new PolicyTargetSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
