/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about the validation errors that occurred when generating or enforcing an <a
 * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_effective.html">effective
 * policy</a>, such as which policies contributed to the error and location of the error.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EffectivePolicyValidationError implements SdkPojo, Serializable,
        ToCopyableBuilder<EffectivePolicyValidationError.Builder, EffectivePolicyValidationError> {
    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorCode").getter(getter(EffectivePolicyValidationError::errorCode)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorCode").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(EffectivePolicyValidationError::errorMessage))
            .setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<String> PATH_TO_ERROR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PathToError").getter(getter(EffectivePolicyValidationError::pathToError))
            .setter(setter(Builder::pathToError))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PathToError").build()).build();

    private static final SdkField<List<String>> CONTRIBUTING_POLICIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ContributingPolicies")
            .getter(getter(EffectivePolicyValidationError::contributingPolicies))
            .setter(setter(Builder::contributingPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContributingPolicies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ERROR_CODE_FIELD,
            ERROR_MESSAGE_FIELD, PATH_TO_ERROR_FIELD, CONTRIBUTING_POLICIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String errorCode;

    private final String errorMessage;

    private final String pathToError;

    private final List<String> contributingPolicies;

    private EffectivePolicyValidationError(BuilderImpl builder) {
        this.errorCode = builder.errorCode;
        this.errorMessage = builder.errorMessage;
        this.pathToError = builder.pathToError;
        this.contributingPolicies = builder.contributingPolicies;
    }

    /**
     * <p>
     * The error code for the validation error. For example, <code>ELEMENTS_TOO_MANY</code>.
     * </p>
     * 
     * @return The error code for the validation error. For example, <code>ELEMENTS_TOO_MANY</code>.
     */
    public final String errorCode() {
        return errorCode;
    }

    /**
     * <p>
     * The error message for the validation error.
     * </p>
     * 
     * @return The error message for the validation error.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * The path within the effective policy where the validation error occurred.
     * </p>
     * 
     * @return The path within the effective policy where the validation error occurred.
     */
    public final String pathToError() {
        return pathToError;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContributingPolicies property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContributingPolicies() {
        return contributingPolicies != null && !(contributingPolicies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The individual policies <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_inheritance_mgmt.html"
     * >inherited</a> and <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_policies_attach.html">attached</a> to the
     * account which contributed to the validation error.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContributingPolicies} method.
     * </p>
     * 
     * @return The individual policies <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_inheritance_mgmt.html"
     *         >inherited</a> and <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_policies_attach.html">attached</a>
     *         to the account which contributed to the validation error.
     */
    public final List<String> contributingPolicies() {
        return contributingPolicies;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(errorCode());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(pathToError());
        hashCode = 31 * hashCode + Objects.hashCode(hasContributingPolicies() ? contributingPolicies() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EffectivePolicyValidationError)) {
            return false;
        }
        EffectivePolicyValidationError other = (EffectivePolicyValidationError) obj;
        return Objects.equals(errorCode(), other.errorCode()) && Objects.equals(errorMessage(), other.errorMessage())
                && Objects.equals(pathToError(), other.pathToError())
                && hasContributingPolicies() == other.hasContributingPolicies()
                && Objects.equals(contributingPolicies(), other.contributingPolicies());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EffectivePolicyValidationError").add("ErrorCode", errorCode())
                .add("ErrorMessage", errorMessage()).add("PathToError", pathToError())
                .add("ContributingPolicies", hasContributingPolicies() ? contributingPolicies() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCode()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "PathToError":
            return Optional.ofNullable(clazz.cast(pathToError()));
        case "ContributingPolicies":
            return Optional.ofNullable(clazz.cast(contributingPolicies()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ErrorCode", ERROR_CODE_FIELD);
        map.put("ErrorMessage", ERROR_MESSAGE_FIELD);
        map.put("PathToError", PATH_TO_ERROR_FIELD);
        map.put("ContributingPolicies", CONTRIBUTING_POLICIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EffectivePolicyValidationError, T> g) {
        return obj -> g.apply((EffectivePolicyValidationError) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EffectivePolicyValidationError> {
        /**
         * <p>
         * The error code for the validation error. For example, <code>ELEMENTS_TOO_MANY</code>.
         * </p>
         * 
         * @param errorCode
         *        The error code for the validation error. For example, <code>ELEMENTS_TOO_MANY</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * The error message for the validation error.
         * </p>
         * 
         * @param errorMessage
         *        The error message for the validation error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * The path within the effective policy where the validation error occurred.
         * </p>
         * 
         * @param pathToError
         *        The path within the effective policy where the validation error occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pathToError(String pathToError);

        /**
         * <p>
         * The individual policies <a href=
         * "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_inheritance_mgmt.html"
         * >inherited</a> and <a
         * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_policies_attach.html">attached</a> to
         * the account which contributed to the validation error.
         * </p>
         * 
         * @param contributingPolicies
         *        The individual policies <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_inheritance_mgmt.html"
         *        >inherited</a> and <a
         *        href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_policies_attach.html"
         *        >attached</a> to the account which contributed to the validation error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contributingPolicies(Collection<String> contributingPolicies);

        /**
         * <p>
         * The individual policies <a href=
         * "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_inheritance_mgmt.html"
         * >inherited</a> and <a
         * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_policies_attach.html">attached</a> to
         * the account which contributed to the validation error.
         * </p>
         * 
         * @param contributingPolicies
         *        The individual policies <a href=
         *        "https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_inheritance_mgmt.html"
         *        >inherited</a> and <a
         *        href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_policies_attach.html"
         *        >attached</a> to the account which contributed to the validation error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contributingPolicies(String... contributingPolicies);
    }

    static final class BuilderImpl implements Builder {
        private String errorCode;

        private String errorMessage;

        private String pathToError;

        private List<String> contributingPolicies = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EffectivePolicyValidationError model) {
            errorCode(model.errorCode);
            errorMessage(model.errorMessage);
            pathToError(model.pathToError);
            contributingPolicies(model.contributingPolicies);
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final String getPathToError() {
            return pathToError;
        }

        public final void setPathToError(String pathToError) {
            this.pathToError = pathToError;
        }

        @Override
        public final Builder pathToError(String pathToError) {
            this.pathToError = pathToError;
            return this;
        }

        public final Collection<String> getContributingPolicies() {
            if (contributingPolicies instanceof SdkAutoConstructList) {
                return null;
            }
            return contributingPolicies;
        }

        public final void setContributingPolicies(Collection<String> contributingPolicies) {
            this.contributingPolicies = PolicyIdsCopier.copy(contributingPolicies);
        }

        @Override
        public final Builder contributingPolicies(Collection<String> contributingPolicies) {
            this.contributingPolicies = PolicyIdsCopier.copy(contributingPolicies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder contributingPolicies(String... contributingPolicies) {
            contributingPolicies(Arrays.asList(contributingPolicies));
            return this;
        }

        @Override
        public EffectivePolicyValidationError build() {
            return new EffectivePolicyValidationError(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
