/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.organizations.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information that must be exchanged to securely establish a relationship between two accounts (an
 * <i>originator</i> and a <i>recipient</i>). For example, when a management account (the originator) invites another
 * account (the recipient) to join its organization, the two accounts exchange information as a series of handshake
 * requests and responses.
 * </p>
 * <p>
 * <b>Note:</b> Handshakes that are <code>CANCELED</code>, <code>ACCEPTED</code>, <code>DECLINED</code>, or
 * <code>EXPIRED</code> show up in lists for only 30 days after entering that state After that they are deleted.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Handshake implements SdkPojo, Serializable, ToCopyableBuilder<Handshake.Builder, Handshake> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Handshake::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Handshake::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<List<HandshakeParty>> PARTIES_FIELD = SdkField
            .<List<HandshakeParty>> builder(MarshallingType.LIST)
            .memberName("Parties")
            .getter(getter(Handshake::parties))
            .setter(setter(Builder::parties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HandshakeParty> builder(MarshallingType.SDK_POJO)
                                            .constructor(HandshakeParty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(Handshake::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<Instant> REQUESTED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RequestedTimestamp").getter(getter(Handshake::requestedTimestamp))
            .setter(setter(Builder::requestedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedTimestamp").build())
            .build();

    private static final SdkField<Instant> EXPIRATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExpirationTimestamp").getter(getter(Handshake::expirationTimestamp))
            .setter(setter(Builder::expirationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpirationTimestamp").build())
            .build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(Handshake::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<List<HandshakeResource>> RESOURCES_FIELD = SdkField
            .<List<HandshakeResource>> builder(MarshallingType.LIST)
            .memberName("Resources")
            .getter(getter(Handshake::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HandshakeResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(HandshakeResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            PARTIES_FIELD, STATE_FIELD, REQUESTED_TIMESTAMP_FIELD, EXPIRATION_TIMESTAMP_FIELD, ACTION_FIELD, RESOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final List<HandshakeParty> parties;

    private final String state;

    private final Instant requestedTimestamp;

    private final Instant expirationTimestamp;

    private final String action;

    private final List<HandshakeResource> resources;

    private Handshake(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.parties = builder.parties;
        this.state = builder.state;
        this.requestedTimestamp = builder.requestedTimestamp;
        this.expirationTimestamp = builder.expirationTimestamp;
        this.action = builder.action;
        this.resources = builder.resources;
    }

    /**
     * <p>
     * The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates the
     * handshake.
     * </p>
     * <p>
     * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-" followed by
     * from 8 to 32 lowercase letters or digits.
     * </p>
     * 
     * @return The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates the
     *         handshake.</p>
     *         <p>
     *         The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-"
     *         followed by from 8 to 32 lowercase letters or digits.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of a handshake.
     * </p>
     * <p>
     * For more information about ARNs in Organizations, see <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
     * >ARN Formats Supported by Organizations</a> in the <i>Amazon Web Services Service Authorization Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of a handshake.</p>
     *         <p>
     *         For more information about ARNs in Organizations, see <a href=
     *         "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
     *         >ARN Formats Supported by Organizations</a> in the <i>Amazon Web Services Service Authorization
     *         Reference</i>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parties property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParties() {
        return parties != null && !(parties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the two accounts that are participating in the handshake.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParties} method.
     * </p>
     * 
     * @return Information about the two accounts that are participating in the handshake.
     */
    public final List<HandshakeParty> parties() {
        return parties;
    }

    /**
     * <p>
     * The current state of the handshake. Use the state to trace the flow of the handshake through the process from its
     * creation to its acceptance. The meaning of each of the valid values is as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types) and
     * not all recipients have responded yet. The request stays in this state until all recipients respond.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
     * recipients have responded, allowing the originator to complete the handshake action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any kind
     * from the recipient before the expiration time (15 days).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link HandshakeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the handshake. Use the state to trace the flow of the handshake through the process
     *         from its creation to its acceptance. The meaning of each of the valid values is as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
     *         types) and not all recipients have responded yet. The request stays in this state until all recipients
     *         respond.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
     *         all recipients have responded, allowing the originator to complete the handshake action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of
     *         any kind from the recipient before the expiration time (15 days).
     *         </p>
     *         </li>
     * @see HandshakeState
     */
    public final HandshakeState state() {
        return HandshakeState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the handshake. Use the state to trace the flow of the handshake through the process from its
     * creation to its acceptance. The meaning of each of the valid values is as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types) and
     * not all recipients have responded yet. The request stays in this state until all recipients respond.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
     * recipients have responded, allowing the originator to complete the handshake action.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any kind
     * from the recipient before the expiration time (15 days).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link HandshakeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the handshake. Use the state to trace the flow of the handshake through the process
     *         from its creation to its acceptance. The meaning of each of the valid values is as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
     *         types) and not all recipients have responded yet. The request stays in this state until all recipients
     *         respond.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
     *         all recipients have responded, allowing the originator to complete the handshake action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of
     *         any kind from the recipient before the expiration time (15 days).
     *         </p>
     *         </li>
     * @see HandshakeState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The date and time that the handshake request was made.
     * </p>
     * 
     * @return The date and time that the handshake request was made.
     */
    public final Instant requestedTimestamp() {
        return requestedTimestamp;
    }

    /**
     * <p>
     * The date and time that the handshake expires. If the recipient of the handshake request fails to respond before
     * the specified date and time, the handshake becomes inactive and is no longer valid.
     * </p>
     * 
     * @return The date and time that the handshake expires. If the recipient of the handshake request fails to respond
     *         before the specified date and time, the handshake becomes inactive and is no longer valid.
     */
    public final Instant expirationTimestamp() {
        return expirationTimestamp;
    }

    /**
     * <p>
     * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
     * handshake types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from the
     * management account to only non-member accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     * organization. It is always sent from the management account to only <i>invited</i> member accounts. Created
     * accounts do not receive this because those accounts were created by the organization's management account and
     * approval is inferred.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
     * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the management account
     * and signals the master that it can finalize the process to enable all features.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link ActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
     *         following handshake types are supported:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
     *         from the management account to only non-member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     *         organization. It is always sent from the management account to only <i>invited</i> member accounts.
     *         Created accounts do not receive this because those accounts were created by the organization's management
     *         account and approval is inferred.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
     *         member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
     *         management account and signals the master that it can finalize the process to enable all features.
     *         </p>
     *         </li>
     * @see ActionType
     */
    public final ActionType action() {
        return ActionType.fromValue(action);
    }

    /**
     * <p>
     * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
     * handshake types are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from the
     * management account to only non-member accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     * organization. It is always sent from the management account to only <i>invited</i> member accounts. Created
     * accounts do not receive this because those accounts were created by the organization's management account and
     * approval is inferred.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
     * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the management account
     * and signals the master that it can finalize the process to enable all features.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link ActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
     *         following handshake types are supported:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
     *         from the management account to only non-member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
     *         organization. It is always sent from the management account to only <i>invited</i> member accounts.
     *         Created accounts do not receive this because those accounts were created by the organization's management
     *         account and approval is inferred.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
     *         member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
     *         management account and signals the master that it can finalize the process to enable all features.
     *         </p>
     *         </li>
     * @see ActionType
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * For responses, this returns true if the service returned a value for the Resources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Additional information that is needed to process the handshake.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResources} method.
     * </p>
     * 
     * @return Additional information that is needed to process the handshake.
     */
    public final List<HandshakeResource> resources() {
        return resources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(hasParties() ? parties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(requestedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(expirationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Handshake)) {
            return false;
        }
        Handshake other = (Handshake) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn()) && hasParties() == other.hasParties()
                && Objects.equals(parties(), other.parties()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(requestedTimestamp(), other.requestedTimestamp())
                && Objects.equals(expirationTimestamp(), other.expirationTimestamp())
                && Objects.equals(actionAsString(), other.actionAsString()) && hasResources() == other.hasResources()
                && Objects.equals(resources(), other.resources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Handshake").add("Id", id()).add("Arn", arn()).add("Parties", hasParties() ? parties() : null)
                .add("State", stateAsString()).add("RequestedTimestamp", requestedTimestamp())
                .add("ExpirationTimestamp", expirationTimestamp()).add("Action", actionAsString())
                .add("Resources", hasResources() ? resources() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Parties":
            return Optional.ofNullable(clazz.cast(parties()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "RequestedTimestamp":
            return Optional.ofNullable(clazz.cast(requestedTimestamp()));
        case "ExpirationTimestamp":
            return Optional.ofNullable(clazz.cast(expirationTimestamp()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("Arn", ARN_FIELD);
        map.put("Parties", PARTIES_FIELD);
        map.put("State", STATE_FIELD);
        map.put("RequestedTimestamp", REQUESTED_TIMESTAMP_FIELD);
        map.put("ExpirationTimestamp", EXPIRATION_TIMESTAMP_FIELD);
        map.put("Action", ACTION_FIELD);
        map.put("Resources", RESOURCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Handshake, T> g) {
        return obj -> g.apply((Handshake) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Handshake> {
        /**
         * <p>
         * The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates the
         * handshake.
         * </p>
         * <p>
         * The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-"
         * followed by from 8 to 32 lowercase letters or digits.
         * </p>
         * 
         * @param id
         *        The unique identifier (ID) of a handshake. The originating account creates the ID when it initiates
         *        the handshake.</p>
         *        <p>
         *        The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for handshake ID string requires "h-"
         *        followed by from 8 to 32 lowercase letters or digits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of a handshake.
         * </p>
         * <p>
         * For more information about ARNs in Organizations, see <a href=
         * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
         * >ARN Formats Supported by Organizations</a> in the <i>Amazon Web Services Service Authorization
         * Reference</i>.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of a handshake.</p>
         *        <p>
         *        For more information about ARNs in Organizations, see <a href=
         *        "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html#awsorganizations-resources-for-iam-policies"
         *        >ARN Formats Supported by Organizations</a> in the <i>Amazon Web Services Service Authorization
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Information about the two accounts that are participating in the handshake.
         * </p>
         * 
         * @param parties
         *        Information about the two accounts that are participating in the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parties(Collection<HandshakeParty> parties);

        /**
         * <p>
         * Information about the two accounts that are participating in the handshake.
         * </p>
         * 
         * @param parties
         *        Information about the two accounts that are participating in the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parties(HandshakeParty... parties);

        /**
         * <p>
         * Information about the two accounts that are participating in the handshake.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.organizations.model.HandshakeParty.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.organizations.model.HandshakeParty#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.organizations.model.HandshakeParty.Builder#build()} is called
         * immediately and its result is passed to {@link #parties(List<HandshakeParty>)}.
         * 
         * @param parties
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.organizations.model.HandshakeParty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parties(java.util.Collection<HandshakeParty>)
         */
        Builder parties(Consumer<HandshakeParty.Builder>... parties);

        /**
         * <p>
         * The current state of the handshake. Use the state to trace the flow of the handshake through the process from
         * its creation to its acceptance. The meaning of each of the valid values is as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types)
         * and not all recipients have responded yet. The request stays in this state until all recipients respond.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
         * recipients have responded, allowing the originator to complete the handshake action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any
         * kind from the recipient before the expiration time (15 days).
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the handshake. Use the state to trace the flow of the handshake through the
         *        process from its creation to its acceptance. The meaning of each of the valid values is as
         *        follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
         *        types) and not all recipients have responded yet. The request stays in this state until all recipients
         *        respond.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
         *        all recipients have responded, allowing the originator to complete the handshake action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating
         *        account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response
         *        of any kind from the recipient before the expiration time (15 days).
         *        </p>
         *        </li>
         * @see HandshakeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HandshakeState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the handshake. Use the state to trace the flow of the handshake through the process from
         * its creation to its acceptance. The meaning of each of the valid values is as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake types)
         * and not all recipients have responded yet. The request stays in this state until all recipients respond.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and all
         * recipients have responded, allowing the originator to complete the handshake action.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response of any
         * kind from the recipient before the expiration time (15 days).
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the handshake. Use the state to trace the flow of the handshake through the
         *        process from its creation to its acceptance. The meaning of each of the valid values is as
         *        follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>REQUESTED</b>: This handshake was sent to multiple recipients (applicable to only some handshake
         *        types) and not all recipients have responded yet. The request stays in this state until all recipients
         *        respond.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>OPEN</b>: This handshake was sent to multiple recipients (applicable to only some policy types) and
         *        all recipients have responded, allowing the originator to complete the handshake action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELED</b>: This handshake is no longer active because it was canceled by the originating
         *        account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACCEPTED</b>: This handshake is complete because it has been accepted by the recipient.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DECLINED</b>: This handshake is no longer active because it was declined by the recipient account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>EXPIRED</b>: This handshake is no longer active because the originator did not receive a response
         *        of any kind from the recipient before the expiration time (15 days).
         *        </p>
         *        </li>
         * @see HandshakeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HandshakeState
         */
        Builder state(HandshakeState state);

        /**
         * <p>
         * The date and time that the handshake request was made.
         * </p>
         * 
         * @param requestedTimestamp
         *        The date and time that the handshake request was made.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedTimestamp(Instant requestedTimestamp);

        /**
         * <p>
         * The date and time that the handshake expires. If the recipient of the handshake request fails to respond
         * before the specified date and time, the handshake becomes inactive and is no longer valid.
         * </p>
         * 
         * @param expirationTimestamp
         *        The date and time that the handshake expires. If the recipient of the handshake request fails to
         *        respond before the specified date and time, the handshake becomes inactive and is no longer valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationTimestamp(Instant expirationTimestamp);

        /**
         * <p>
         * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
         * handshake types are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from
         * the management account to only non-member accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         * organization. It is always sent from the management account to only <i>invited</i> member accounts. Created
         * accounts do not receive this because those accounts were created by the organization's management account and
         * approval is inferred.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
         * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the management
         * account and signals the master that it can finalize the process to enable all features.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
         *        following handshake types are supported:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
         *        from the management account to only non-member accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         *        organization. It is always sent from the management account to only <i>invited</i> member accounts.
         *        Created accounts do not receive this because those accounts were created by the organization's
         *        management account and approval is inferred.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
         *        member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
         *        management account and signals the master that it can finalize the process to enable all features.
         *        </p>
         *        </li>
         * @see ActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionType
         */
        Builder action(String action);

        /**
         * <p>
         * The type of handshake, indicating what action occurs when the recipient accepts the handshake. The following
         * handshake types are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent from
         * the management account to only non-member accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         * organization. It is always sent from the management account to only <i>invited</i> member accounts. Created
         * accounts do not receive this because those accounts were created by the organization's management account and
         * approval is inferred.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all member
         * accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the management
         * account and signals the master that it can finalize the process to enable all features.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The type of handshake, indicating what action occurs when the recipient accepts the handshake. The
         *        following handshake types are supported:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>INVITE</b>: This type of handshake represents a request to join an organization. It is always sent
         *        from the management account to only non-member accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ENABLE_ALL_FEATURES</b>: This type of handshake represents a request to enable all features in an
         *        organization. It is always sent from the management account to only <i>invited</i> member accounts.
         *        Created accounts do not receive this because those accounts were created by the organization's
         *        management account and approval is inferred.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>APPROVE_ALL_FEATURES</b>: This type of handshake is sent from the Organizations service when all
         *        member accounts have approved the <code>ENABLE_ALL_FEATURES</code> invitation. It is sent only to the
         *        management account and signals the master that it can finalize the process to enable all features.
         *        </p>
         *        </li>
         * @see ActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionType
         */
        Builder action(ActionType action);

        /**
         * <p>
         * Additional information that is needed to process the handshake.
         * </p>
         * 
         * @param resources
         *        Additional information that is needed to process the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<HandshakeResource> resources);

        /**
         * <p>
         * Additional information that is needed to process the handshake.
         * </p>
         * 
         * @param resources
         *        Additional information that is needed to process the handshake.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(HandshakeResource... resources);

        /**
         * <p>
         * Additional information that is needed to process the handshake.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.organizations.model.HandshakeResource.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.organizations.model.HandshakeResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.organizations.model.HandshakeResource.Builder#build()} is called
         * immediately and its result is passed to {@link #resources(List<HandshakeResource>)}.
         * 
         * @param resources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.organizations.model.HandshakeResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(java.util.Collection<HandshakeResource>)
         */
        Builder resources(Consumer<HandshakeResource.Builder>... resources);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private List<HandshakeParty> parties = DefaultSdkAutoConstructList.getInstance();

        private String state;

        private Instant requestedTimestamp;

        private Instant expirationTimestamp;

        private String action;

        private List<HandshakeResource> resources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Handshake model) {
            id(model.id);
            arn(model.arn);
            parties(model.parties);
            state(model.state);
            requestedTimestamp(model.requestedTimestamp);
            expirationTimestamp(model.expirationTimestamp);
            action(model.action);
            resources(model.resources);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final List<HandshakeParty.Builder> getParties() {
            List<HandshakeParty.Builder> result = HandshakePartiesCopier.copyToBuilder(this.parties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParties(Collection<HandshakeParty.BuilderImpl> parties) {
            this.parties = HandshakePartiesCopier.copyFromBuilder(parties);
        }

        @Override
        public final Builder parties(Collection<HandshakeParty> parties) {
            this.parties = HandshakePartiesCopier.copy(parties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parties(HandshakeParty... parties) {
            parties(Arrays.asList(parties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parties(Consumer<HandshakeParty.Builder>... parties) {
            parties(Stream.of(parties).map(c -> HandshakeParty.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(HandshakeState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Instant getRequestedTimestamp() {
            return requestedTimestamp;
        }

        public final void setRequestedTimestamp(Instant requestedTimestamp) {
            this.requestedTimestamp = requestedTimestamp;
        }

        @Override
        public final Builder requestedTimestamp(Instant requestedTimestamp) {
            this.requestedTimestamp = requestedTimestamp;
            return this;
        }

        public final Instant getExpirationTimestamp() {
            return expirationTimestamp;
        }

        public final void setExpirationTimestamp(Instant expirationTimestamp) {
            this.expirationTimestamp = expirationTimestamp;
        }

        @Override
        public final Builder expirationTimestamp(Instant expirationTimestamp) {
            this.expirationTimestamp = expirationTimestamp;
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(ActionType action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final List<HandshakeResource.Builder> getResources() {
            List<HandshakeResource.Builder> result = HandshakeResourcesCopier.copyToBuilder(this.resources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResources(Collection<HandshakeResource.BuilderImpl> resources) {
            this.resources = HandshakeResourcesCopier.copyFromBuilder(resources);
        }

        @Override
        public final Builder resources(Collection<HandshakeResource> resources) {
            this.resources = HandshakeResourcesCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(HandshakeResource... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Consumer<HandshakeResource.Builder>... resources) {
            resources(Stream.of(resources).map(c -> HandshakeResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Handshake build() {
            return new Handshake(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
