/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.panorama;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.panorama.model.AccessDeniedException;
import software.amazon.awssdk.services.panorama.model.ConflictException;
import software.amazon.awssdk.services.panorama.model.CreateApplicationInstanceRequest;
import software.amazon.awssdk.services.panorama.model.CreateApplicationInstanceResponse;
import software.amazon.awssdk.services.panorama.model.CreateJobForDevicesRequest;
import software.amazon.awssdk.services.panorama.model.CreateJobForDevicesResponse;
import software.amazon.awssdk.services.panorama.model.CreateNodeFromTemplateJobRequest;
import software.amazon.awssdk.services.panorama.model.CreateNodeFromTemplateJobResponse;
import software.amazon.awssdk.services.panorama.model.CreatePackageImportJobRequest;
import software.amazon.awssdk.services.panorama.model.CreatePackageImportJobResponse;
import software.amazon.awssdk.services.panorama.model.CreatePackageRequest;
import software.amazon.awssdk.services.panorama.model.CreatePackageResponse;
import software.amazon.awssdk.services.panorama.model.DeleteDeviceRequest;
import software.amazon.awssdk.services.panorama.model.DeleteDeviceResponse;
import software.amazon.awssdk.services.panorama.model.DeletePackageRequest;
import software.amazon.awssdk.services.panorama.model.DeletePackageResponse;
import software.amazon.awssdk.services.panorama.model.DeregisterPackageVersionRequest;
import software.amazon.awssdk.services.panorama.model.DeregisterPackageVersionResponse;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceDetailsRequest;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceDetailsResponse;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceRequest;
import software.amazon.awssdk.services.panorama.model.DescribeApplicationInstanceResponse;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceJobRequest;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceJobResponse;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceRequest;
import software.amazon.awssdk.services.panorama.model.DescribeDeviceResponse;
import software.amazon.awssdk.services.panorama.model.DescribeNodeFromTemplateJobRequest;
import software.amazon.awssdk.services.panorama.model.DescribeNodeFromTemplateJobResponse;
import software.amazon.awssdk.services.panorama.model.DescribeNodeRequest;
import software.amazon.awssdk.services.panorama.model.DescribeNodeResponse;
import software.amazon.awssdk.services.panorama.model.DescribePackageImportJobRequest;
import software.amazon.awssdk.services.panorama.model.DescribePackageImportJobResponse;
import software.amazon.awssdk.services.panorama.model.DescribePackageRequest;
import software.amazon.awssdk.services.panorama.model.DescribePackageResponse;
import software.amazon.awssdk.services.panorama.model.DescribePackageVersionRequest;
import software.amazon.awssdk.services.panorama.model.DescribePackageVersionResponse;
import software.amazon.awssdk.services.panorama.model.InternalServerException;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesRequest;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesResponse;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesRequest;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesResponse;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstancesRequest;
import software.amazon.awssdk.services.panorama.model.ListApplicationInstancesResponse;
import software.amazon.awssdk.services.panorama.model.ListDevicesJobsRequest;
import software.amazon.awssdk.services.panorama.model.ListDevicesJobsResponse;
import software.amazon.awssdk.services.panorama.model.ListDevicesRequest;
import software.amazon.awssdk.services.panorama.model.ListDevicesResponse;
import software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsRequest;
import software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsResponse;
import software.amazon.awssdk.services.panorama.model.ListNodesRequest;
import software.amazon.awssdk.services.panorama.model.ListNodesResponse;
import software.amazon.awssdk.services.panorama.model.ListPackageImportJobsRequest;
import software.amazon.awssdk.services.panorama.model.ListPackageImportJobsResponse;
import software.amazon.awssdk.services.panorama.model.ListPackagesRequest;
import software.amazon.awssdk.services.panorama.model.ListPackagesResponse;
import software.amazon.awssdk.services.panorama.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.panorama.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.panorama.model.PanoramaException;
import software.amazon.awssdk.services.panorama.model.PanoramaRequest;
import software.amazon.awssdk.services.panorama.model.ProvisionDeviceRequest;
import software.amazon.awssdk.services.panorama.model.ProvisionDeviceResponse;
import software.amazon.awssdk.services.panorama.model.RegisterPackageVersionRequest;
import software.amazon.awssdk.services.panorama.model.RegisterPackageVersionResponse;
import software.amazon.awssdk.services.panorama.model.RemoveApplicationInstanceRequest;
import software.amazon.awssdk.services.panorama.model.RemoveApplicationInstanceResponse;
import software.amazon.awssdk.services.panorama.model.ResourceNotFoundException;
import software.amazon.awssdk.services.panorama.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.panorama.model.SignalApplicationInstanceNodeInstancesRequest;
import software.amazon.awssdk.services.panorama.model.SignalApplicationInstanceNodeInstancesResponse;
import software.amazon.awssdk.services.panorama.model.TagResourceRequest;
import software.amazon.awssdk.services.panorama.model.TagResourceResponse;
import software.amazon.awssdk.services.panorama.model.UntagResourceRequest;
import software.amazon.awssdk.services.panorama.model.UntagResourceResponse;
import software.amazon.awssdk.services.panorama.model.UpdateDeviceMetadataRequest;
import software.amazon.awssdk.services.panorama.model.UpdateDeviceMetadataResponse;
import software.amazon.awssdk.services.panorama.model.ValidationException;
import software.amazon.awssdk.services.panorama.paginators.ListApplicationInstanceDependenciesPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListApplicationInstanceNodeInstancesPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListApplicationInstancesPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListDevicesJobsPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListDevicesPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListNodeFromTemplateJobsPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListNodesPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListPackageImportJobsPublisher;
import software.amazon.awssdk.services.panorama.paginators.ListPackagesPublisher;
import software.amazon.awssdk.services.panorama.transform.CreateApplicationInstanceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreateJobForDevicesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreateNodeFromTemplateJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreatePackageImportJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.CreatePackageRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DeleteDeviceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DeletePackageRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DeregisterPackageVersionRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeApplicationInstanceDetailsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeApplicationInstanceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeDeviceJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeDeviceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeNodeFromTemplateJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribeNodeRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribePackageImportJobRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribePackageRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.DescribePackageVersionRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListApplicationInstanceDependenciesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListApplicationInstanceNodeInstancesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListApplicationInstancesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListDevicesJobsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListNodeFromTemplateJobsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListNodesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListPackageImportJobsRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListPackagesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.ProvisionDeviceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.RegisterPackageVersionRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.RemoveApplicationInstanceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.SignalApplicationInstanceNodeInstancesRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.panorama.transform.UpdateDeviceMetadataRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PanoramaAsyncClient}.
 *
 * @see PanoramaAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPanoramaAsyncClient implements PanoramaAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPanoramaAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPanoramaAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an application instance and deploys it to a device.
     * </p>
     *
     * @param createApplicationInstanceRequest
     * @return A Java Future containing the result of the CreateApplicationInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ServiceQuotaExceededException The request would cause a limit to be exceeded.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.CreateApplicationInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreateApplicationInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationInstanceResponse> createApplicationInstance(
            CreateApplicationInstanceRequest createApplicationInstanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplicationInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationInstanceRequest, CreateApplicationInstanceResponse>()
                            .withOperationName("CreateApplicationInstance")
                            .withMarshaller(new CreateApplicationInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApplicationInstanceRequest));
            CompletableFuture<CreateApplicationInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a job to run on one or more devices. A job can update a device's software or reboot it.
     * </p>
     *
     * @param createJobForDevicesRequest
     * @return A Java Future containing the result of the CreateJobForDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.CreateJobForDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreateJobForDevices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobForDevicesResponse> createJobForDevices(
            CreateJobForDevicesRequest createJobForDevicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobForDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJobForDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobForDevicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateJobForDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobForDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobForDevicesRequest, CreateJobForDevicesResponse>()
                            .withOperationName("CreateJobForDevices")
                            .withMarshaller(new CreateJobForDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createJobForDevicesRequest));
            CompletableFuture<CreateJobForDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a camera stream node.
     * </p>
     *
     * @param createNodeFromTemplateJobRequest
     * @return A Java Future containing the result of the CreateNodeFromTemplateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.CreateNodeFromTemplateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreateNodeFromTemplateJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNodeFromTemplateJobResponse> createNodeFromTemplateJob(
            CreateNodeFromTemplateJobRequest createNodeFromTemplateJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNodeFromTemplateJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNodeFromTemplateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNodeFromTemplateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNodeFromTemplateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNodeFromTemplateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNodeFromTemplateJobRequest, CreateNodeFromTemplateJobResponse>()
                            .withOperationName("CreateNodeFromTemplateJob")
                            .withMarshaller(new CreateNodeFromTemplateJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createNodeFromTemplateJobRequest));
            CompletableFuture<CreateNodeFromTemplateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a package and storage location in an Amazon S3 access point.
     * </p>
     *
     * @param createPackageRequest
     * @return A Java Future containing the result of the CreatePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.CreatePackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreatePackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePackageResponse> createPackage(CreatePackageRequest createPackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreatePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePackageRequest, CreatePackageResponse>()
                            .withOperationName("CreatePackage")
                            .withMarshaller(new CreatePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPackageRequest));
            CompletableFuture<CreatePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports a node package.
     * </p>
     *
     * @param createPackageImportJobRequest
     * @return A Java Future containing the result of the CreatePackageImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.CreatePackageImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/CreatePackageImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePackageImportJobResponse> createPackageImportJob(
            CreatePackageImportJobRequest createPackageImportJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPackageImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePackageImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePackageImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePackageImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePackageImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePackageImportJobRequest, CreatePackageImportJobResponse>()
                            .withOperationName("CreatePackageImportJob")
                            .withMarshaller(new CreatePackageImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPackageImportJobRequest));
            CompletableFuture<CreatePackageImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a device.
     * </p>
     *
     * @param deleteDeviceRequest
     * @return A Java Future containing the result of the DeleteDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DeleteDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DeleteDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDeviceResponse> deleteDevice(DeleteDeviceRequest deleteDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDeviceRequest, DeleteDeviceResponse>()
                            .withOperationName("DeleteDevice").withMarshaller(new DeleteDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDeviceRequest));
            CompletableFuture<DeleteDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a package.
     * </p>
     * <note>
     * <p>
     * To delete a package, you need permission to call <code>s3:DeleteObject</code> in addition to permissions for the
     * AWS Panorama API.
     * </p>
     * </note>
     *
     * @param deletePackageRequest
     * @return A Java Future containing the result of the DeletePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DeletePackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DeletePackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePackageResponse> deletePackage(DeletePackageRequest deletePackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePackageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePackageRequest, DeletePackageResponse>()
                            .withOperationName("DeletePackage")
                            .withMarshaller(new DeletePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePackageRequest));
            CompletableFuture<DeletePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters a package version.
     * </p>
     *
     * @param deregisterPackageVersionRequest
     * @return A Java Future containing the result of the DeregisterPackageVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DeregisterPackageVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DeregisterPackageVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterPackageVersionResponse> deregisterPackageVersion(
            DeregisterPackageVersionRequest deregisterPackageVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterPackageVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterPackageVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterPackageVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterPackageVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterPackageVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterPackageVersionRequest, DeregisterPackageVersionResponse>()
                            .withOperationName("DeregisterPackageVersion")
                            .withMarshaller(new DeregisterPackageVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterPackageVersionRequest));
            CompletableFuture<DeregisterPackageVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an application instance on a device.
     * </p>
     *
     * @param describeApplicationInstanceRequest
     * @return A Java Future containing the result of the DescribeApplicationInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribeApplicationInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeApplicationInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeApplicationInstanceResponse> describeApplicationInstance(
            DescribeApplicationInstanceRequest describeApplicationInstanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeApplicationInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplicationInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeApplicationInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeApplicationInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeApplicationInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationInstanceRequest, DescribeApplicationInstanceResponse>()
                            .withOperationName("DescribeApplicationInstance")
                            .withMarshaller(new DescribeApplicationInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeApplicationInstanceRequest));
            CompletableFuture<DescribeApplicationInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an application instance's configuration manifest.
     * </p>
     *
     * @param describeApplicationInstanceDetailsRequest
     * @return A Java Future containing the result of the DescribeApplicationInstanceDetails operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribeApplicationInstanceDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeApplicationInstanceDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeApplicationInstanceDetailsResponse> describeApplicationInstanceDetails(
            DescribeApplicationInstanceDetailsRequest describeApplicationInstanceDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeApplicationInstanceDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplicationInstanceDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeApplicationInstanceDetailsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeApplicationInstanceDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeApplicationInstanceDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationInstanceDetailsRequest, DescribeApplicationInstanceDetailsResponse>()
                            .withOperationName("DescribeApplicationInstanceDetails")
                            .withMarshaller(new DescribeApplicationInstanceDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeApplicationInstanceDetailsRequest));
            CompletableFuture<DescribeApplicationInstanceDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a device.
     * </p>
     *
     * @param describeDeviceRequest
     * @return A Java Future containing the result of the DescribeDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribeDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDeviceResponse> describeDevice(DescribeDeviceRequest describeDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDeviceRequest, DescribeDeviceResponse>()
                            .withOperationName("DescribeDevice")
                            .withMarshaller(new DescribeDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDeviceRequest));
            CompletableFuture<DescribeDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a device job.
     * </p>
     *
     * @param describeDeviceJobRequest
     * @return A Java Future containing the result of the DescribeDeviceJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribeDeviceJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeDeviceJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDeviceJobResponse> describeDeviceJob(DescribeDeviceJobRequest describeDeviceJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDeviceJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDeviceJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDeviceJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDeviceJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDeviceJobRequest, DescribeDeviceJobResponse>()
                            .withOperationName("DescribeDeviceJob")
                            .withMarshaller(new DescribeDeviceJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDeviceJobRequest));
            CompletableFuture<DescribeDeviceJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a node.
     * </p>
     *
     * @param describeNodeRequest
     * @return A Java Future containing the result of the DescribeNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribeNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeNodeResponse> describeNode(DescribeNodeRequest describeNodeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNodeRequest, DescribeNodeResponse>()
                            .withOperationName("DescribeNode").withMarshaller(new DescribeNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeNodeRequest));
            CompletableFuture<DescribeNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a job to create a camera stream node.
     * </p>
     *
     * @param describeNodeFromTemplateJobRequest
     * @return A Java Future containing the result of the DescribeNodeFromTemplateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribeNodeFromTemplateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribeNodeFromTemplateJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeNodeFromTemplateJobResponse> describeNodeFromTemplateJob(
            DescribeNodeFromTemplateJobRequest describeNodeFromTemplateJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeNodeFromTemplateJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNodeFromTemplateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeNodeFromTemplateJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeNodeFromTemplateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeNodeFromTemplateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNodeFromTemplateJobRequest, DescribeNodeFromTemplateJobResponse>()
                            .withOperationName("DescribeNodeFromTemplateJob")
                            .withMarshaller(new DescribeNodeFromTemplateJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeNodeFromTemplateJobRequest));
            CompletableFuture<DescribeNodeFromTemplateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a package.
     * </p>
     *
     * @param describePackageRequest
     * @return A Java Future containing the result of the DescribePackage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribePackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribePackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackageResponse> describePackage(DescribePackageRequest describePackageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackageRequest, DescribePackageResponse>()
                            .withOperationName("DescribePackage")
                            .withMarshaller(new DescribePackageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePackageRequest));
            CompletableFuture<DescribePackageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a package import job.
     * </p>
     *
     * @param describePackageImportJobRequest
     * @return A Java Future containing the result of the DescribePackageImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribePackageImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribePackageImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackageImportJobResponse> describePackageImportJob(
            DescribePackageImportJobRequest describePackageImportJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackageImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackageImportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackageImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackageImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackageImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackageImportJobRequest, DescribePackageImportJobResponse>()
                            .withOperationName("DescribePackageImportJob")
                            .withMarshaller(new DescribePackageImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePackageImportJobRequest));
            CompletableFuture<DescribePackageImportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a package version.
     * </p>
     *
     * @param describePackageVersionRequest
     * @return A Java Future containing the result of the DescribePackageVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.DescribePackageVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/DescribePackageVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackageVersionResponse> describePackageVersion(
            DescribePackageVersionRequest describePackageVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackageVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackageVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackageVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackageVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackageVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackageVersionRequest, DescribePackageVersionResponse>()
                            .withOperationName("DescribePackageVersion")
                            .withMarshaller(new DescribePackageVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePackageVersionRequest));
            CompletableFuture<DescribePackageVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of application instance dependencies.
     * </p>
     *
     * @param listApplicationInstanceDependenciesRequest
     * @return A Java Future containing the result of the ListApplicationInstanceDependencies operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListApplicationInstanceDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstanceDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationInstanceDependenciesResponse> listApplicationInstanceDependencies(
            ListApplicationInstanceDependenciesRequest listApplicationInstanceDependenciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listApplicationInstanceDependenciesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationInstanceDependencies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationInstanceDependenciesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListApplicationInstanceDependenciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationInstanceDependenciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationInstanceDependenciesRequest, ListApplicationInstanceDependenciesResponse>()
                            .withOperationName("ListApplicationInstanceDependencies")
                            .withMarshaller(new ListApplicationInstanceDependenciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationInstanceDependenciesRequest));
            CompletableFuture<ListApplicationInstanceDependenciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of application instance dependencies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationInstanceDependencies(software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListApplicationInstanceDependenciesPublisher publisher = client.listApplicationInstanceDependenciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListApplicationInstanceDependenciesPublisher publisher = client.listApplicationInstanceDependenciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationInstanceDependencies(software.amazon.awssdk.services.panorama.model.ListApplicationInstanceDependenciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationInstanceDependenciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListApplicationInstanceDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstanceDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationInstanceDependenciesPublisher listApplicationInstanceDependenciesPaginator(
            ListApplicationInstanceDependenciesRequest listApplicationInstanceDependenciesRequest) {
        return new ListApplicationInstanceDependenciesPublisher(this,
                applyPaginatorUserAgent(listApplicationInstanceDependenciesRequest));
    }

    /**
     * <p>
     * Returns a list of application node instances.
     * </p>
     *
     * @param listApplicationInstanceNodeInstancesRequest
     * @return A Java Future containing the result of the ListApplicationInstanceNodeInstances operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListApplicationInstanceNodeInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstanceNodeInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationInstanceNodeInstancesResponse> listApplicationInstanceNodeInstances(
            ListApplicationInstanceNodeInstancesRequest listApplicationInstanceNodeInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listApplicationInstanceNodeInstancesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationInstanceNodeInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationInstanceNodeInstancesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListApplicationInstanceNodeInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationInstanceNodeInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationInstanceNodeInstancesRequest, ListApplicationInstanceNodeInstancesResponse>()
                            .withOperationName("ListApplicationInstanceNodeInstances")
                            .withMarshaller(new ListApplicationInstanceNodeInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationInstanceNodeInstancesRequest));
            CompletableFuture<ListApplicationInstanceNodeInstancesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of application node instances.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationInstanceNodeInstances(software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListApplicationInstanceNodeInstancesPublisher publisher = client.listApplicationInstanceNodeInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListApplicationInstanceNodeInstancesPublisher publisher = client.listApplicationInstanceNodeInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationInstanceNodeInstances(software.amazon.awssdk.services.panorama.model.ListApplicationInstanceNodeInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationInstanceNodeInstancesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListApplicationInstanceNodeInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstanceNodeInstances"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationInstanceNodeInstancesPublisher listApplicationInstanceNodeInstancesPaginator(
            ListApplicationInstanceNodeInstancesRequest listApplicationInstanceNodeInstancesRequest) {
        return new ListApplicationInstanceNodeInstancesPublisher(this,
                applyPaginatorUserAgent(listApplicationInstanceNodeInstancesRequest));
    }

    /**
     * <p>
     * Returns a list of application instances.
     * </p>
     *
     * @param listApplicationInstancesRequest
     * @return A Java Future containing the result of the ListApplicationInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListApplicationInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationInstancesResponse> listApplicationInstances(
            ListApplicationInstancesRequest listApplicationInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplicationInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationInstancesRequest, ListApplicationInstancesResponse>()
                            .withOperationName("ListApplicationInstances")
                            .withMarshaller(new ListApplicationInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationInstancesRequest));
            CompletableFuture<ListApplicationInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of application instances.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplicationInstances(software.amazon.awssdk.services.panorama.model.ListApplicationInstancesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListApplicationInstancesPublisher publisher = client.listApplicationInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListApplicationInstancesPublisher publisher = client.listApplicationInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListApplicationInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListApplicationInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplicationInstances(software.amazon.awssdk.services.panorama.model.ListApplicationInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationInstancesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListApplicationInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListApplicationInstances"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationInstancesPublisher listApplicationInstancesPaginator(
            ListApplicationInstancesRequest listApplicationInstancesRequest) {
        return new ListApplicationInstancesPublisher(this, applyPaginatorUserAgent(listApplicationInstancesRequest));
    }

    /**
     * <p>
     * Returns a list of devices.
     * </p>
     *
     * @param listDevicesRequest
     * @return A Java Future containing the result of the ListDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDevicesResponse> listDevices(ListDevicesRequest listDevicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                            .withOperationName("ListDevices").withMarshaller(new ListDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDevicesRequest));
            CompletableFuture<ListDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of jobs.
     * </p>
     *
     * @param listDevicesJobsRequest
     * @return A Java Future containing the result of the ListDevicesJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListDevicesJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListDevicesJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDevicesJobsResponse> listDevicesJobs(ListDevicesJobsRequest listDevicesJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevicesJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDevicesJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDevicesJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDevicesJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDevicesJobsRequest, ListDevicesJobsResponse>()
                            .withOperationName("ListDevicesJobs")
                            .withMarshaller(new ListDevicesJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDevicesJobsRequest));
            CompletableFuture<ListDevicesJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of jobs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDevicesJobs(software.amazon.awssdk.services.panorama.model.ListDevicesJobsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListDevicesJobsPublisher publisher = client.listDevicesJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListDevicesJobsPublisher publisher = client.listDevicesJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListDevicesJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListDevicesJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDevicesJobs(software.amazon.awssdk.services.panorama.model.ListDevicesJobsRequest)} operation.</b>
     * </p>
     *
     * @param listDevicesJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListDevicesJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListDevicesJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDevicesJobsPublisher listDevicesJobsPaginator(ListDevicesJobsRequest listDevicesJobsRequest) {
        return new ListDevicesJobsPublisher(this, applyPaginatorUserAgent(listDevicesJobsRequest));
    }

    /**
     * <p>
     * Returns a list of devices.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDevices(software.amazon.awssdk.services.panorama.model.ListDevicesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListDevicesPublisher publisher = client.listDevicesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListDevicesPublisher publisher = client.listDevicesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListDevicesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListDevicesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDevices(software.amazon.awssdk.services.panorama.model.ListDevicesRequest)} operation.</b>
     * </p>
     *
     * @param listDevicesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListDevices" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDevicesPublisher listDevicesPaginator(ListDevicesRequest listDevicesRequest) {
        return new ListDevicesPublisher(this, applyPaginatorUserAgent(listDevicesRequest));
    }

    /**
     * <p>
     * Returns a list of camera stream node jobs.
     * </p>
     *
     * @param listNodeFromTemplateJobsRequest
     * @return A Java Future containing the result of the ListNodeFromTemplateJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListNodeFromTemplateJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListNodeFromTemplateJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNodeFromTemplateJobsResponse> listNodeFromTemplateJobs(
            ListNodeFromTemplateJobsRequest listNodeFromTemplateJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNodeFromTemplateJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNodeFromTemplateJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNodeFromTemplateJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNodeFromTemplateJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNodeFromTemplateJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNodeFromTemplateJobsRequest, ListNodeFromTemplateJobsResponse>()
                            .withOperationName("ListNodeFromTemplateJobs")
                            .withMarshaller(new ListNodeFromTemplateJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listNodeFromTemplateJobsRequest));
            CompletableFuture<ListNodeFromTemplateJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of camera stream node jobs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNodeFromTemplateJobs(software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListNodeFromTemplateJobsPublisher publisher = client.listNodeFromTemplateJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListNodeFromTemplateJobsPublisher publisher = client.listNodeFromTemplateJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNodeFromTemplateJobs(software.amazon.awssdk.services.panorama.model.ListNodeFromTemplateJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNodeFromTemplateJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListNodeFromTemplateJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListNodeFromTemplateJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNodeFromTemplateJobsPublisher listNodeFromTemplateJobsPaginator(
            ListNodeFromTemplateJobsRequest listNodeFromTemplateJobsRequest) {
        return new ListNodeFromTemplateJobsPublisher(this, applyPaginatorUserAgent(listNodeFromTemplateJobsRequest));
    }

    /**
     * <p>
     * Returns a list of nodes.
     * </p>
     *
     * @param listNodesRequest
     * @return A Java Future containing the result of the ListNodes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListNodes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListNodesResponse> listNodes(ListNodesRequest listNodesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNodes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNodesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListNodesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNodesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNodesRequest, ListNodesResponse>().withOperationName("ListNodes")
                            .withMarshaller(new ListNodesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNodesRequest));
            CompletableFuture<ListNodesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of nodes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listNodes(software.amazon.awssdk.services.panorama.model.ListNodesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListNodesPublisher publisher = client.listNodesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListNodesPublisher publisher = client.listNodesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListNodesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListNodesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNodes(software.amazon.awssdk.services.panorama.model.ListNodesRequest)} operation.</b>
     * </p>
     *
     * @param listNodesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListNodes" target="_top">AWS API
     *      Documentation</a>
     */
    public ListNodesPublisher listNodesPaginator(ListNodesRequest listNodesRequest) {
        return new ListNodesPublisher(this, applyPaginatorUserAgent(listNodesRequest));
    }

    /**
     * <p>
     * Returns a list of package import jobs.
     * </p>
     *
     * @param listPackageImportJobsRequest
     * @return A Java Future containing the result of the ListPackageImportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListPackageImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListPackageImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackageImportJobsResponse> listPackageImportJobs(
            ListPackageImportJobsRequest listPackageImportJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackageImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackageImportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackageImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackageImportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackageImportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackageImportJobsRequest, ListPackageImportJobsResponse>()
                            .withOperationName("ListPackageImportJobs")
                            .withMarshaller(new ListPackageImportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackageImportJobsRequest));
            CompletableFuture<ListPackageImportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of package import jobs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPackageImportJobs(software.amazon.awssdk.services.panorama.model.ListPackageImportJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListPackageImportJobsPublisher publisher = client.listPackageImportJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListPackageImportJobsPublisher publisher = client.listPackageImportJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListPackageImportJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListPackageImportJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackageImportJobs(software.amazon.awssdk.services.panorama.model.ListPackageImportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackageImportJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListPackageImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListPackageImportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPackageImportJobsPublisher listPackageImportJobsPaginator(ListPackageImportJobsRequest listPackageImportJobsRequest) {
        return new ListPackageImportJobsPublisher(this, applyPaginatorUserAgent(listPackageImportJobsRequest));
    }

    /**
     * <p>
     * Returns a list of packages.
     * </p>
     *
     * @param listPackagesRequest
     * @return A Java Future containing the result of the ListPackages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListPackages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackagesResponse> listPackages(ListPackagesRequest listPackagesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPackagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackagesRequest, ListPackagesResponse>()
                            .withOperationName("ListPackages").withMarshaller(new ListPackagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackagesRequest));
            CompletableFuture<ListPackagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of packages.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listPackages(software.amazon.awssdk.services.panorama.model.ListPackagesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListPackagesPublisher publisher = client.listPackagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.panorama.paginators.ListPackagesPublisher publisher = client.listPackagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.panorama.model.ListPackagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.panorama.model.ListPackagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackages(software.amazon.awssdk.services.panorama.model.ListPackagesRequest)} operation.</b>
     * </p>
     *
     * @param listPackagesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListPackages" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPackagesPublisher listPackagesPaginator(ListPackagesRequest listPackagesRequest) {
        return new ListPackagesPublisher(this, applyPaginatorUserAgent(listPackagesRequest));
    }

    /**
     * <p>
     * Returns a list of tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a device and returns a configuration archive. The configuration archive is a ZIP file that contains a
     * provisioning certificate that is valid for 5 minutes. Name the configuration archive
     * <code>certificates-omni_<i>device-name</i>.zip</code> and transfer it to the device within 5 minutes. Use the
     * included USB storage device and connect it to the USB 3.0 port next to the HDMI output.
     * </p>
     *
     * @param provisionDeviceRequest
     * @return A Java Future containing the result of the ProvisionDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ServiceQuotaExceededException The request would cause a limit to be exceeded.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.ProvisionDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/ProvisionDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ProvisionDeviceResponse> provisionDevice(ProvisionDeviceRequest provisionDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, provisionDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ProvisionDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ProvisionDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ProvisionDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ProvisionDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ProvisionDeviceRequest, ProvisionDeviceResponse>()
                            .withOperationName("ProvisionDevice")
                            .withMarshaller(new ProvisionDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(provisionDeviceRequest));
            CompletableFuture<ProvisionDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a package version.
     * </p>
     *
     * @param registerPackageVersionRequest
     * @return A Java Future containing the result of the RegisterPackageVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.RegisterPackageVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/RegisterPackageVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterPackageVersionResponse> registerPackageVersion(
            RegisterPackageVersionRequest registerPackageVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerPackageVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterPackageVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterPackageVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterPackageVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterPackageVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterPackageVersionRequest, RegisterPackageVersionResponse>()
                            .withOperationName("RegisterPackageVersion")
                            .withMarshaller(new RegisterPackageVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerPackageVersionRequest));
            CompletableFuture<RegisterPackageVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an application instance.
     * </p>
     *
     * @param removeApplicationInstanceRequest
     * @return A Java Future containing the result of the RemoveApplicationInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.RemoveApplicationInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/RemoveApplicationInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveApplicationInstanceResponse> removeApplicationInstance(
            RemoveApplicationInstanceRequest removeApplicationInstanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeApplicationInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveApplicationInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveApplicationInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveApplicationInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveApplicationInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveApplicationInstanceRequest, RemoveApplicationInstanceResponse>()
                            .withOperationName("RemoveApplicationInstance")
                            .withMarshaller(new RemoveApplicationInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeApplicationInstanceRequest));
            CompletableFuture<RemoveApplicationInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Signal camera nodes to stop or resume.
     * </p>
     *
     * @param signalApplicationInstanceNodeInstancesRequest
     * @return A Java Future containing the result of the SignalApplicationInstanceNodeInstances operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ServiceQuotaExceededException The request would cause a limit to be exceeded.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.SignalApplicationInstanceNodeInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/SignalApplicationInstanceNodeInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SignalApplicationInstanceNodeInstancesResponse> signalApplicationInstanceNodeInstances(
            SignalApplicationInstanceNodeInstancesRequest signalApplicationInstanceNodeInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                signalApplicationInstanceNodeInstancesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SignalApplicationInstanceNodeInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SignalApplicationInstanceNodeInstancesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, SignalApplicationInstanceNodeInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SignalApplicationInstanceNodeInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SignalApplicationInstanceNodeInstancesRequest, SignalApplicationInstanceNodeInstancesResponse>()
                            .withOperationName("SignalApplicationInstanceNodeInstances")
                            .withMarshaller(new SignalApplicationInstanceNodeInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(signalApplicationInstanceNodeInstancesRequest));
            CompletableFuture<SignalApplicationInstanceNodeInstancesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a device's metadata.
     * </p>
     *
     * @param updateDeviceMetadataRequest
     * @return A Java Future containing the result of the UpdateDeviceMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The target resource is in use.</li>
     *         <li>ValidationException The request contains an invalid parameter value.</li>
     *         <li>AccessDeniedException The requestor does not have permission to access the target action or resource.
     *         </li>
     *         <li>ResourceNotFoundException The target resource was not found.</li>
     *         <li>InternalServerException An internal error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PanoramaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PanoramaAsyncClient.UpdateDeviceMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/panorama-2019-07-24/UpdateDeviceMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDeviceMetadataResponse> updateDeviceMetadata(
            UpdateDeviceMetadataRequest updateDeviceMetadataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeviceMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Panorama");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDeviceMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDeviceMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDeviceMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDeviceMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDeviceMetadataRequest, UpdateDeviceMetadataResponse>()
                            .withOperationName("UpdateDeviceMetadata")
                            .withMarshaller(new UpdateDeviceMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDeviceMetadataRequest));
            CompletableFuture<UpdateDeviceMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PanoramaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends PanoramaRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
