/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.paymentcryptography.internal.PaymentCryptographyServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.paymentcryptography.model.AccessDeniedException;
import software.amazon.awssdk.services.paymentcryptography.model.ConflictException;
import software.amazon.awssdk.services.paymentcryptography.model.CreateAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.CreateAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.CreateKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.CreateKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ExportKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ExportKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForExportRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForExportResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForImportRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForImportResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetPublicKeyCertificateRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetPublicKeyCertificateResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ImportKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ImportKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.InternalServerException;
import software.amazon.awssdk.services.paymentcryptography.model.ListAliasesRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListAliasesResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ListKeysRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListKeysResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.PaymentCryptographyException;
import software.amazon.awssdk.services.paymentcryptography.model.ResourceNotFoundException;
import software.amazon.awssdk.services.paymentcryptography.model.RestoreKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.RestoreKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.paymentcryptography.model.ServiceUnavailableException;
import software.amazon.awssdk.services.paymentcryptography.model.StartKeyUsageRequest;
import software.amazon.awssdk.services.paymentcryptography.model.StartKeyUsageResponse;
import software.amazon.awssdk.services.paymentcryptography.model.StopKeyUsageRequest;
import software.amazon.awssdk.services.paymentcryptography.model.StopKeyUsageResponse;
import software.amazon.awssdk.services.paymentcryptography.model.TagResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.TagResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ThrottlingException;
import software.amazon.awssdk.services.paymentcryptography.model.UntagResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.UntagResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.UpdateAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.UpdateAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ValidationException;
import software.amazon.awssdk.services.paymentcryptography.transform.CreateAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.CreateKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.DeleteAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.DeleteKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ExportKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetParametersForExportRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetParametersForImportRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetPublicKeyCertificateRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ImportKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListAliasesRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListKeysRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.RestoreKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.StartKeyUsageRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.StopKeyUsageRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.UpdateAliasRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PaymentCryptographyAsyncClient}.
 *
 * @see PaymentCryptographyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPaymentCryptographyAsyncClient implements PaymentCryptographyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPaymentCryptographyAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PaymentCryptographyServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPaymentCryptographyAsyncClient(PaymentCryptographyServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an <i>alias</i>, or a friendly name, for an Amazon Web Services Payment Cryptography key. You can use an
     * alias to identify a key in the console and when you call cryptographic operations such as <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_EncryptData.html"
     * >EncryptData</a> or <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_DecryptData.html"
     * >DecryptData</a>.
     * </p>
     * <p>
     * You can associate the alias with any key in the same Amazon Web Services Region. Each alias is associated with
     * only one key at a time, but a key can have multiple aliases. You can't create an alias without a key. The alias
     * must be unique in the account and Amazon Web Services Region, but you can create another alias with the same name
     * in a different Amazon Web Services Region.
     * </p>
     * <p>
     * To change the key that's associated with the alias, call <a>UpdateAlias</a>. To delete the alias, call
     * <a>DeleteAlias</a>. These operations don't affect the underlying key. To get the alias that you created, call
     * <a>ListAliases</a>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createAliasRequest
     * @return A Java Future containing the result of the CreateAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.CreateAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/CreateAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAliasResponse> createAlias(CreateAliasRequest createAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAliasRequest, CreateAliasResponse>()
                            .withOperationName("CreateAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAliasRequest));
            CompletableFuture<CreateAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Web Services Payment Cryptography key, a logical representation of a cryptographic key, that is
     * unique in your account and Amazon Web Services Region. You use keys for cryptographic functions such as
     * encryption and decryption.
     * </p>
     * <p>
     * In addition to the key material used in cryptographic operations, an Amazon Web Services Payment Cryptography key
     * includes metadata such as the key ARN, key usage, key origin, creation date, description, and key state.
     * </p>
     * <p>
     * When you create a key, you specify both immutable and mutable data about the key. The immutable data contains key
     * attributes that defines the scope and cryptographic operations that you can perform using the key, for example
     * key class (example: <code>SYMMETRIC_KEY</code>), key algorithm (example: <code>TDES_2KEY</code>), key usage
     * (example: <code>TR31_P0_PIN_ENCRYPTION_KEY</code>) and key modes of use (example: <code>Encrypt</code>). For
     * information about valid combinations of key attributes, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>. The mutable data contained
     * within a key includes usage timestamp and key deletion timestamp and can be modified after creation.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography binds key attributes to keys using key blocks when you store or export
     * them. Amazon Web Services Payment Cryptography stores the key contents wrapped and never stores or transmits them
     * in the clear.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListKeys</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createKeyRequest
     * @return A Java Future containing the result of the CreateKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.CreateKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/CreateKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKeyResponse> createKey(CreateKeyRequest createKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKeyRequest, CreateKeyResponse>().withOperationName("CreateKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createKeyRequest));
            CompletableFuture<CreateKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the alias, but doesn't affect the underlying key.
     * </p>
     * <p>
     * Each key can have multiple aliases. To get the aliases of all keys, use the <a>ListAliases</a> operation. To
     * change the alias of a key, first use <a>DeleteAlias</a> to delete the current alias and then use
     * <a>CreateAlias</a> to create a new alias. To associate an existing alias with a different key, call
     * <a>UpdateAlias</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAliasRequest
     * @return A Java Future containing the result of the DeleteAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.DeleteAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/DeleteAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAliasResponse> deleteAlias(DeleteAliasRequest deleteAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAliasRequest, DeleteAliasResponse>()
                            .withOperationName("DeleteAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAliasRequest));
            CompletableFuture<DeleteAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the key material and all metadata associated with Amazon Web Services Payment Cryptography key.
     * </p>
     * <p>
     * Key deletion is irreversible. After a key is deleted, you can't perform cryptographic operations using the key.
     * For example, you can't decrypt data that was encrypted by a deleted Amazon Web Services Payment Cryptography key,
     * and the data may become unrecoverable. Because key deletion is destructive, Amazon Web Services Payment
     * Cryptography has a safety mechanism to prevent accidental deletion of a key. When you call this operation, Amazon
     * Web Services Payment Cryptography disables the specified key but doesn't delete it until after a waiting period.
     * The default waiting period is 7 days. To set a different waiting period, set <code>DeleteKeyInDays</code>. During
     * the waiting period, the <code>KeyState</code> is <code>DELETE_PENDING</code>. After the key is deleted, the
     * <code>KeyState</code> is <code>DELETE_COMPLETE</code>.
     * </p>
     * <p>
     * If you delete key material, you can use <a>ImportKey</a> to reimport the same key material into the Amazon Web
     * Services Payment Cryptography key.
     * </p>
     * <p>
     * You should delete a key only when you are sure that you don't need to use it anymore and no other parties are
     * utilizing this key. If you aren't sure, consider deactivating it instead by calling <a>StopKeyUsage</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RestoreKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartKeyUsage</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteKeyRequest
     * @return A Java Future containing the result of the DeleteKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.DeleteKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/DeleteKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKeyResponse> deleteKey(DeleteKeyRequest deleteKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKeyRequest, DeleteKeyResponse>().withOperationName("DeleteKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteKeyRequest));
            CompletableFuture<DeleteKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports a key from Amazon Web Services Payment Cryptography using either ANSI X9 TR-34 or TR-31 key export
     * standard.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography simplifies main or root key exchange process by eliminating the need of
     * a paper-based key exchange process. It takes a modern and secure approach based of the ANSI X9 TR-34 key exchange
     * standard.
     * </p>
     * <p>
     * You can use <code>ExportKey</code> to export main or root keys such as KEK (Key Encryption Key), using asymmetric
     * key exchange technique following ANSI X9 TR-34 standard. The ANSI X9 TR-34 standard uses asymmetric keys to
     * establishes bi-directional trust between the two parties exchanging keys. After which you can export working keys
     * using the ANSI X9 TR-31 symmetric key exchange standard as mandated by PCI PIN. Using this operation, you can
     * share your Amazon Web Services Payment Cryptography generated keys with other service partners to perform
     * cryptographic operations outside of Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * <b>TR-34 key export</b>
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography uses TR-34 asymmetric key exchange standard to export main keys such as
     * KEK. In TR-34 terminology, the sending party of the key is called Key Distribution Host (KDH) and the receiving
     * party of the key is called Key Receiving Host (KRH). In key export process, KDH is Amazon Web Services Payment
     * Cryptography which initiates key export. KRH is the user receiving the key. Before you initiate TR-34 key export,
     * you must obtain an export token by calling <a>GetParametersForExport</a>. This operation also returns the signing
     * key certificate that KDH uses to sign the wrapped key to generate a TR-34 wrapped key block. The export token
     * expires after 7 days.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <dl>
     * <dt>CertificateAuthorityPublicKeyIdentifier</dt>
     * <dd>
     * <p>
     * The <code>KeyARN</code> of the certificate chain that will sign the wrapping key certificate. This must exist
     * within Amazon Web Services Payment Cryptography before you initiate TR-34 key export. If it does not exist, you
     * can import it by calling <a>ImportKey</a> for <code>RootCertificatePublicKey</code>.
     * </p>
     * </dd>
     * <dt>ExportToken</dt>
     * <dd>
     * <p>
     * Obtained from KDH by calling <a>GetParametersForExport</a>.
     * </p>
     * </dd>
     * <dt>WrappingKeyCertificate</dt>
     * <dd>
     * <p>
     * Amazon Web Services Payment Cryptography uses this to wrap the key under export.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * When this operation is successful, Amazon Web Services Payment Cryptography returns the TR-34 wrapped key block.
     * </p>
     * <p>
     * <b>TR-31 key export</b>
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography uses TR-31 symmetric key exchange standard to export working keys. In
     * TR-31, you must use a main key such as KEK to encrypt or wrap the key under export. To establish a KEK, you can
     * use <a>CreateKey</a> or <a>ImportKey</a>. When this operation is successful, Amazon Web Services Payment
     * Cryptography returns a TR-31 wrapped key block.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GetParametersForExport</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param exportKeyRequest
     * @return A Java Future containing the result of the ExportKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ExportKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ExportKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportKeyResponse> exportKey(ExportKeyRequest exportKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportKeyRequest, ExportKeyResponse>().withOperationName("ExportKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportKeyRequest));
            CompletableFuture<ExportKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the Amazon Web Services Payment Cryptography key associated with the alias.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAliasRequest
     * @return A Java Future containing the result of the GetAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetAlias" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAliasResponse> getAlias(GetAliasRequest getAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAliasRequest, GetAliasResponse>().withOperationName("GetAlias")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAliasRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAliasRequest));
            CompletableFuture<GetAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the key material for an Amazon Web Services Payment Cryptography key, including the immutable and mutable
     * data specified when the key was created.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListKeys</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getKeyRequest
     * @return A Java Future containing the result of the GetKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetKeyResponse> getKey(GetKeyRequest getKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKeyRequest, GetKeyResponse>().withOperationName("GetKey")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKeyRequest));
            CompletableFuture<GetKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the export token and the signing key certificate to initiate a TR-34 key export from Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * The signing key certificate signs the wrapped key under export within the TR-34 key payload. The export token and
     * signing key certificate must be in place and operational before calling <a>ExportKey</a>. The export token
     * expires in 7 days. You can use the same export token to export multiple keys from your service account.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ExportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetParametersForImport</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getParametersForExportRequest
     * @return A Java Future containing the result of the GetParametersForExport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetParametersForExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetParametersForExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetParametersForExportResponse> getParametersForExport(
            GetParametersForExportRequest getParametersForExportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParametersForExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParametersForExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParametersForExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetParametersForExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetParametersForExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetParametersForExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetParametersForExportRequest, GetParametersForExportResponse>()
                            .withOperationName("GetParametersForExport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetParametersForExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getParametersForExportRequest));
            CompletableFuture<GetParametersForExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the import token and the wrapping key certificate to initiate a TR-34 key import into Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * The wrapping key certificate wraps the key under import within the TR-34 key payload. The import token and
     * wrapping key certificate must be in place and operational before calling <a>ImportKey</a>. The import token
     * expires in 7 days. The same import token can be used to import multiple keys into your service account.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GetParametersForExport</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getParametersForImportRequest
     * @return A Java Future containing the result of the GetParametersForImport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetParametersForImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetParametersForImport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetParametersForImportResponse> getParametersForImport(
            GetParametersForImportRequest getParametersForImportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParametersForImportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParametersForImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParametersForImport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetParametersForImportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetParametersForImportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetParametersForImportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetParametersForImportRequest, GetParametersForImportResponse>()
                            .withOperationName("GetParametersForImport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetParametersForImportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getParametersForImportRequest));
            CompletableFuture<GetParametersForImportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the public key certificate of the asymmetric key pair that exists within Amazon Web Services Payment
     * Cryptography.
     * </p>
     * <p>
     * Unlike the private key of an asymmetric key, which never leaves Amazon Web Services Payment Cryptography
     * unencrypted, callers with <code>GetPublicKeyCertificate</code> permission can download the public key certificate
     * of the asymmetric key. You can share the public key certificate to allow others to encrypt messages and verify
     * signatures outside of Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     *
     * @param getPublicKeyCertificateRequest
     * @return A Java Future containing the result of the GetPublicKeyCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.GetPublicKeyCertificate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetPublicKeyCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPublicKeyCertificateResponse> getPublicKeyCertificate(
            GetPublicKeyCertificateRequest getPublicKeyCertificateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPublicKeyCertificateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPublicKeyCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPublicKeyCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPublicKeyCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPublicKeyCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPublicKeyCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPublicKeyCertificateRequest, GetPublicKeyCertificateResponse>()
                            .withOperationName("GetPublicKeyCertificate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPublicKeyCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPublicKeyCertificateRequest));
            CompletableFuture<GetPublicKeyCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports keys and public key certificates into Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography simplifies main or root key exchange process by eliminating the need of
     * a paper-based key exchange process. It takes a modern and secure approach based of the ANSI X9 TR-34 key exchange
     * standard.
     * </p>
     * <p>
     * You can use <code>ImportKey</code> to import main or root keys such as KEK (Key Encryption Key) using asymmetric
     * key exchange technique following the ANSI X9 TR-34 standard. The ANSI X9 TR-34 standard uses asymmetric keys to
     * establishes bi-directional trust between the two parties exchanging keys.
     * </p>
     * <p>
     * After you have imported a main or root key, you can import working keys to perform various cryptographic
     * operations within Amazon Web Services Payment Cryptography using the ANSI X9 TR-31 symmetric key exchange
     * standard as mandated by PCI PIN.
     * </p>
     * <p>
     * You can also import a <i>root public key certificate</i>, a self-signed certificate used to sign other public key
     * certificates, or a <i>trusted public key certificate</i> under an already established root public key
     * certificate.
     * </p>
     * <p>
     * <b>To import a public root key certificate</b>
     * </p>
     * <p>
     * Using this operation, you can import the public component (in PEM cerificate format) of your private root key.
     * You can use the imported public root key certificate for digital signatures, for example signing wrapping key or
     * signing key in TR-34, within your Amazon Web Services Payment Cryptography account.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: <code>RootCertificatePublicKey</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyClass</code>: <code>PUBLIC_KEY</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyModesOfUse</code>: <code>Verify</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyUsage</code>: <code>TR31_S0_ASYMMETRIC_KEY_FOR_DIGITAL_SIGNATURE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The certificate authority used to sign the root public key certificate.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import a trusted public key certificate</b>
     * </p>
     * <p>
     * The root public key certificate must be in place and operational before you import a trusted public key
     * certificate. Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: <code>TrustedCertificatePublicKey</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: <code>KeyArn</code> of the
     * <code>RootCertificatePublicKey</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyModesOfUse</code> and <code>KeyUsage</code>: Corresponding to the cryptographic operations such as wrap,
     * sign, or encrypt that you will allow the trusted public key certificate to perform.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The certificate authority used to sign the trusted public key certificate.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Import main keys</b>
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography uses TR-34 asymmetric key exchange standard to import main keys such as
     * KEK. In TR-34 terminology, the sending party of the key is called Key Distribution Host (KDH) and the receiving
     * party of the key is called Key Receiving Host (KRH). During the key import process, KDH is the user who initiates
     * the key import and KRH is Amazon Web Services Payment Cryptography who receives the key. Before initiating TR-34
     * key import, you must obtain an import token by calling <a>GetParametersForImport</a>. This operation also returns
     * the wrapping key certificate that KDH uses wrap key under import to generate a TR-34 wrapped key block. The
     * import token expires after 7 days.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>KeyArn</code> of the certificate chain that will
     * sign the signing key certificate and should exist within Amazon Web Services Payment Cryptography before
     * initiating TR-34 key import. If it does not exist, you can import it by calling by calling <code>ImportKey</code>
     * for <code>RootCertificatePublicKey</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ImportToken</code>: Obtained from KRH by calling <a>GetParametersForImport</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappedKeyBlock</code>: The TR-34 wrapped key block from KDH. It contains the KDH key under import, wrapped
     * with KRH provided wrapping key certificate and signed by the KDH private signing key. This TR-34 key block is
     * generated by the KDH Hardware Security Module (HSM) outside of Amazon Web Services Payment Cryptography.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SigningKeyCertificate</code>: The public component of the private key that signed the KDH TR-34 wrapped key
     * block. In PEM certificate format.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * TR-34 is intended primarily to exchange 3DES keys. Your ability to export AES-128 and larger AES keys may be
     * dependent on your source system.
     * </p>
     * </note>
     * <p>
     * <b>Import working keys</b>
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography uses TR-31 symmetric key exchange standard to import working keys. A KEK
     * must be established within Amazon Web Services Payment Cryptography by using TR-34 key import. To initiate a
     * TR-31 key import, set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>WrappedKeyBlock</code>: The key under import and encrypted using KEK. The TR-31 key block generated by your
     * HSM outside of Amazon Web Services Payment Cryptography.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappingKeyIdentifier</code>: The <code>KeyArn</code> of the KEK that Amazon Web Services Payment
     * Cryptography uses to decrypt or unwrap the key under import.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ExportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetParametersForImport</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param importKeyRequest
     * @return A Java Future containing the result of the ImportKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ImportKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ImportKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportKeyResponse> importKey(ImportKeyRequest importKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ImportKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportKeyRequest, ImportKeyResponse>().withOperationName("ImportKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ImportKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(importKeyRequest));
            CompletableFuture<ImportKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the aliases for all keys in the caller's Amazon Web Services account and Amazon Web Services Region. You
     * can filter the list of aliases. For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-managealias.html">Using aliases</a>
     * in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the aliases.
     * When the response contains only a subset of aliases, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>ListAliases</code> request to get more aliases. When you receive a response with no
     * NextToken (or an empty or null value), that means there are no more aliases to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listAliasesRequest
     * @return A Java Future containing the result of the ListAliases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ListAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAliasesResponse> listAliases(ListAliasesRequest listAliasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAliasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAliases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAliasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAliasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAliasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAliasesRequest, ListAliasesResponse>()
                            .withOperationName("ListAliases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAliasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAliasesRequest));
            CompletableFuture<ListAliasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the keys in the caller's Amazon Web Services account and Amazon Web Services Region. You can filter the
     * list of keys.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the keys. When
     * the response contains only a subset of keys, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListKeys</code> request to get more keys. When you receive a response with no NextToken (or an
     * empty or null value), that means there are no more keys to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listKeysRequest
     * @return A Java Future containing the result of the ListKeys operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ListKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListKeys" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListKeysResponse> listKeys(ListKeysRequest listKeysRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKeysRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKeys");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListKeysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKeysRequest, ListKeysResponse>().withOperationName("ListKeys")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKeysRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listKeysRequest));
            CompletableFuture<ListKeysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for an Amazon Web Services resource.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTagsForResource</code> request to get more tags. When you receive a response with no
     * NextToken (or an empty or null value), that means there are no more tags to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UntagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a scheduled key deletion during the waiting period. Use this operation to restore a <code>Key</code> that
     * is scheduled for deletion.
     * </p>
     * <p>
     * During the waiting period, the <code>KeyState</code> is <code>DELETE_PENDING</code> and
     * <code>deletePendingTimestamp</code> contains the date and time after which the <code>Key</code> will be deleted.
     * After <code>Key</code> is restored, the <code>KeyState</code> is <code>CREATE_COMPLETE</code>, and the value for
     * <code>deletePendingTimestamp</code> is removed.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartKeyUsage</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreKeyRequest
     * @return A Java Future containing the result of the RestoreKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.RestoreKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/RestoreKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreKeyResponse> restoreKey(RestoreKeyRequest restoreKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RestoreKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreKeyRequest, RestoreKeyResponse>().withOperationName("RestoreKey")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestoreKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restoreKeyRequest));
            CompletableFuture<RestoreKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables an Amazon Web Services Payment Cryptography key, which makes it active for cryptographic operations
     * within Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StopKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startKeyUsageRequest
     * @return A Java Future containing the result of the StartKeyUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.StartKeyUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/StartKeyUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartKeyUsageResponse> startKeyUsage(StartKeyUsageRequest startKeyUsageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startKeyUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startKeyUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartKeyUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartKeyUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartKeyUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartKeyUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartKeyUsageRequest, StartKeyUsageResponse>()
                            .withOperationName("StartKeyUsage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartKeyUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startKeyUsageRequest));
            CompletableFuture<StartKeyUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables an Amazon Web Services Payment Cryptography key, which makes it inactive within Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * You can use this operation instead of <a>DeleteKey</a> to deactivate a key. You can enable the key in the future
     * by calling <a>StartKeyUsage</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param stopKeyUsageRequest
     * @return A Java Future containing the result of the StopKeyUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.StopKeyUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/StopKeyUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopKeyUsageResponse> stopKeyUsage(StopKeyUsageRequest stopKeyUsageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopKeyUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopKeyUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopKeyUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopKeyUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopKeyUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopKeyUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopKeyUsageRequest, StopKeyUsageResponse>()
                            .withOperationName("StopKeyUsage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopKeyUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopKeyUsageRequest));
            CompletableFuture<StopKeyUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or edits tags on an Amazon Web Services Payment Cryptography key.
     * </p>
     * <note>
     * <p>
     * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
     * </p>
     * </note>
     * <p>
     * Each tag consists of a tag key and a tag value, both of which are case-sensitive strings. The tag value can be an
     * empty (null) string. To add a tag, specify a new tag key and a tag value. To edit a tag, specify an existing tag
     * key and a new tag value. You can also add tags to an Amazon Web Services Payment Cryptography key when you create
     * it with <a>CreateKey</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ListTagsForResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UntagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException This request would cause a service quota to be exceeded.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a tag from an Amazon Web Services Payment Cryptography key.
     * </p>
     * <note>
     * <p>
     * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
     * </p>
     * </note>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ListTagsForResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates an existing Amazon Web Services Payment Cryptography alias with a different key. Each alias is
     * associated with only one Amazon Web Services Payment Cryptography key at a time, although a key can have multiple
     * aliases. The alias and the Amazon Web Services Payment Cryptography key must be in the same Amazon Web Services
     * account and Amazon Web Services Region
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateAliasRequest
     * @return A Java Future containing the result of the UpdateAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ValidationException The request was denied due to an invalid request error.</li>
     *         <li>ConflictException This request can cause an inconsistent state for the resource.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The request was denied due to an invalid resource error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PaymentCryptographyException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample PaymentCryptographyAsyncClient.UpdateAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/UpdateAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAliasResponse> updateAlias(UpdateAliasRequest updateAliasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAliasRequest, UpdateAliasResponse>()
                            .withOperationName("UpdateAlias").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAliasRequest));
            CompletableFuture<UpdateAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PaymentCryptographyServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PaymentCryptographyException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        PaymentCryptographyServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = PaymentCryptographyServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
