/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.paymentcryptography.internal.PaymentCryptographyServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.paymentcryptography.model.AccessDeniedException;
import software.amazon.awssdk.services.paymentcryptography.model.ConflictException;
import software.amazon.awssdk.services.paymentcryptography.model.CreateAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.CreateAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.CreateKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.CreateKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.DeleteKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ExportKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ExportKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForExportRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForExportResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForImportRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetParametersForImportResponse;
import software.amazon.awssdk.services.paymentcryptography.model.GetPublicKeyCertificateRequest;
import software.amazon.awssdk.services.paymentcryptography.model.GetPublicKeyCertificateResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ImportKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ImportKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.InternalServerException;
import software.amazon.awssdk.services.paymentcryptography.model.ListAliasesRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListAliasesResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ListKeysRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListKeysResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.PaymentCryptographyException;
import software.amazon.awssdk.services.paymentcryptography.model.ResourceNotFoundException;
import software.amazon.awssdk.services.paymentcryptography.model.RestoreKeyRequest;
import software.amazon.awssdk.services.paymentcryptography.model.RestoreKeyResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.paymentcryptography.model.ServiceUnavailableException;
import software.amazon.awssdk.services.paymentcryptography.model.StartKeyUsageRequest;
import software.amazon.awssdk.services.paymentcryptography.model.StartKeyUsageResponse;
import software.amazon.awssdk.services.paymentcryptography.model.StopKeyUsageRequest;
import software.amazon.awssdk.services.paymentcryptography.model.StopKeyUsageResponse;
import software.amazon.awssdk.services.paymentcryptography.model.TagResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.TagResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ThrottlingException;
import software.amazon.awssdk.services.paymentcryptography.model.UntagResourceRequest;
import software.amazon.awssdk.services.paymentcryptography.model.UntagResourceResponse;
import software.amazon.awssdk.services.paymentcryptography.model.UpdateAliasRequest;
import software.amazon.awssdk.services.paymentcryptography.model.UpdateAliasResponse;
import software.amazon.awssdk.services.paymentcryptography.model.ValidationException;
import software.amazon.awssdk.services.paymentcryptography.transform.CreateAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.CreateKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.DeleteAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.DeleteKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ExportKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetAliasRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetParametersForExportRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetParametersForImportRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.GetPublicKeyCertificateRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ImportKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListAliasesRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListKeysRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.RestoreKeyRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.StartKeyUsageRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.StopKeyUsageRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.paymentcryptography.transform.UpdateAliasRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PaymentCryptographyClient}.
 *
 * @see PaymentCryptographyClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPaymentCryptographyClient implements PaymentCryptographyClient {
    private static final Logger log = Logger.loggerFor(DefaultPaymentCryptographyClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPaymentCryptographyClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an <i>alias</i>, or a friendly name, for an Amazon Web Services Payment Cryptography key. You can use an
     * alias to identify a key in the console and when you call cryptographic operations such as <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_EncryptData.html"
     * >EncryptData</a> or <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/DataAPIReference/API_DecryptData.html"
     * >DecryptData</a>.
     * </p>
     * <p>
     * You can associate the alias with any key in the same Amazon Web Services Region. Each alias is associated with
     * only one key at a time, but a key can have multiple aliases. You can't create an alias without a key. The alias
     * must be unique in the account and Amazon Web Services Region, but you can create another alias with the same name
     * in a different Amazon Web Services Region.
     * </p>
     * <p>
     * To change the key that's associated with the alias, call <a>UpdateAlias</a>. To delete the alias, call
     * <a>DeleteAlias</a>. These operations don't affect the underlying key. To get the alias that you created, call
     * <a>ListAliases</a>.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createAliasRequest
     * @return Result of the CreateAlias operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.CreateAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/CreateAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAliasResponse createAlias(CreateAliasRequest createAliasRequest) throws ServiceQuotaExceededException,
            ServiceUnavailableException, ValidationException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlias");

            return clientHandler.execute(new ClientExecutionParams<CreateAliasRequest, CreateAliasResponse>()
                    .withOperationName("CreateAlias").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createAliasRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon Web Services Payment Cryptography key, a logical representation of a cryptographic key, that is
     * unique in your account and Amazon Web Services Region. You use keys for cryptographic functions such as
     * encryption and decryption.
     * </p>
     * <p>
     * In addition to the key material used in cryptographic operations, an Amazon Web Services Payment Cryptography key
     * includes metadata such as the key ARN, key usage, key origin, creation date, description, and key state.
     * </p>
     * <p>
     * When you create a key, you specify both immutable and mutable data about the key. The immutable data contains key
     * attributes that define the scope and cryptographic operations that you can perform using the key, for example key
     * class (example: <code>SYMMETRIC_KEY</code>), key algorithm (example: <code>TDES_2KEY</code>), key usage (example:
     * <code>TR31_P0_PIN_ENCRYPTION_KEY</code>) and key modes of use (example: <code>Encrypt</code>). For information
     * about valid combinations of key attributes, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-validattributes.html">Understanding
     * key attributes</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>. The mutable data contained
     * within a key includes usage timestamp and key deletion timestamp and can be modified after creation.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography binds key attributes to keys using key blocks when you store or export
     * them. Amazon Web Services Payment Cryptography stores the key contents wrapped and never stores or transmits them
     * in the clear.
     * </p>
     * <p>
     * <b>Cross-account use</b>: This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListKeys</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createKeyRequest
     * @return Result of the CreateKey operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.CreateKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/CreateKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateKeyResponse createKey(CreateKeyRequest createKeyRequest) throws ServiceQuotaExceededException,
            ServiceUnavailableException, ValidationException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKey");

            return clientHandler.execute(new ClientExecutionParams<CreateKeyRequest, CreateKeyResponse>()
                    .withOperationName("CreateKey").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createKeyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the alias, but doesn't affect the underlying key.
     * </p>
     * <p>
     * Each key can have multiple aliases. To get the aliases of all keys, use the <a>ListAliases</a> operation. To
     * change the alias of a key, first use <a>DeleteAlias</a> to delete the current alias and then use
     * <a>CreateAlias</a> to create a new alias. To associate an existing alias with a different key, call
     * <a>UpdateAlias</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAliasRequest
     * @return Result of the DeleteAlias operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.DeleteAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/DeleteAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAliasResponse deleteAlias(DeleteAliasRequest deleteAliasRequest) throws ServiceUnavailableException,
            ValidationException, ConflictException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlias");

            return clientHandler.execute(new ClientExecutionParams<DeleteAliasRequest, DeleteAliasResponse>()
                    .withOperationName("DeleteAlias").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteAliasRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the key material and metadata associated with Amazon Web Services Payment Cryptography key.
     * </p>
     * <p>
     * Key deletion is irreversible. After a key is deleted, you can't perform cryptographic operations using the key.
     * For example, you can't decrypt data that was encrypted by a deleted Amazon Web Services Payment Cryptography key,
     * and the data may become unrecoverable. Because key deletion is destructive, Amazon Web Services Payment
     * Cryptography has a safety mechanism to prevent accidental deletion of a key. When you call this operation, Amazon
     * Web Services Payment Cryptography disables the specified key but doesn't delete it until after a waiting period
     * set using <code>DeleteKeyInDays</code>. The default waiting period is 7 days. During the waiting period, the
     * <code>KeyState</code> is <code>DELETE_PENDING</code>. After the key is deleted, the <code>KeyState</code> is
     * <code>DELETE_COMPLETE</code>.
     * </p>
     * <p>
     * You should delete a key only when you are sure that you don't need to use it anymore and no other parties are
     * utilizing this key. If you aren't sure, consider deactivating it instead by calling <a>StopKeyUsage</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RestoreKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartKeyUsage</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteKeyRequest
     * @return Result of the DeleteKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.DeleteKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/DeleteKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteKeyResponse deleteKey(DeleteKeyRequest deleteKeyRequest) throws ServiceUnavailableException,
            ValidationException, ConflictException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKey");

            return clientHandler.execute(new ClientExecutionParams<DeleteKeyRequest, DeleteKeyResponse>()
                    .withOperationName("DeleteKey").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteKeyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports a key from Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography simplifies key exchange by replacing the existing paper-based approach
     * with a modern electronic approach. With <code>ExportKey</code> you can export symmetric keys using either
     * symmetric and asymmetric key exchange mechanisms. Using this operation, you can share your Amazon Web Services
     * Payment Cryptography generated keys with other service partners to perform cryptographic operations outside of
     * Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * For symmetric key exchange, Amazon Web Services Payment Cryptography uses the ANSI X9 TR-31 norm in accordance
     * with PCI PIN guidelines. And for asymmetric key exchange, Amazon Web Services Payment Cryptography supports ANSI
     * X9 TR-34 norm . Asymmetric key exchange methods are typically used to establish bi-directional trust between the
     * two parties exhanging keys and are used for initial key exchange such as Key Encryption Key (KEK). After which
     * you can export working keys using symmetric method to perform various cryptographic operations within Amazon Web
     * Services Payment Cryptography.
     * </p>
     * <p>
     * The TR-34 norm is intended for exchanging 3DES keys only and keys are imported in a WrappedKeyBlock format. Key
     * attributes (such as KeyUsage, KeyAlgorithm, KeyModesOfUse, Exportability) are contained within the key block.
     * </p>
     * <p>
     * You can also use <code>ExportKey</code> functionality to generate and export an IPEK (Initial Pin Encryption Key)
     * from Amazon Web Services Payment Cryptography using either TR-31 or TR-34 export key exchange. IPEK is generated
     * from BDK (Base Derivation Key) and <code>ExportDukptInitialKey</code> attribute KSN (<code>KeySerialNumber</code>
     * ). The generated IPEK does not persist within Amazon Web Services Payment Cryptography and has to be re-generated
     * each time during export.
     * </p>
     * <p>
     * <b>To export KEK or IPEK using TR-34</b>
     * </p>
     * <p>
     * Using this operation, you can export initial key using TR-34 asymmetric key exchange. You can only export KEK
     * generated within Amazon Web Services Payment Cryptography. In TR-34 terminology, the sending party of the key is
     * called Key Distribution Host (KDH) and the receiving party of the key is called Key Receiving Device (KRD).
     * During key export process, KDH is Amazon Web Services Payment Cryptography which initiates key export and KRD is
     * the user receiving the key.
     * </p>
     * <p>
     * To initiate TR-34 key export, the KRD must obtain an export token by calling <a>GetParametersForExport</a>. This
     * operation also generates a key pair for the purpose of key export, signs the key and returns back the signing
     * public key certificate (also known as KDH signing certificate) and root certificate chain. The KDH uses the
     * private key to sign the the export payload and the signing public key certificate is provided to KRD to verify
     * the signature. The KRD can import the root certificate into its Hardware Security Module (HSM), as required. The
     * export token and the associated KDH signing certificate expires after 7 days.
     * </p>
     * <p>
     * Next the KRD generates a key pair for the the purpose of encrypting the KDH key and provides the public key
     * cerificate (also known as KRD wrapping certificate) back to KDH. The KRD will also import the root cerificate
     * chain into Amazon Web Services Payment Cryptography by calling <a>ImportKey</a> for
     * <code>RootCertificatePublicKey</code>. The KDH, Amazon Web Services Payment Cryptography, will use the KRD
     * wrapping cerificate to encrypt (wrap) the key under export and signs it with signing private key to generate a
     * TR-34 WrappedKeyBlock. For more information on TR-34 key export, see section <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-export.html">Exporting symmetric
     * keys</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ExportAttributes</code>: Specify export attributes in case of IPEK export. This parameter is optional for
     * KEK export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ExportKeyIdentifier</code>: The <code>KeyARN</code> of the KEK or BDK (in case of IPEK) under export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr34KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>KeyARN</code> of the certificate chain that
     * signed the KRD wrapping key certificate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ExportToken</code>: Obtained from KDH by calling <a>GetParametersForImport</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappingKeyCertificate</code>: The public key certificate in PEM format (base64 encoded) of the KRD
     * wrapping key Amazon Web Services Payment Cryptography uses for encryption of the TR-34 export payload. This
     * certificate must be signed by the root certificate (CertificateAuthorityPublicKeyIdentifier) imported into Amazon
     * Web Services Payment Cryptography.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When this operation is successful, Amazon Web Services Payment Cryptography returns the KEK or IPEK as a TR-34
     * WrappedKeyBlock.
     * </p>
     * <p>
     * <b>To export WK (Working Key) or IPEK using TR-31</b>
     * </p>
     * <p>
     * Using this operation, you can export working keys or IPEK using TR-31 symmetric key exchange. In TR-31, you must
     * use an initial key such as KEK to encrypt or wrap the key under export. To establish a KEK, you can use
     * <a>CreateKey</a> or <a>ImportKey</a>.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ExportAttributes</code>: Specify export attributes in case of IPEK export. This parameter is optional for
     * KEK export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ExportKeyIdentifier</code>: The <code>KeyARN</code> of the KEK or BDK (in case of IPEK) under export.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr31KeyBlock</code> parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When this operation is successful, Amazon Web Services Payment Cryptography returns the WK or IPEK as a TR-31
     * WrappedKeyBlock.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GetParametersForExport</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param exportKeyRequest
     * @return Result of the ExportKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.ExportKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ExportKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExportKeyResponse exportKey(ExportKeyRequest exportKeyRequest) throws ServiceUnavailableException,
            ValidationException, ConflictException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExportKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportKey");

            return clientHandler.execute(new ClientExecutionParams<ExportKeyRequest, ExportKeyResponse>()
                    .withOperationName("ExportKey").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(exportKeyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExportKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the Amazon Web Services Payment Cryptography key associated with the alias.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAliasRequest
     * @return Result of the GetAlias operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.GetAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetAlias" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAliasResponse getAlias(GetAliasRequest getAliasRequest) throws ServiceUnavailableException, ValidationException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAlias");

            return clientHandler.execute(new ClientExecutionParams<GetAliasRequest, GetAliasResponse>()
                    .withOperationName("GetAlias").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getAliasRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the key material for an Amazon Web Services Payment Cryptography key, including the immutable and mutable
     * data specified when the key was created.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListKeys</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getKeyRequest
     * @return Result of the GetKey operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.GetKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetKeyResponse getKey(GetKeyRequest getKeyRequest) throws ServiceUnavailableException, ValidationException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKey");

            return clientHandler.execute(new ClientExecutionParams<GetKeyRequest, GetKeyResponse>().withOperationName("GetKey")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getKeyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the export token and the signing key certificate to initiate a TR-34 key export from Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * The signing key certificate signs the wrapped key under export within the TR-34 key payload. The export token and
     * signing key certificate must be in place and operational before calling <a>ExportKey</a>. The export token
     * expires in 7 days. You can use the same export token to export multiple keys from your service account.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ExportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetParametersForImport</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getParametersForExportRequest
     * @return Result of the GetParametersForExport operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.GetParametersForExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetParametersForExport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetParametersForExportResponse getParametersForExport(GetParametersForExportRequest getParametersForExportRequest)
            throws ServiceQuotaExceededException, ServiceUnavailableException, ValidationException, ConflictException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetParametersForExportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetParametersForExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParametersForExportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParametersForExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParametersForExport");

            return clientHandler
                    .execute(new ClientExecutionParams<GetParametersForExportRequest, GetParametersForExportResponse>()
                            .withOperationName("GetParametersForExport").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getParametersForExportRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetParametersForExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the import token and the wrapping key certificate in PEM format (base64 encoded) to initiate a TR-34
     * WrappedKeyBlock.
     * </p>
     * <p>
     * The wrapping key certificate wraps the key under import. The import token and wrapping key certificate must be in
     * place and operational before calling <a>ImportKey</a>. The import token expires in 7 days. You can use the same
     * import token to import multiple keys into your service account.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>GetParametersForExport</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ImportKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getParametersForImportRequest
     * @return Result of the GetParametersForImport operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.GetParametersForImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetParametersForImport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetParametersForImportResponse getParametersForImport(GetParametersForImportRequest getParametersForImportRequest)
            throws ServiceQuotaExceededException, ServiceUnavailableException, ValidationException, ConflictException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetParametersForImportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetParametersForImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParametersForImportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParametersForImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParametersForImport");

            return clientHandler
                    .execute(new ClientExecutionParams<GetParametersForImportRequest, GetParametersForImportResponse>()
                            .withOperationName("GetParametersForImport").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getParametersForImportRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetParametersForImportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the public key certificate of the asymmetric key pair that exists within Amazon Web Services Payment
     * Cryptography.
     * </p>
     * <p>
     * Unlike the private key of an asymmetric key, which never leaves Amazon Web Services Payment Cryptography
     * unencrypted, callers with <code>GetPublicKeyCertificate</code> permission can download the public key certificate
     * of the asymmetric key. You can share the public key certificate to allow others to encrypt messages and verify
     * signatures outside of Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     *
     * @param getPublicKeyCertificateRequest
     * @return Result of the GetPublicKeyCertificate operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.GetPublicKeyCertificate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/GetPublicKeyCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPublicKeyCertificateResponse getPublicKeyCertificate(GetPublicKeyCertificateRequest getPublicKeyCertificateRequest)
            throws ServiceUnavailableException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPublicKeyCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPublicKeyCertificateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPublicKeyCertificateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPublicKeyCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPublicKeyCertificate");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPublicKeyCertificateRequest, GetPublicKeyCertificateResponse>()
                            .withOperationName("GetPublicKeyCertificate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getPublicKeyCertificateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPublicKeyCertificateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Imports symmetric keys and public key certificates in PEM format (base64 encoded) into Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography simplifies key exchange by replacing the existing paper-based approach
     * with a modern electronic approach. With <code>ImportKey</code> you can import symmetric keys using either
     * symmetric and asymmetric key exchange mechanisms.
     * </p>
     * <p>
     * For symmetric key exchange, Amazon Web Services Payment Cryptography uses the ANSI X9 TR-31 norm in accordance
     * with PCI PIN guidelines. And for asymmetric key exchange, Amazon Web Services Payment Cryptography supports ANSI
     * X9 TR-34 norm . Asymmetric key exchange methods are typically used to establish bi-directional trust between the
     * two parties exhanging keys and are used for initial key exchange such as Key Encryption Key (KEK) or Zone Master
     * Key (ZMK). After which you can import working keys using symmetric method to perform various cryptographic
     * operations within Amazon Web Services Payment Cryptography.
     * </p>
     * <p>
     * The TR-34 norm is intended for exchanging 3DES keys only and keys are imported in a WrappedKeyBlock format. Key
     * attributes (such as KeyUsage, KeyAlgorithm, KeyModesOfUse, Exportability) are contained within the key block.
     * </p>
     * <p>
     * You can also import a <i>root public key certificate</i>, used to sign other public key certificates, or a
     * <i>trusted public key certificate</i> under an already established root public key certificate.
     * </p>
     * <p>
     * <b>To import a public root key certificate</b>
     * </p>
     * <p>
     * You can also import a <i>root public key certificate</i>, used to sign other public key certificates, or a
     * <i>trusted public key certificate</i> under an already established root public key certificate.
     * </p>
     * <p>
     * <b>To import a public root key certificate</b>
     * </p>
     * <p>
     * Using this operation, you can import the public component (in PEM cerificate format) of your private root key.
     * You can use the imported public root key certificate for digital signatures, for example signing wrapping key or
     * signing key in TR-34, within your Amazon Web Services Payment Cryptography account.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: <code>RootCertificatePublicKey</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyClass</code>: <code>PUBLIC_KEY</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyModesOfUse</code>: <code>Verify</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyUsage</code>: <code>TR31_S0_ASYMMETRIC_KEY_FOR_DIGITAL_SIGNATURE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The public key certificate in PEM format (base64 encoded) of the private root
     * key under import.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import a trusted public key certificate</b>
     * </p>
     * <p>
     * The root public key certificate must be in place and operational before you import a trusted public key
     * certificate. Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: <code>TrustedCertificatePublicKey</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: <code>KeyArn</code> of the
     * <code>RootCertificatePublicKey</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyModesOfUse</code> and <code>KeyUsage</code>: Corresponding to the cryptographic operations such as wrap,
     * sign, or encrypt that you will allow the trusted public key certificate to perform.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PublicKeyCertificate</code>: The trusted public key certificate in PEM format (base64 encoded) under
     * import.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import KEK or ZMK using TR-34</b>
     * </p>
     * <p>
     * Using this operation, you can import initial key using TR-34 asymmetric key exchange. In TR-34 terminology, the
     * sending party of the key is called Key Distribution Host (KDH) and the receiving party of the key is called Key
     * Receiving Device (KRD). During the key import process, KDH is the user who initiates the key import and KRD is
     * Amazon Web Services Payment Cryptography who receives the key.
     * </p>
     * <p>
     * To initiate TR-34 key import, the KDH must obtain an import token by calling <a>GetParametersForImport</a>. This
     * operation generates an encryption keypair for the purpose of key import, signs the key and returns back the
     * wrapping key certificate (also known as KRD wrapping certificate) and the root certificate chain. The KDH must
     * trust and install the KRD wrapping certificate on its HSM and use it to encrypt (wrap) the KDH key during TR-34
     * WrappedKeyBlock generation. The import token and associated KRD wrapping certificate expires after 7 days.
     * </p>
     * <p>
     * Next the KDH generates a key pair for the purpose of signing the encrypted KDH key and provides the public
     * certificate of the signing key to Amazon Web Services Payment Cryptography. The KDH will also need to import the
     * root certificate chain of the KDH signing certificate by calling <code>ImportKey</code> for
     * <code>RootCertificatePublicKey</code>. For more information on TR-34 key import, see section <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-import.html">Importing symmetric
     * keys</a> in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * Set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr34KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CertificateAuthorityPublicKeyIdentifier</code>: The <code>KeyARN</code> of the certificate chain that
     * signed the KDH signing key certificate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ImportToken</code>: Obtained from KRD by calling <a>GetParametersForImport</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappedKeyBlock</code>: The TR-34 wrapped key material from KDH. It contains the KDH key under import,
     * wrapped with KRD wrapping certificate and signed by KDH signing private key. This TR-34 key block is typically
     * generated by the KDH Hardware Security Module (HSM) outside of Amazon Web Services Payment Cryptography.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SigningKeyCertificate</code>: The public key certificate in PEM format (base64 encoded) of the KDH signing
     * key generated under the root certificate (CertificateAuthorityPublicKeyIdentifier) imported in Amazon Web
     * Services Payment Cryptography.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>To import WK (Working Key) using TR-31</b>
     * </p>
     * <p>
     * Amazon Web Services Payment Cryptography uses TR-31 symmetric key exchange norm to import working keys. A KEK
     * must be established within Amazon Web Services Payment Cryptography by using TR-34 key import or by using
     * <a>CreateKey</a>. To initiate a TR-31 key import, set the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KeyMaterial</code>: Use <code>Tr31KeyBlock</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappedKeyBlock</code>: The TR-31 wrapped key material. It contains the key under import, encrypted using
     * KEK. The TR-31 key block is typically generated by a HSM outside of Amazon Web Services Payment Cryptography.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WrappingKeyIdentifier</code>: The <code>KeyArn</code> of the KEK that Amazon Web Services Payment
     * Cryptography uses to decrypt or unwrap the key under import.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ExportKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetParametersForImport</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param importKeyRequest
     * @return Result of the ImportKey operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.ImportKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ImportKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ImportKeyResponse importKey(ImportKeyRequest importKeyRequest) throws ServiceQuotaExceededException,
            ServiceUnavailableException, ValidationException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ImportKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportKey");

            return clientHandler.execute(new ClientExecutionParams<ImportKeyRequest, ImportKeyResponse>()
                    .withOperationName("ImportKey").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(importKeyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ImportKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the aliases for all keys in the caller's Amazon Web Services account and Amazon Web Services Region. You
     * can filter the list of aliases. For more information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-managealias.html">Using aliases</a>
     * in the <i>Amazon Web Services Payment Cryptography User Guide</i>.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the aliases.
     * When the response contains only a subset of aliases, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>ListAliases</code> request to get more aliases. When you receive a response with no
     * NextToken (or an empty or null value), that means there are no more aliases to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listAliasesRequest
     * @return Result of the ListAliases operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.ListAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAliasesResponse listAliases(ListAliasesRequest listAliasesRequest) throws ServiceUnavailableException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAliasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAliasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAliasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAliases");

            return clientHandler.execute(new ClientExecutionParams<ListAliasesRequest, ListAliasesResponse>()
                    .withOperationName("ListAliases").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listAliasesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the keys in the caller's Amazon Web Services account and Amazon Web Services Region. You can filter the
     * list of keys.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the keys. When
     * the response contains only a subset of keys, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListKeys</code> request to get more keys. When you receive a response with no NextToken (or an
     * empty or null value), that means there are no more keys to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetKey</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listKeysRequest
     * @return Result of the ListKeys operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.ListKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListKeys" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListKeysResponse listKeys(ListKeysRequest listKeysRequest) throws ServiceUnavailableException, ValidationException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListKeysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKeysRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKeys");

            return clientHandler.execute(new ClientExecutionParams<ListKeysRequest, ListKeysResponse>()
                    .withOperationName("ListKeys").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listKeysRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListKeysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for an Amazon Web Services resource.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTagsForResource</code> request to get more tags. When you receive a response with no
     * NextToken (or an empty or null value), that means there are no more tags to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UntagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ServiceUnavailableException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, InternalServerException, AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels a scheduled key deletion during the waiting period. Use this operation to restore a <code>Key</code> that
     * is scheduled for deletion.
     * </p>
     * <p>
     * During the waiting period, the <code>KeyState</code> is <code>DELETE_PENDING</code> and
     * <code>deletePendingTimestamp</code> contains the date and time after which the <code>Key</code> will be deleted.
     * After <code>Key</code> is restored, the <code>KeyState</code> is <code>CREATE_COMPLETE</code>, and the value for
     * <code>deletePendingTimestamp</code> is removed.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartKeyUsage</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreKeyRequest
     * @return Result of the RestoreKey operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.RestoreKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/RestoreKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RestoreKeyResponse restoreKey(RestoreKeyRequest restoreKeyRequest) throws ServiceQuotaExceededException,
            ServiceUnavailableException, ValidationException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RestoreKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreKey");

            return clientHandler.execute(new ClientExecutionParams<RestoreKeyRequest, RestoreKeyResponse>()
                    .withOperationName("RestoreKey").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(restoreKeyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RestoreKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables an Amazon Web Services Payment Cryptography key, which makes it active for cryptographic operations
     * within Amazon Web Services Payment Cryptography
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StopKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startKeyUsageRequest
     * @return Result of the StartKeyUsage operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.StartKeyUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/StartKeyUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartKeyUsageResponse startKeyUsage(StartKeyUsageRequest startKeyUsageRequest) throws ServiceQuotaExceededException,
            ServiceUnavailableException, ValidationException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartKeyUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartKeyUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startKeyUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startKeyUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartKeyUsage");

            return clientHandler.execute(new ClientExecutionParams<StartKeyUsageRequest, StartKeyUsageResponse>()
                    .withOperationName("StartKeyUsage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startKeyUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartKeyUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables an Amazon Web Services Payment Cryptography key, which makes it inactive within Amazon Web Services
     * Payment Cryptography.
     * </p>
     * <p>
     * You can use this operation instead of <a>DeleteKey</a> to deactivate a key. You can enable the key in the future
     * by calling <a>StartKeyUsage</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DeleteKey</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartKeyUsage</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param stopKeyUsageRequest
     * @return Result of the StopKeyUsage operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.StopKeyUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/StopKeyUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopKeyUsageResponse stopKeyUsage(StopKeyUsageRequest stopKeyUsageRequest) throws ServiceQuotaExceededException,
            ServiceUnavailableException, ValidationException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopKeyUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopKeyUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopKeyUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopKeyUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopKeyUsage");

            return clientHandler.execute(new ClientExecutionParams<StopKeyUsageRequest, StopKeyUsageResponse>()
                    .withOperationName("StopKeyUsage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopKeyUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopKeyUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or edits tags on an Amazon Web Services Payment Cryptography key.
     * </p>
     * <note>
     * <p>
     * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
     * </p>
     * </note>
     * <p>
     * Each tag consists of a tag key and a tag value, both of which are case-sensitive strings. The tag value can be an
     * empty (null) string. To add a tag, specify a new tag key and a tag value. To edit a tag, specify an existing tag
     * key and a new tag value. You can also add tags to an Amazon Web Services Payment Cryptography key when you create
     * it with <a>CreateKey</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ListTagsForResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UntagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This request would cause a service quota to be exceeded.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceQuotaExceededException,
            ServiceUnavailableException, ValidationException, ConflictException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a tag from an Amazon Web Services Payment Cryptography key.
     * </p>
     * <note>
     * <p>
     * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
     * </p>
     * </note>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>ListTagsForResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>TagResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ServiceUnavailableException,
            ValidationException, ConflictException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates an existing Amazon Web Services Payment Cryptography alias with a different key. Each alias is
     * associated with only one Amazon Web Services Payment Cryptography key at a time, although a key can have multiple
     * aliases. The alias and the Amazon Web Services Payment Cryptography key must be in the same Amazon Web Services
     * account and Amazon Web Services Region
     * </p>
     * <p>
     * <b>Cross-account use:</b> This operation can't be used across different Amazon Web Services accounts.
     * </p>
     * <p>
     * <b>Related operations:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateAliasRequest
     * @return Result of the UpdateAlias operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service cannot complete the request.
     * @throws ValidationException
     *         The request was denied due to an invalid request error.
     * @throws ConflictException
     *         This request can cause an inconsistent state for the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request was denied due to an invalid resource error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PaymentCryptographyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PaymentCryptographyClient.UpdateAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/payment-cryptography-2021-09-14/UpdateAlias"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAliasResponse updateAlias(UpdateAliasRequest updateAliasRequest) throws ServiceUnavailableException,
            ValidationException, ConflictException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, PaymentCryptographyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Payment Cryptography");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAlias");

            return clientHandler.execute(new ClientExecutionParams<UpdateAliasRequest, UpdateAliasResponse>()
                    .withOperationName("UpdateAlias").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateAliasRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        PaymentCryptographyServiceClientConfigurationBuilder serviceConfigBuilder = new PaymentCryptographyServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PaymentCryptographyException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final PaymentCryptographyServiceClientConfiguration serviceClientConfiguration() {
        return new PaymentCryptographyServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
