/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameter information for key material export using the asymmetric TR-34 key exchange method.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExportTr34KeyBlock implements SdkPojo, Serializable,
        ToCopyableBuilder<ExportTr34KeyBlock.Builder, ExportTr34KeyBlock> {
    private static final SdkField<String> CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CertificateAuthorityPublicKeyIdentifier")
            .getter(getter(ExportTr34KeyBlock::certificateAuthorityPublicKeyIdentifier))
            .setter(setter(Builder::certificateAuthorityPublicKeyIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("CertificateAuthorityPublicKeyIdentifier").build()).build();

    private static final SdkField<String> EXPORT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExportToken").getter(getter(ExportTr34KeyBlock::exportToken)).setter(setter(Builder::exportToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportToken").build()).build();

    private static final SdkField<String> KEY_BLOCK_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyBlockFormat").getter(getter(ExportTr34KeyBlock::keyBlockFormatAsString))
            .setter(setter(Builder::keyBlockFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyBlockFormat").build()).build();

    private static final SdkField<String> RANDOM_NONCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RandomNonce").getter(getter(ExportTr34KeyBlock::randomNonce)).setter(setter(Builder::randomNonce))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RandomNonce").build()).build();

    private static final SdkField<String> WRAPPING_KEY_CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappingKeyCertificate").getter(getter(ExportTr34KeyBlock::wrappingKeyCertificate))
            .setter(setter(Builder::wrappingKeyCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappingKeyCertificate").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD, EXPORT_TOKEN_FIELD, KEY_BLOCK_FORMAT_FIELD, RANDOM_NONCE_FIELD,
            WRAPPING_KEY_CERTIFICATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String certificateAuthorityPublicKeyIdentifier;

    private final String exportToken;

    private final String keyBlockFormat;

    private final String randomNonce;

    private final String wrappingKeyCertificate;

    private ExportTr34KeyBlock(BuilderImpl builder) {
        this.certificateAuthorityPublicKeyIdentifier = builder.certificateAuthorityPublicKeyIdentifier;
        this.exportToken = builder.exportToken;
        this.keyBlockFormat = builder.keyBlockFormat;
        this.randomNonce = builder.randomNonce;
        this.wrappingKeyCertificate = builder.wrappingKeyCertificate;
    }

    /**
     * <p>
     * The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34 key export.
     * </p>
     * 
     * @return The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34 key
     *         export.
     */
    public final String certificateAuthorityPublicKeyIdentifier() {
        return certificateAuthorityPublicKeyIdentifier;
    }

    /**
     * <p>
     * The export token to initiate key export from Amazon Web Services Payment Cryptography. It also contains the
     * signing key certificate that will sign the wrapped key during TR-34 key block generation. Call
     * <a>GetParametersForExport</a> to receive an export token. It expires after 7 days. You can use the same export
     * token to export multiple keys from the same service account.
     * </p>
     * 
     * @return The export token to initiate key export from Amazon Web Services Payment Cryptography. It also contains
     *         the signing key certificate that will sign the wrapped key during TR-34 key block generation. Call
     *         <a>GetParametersForExport</a> to receive an export token. It expires after 7 days. You can use the same
     *         export token to export multiple keys from the same service account.
     */
    public final String exportToken() {
        return exportToken;
    }

    /**
     * <p>
     * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyBlockFormat}
     * will return {@link Tr34KeyBlockFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyBlockFormatAsString}.
     * </p>
     * 
     * @return The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * @see Tr34KeyBlockFormat
     */
    public final Tr34KeyBlockFormat keyBlockFormat() {
        return Tr34KeyBlockFormat.fromValue(keyBlockFormat);
    }

    /**
     * <p>
     * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyBlockFormat}
     * will return {@link Tr34KeyBlockFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyBlockFormatAsString}.
     * </p>
     * 
     * @return The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * @see Tr34KeyBlockFormat
     */
    public final String keyBlockFormatAsString() {
        return keyBlockFormat;
    }

    /**
     * <p>
     * A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will fail, if a
     * random nonce value is not provided for a TR-34 key block generated using 2 pass.
     * </p>
     * 
     * @return A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will
     *         fail, if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
     */
    public final String randomNonce() {
        return randomNonce;
    }

    /**
     * <p>
     * The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses this
     * certificate to wrap the key under export.
     * </p>
     * 
     * @return The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses
     *         this certificate to wrap the key under export.
     */
    public final String wrappingKeyCertificate() {
        return wrappingKeyCertificate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityPublicKeyIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(exportToken());
        hashCode = 31 * hashCode + Objects.hashCode(keyBlockFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(randomNonce());
        hashCode = 31 * hashCode + Objects.hashCode(wrappingKeyCertificate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExportTr34KeyBlock)) {
            return false;
        }
        ExportTr34KeyBlock other = (ExportTr34KeyBlock) obj;
        return Objects.equals(certificateAuthorityPublicKeyIdentifier(), other.certificateAuthorityPublicKeyIdentifier())
                && Objects.equals(exportToken(), other.exportToken())
                && Objects.equals(keyBlockFormatAsString(), other.keyBlockFormatAsString())
                && Objects.equals(randomNonce(), other.randomNonce())
                && Objects.equals(wrappingKeyCertificate(), other.wrappingKeyCertificate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExportTr34KeyBlock")
                .add("CertificateAuthorityPublicKeyIdentifier", certificateAuthorityPublicKeyIdentifier())
                .add("ExportToken", exportToken()).add("KeyBlockFormat", keyBlockFormatAsString())
                .add("RandomNonce", randomNonce())
                .add("WrappingKeyCertificate", wrappingKeyCertificate() == null ? null : "*** Sensitive Data Redacted ***")
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateAuthorityPublicKeyIdentifier":
            return Optional.ofNullable(clazz.cast(certificateAuthorityPublicKeyIdentifier()));
        case "ExportToken":
            return Optional.ofNullable(clazz.cast(exportToken()));
        case "KeyBlockFormat":
            return Optional.ofNullable(clazz.cast(keyBlockFormatAsString()));
        case "RandomNonce":
            return Optional.ofNullable(clazz.cast(randomNonce()));
        case "WrappingKeyCertificate":
            return Optional.ofNullable(clazz.cast(wrappingKeyCertificate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ExportTr34KeyBlock, T> g) {
        return obj -> g.apply((ExportTr34KeyBlock) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExportTr34KeyBlock> {
        /**
         * <p>
         * The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34 key
         * export.
         * </p>
         * 
         * @param certificateAuthorityPublicKeyIdentifier
         *        The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34
         *        key export.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier);

        /**
         * <p>
         * The export token to initiate key export from Amazon Web Services Payment Cryptography. It also contains the
         * signing key certificate that will sign the wrapped key during TR-34 key block generation. Call
         * <a>GetParametersForExport</a> to receive an export token. It expires after 7 days. You can use the same
         * export token to export multiple keys from the same service account.
         * </p>
         * 
         * @param exportToken
         *        The export token to initiate key export from Amazon Web Services Payment Cryptography. It also
         *        contains the signing key certificate that will sign the wrapped key during TR-34 key block generation.
         *        Call <a>GetParametersForExport</a> to receive an export token. It expires after 7 days. You can use
         *        the same export token to export multiple keys from the same service account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportToken(String exportToken);

        /**
         * <p>
         * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * </p>
         * 
         * @param keyBlockFormat
         *        The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * @see Tr34KeyBlockFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tr34KeyBlockFormat
         */
        Builder keyBlockFormat(String keyBlockFormat);

        /**
         * <p>
         * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * </p>
         * 
         * @param keyBlockFormat
         *        The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * @see Tr34KeyBlockFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tr34KeyBlockFormat
         */
        Builder keyBlockFormat(Tr34KeyBlockFormat keyBlockFormat);

        /**
         * <p>
         * A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will fail,
         * if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
         * </p>
         * 
         * @param randomNonce
         *        A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will
         *        fail, if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder randomNonce(String randomNonce);

        /**
         * <p>
         * The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses this
         * certificate to wrap the key under export.
         * </p>
         * 
         * @param wrappingKeyCertificate
         *        The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses
         *        this certificate to wrap the key under export.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wrappingKeyCertificate(String wrappingKeyCertificate);
    }

    static final class BuilderImpl implements Builder {
        private String certificateAuthorityPublicKeyIdentifier;

        private String exportToken;

        private String keyBlockFormat;

        private String randomNonce;

        private String wrappingKeyCertificate;

        private BuilderImpl() {
        }

        private BuilderImpl(ExportTr34KeyBlock model) {
            certificateAuthorityPublicKeyIdentifier(model.certificateAuthorityPublicKeyIdentifier);
            exportToken(model.exportToken);
            keyBlockFormat(model.keyBlockFormat);
            randomNonce(model.randomNonce);
            wrappingKeyCertificate(model.wrappingKeyCertificate);
        }

        public final String getCertificateAuthorityPublicKeyIdentifier() {
            return certificateAuthorityPublicKeyIdentifier;
        }

        public final void setCertificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
        }

        @Override
        public final Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
            return this;
        }

        public final String getExportToken() {
            return exportToken;
        }

        public final void setExportToken(String exportToken) {
            this.exportToken = exportToken;
        }

        @Override
        public final Builder exportToken(String exportToken) {
            this.exportToken = exportToken;
            return this;
        }

        public final String getKeyBlockFormat() {
            return keyBlockFormat;
        }

        public final void setKeyBlockFormat(String keyBlockFormat) {
            this.keyBlockFormat = keyBlockFormat;
        }

        @Override
        public final Builder keyBlockFormat(String keyBlockFormat) {
            this.keyBlockFormat = keyBlockFormat;
            return this;
        }

        @Override
        public final Builder keyBlockFormat(Tr34KeyBlockFormat keyBlockFormat) {
            this.keyBlockFormat(keyBlockFormat == null ? null : keyBlockFormat.toString());
            return this;
        }

        public final String getRandomNonce() {
            return randomNonce;
        }

        public final void setRandomNonce(String randomNonce) {
            this.randomNonce = randomNonce;
        }

        @Override
        public final Builder randomNonce(String randomNonce) {
            this.randomNonce = randomNonce;
            return this;
        }

        public final String getWrappingKeyCertificate() {
            return wrappingKeyCertificate;
        }

        public final void setWrappingKeyCertificate(String wrappingKeyCertificate) {
            this.wrappingKeyCertificate = wrappingKeyCertificate;
        }

        @Override
        public final Builder wrappingKeyCertificate(String wrappingKeyCertificate) {
            this.wrappingKeyCertificate = wrappingKeyCertificate;
            return this;
        }

        @Override
        public ExportTr34KeyBlock build() {
            return new ExportTr34KeyBlock(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
