/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Metadata about an Amazon Web Services Payment Cryptography key.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Key implements SdkPojo, Serializable, ToCopyableBuilder<Key.Builder, Key> {
    private static final SdkField<Instant> CREATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTimestamp").getter(getter(Key::createTimestamp)).setter(setter(Builder::createTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTimestamp").build()).build();

    private static final SdkField<Instant> DELETE_PENDING_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeletePendingTimestamp").getter(getter(Key::deletePendingTimestamp))
            .setter(setter(Builder::deletePendingTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletePendingTimestamp").build())
            .build();

    private static final SdkField<Instant> DELETE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeleteTimestamp").getter(getter(Key::deleteTimestamp)).setter(setter(Builder::deleteTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteTimestamp").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(Key::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<Boolean> EXPORTABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Exportable").getter(getter(Key::exportable)).setter(setter(Builder::exportable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exportable").build()).build();

    private static final SdkField<String> KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyArn")
            .getter(getter(Key::keyArn)).setter(setter(Builder::keyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyArn").build()).build();

    private static final SdkField<KeyAttributes> KEY_ATTRIBUTES_FIELD = SdkField
            .<KeyAttributes> builder(MarshallingType.SDK_POJO).memberName("KeyAttributes").getter(getter(Key::keyAttributes))
            .setter(setter(Builder::keyAttributes)).constructor(KeyAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyAttributes").build()).build();

    private static final SdkField<String> KEY_CHECK_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyCheckValue").getter(getter(Key::keyCheckValue)).setter(setter(Builder::keyCheckValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyCheckValue").build()).build();

    private static final SdkField<String> KEY_CHECK_VALUE_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyCheckValueAlgorithm").getter(getter(Key::keyCheckValueAlgorithmAsString))
            .setter(setter(Builder::keyCheckValueAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyCheckValueAlgorithm").build())
            .build();

    private static final SdkField<String> KEY_ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyOrigin").getter(getter(Key::keyOriginAsString)).setter(setter(Builder::keyOrigin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyOrigin").build()).build();

    private static final SdkField<String> KEY_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyState").getter(getter(Key::keyStateAsString)).setter(setter(Builder::keyState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyState").build()).build();

    private static final SdkField<Instant> USAGE_START_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UsageStartTimestamp").getter(getter(Key::usageStartTimestamp))
            .setter(setter(Builder::usageStartTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsageStartTimestamp").build())
            .build();

    private static final SdkField<Instant> USAGE_STOP_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UsageStopTimestamp").getter(getter(Key::usageStopTimestamp)).setter(setter(Builder::usageStopTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsageStopTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATE_TIMESTAMP_FIELD,
            DELETE_PENDING_TIMESTAMP_FIELD, DELETE_TIMESTAMP_FIELD, ENABLED_FIELD, EXPORTABLE_FIELD, KEY_ARN_FIELD,
            KEY_ATTRIBUTES_FIELD, KEY_CHECK_VALUE_FIELD, KEY_CHECK_VALUE_ALGORITHM_FIELD, KEY_ORIGIN_FIELD, KEY_STATE_FIELD,
            USAGE_START_TIMESTAMP_FIELD, USAGE_STOP_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant createTimestamp;

    private final Instant deletePendingTimestamp;

    private final Instant deleteTimestamp;

    private final Boolean enabled;

    private final Boolean exportable;

    private final String keyArn;

    private final KeyAttributes keyAttributes;

    private final String keyCheckValue;

    private final String keyCheckValueAlgorithm;

    private final String keyOrigin;

    private final String keyState;

    private final Instant usageStartTimestamp;

    private final Instant usageStopTimestamp;

    private Key(BuilderImpl builder) {
        this.createTimestamp = builder.createTimestamp;
        this.deletePendingTimestamp = builder.deletePendingTimestamp;
        this.deleteTimestamp = builder.deleteTimestamp;
        this.enabled = builder.enabled;
        this.exportable = builder.exportable;
        this.keyArn = builder.keyArn;
        this.keyAttributes = builder.keyAttributes;
        this.keyCheckValue = builder.keyCheckValue;
        this.keyCheckValueAlgorithm = builder.keyCheckValueAlgorithm;
        this.keyOrigin = builder.keyOrigin;
        this.keyState = builder.keyState;
        this.usageStartTimestamp = builder.usageStartTimestamp;
        this.usageStopTimestamp = builder.usageStopTimestamp;
    }

    /**
     * <p>
     * The date and time when the key was created.
     * </p>
     * 
     * @return The date and time when the key was created.
     */
    public final Instant createTimestamp() {
        return createTimestamp;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is present
     * only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for deletion.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
     *         present only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for
     *         deletion.
     */
    public final Instant deletePendingTimestamp() {
        return deletePendingTimestamp;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is present
     * only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web Services Payment
     * Cryptography key is deleted.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
     *         present only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web
     *         Services Payment Cryptography key is deleted.
     */
    public final Instant deleteTimestamp() {
        return deleteTimestamp;
    }

    /**
     * <p>
     * Specifies whether the key is enabled.
     * </p>
     * 
     * @return Specifies whether the key is enabled.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Specifies whether the key is exportable. This data is immutable after the key is created.
     * </p>
     * 
     * @return Specifies whether the key is exportable. This data is immutable after the key is created.
     */
    public final Boolean exportable() {
        return exportable;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the key.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the key.
     */
    public final String keyArn() {
        return keyArn;
    }

    /**
     * <p>
     * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This data
     * is immutable after the key is created.
     * </p>
     * 
     * @return The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key.
     *         This data is immutable after the key is created.
     */
    public final KeyAttributes keyAttributes() {
        return keyAttributes;
    }

    /**
     * <p>
     * The key check value (KCV) is used to check if all parties holding a given key have the same key or to detect that
     * a key has changed.
     * </p>
     * 
     * @return The key check value (KCV) is used to check if all parties holding a given key have the same key or to
     *         detect that a key has changed.
     */
    public final String keyCheckValue() {
        return keyCheckValue;
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final KeyCheckValueAlgorithm keyCheckValueAlgorithm() {
        return KeyCheckValueAlgorithm.fromValue(keyCheckValueAlgorithm);
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final String keyCheckValueAlgorithmAsString() {
        return keyCheckValueAlgorithm;
    }

    /**
     * <p>
     * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value is
     * <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography, the value
     * is <code>EXTERNAL</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyOrigin} will
     * return {@link KeyOrigin#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyOriginAsString}.
     * </p>
     * 
     * @return The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
     *         value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
     *         Cryptography, the value is <code>EXTERNAL</code>.
     * @see KeyOrigin
     */
    public final KeyOrigin keyOrigin() {
        return KeyOrigin.fromValue(keyOrigin);
    }

    /**
     * <p>
     * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value is
     * <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography, the value
     * is <code>EXTERNAL</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyOrigin} will
     * return {@link KeyOrigin#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyOriginAsString}.
     * </p>
     * 
     * @return The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
     *         value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
     *         Cryptography, the value is <code>EXTERNAL</code>.
     * @see KeyOrigin
     */
    public final String keyOriginAsString() {
        return keyOrigin;
    }

    /**
     * <p>
     * The state of key that is being created or deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyState} will
     * return {@link KeyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyStateAsString}.
     * </p>
     * 
     * @return The state of key that is being created or deleted.
     * @see KeyState
     */
    public final KeyState keyState() {
        return KeyState.fromValue(keyState);
    }

    /**
     * <p>
     * The state of key that is being created or deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyState} will
     * return {@link KeyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyStateAsString}.
     * </p>
     * 
     * @return The state of key that is being created or deleted.
     * @see KeyState
     */
    public final String keyStateAsString() {
        return keyState;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will start using the key material for
     * cryptographic operations.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will start using the key material
     *         for cryptographic operations.
     */
    public final Instant usageStartTimestamp() {
        return usageStartTimestamp;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will stop using the key material for
     * cryptographic operations.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will stop using the key material
     *         for cryptographic operations.
     */
    public final Instant usageStopTimestamp() {
        return usageStopTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(deletePendingTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(deleteTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(exportable());
        hashCode = 31 * hashCode + Objects.hashCode(keyArn());
        hashCode = 31 * hashCode + Objects.hashCode(keyAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(keyCheckValue());
        hashCode = 31 * hashCode + Objects.hashCode(keyCheckValueAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyOriginAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(usageStartTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(usageStopTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Key)) {
            return false;
        }
        Key other = (Key) obj;
        return Objects.equals(createTimestamp(), other.createTimestamp())
                && Objects.equals(deletePendingTimestamp(), other.deletePendingTimestamp())
                && Objects.equals(deleteTimestamp(), other.deleteTimestamp()) && Objects.equals(enabled(), other.enabled())
                && Objects.equals(exportable(), other.exportable()) && Objects.equals(keyArn(), other.keyArn())
                && Objects.equals(keyAttributes(), other.keyAttributes())
                && Objects.equals(keyCheckValue(), other.keyCheckValue())
                && Objects.equals(keyCheckValueAlgorithmAsString(), other.keyCheckValueAlgorithmAsString())
                && Objects.equals(keyOriginAsString(), other.keyOriginAsString())
                && Objects.equals(keyStateAsString(), other.keyStateAsString())
                && Objects.equals(usageStartTimestamp(), other.usageStartTimestamp())
                && Objects.equals(usageStopTimestamp(), other.usageStopTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Key").add("CreateTimestamp", createTimestamp())
                .add("DeletePendingTimestamp", deletePendingTimestamp()).add("DeleteTimestamp", deleteTimestamp())
                .add("Enabled", enabled()).add("Exportable", exportable()).add("KeyArn", keyArn())
                .add("KeyAttributes", keyAttributes()).add("KeyCheckValue", keyCheckValue())
                .add("KeyCheckValueAlgorithm", keyCheckValueAlgorithmAsString()).add("KeyOrigin", keyOriginAsString())
                .add("KeyState", keyStateAsString()).add("UsageStartTimestamp", usageStartTimestamp())
                .add("UsageStopTimestamp", usageStopTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreateTimestamp":
            return Optional.ofNullable(clazz.cast(createTimestamp()));
        case "DeletePendingTimestamp":
            return Optional.ofNullable(clazz.cast(deletePendingTimestamp()));
        case "DeleteTimestamp":
            return Optional.ofNullable(clazz.cast(deleteTimestamp()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Exportable":
            return Optional.ofNullable(clazz.cast(exportable()));
        case "KeyArn":
            return Optional.ofNullable(clazz.cast(keyArn()));
        case "KeyAttributes":
            return Optional.ofNullable(clazz.cast(keyAttributes()));
        case "KeyCheckValue":
            return Optional.ofNullable(clazz.cast(keyCheckValue()));
        case "KeyCheckValueAlgorithm":
            return Optional.ofNullable(clazz.cast(keyCheckValueAlgorithmAsString()));
        case "KeyOrigin":
            return Optional.ofNullable(clazz.cast(keyOriginAsString()));
        case "KeyState":
            return Optional.ofNullable(clazz.cast(keyStateAsString()));
        case "UsageStartTimestamp":
            return Optional.ofNullable(clazz.cast(usageStartTimestamp()));
        case "UsageStopTimestamp":
            return Optional.ofNullable(clazz.cast(usageStopTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Key, T> g) {
        return obj -> g.apply((Key) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Key> {
        /**
         * <p>
         * The date and time when the key was created.
         * </p>
         * 
         * @param createTimestamp
         *        The date and time when the key was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTimestamp(Instant createTimestamp);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
         * present only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for deletion.
         * </p>
         * 
         * @param deletePendingTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value
         *        is present only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for
         *        deletion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletePendingTimestamp(Instant deletePendingTimestamp);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
         * present only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web Services
         * Payment Cryptography key is deleted.
         * </p>
         * 
         * @param deleteTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value
         *        is present only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web
         *        Services Payment Cryptography key is deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteTimestamp(Instant deleteTimestamp);

        /**
         * <p>
         * Specifies whether the key is enabled.
         * </p>
         * 
         * @param enabled
         *        Specifies whether the key is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Specifies whether the key is exportable. This data is immutable after the key is created.
         * </p>
         * 
         * @param exportable
         *        Specifies whether the key is exportable. This data is immutable after the key is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportable(Boolean exportable);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the key.
         * </p>
         * 
         * @param keyArn
         *        The Amazon Resource Name (ARN) of the key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyArn(String keyArn);

        /**
         * <p>
         * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This
         * data is immutable after the key is created.
         * </p>
         * 
         * @param keyAttributes
         *        The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key.
         *        This data is immutable after the key is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyAttributes(KeyAttributes keyAttributes);

        /**
         * <p>
         * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This
         * data is immutable after the key is created.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyAttributes.Builder} avoiding the need
         * to create one manually via {@link KeyAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyAttributes.Builder#build()} is called immediately and its
         * result is passed to {@link #keyAttributes(KeyAttributes)}.
         * 
         * @param keyAttributes
         *        a consumer that will call methods on {@link KeyAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyAttributes(KeyAttributes)
         */
        default Builder keyAttributes(Consumer<KeyAttributes.Builder> keyAttributes) {
            return keyAttributes(KeyAttributes.builder().applyMutation(keyAttributes).build());
        }

        /**
         * <p>
         * The key check value (KCV) is used to check if all parties holding a given key have the same key or to detect
         * that a key has changed.
         * </p>
         * 
         * @param keyCheckValue
         *        The key check value (KCV) is used to check if all parties holding a given key have the same key or to
         *        detect that a key has changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyCheckValue(String keyCheckValue);

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm);

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm);

        /**
         * <p>
         * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value
         * is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography,
         * the value is <code>EXTERNAL</code>.
         * </p>
         * 
         * @param keyOrigin
         *        The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
         *        value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
         *        Cryptography, the value is <code>EXTERNAL</code>.
         * @see KeyOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyOrigin
         */
        Builder keyOrigin(String keyOrigin);

        /**
         * <p>
         * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value
         * is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography,
         * the value is <code>EXTERNAL</code>.
         * </p>
         * 
         * @param keyOrigin
         *        The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
         *        value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
         *        Cryptography, the value is <code>EXTERNAL</code>.
         * @see KeyOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyOrigin
         */
        Builder keyOrigin(KeyOrigin keyOrigin);

        /**
         * <p>
         * The state of key that is being created or deleted.
         * </p>
         * 
         * @param keyState
         *        The state of key that is being created or deleted.
         * @see KeyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyState
         */
        Builder keyState(String keyState);

        /**
         * <p>
         * The state of key that is being created or deleted.
         * </p>
         * 
         * @param keyState
         *        The state of key that is being created or deleted.
         * @see KeyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyState
         */
        Builder keyState(KeyState keyState);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will start using the key material for
         * cryptographic operations.
         * </p>
         * 
         * @param usageStartTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will start using the key
         *        material for cryptographic operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usageStartTimestamp(Instant usageStartTimestamp);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will stop using the key material for
         * cryptographic operations.
         * </p>
         * 
         * @param usageStopTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will stop using the key
         *        material for cryptographic operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usageStopTimestamp(Instant usageStopTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private Instant createTimestamp;

        private Instant deletePendingTimestamp;

        private Instant deleteTimestamp;

        private Boolean enabled;

        private Boolean exportable;

        private String keyArn;

        private KeyAttributes keyAttributes;

        private String keyCheckValue;

        private String keyCheckValueAlgorithm;

        private String keyOrigin;

        private String keyState;

        private Instant usageStartTimestamp;

        private Instant usageStopTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(Key model) {
            createTimestamp(model.createTimestamp);
            deletePendingTimestamp(model.deletePendingTimestamp);
            deleteTimestamp(model.deleteTimestamp);
            enabled(model.enabled);
            exportable(model.exportable);
            keyArn(model.keyArn);
            keyAttributes(model.keyAttributes);
            keyCheckValue(model.keyCheckValue);
            keyCheckValueAlgorithm(model.keyCheckValueAlgorithm);
            keyOrigin(model.keyOrigin);
            keyState(model.keyState);
            usageStartTimestamp(model.usageStartTimestamp);
            usageStopTimestamp(model.usageStopTimestamp);
        }

        public final Instant getCreateTimestamp() {
            return createTimestamp;
        }

        public final void setCreateTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
        }

        @Override
        public final Builder createTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
            return this;
        }

        public final Instant getDeletePendingTimestamp() {
            return deletePendingTimestamp;
        }

        public final void setDeletePendingTimestamp(Instant deletePendingTimestamp) {
            this.deletePendingTimestamp = deletePendingTimestamp;
        }

        @Override
        public final Builder deletePendingTimestamp(Instant deletePendingTimestamp) {
            this.deletePendingTimestamp = deletePendingTimestamp;
            return this;
        }

        public final Instant getDeleteTimestamp() {
            return deleteTimestamp;
        }

        public final void setDeleteTimestamp(Instant deleteTimestamp) {
            this.deleteTimestamp = deleteTimestamp;
        }

        @Override
        public final Builder deleteTimestamp(Instant deleteTimestamp) {
            this.deleteTimestamp = deleteTimestamp;
            return this;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Boolean getExportable() {
            return exportable;
        }

        public final void setExportable(Boolean exportable) {
            this.exportable = exportable;
        }

        @Override
        public final Builder exportable(Boolean exportable) {
            this.exportable = exportable;
            return this;
        }

        public final String getKeyArn() {
            return keyArn;
        }

        public final void setKeyArn(String keyArn) {
            this.keyArn = keyArn;
        }

        @Override
        public final Builder keyArn(String keyArn) {
            this.keyArn = keyArn;
            return this;
        }

        public final KeyAttributes.Builder getKeyAttributes() {
            return keyAttributes != null ? keyAttributes.toBuilder() : null;
        }

        public final void setKeyAttributes(KeyAttributes.BuilderImpl keyAttributes) {
            this.keyAttributes = keyAttributes != null ? keyAttributes.build() : null;
        }

        @Override
        public final Builder keyAttributes(KeyAttributes keyAttributes) {
            this.keyAttributes = keyAttributes;
            return this;
        }

        public final String getKeyCheckValue() {
            return keyCheckValue;
        }

        public final void setKeyCheckValue(String keyCheckValue) {
            this.keyCheckValue = keyCheckValue;
        }

        @Override
        public final Builder keyCheckValue(String keyCheckValue) {
            this.keyCheckValue = keyCheckValue;
            return this;
        }

        public final String getKeyCheckValueAlgorithm() {
            return keyCheckValueAlgorithm;
        }

        public final void setKeyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
            return this;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm(keyCheckValueAlgorithm == null ? null : keyCheckValueAlgorithm.toString());
            return this;
        }

        public final String getKeyOrigin() {
            return keyOrigin;
        }

        public final void setKeyOrigin(String keyOrigin) {
            this.keyOrigin = keyOrigin;
        }

        @Override
        public final Builder keyOrigin(String keyOrigin) {
            this.keyOrigin = keyOrigin;
            return this;
        }

        @Override
        public final Builder keyOrigin(KeyOrigin keyOrigin) {
            this.keyOrigin(keyOrigin == null ? null : keyOrigin.toString());
            return this;
        }

        public final String getKeyState() {
            return keyState;
        }

        public final void setKeyState(String keyState) {
            this.keyState = keyState;
        }

        @Override
        public final Builder keyState(String keyState) {
            this.keyState = keyState;
            return this;
        }

        @Override
        public final Builder keyState(KeyState keyState) {
            this.keyState(keyState == null ? null : keyState.toString());
            return this;
        }

        public final Instant getUsageStartTimestamp() {
            return usageStartTimestamp;
        }

        public final void setUsageStartTimestamp(Instant usageStartTimestamp) {
            this.usageStartTimestamp = usageStartTimestamp;
        }

        @Override
        public final Builder usageStartTimestamp(Instant usageStartTimestamp) {
            this.usageStartTimestamp = usageStartTimestamp;
            return this;
        }

        public final Instant getUsageStopTimestamp() {
            return usageStopTimestamp;
        }

        public final void setUsageStopTimestamp(Instant usageStopTimestamp) {
            this.usageStopTimestamp = usageStopTimestamp;
        }

        @Override
        public final Builder usageStopTimestamp(Instant usageStopTimestamp) {
            this.usageStopTimestamp = usageStopTimestamp;
            return this;
        }

        @Override
        public Key build() {
            return new Key(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
