/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameter information for key material import using asymmetric RSA wrap and unwrap key exchange method.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImportKeyCryptogram implements SdkPojo, Serializable,
        ToCopyableBuilder<ImportKeyCryptogram.Builder, ImportKeyCryptogram> {
    private static final SdkField<Boolean> EXPORTABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Exportable").getter(getter(ImportKeyCryptogram::exportable)).setter(setter(Builder::exportable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exportable").build()).build();

    private static final SdkField<String> IMPORT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImportToken").getter(getter(ImportKeyCryptogram::importToken)).setter(setter(Builder::importToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportToken").build()).build();

    private static final SdkField<KeyAttributes> KEY_ATTRIBUTES_FIELD = SdkField
            .<KeyAttributes> builder(MarshallingType.SDK_POJO).memberName("KeyAttributes")
            .getter(getter(ImportKeyCryptogram::keyAttributes)).setter(setter(Builder::keyAttributes))
            .constructor(KeyAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyAttributes").build()).build();

    private static final SdkField<String> WRAPPED_KEY_CRYPTOGRAM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappedKeyCryptogram").getter(getter(ImportKeyCryptogram::wrappedKeyCryptogram))
            .setter(setter(Builder::wrappedKeyCryptogram))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappedKeyCryptogram").build())
            .build();

    private static final SdkField<String> WRAPPING_SPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappingSpec").getter(getter(ImportKeyCryptogram::wrappingSpecAsString))
            .setter(setter(Builder::wrappingSpec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappingSpec").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPORTABLE_FIELD,
            IMPORT_TOKEN_FIELD, KEY_ATTRIBUTES_FIELD, WRAPPED_KEY_CRYPTOGRAM_FIELD, WRAPPING_SPEC_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean exportable;

    private final String importToken;

    private final KeyAttributes keyAttributes;

    private final String wrappedKeyCryptogram;

    private final String wrappingSpec;

    private ImportKeyCryptogram(BuilderImpl builder) {
        this.exportable = builder.exportable;
        this.importToken = builder.importToken;
        this.keyAttributes = builder.keyAttributes;
        this.wrappedKeyCryptogram = builder.wrappedKeyCryptogram;
        this.wrappingSpec = builder.wrappingSpec;
    }

    /**
     * <p>
     * Specifies whether the key is exportable from the service.
     * </p>
     * 
     * @return Specifies whether the key is exportable from the service.
     */
    public final Boolean exportable() {
        return exportable;
    }

    /**
     * <p>
     * The import token that initiates key import using the asymmetric RSA wrap and unwrap key exchange method into AWS
     * Payment Cryptography. It expires after 7 days. You can use the same import token to import multiple keys to the
     * same service account.
     * </p>
     * 
     * @return The import token that initiates key import using the asymmetric RSA wrap and unwrap key exchange method
     *         into AWS Payment Cryptography. It expires after 7 days. You can use the same import token to import
     *         multiple keys to the same service account.
     */
    public final String importToken() {
        return importToken;
    }

    /**
     * Returns the value of the KeyAttributes property for this object.
     * 
     * @return The value of the KeyAttributes property for this object.
     */
    public final KeyAttributes keyAttributes() {
        return keyAttributes;
    }

    /**
     * <p>
     * The RSA wrapped key cryptogram under import.
     * </p>
     * 
     * @return The RSA wrapped key cryptogram under import.
     */
    public final String wrappedKeyCryptogram() {
        return wrappedKeyCryptogram;
    }

    /**
     * <p>
     * The wrapping spec for the wrapped key cryptogram.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #wrappingSpec} will
     * return {@link WrappingKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #wrappingSpecAsString}.
     * </p>
     * 
     * @return The wrapping spec for the wrapped key cryptogram.
     * @see WrappingKeySpec
     */
    public final WrappingKeySpec wrappingSpec() {
        return WrappingKeySpec.fromValue(wrappingSpec);
    }

    /**
     * <p>
     * The wrapping spec for the wrapped key cryptogram.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #wrappingSpec} will
     * return {@link WrappingKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #wrappingSpecAsString}.
     * </p>
     * 
     * @return The wrapping spec for the wrapped key cryptogram.
     * @see WrappingKeySpec
     */
    public final String wrappingSpecAsString() {
        return wrappingSpec;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(exportable());
        hashCode = 31 * hashCode + Objects.hashCode(importToken());
        hashCode = 31 * hashCode + Objects.hashCode(keyAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(wrappedKeyCryptogram());
        hashCode = 31 * hashCode + Objects.hashCode(wrappingSpecAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportKeyCryptogram)) {
            return false;
        }
        ImportKeyCryptogram other = (ImportKeyCryptogram) obj;
        return Objects.equals(exportable(), other.exportable()) && Objects.equals(importToken(), other.importToken())
                && Objects.equals(keyAttributes(), other.keyAttributes())
                && Objects.equals(wrappedKeyCryptogram(), other.wrappedKeyCryptogram())
                && Objects.equals(wrappingSpecAsString(), other.wrappingSpecAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImportKeyCryptogram").add("Exportable", exportable()).add("ImportToken", importToken())
                .add("KeyAttributes", keyAttributes()).add("WrappedKeyCryptogram", wrappedKeyCryptogram())
                .add("WrappingSpec", wrappingSpecAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Exportable":
            return Optional.ofNullable(clazz.cast(exportable()));
        case "ImportToken":
            return Optional.ofNullable(clazz.cast(importToken()));
        case "KeyAttributes":
            return Optional.ofNullable(clazz.cast(keyAttributes()));
        case "WrappedKeyCryptogram":
            return Optional.ofNullable(clazz.cast(wrappedKeyCryptogram()));
        case "WrappingSpec":
            return Optional.ofNullable(clazz.cast(wrappingSpecAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImportKeyCryptogram, T> g) {
        return obj -> g.apply((ImportKeyCryptogram) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImportKeyCryptogram> {
        /**
         * <p>
         * Specifies whether the key is exportable from the service.
         * </p>
         * 
         * @param exportable
         *        Specifies whether the key is exportable from the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportable(Boolean exportable);

        /**
         * <p>
         * The import token that initiates key import using the asymmetric RSA wrap and unwrap key exchange method into
         * AWS Payment Cryptography. It expires after 7 days. You can use the same import token to import multiple keys
         * to the same service account.
         * </p>
         * 
         * @param importToken
         *        The import token that initiates key import using the asymmetric RSA wrap and unwrap key exchange
         *        method into AWS Payment Cryptography. It expires after 7 days. You can use the same import token to
         *        import multiple keys to the same service account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importToken(String importToken);

        /**
         * Sets the value of the KeyAttributes property for this object.
         *
         * @param keyAttributes
         *        The new value for the KeyAttributes property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyAttributes(KeyAttributes keyAttributes);

        /**
         * Sets the value of the KeyAttributes property for this object.
         *
         * This is a convenience method that creates an instance of the {@link KeyAttributes.Builder} avoiding the need
         * to create one manually via {@link KeyAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyAttributes.Builder#build()} is called immediately and its
         * result is passed to {@link #keyAttributes(KeyAttributes)}.
         * 
         * @param keyAttributes
         *        a consumer that will call methods on {@link KeyAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyAttributes(KeyAttributes)
         */
        default Builder keyAttributes(Consumer<KeyAttributes.Builder> keyAttributes) {
            return keyAttributes(KeyAttributes.builder().applyMutation(keyAttributes).build());
        }

        /**
         * <p>
         * The RSA wrapped key cryptogram under import.
         * </p>
         * 
         * @param wrappedKeyCryptogram
         *        The RSA wrapped key cryptogram under import.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wrappedKeyCryptogram(String wrappedKeyCryptogram);

        /**
         * <p>
         * The wrapping spec for the wrapped key cryptogram.
         * </p>
         * 
         * @param wrappingSpec
         *        The wrapping spec for the wrapped key cryptogram.
         * @see WrappingKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WrappingKeySpec
         */
        Builder wrappingSpec(String wrappingSpec);

        /**
         * <p>
         * The wrapping spec for the wrapped key cryptogram.
         * </p>
         * 
         * @param wrappingSpec
         *        The wrapping spec for the wrapped key cryptogram.
         * @see WrappingKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WrappingKeySpec
         */
        Builder wrappingSpec(WrappingKeySpec wrappingSpec);
    }

    static final class BuilderImpl implements Builder {
        private Boolean exportable;

        private String importToken;

        private KeyAttributes keyAttributes;

        private String wrappedKeyCryptogram;

        private String wrappingSpec;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportKeyCryptogram model) {
            exportable(model.exportable);
            importToken(model.importToken);
            keyAttributes(model.keyAttributes);
            wrappedKeyCryptogram(model.wrappedKeyCryptogram);
            wrappingSpec(model.wrappingSpec);
        }

        public final Boolean getExportable() {
            return exportable;
        }

        public final void setExportable(Boolean exportable) {
            this.exportable = exportable;
        }

        @Override
        public final Builder exportable(Boolean exportable) {
            this.exportable = exportable;
            return this;
        }

        public final String getImportToken() {
            return importToken;
        }

        public final void setImportToken(String importToken) {
            this.importToken = importToken;
        }

        @Override
        public final Builder importToken(String importToken) {
            this.importToken = importToken;
            return this;
        }

        public final KeyAttributes.Builder getKeyAttributes() {
            return keyAttributes != null ? keyAttributes.toBuilder() : null;
        }

        public final void setKeyAttributes(KeyAttributes.BuilderImpl keyAttributes) {
            this.keyAttributes = keyAttributes != null ? keyAttributes.build() : null;
        }

        @Override
        public final Builder keyAttributes(KeyAttributes keyAttributes) {
            this.keyAttributes = keyAttributes;
            return this;
        }

        public final String getWrappedKeyCryptogram() {
            return wrappedKeyCryptogram;
        }

        public final void setWrappedKeyCryptogram(String wrappedKeyCryptogram) {
            this.wrappedKeyCryptogram = wrappedKeyCryptogram;
        }

        @Override
        public final Builder wrappedKeyCryptogram(String wrappedKeyCryptogram) {
            this.wrappedKeyCryptogram = wrappedKeyCryptogram;
            return this;
        }

        public final String getWrappingSpec() {
            return wrappingSpec;
        }

        public final void setWrappingSpec(String wrappingSpec) {
            this.wrappingSpec = wrappingSpec;
        }

        @Override
        public final Builder wrappingSpec(String wrappingSpec) {
            this.wrappingSpec = wrappingSpec;
            return this;
        }

        @Override
        public final Builder wrappingSpec(WrappingKeySpec wrappingSpec) {
            this.wrappingSpec(wrappingSpec == null ? null : wrappingSpec.toString());
            return this;
        }

        @Override
        public ImportKeyCryptogram build() {
            return new ImportKeyCryptogram(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
