/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateKeyRequest extends PaymentCryptographyRequest implements
        ToCopyableBuilder<CreateKeyRequest.Builder, CreateKeyRequest> {
    private static final SdkField<KeyAttributes> KEY_ATTRIBUTES_FIELD = SdkField
            .<KeyAttributes> builder(MarshallingType.SDK_POJO).memberName("KeyAttributes")
            .getter(getter(CreateKeyRequest::keyAttributes)).setter(setter(Builder::keyAttributes))
            .constructor(KeyAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyAttributes").build()).build();

    private static final SdkField<String> KEY_CHECK_VALUE_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyCheckValueAlgorithm").getter(getter(CreateKeyRequest::keyCheckValueAlgorithmAsString))
            .setter(setter(Builder::keyCheckValueAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyCheckValueAlgorithm").build())
            .build();

    private static final SdkField<Boolean> EXPORTABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Exportable").getter(getter(CreateKeyRequest::exportable)).setter(setter(Builder::exportable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exportable").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(CreateKeyRequest::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateKeyRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DERIVE_KEY_USAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeriveKeyUsage").getter(getter(CreateKeyRequest::deriveKeyUsageAsString))
            .setter(setter(Builder::deriveKeyUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeriveKeyUsage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_ATTRIBUTES_FIELD,
            KEY_CHECK_VALUE_ALGORITHM_FIELD, EXPORTABLE_FIELD, ENABLED_FIELD, TAGS_FIELD, DERIVE_KEY_USAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final KeyAttributes keyAttributes;

    private final String keyCheckValueAlgorithm;

    private final Boolean exportable;

    private final Boolean enabled;

    private final List<Tag> tags;

    private final String deriveKeyUsage;

    private CreateKeyRequest(BuilderImpl builder) {
        super(builder);
        this.keyAttributes = builder.keyAttributes;
        this.keyCheckValueAlgorithm = builder.keyCheckValueAlgorithm;
        this.exportable = builder.exportable;
        this.enabled = builder.enabled;
        this.tags = builder.tags;
        this.deriveKeyUsage = builder.deriveKeyUsage;
    }

    /**
     * <p>
     * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This data
     * is immutable after the key is created.
     * </p>
     * 
     * @return The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key.
     *         This data is immutable after the key is created.
     */
    public final KeyAttributes keyAttributes() {
        return keyAttributes;
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final KeyCheckValueAlgorithm keyCheckValueAlgorithm() {
        return KeyCheckValueAlgorithm.fromValue(keyCheckValueAlgorithm);
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final String keyCheckValueAlgorithmAsString() {
        return keyCheckValueAlgorithm;
    }

    /**
     * <p>
     * Specifies whether the key is exportable from the service.
     * </p>
     * 
     * @return Specifies whether the key is exportable from the service.
     */
    public final Boolean exportable() {
        return exportable;
    }

    /**
     * <p>
     * Specifies whether to enable the key. If the key is enabled, it is activated for use within the service. If the
     * key is not enabled, then it is created but not activated. The default value is enabled.
     * </p>
     * 
     * @return Specifies whether to enable the key. If the key is enabled, it is activated for use within the service.
     *         If the key is not enabled, then it is created but not activated. The default value is enabled.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Assigns one or more tags to the Amazon Web Services Payment Cryptography key. Use this parameter to tag a key
     * when it is created. To tag an existing Amazon Web Services Payment Cryptography key, use the <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html">TagResource</a>
     * operation.
     * </p>
     * <p>
     * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
     * value can be an empty (null) string. You can't have more than one tag on an Amazon Web Services Payment
     * Cryptography key with the same tag key.
     * </p>
     * <important>
     * <p>
     * Don't include personal, confidential or sensitive information in this field. This field may be displayed in
     * plaintext in CloudTrail logs and other output.
     * </p>
     * </important> <note>
     * <p>
     * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Assigns one or more tags to the Amazon Web Services Payment Cryptography key. Use this parameter to tag a
     *         key when it is created. To tag an existing Amazon Web Services Payment Cryptography key, use the <a
     *         href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html"
     *         >TagResource</a> operation.</p>
     *         <p>
     *         Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the
     *         tag value can be an empty (null) string. You can't have more than one tag on an Amazon Web Services
     *         Payment Cryptography key with the same tag key.
     *         </p>
     *         <important>
     *         <p>
     *         Don't include personal, confidential or sensitive information in this field. This field may be displayed
     *         in plaintext in CloudTrail logs and other output.
     *         </p>
     *         </important> <note>
     *         <p>
     *         Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the
     *         key.
     *         </p>
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The intended cryptographic usage of keys derived from the ECC key pair to be created.
     * </p>
     * <p>
     * After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived from it using
     * ECDH.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deriveKeyUsage}
     * will return {@link DeriveKeyUsage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deriveKeyUsageAsString}.
     * </p>
     * 
     * @return The intended cryptographic usage of keys derived from the ECC key pair to be created.</p>
     *         <p>
     *         After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived from
     *         it using ECDH.
     * @see DeriveKeyUsage
     */
    public final DeriveKeyUsage deriveKeyUsage() {
        return DeriveKeyUsage.fromValue(deriveKeyUsage);
    }

    /**
     * <p>
     * The intended cryptographic usage of keys derived from the ECC key pair to be created.
     * </p>
     * <p>
     * After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived from it using
     * ECDH.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deriveKeyUsage}
     * will return {@link DeriveKeyUsage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deriveKeyUsageAsString}.
     * </p>
     * 
     * @return The intended cryptographic usage of keys derived from the ECC key pair to be created.</p>
     *         <p>
     *         After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived from
     *         it using ECDH.
     * @see DeriveKeyUsage
     */
    public final String deriveKeyUsageAsString() {
        return deriveKeyUsage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(keyAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(keyCheckValueAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(exportable());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(deriveKeyUsageAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateKeyRequest)) {
            return false;
        }
        CreateKeyRequest other = (CreateKeyRequest) obj;
        return Objects.equals(keyAttributes(), other.keyAttributes())
                && Objects.equals(keyCheckValueAlgorithmAsString(), other.keyCheckValueAlgorithmAsString())
                && Objects.equals(exportable(), other.exportable()) && Objects.equals(enabled(), other.enabled())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(deriveKeyUsageAsString(), other.deriveKeyUsageAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateKeyRequest").add("KeyAttributes", keyAttributes())
                .add("KeyCheckValueAlgorithm", keyCheckValueAlgorithmAsString()).add("Exportable", exportable())
                .add("Enabled", enabled()).add("Tags", hasTags() ? tags() : null).add("DeriveKeyUsage", deriveKeyUsageAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyAttributes":
            return Optional.ofNullable(clazz.cast(keyAttributes()));
        case "KeyCheckValueAlgorithm":
            return Optional.ofNullable(clazz.cast(keyCheckValueAlgorithmAsString()));
        case "Exportable":
            return Optional.ofNullable(clazz.cast(exportable()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "DeriveKeyUsage":
            return Optional.ofNullable(clazz.cast(deriveKeyUsageAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("KeyAttributes", KEY_ATTRIBUTES_FIELD);
        map.put("KeyCheckValueAlgorithm", KEY_CHECK_VALUE_ALGORITHM_FIELD);
        map.put("Exportable", EXPORTABLE_FIELD);
        map.put("Enabled", ENABLED_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("DeriveKeyUsage", DERIVE_KEY_USAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateKeyRequest, T> g) {
        return obj -> g.apply((CreateKeyRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends PaymentCryptographyRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateKeyRequest> {
        /**
         * <p>
         * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This
         * data is immutable after the key is created.
         * </p>
         * 
         * @param keyAttributes
         *        The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key.
         *        This data is immutable after the key is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyAttributes(KeyAttributes keyAttributes);

        /**
         * <p>
         * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This
         * data is immutable after the key is created.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyAttributes.Builder} avoiding the need
         * to create one manually via {@link KeyAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyAttributes.Builder#build()} is called immediately and its
         * result is passed to {@link #keyAttributes(KeyAttributes)}.
         * 
         * @param keyAttributes
         *        a consumer that will call methods on {@link KeyAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyAttributes(KeyAttributes)
         */
        default Builder keyAttributes(Consumer<KeyAttributes.Builder> keyAttributes) {
            return keyAttributes(KeyAttributes.builder().applyMutation(keyAttributes).build());
        }

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm);

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm);

        /**
         * <p>
         * Specifies whether the key is exportable from the service.
         * </p>
         * 
         * @param exportable
         *        Specifies whether the key is exportable from the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportable(Boolean exportable);

        /**
         * <p>
         * Specifies whether to enable the key. If the key is enabled, it is activated for use within the service. If
         * the key is not enabled, then it is created but not activated. The default value is enabled.
         * </p>
         * 
         * @param enabled
         *        Specifies whether to enable the key. If the key is enabled, it is activated for use within the
         *        service. If the key is not enabled, then it is created but not activated. The default value is
         *        enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Assigns one or more tags to the Amazon Web Services Payment Cryptography key. Use this parameter to tag a key
         * when it is created. To tag an existing Amazon Web Services Payment Cryptography key, use the <a
         * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html"
         * >TagResource</a> operation.
         * </p>
         * <p>
         * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
         * value can be an empty (null) string. You can't have more than one tag on an Amazon Web Services Payment
         * Cryptography key with the same tag key.
         * </p>
         * <important>
         * <p>
         * Don't include personal, confidential or sensitive information in this field. This field may be displayed in
         * plaintext in CloudTrail logs and other output.
         * </p>
         * </important> <note>
         * <p>
         * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
         * </p>
         * </note>
         * 
         * @param tags
         *        Assigns one or more tags to the Amazon Web Services Payment Cryptography key. Use this parameter to
         *        tag a key when it is created. To tag an existing Amazon Web Services Payment Cryptography key, use the
         *        <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html">
         *        TagResource</a> operation.</p>
         *        <p>
         *        Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but
         *        the tag value can be an empty (null) string. You can't have more than one tag on an Amazon Web
         *        Services Payment Cryptography key with the same tag key.
         *        </p>
         *        <important>
         *        <p>
         *        Don't include personal, confidential or sensitive information in this field. This field may be
         *        displayed in plaintext in CloudTrail logs and other output.
         *        </p>
         *        </important> <note>
         *        <p>
         *        Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to
         *        the key.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Assigns one or more tags to the Amazon Web Services Payment Cryptography key. Use this parameter to tag a key
         * when it is created. To tag an existing Amazon Web Services Payment Cryptography key, use the <a
         * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html"
         * >TagResource</a> operation.
         * </p>
         * <p>
         * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
         * value can be an empty (null) string. You can't have more than one tag on an Amazon Web Services Payment
         * Cryptography key with the same tag key.
         * </p>
         * <important>
         * <p>
         * Don't include personal, confidential or sensitive information in this field. This field may be displayed in
         * plaintext in CloudTrail logs and other output.
         * </p>
         * </important> <note>
         * <p>
         * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
         * </p>
         * </note>
         * 
         * @param tags
         *        Assigns one or more tags to the Amazon Web Services Payment Cryptography key. Use this parameter to
         *        tag a key when it is created. To tag an existing Amazon Web Services Payment Cryptography key, use the
         *        <a href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html">
         *        TagResource</a> operation.</p>
         *        <p>
         *        Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but
         *        the tag value can be an empty (null) string. You can't have more than one tag on an Amazon Web
         *        Services Payment Cryptography key with the same tag key.
         *        </p>
         *        <important>
         *        <p>
         *        Don't include personal, confidential or sensitive information in this field. This field may be
         *        displayed in plaintext in CloudTrail logs and other output.
         *        </p>
         *        </important> <note>
         *        <p>
         *        Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to
         *        the key.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Assigns one or more tags to the Amazon Web Services Payment Cryptography key. Use this parameter to tag a key
         * when it is created. To tag an existing Amazon Web Services Payment Cryptography key, use the <a
         * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_TagResource.html"
         * >TagResource</a> operation.
         * </p>
         * <p>
         * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
         * value can be an empty (null) string. You can't have more than one tag on an Amazon Web Services Payment
         * Cryptography key with the same tag key.
         * </p>
         * <important>
         * <p>
         * Don't include personal, confidential or sensitive information in this field. This field may be displayed in
         * plaintext in CloudTrail logs and other output.
         * </p>
         * </important> <note>
         * <p>
         * Tagging or untagging an Amazon Web Services Payment Cryptography key can allow or deny permission to the key.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.paymentcryptography.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.paymentcryptography.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.paymentcryptography.model.Tag.Builder#build()} is called immediately
         * and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.paymentcryptography.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The intended cryptographic usage of keys derived from the ECC key pair to be created.
         * </p>
         * <p>
         * After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived from it
         * using ECDH.
         * </p>
         * 
         * @param deriveKeyUsage
         *        The intended cryptographic usage of keys derived from the ECC key pair to be created.</p>
         *        <p>
         *        After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived
         *        from it using ECDH.
         * @see DeriveKeyUsage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeriveKeyUsage
         */
        Builder deriveKeyUsage(String deriveKeyUsage);

        /**
         * <p>
         * The intended cryptographic usage of keys derived from the ECC key pair to be created.
         * </p>
         * <p>
         * After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived from it
         * using ECDH.
         * </p>
         * 
         * @param deriveKeyUsage
         *        The intended cryptographic usage of keys derived from the ECC key pair to be created.</p>
         *        <p>
         *        After creating an ECC key pair, you cannot change the intended cryptographic usage of keys derived
         *        from it using ECDH.
         * @see DeriveKeyUsage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeriveKeyUsage
         */
        Builder deriveKeyUsage(DeriveKeyUsage deriveKeyUsage);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends PaymentCryptographyRequest.BuilderImpl implements Builder {
        private KeyAttributes keyAttributes;

        private String keyCheckValueAlgorithm;

        private Boolean exportable;

        private Boolean enabled;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String deriveKeyUsage;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateKeyRequest model) {
            super(model);
            keyAttributes(model.keyAttributes);
            keyCheckValueAlgorithm(model.keyCheckValueAlgorithm);
            exportable(model.exportable);
            enabled(model.enabled);
            tags(model.tags);
            deriveKeyUsage(model.deriveKeyUsage);
        }

        public final KeyAttributes.Builder getKeyAttributes() {
            return keyAttributes != null ? keyAttributes.toBuilder() : null;
        }

        public final void setKeyAttributes(KeyAttributes.BuilderImpl keyAttributes) {
            this.keyAttributes = keyAttributes != null ? keyAttributes.build() : null;
        }

        @Override
        public final Builder keyAttributes(KeyAttributes keyAttributes) {
            this.keyAttributes = keyAttributes;
            return this;
        }

        public final String getKeyCheckValueAlgorithm() {
            return keyCheckValueAlgorithm;
        }

        public final void setKeyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
            return this;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm(keyCheckValueAlgorithm == null ? null : keyCheckValueAlgorithm.toString());
            return this;
        }

        public final Boolean getExportable() {
            return exportable;
        }

        public final void setExportable(Boolean exportable) {
            this.exportable = exportable;
        }

        @Override
        public final Builder exportable(Boolean exportable) {
            this.exportable = exportable;
            return this;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDeriveKeyUsage() {
            return deriveKeyUsage;
        }

        public final void setDeriveKeyUsage(String deriveKeyUsage) {
            this.deriveKeyUsage = deriveKeyUsage;
        }

        @Override
        public final Builder deriveKeyUsage(String deriveKeyUsage) {
            this.deriveKeyUsage = deriveKeyUsage;
            return this;
        }

        @Override
        public final Builder deriveKeyUsage(DeriveKeyUsage deriveKeyUsage) {
            this.deriveKeyUsage(deriveKeyUsage == null ? null : deriveKeyUsage.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateKeyRequest build() {
            return new CreateKeyRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
