/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameter information for key material import using the asymmetric TR-34 key exchange method.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImportTr34KeyBlock implements SdkPojo, Serializable,
        ToCopyableBuilder<ImportTr34KeyBlock.Builder, ImportTr34KeyBlock> {
    private static final SdkField<String> CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CertificateAuthorityPublicKeyIdentifier")
            .getter(getter(ImportTr34KeyBlock::certificateAuthorityPublicKeyIdentifier))
            .setter(setter(Builder::certificateAuthorityPublicKeyIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("CertificateAuthorityPublicKeyIdentifier").build()).build();

    private static final SdkField<String> SIGNING_KEY_CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SigningKeyCertificate").getter(getter(ImportTr34KeyBlock::signingKeyCertificate))
            .setter(setter(Builder::signingKeyCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningKeyCertificate").build())
            .build();

    private static final SdkField<String> IMPORT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImportToken").getter(getter(ImportTr34KeyBlock::importToken)).setter(setter(Builder::importToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportToken").build()).build();

    private static final SdkField<String> WRAPPED_KEY_BLOCK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappedKeyBlock").getter(getter(ImportTr34KeyBlock::wrappedKeyBlock))
            .setter(setter(Builder::wrappedKeyBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappedKeyBlock").build()).build();

    private static final SdkField<String> KEY_BLOCK_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyBlockFormat").getter(getter(ImportTr34KeyBlock::keyBlockFormatAsString))
            .setter(setter(Builder::keyBlockFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyBlockFormat").build()).build();

    private static final SdkField<String> RANDOM_NONCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RandomNonce").getter(getter(ImportTr34KeyBlock::randomNonce)).setter(setter(Builder::randomNonce))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RandomNonce").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD, SIGNING_KEY_CERTIFICATE_FIELD, IMPORT_TOKEN_FIELD,
            WRAPPED_KEY_BLOCK_FIELD, KEY_BLOCK_FORMAT_FIELD, RANDOM_NONCE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String certificateAuthorityPublicKeyIdentifier;

    private final String signingKeyCertificate;

    private final String importToken;

    private final String wrappedKeyBlock;

    private final String keyBlockFormat;

    private final String randomNonce;

    private ImportTr34KeyBlock(BuilderImpl builder) {
        this.certificateAuthorityPublicKeyIdentifier = builder.certificateAuthorityPublicKeyIdentifier;
        this.signingKeyCertificate = builder.signingKeyCertificate;
        this.importToken = builder.importToken;
        this.wrappedKeyBlock = builder.wrappedKeyBlock;
        this.keyBlockFormat = builder.keyBlockFormat;
        this.randomNonce = builder.randomNonce;
    }

    /**
     * <p>
     * The <code>KeyARN</code> of the certificate chain that signs the signing key certificate during TR-34 key import.
     * </p>
     * 
     * @return The <code>KeyARN</code> of the certificate chain that signs the signing key certificate during TR-34 key
     *         import.
     */
    public final String certificateAuthorityPublicKeyIdentifier() {
        return certificateAuthorityPublicKeyIdentifier;
    }

    /**
     * <p>
     * The public key component in PEM certificate format of the private key that signs the KDH TR-34 WrappedKeyBlock.
     * </p>
     * 
     * @return The public key component in PEM certificate format of the private key that signs the KDH TR-34
     *         WrappedKeyBlock.
     */
    public final String signingKeyCertificate() {
        return signingKeyCertificate;
    }

    /**
     * <p>
     * The import token that initiates key import using the asymmetric TR-34 key exchange method into Amazon Web
     * Services Payment Cryptography. It expires after 30 days. You can use the same import token to import multiple
     * keys to the same service account.
     * </p>
     * 
     * @return The import token that initiates key import using the asymmetric TR-34 key exchange method into Amazon Web
     *         Services Payment Cryptography. It expires after 30 days. You can use the same import token to import
     *         multiple keys to the same service account.
     */
    public final String importToken() {
        return importToken;
    }

    /**
     * <p>
     * The TR-34 wrapped key block to import.
     * </p>
     * 
     * @return The TR-34 wrapped key block to import.
     */
    public final String wrappedKeyBlock() {
        return wrappedKeyBlock;
    }

    /**
     * <p>
     * The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyBlockFormat}
     * will return {@link Tr34KeyBlockFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyBlockFormatAsString}.
     * </p>
     * 
     * @return The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
     * @see Tr34KeyBlockFormat
     */
    public final Tr34KeyBlockFormat keyBlockFormat() {
        return Tr34KeyBlockFormat.fromValue(keyBlockFormat);
    }

    /**
     * <p>
     * The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyBlockFormat}
     * will return {@link Tr34KeyBlockFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyBlockFormatAsString}.
     * </p>
     * 
     * @return The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
     * @see Tr34KeyBlockFormat
     */
    public final String keyBlockFormatAsString() {
        return keyBlockFormat;
    }

    /**
     * <p>
     * A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will fail, if a
     * random nonce value is not provided for a TR-34 key block generated using 2 pass.
     * </p>
     * 
     * @return A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will
     *         fail, if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
     */
    public final String randomNonce() {
        return randomNonce;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityPublicKeyIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(signingKeyCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(importToken());
        hashCode = 31 * hashCode + Objects.hashCode(wrappedKeyBlock());
        hashCode = 31 * hashCode + Objects.hashCode(keyBlockFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(randomNonce());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportTr34KeyBlock)) {
            return false;
        }
        ImportTr34KeyBlock other = (ImportTr34KeyBlock) obj;
        return Objects.equals(certificateAuthorityPublicKeyIdentifier(), other.certificateAuthorityPublicKeyIdentifier())
                && Objects.equals(signingKeyCertificate(), other.signingKeyCertificate())
                && Objects.equals(importToken(), other.importToken())
                && Objects.equals(wrappedKeyBlock(), other.wrappedKeyBlock())
                && Objects.equals(keyBlockFormatAsString(), other.keyBlockFormatAsString())
                && Objects.equals(randomNonce(), other.randomNonce());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImportTr34KeyBlock")
                .add("CertificateAuthorityPublicKeyIdentifier", certificateAuthorityPublicKeyIdentifier())
                .add("SigningKeyCertificate", signingKeyCertificate() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ImportToken", importToken())
                .add("WrappedKeyBlock", wrappedKeyBlock() == null ? null : "*** Sensitive Data Redacted ***")
                .add("KeyBlockFormat", keyBlockFormatAsString()).add("RandomNonce", randomNonce()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateAuthorityPublicKeyIdentifier":
            return Optional.ofNullable(clazz.cast(certificateAuthorityPublicKeyIdentifier()));
        case "SigningKeyCertificate":
            return Optional.ofNullable(clazz.cast(signingKeyCertificate()));
        case "ImportToken":
            return Optional.ofNullable(clazz.cast(importToken()));
        case "WrappedKeyBlock":
            return Optional.ofNullable(clazz.cast(wrappedKeyBlock()));
        case "KeyBlockFormat":
            return Optional.ofNullable(clazz.cast(keyBlockFormatAsString()));
        case "RandomNonce":
            return Optional.ofNullable(clazz.cast(randomNonce()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CertificateAuthorityPublicKeyIdentifier", CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD);
        map.put("SigningKeyCertificate", SIGNING_KEY_CERTIFICATE_FIELD);
        map.put("ImportToken", IMPORT_TOKEN_FIELD);
        map.put("WrappedKeyBlock", WRAPPED_KEY_BLOCK_FIELD);
        map.put("KeyBlockFormat", KEY_BLOCK_FORMAT_FIELD);
        map.put("RandomNonce", RANDOM_NONCE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ImportTr34KeyBlock, T> g) {
        return obj -> g.apply((ImportTr34KeyBlock) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImportTr34KeyBlock> {
        /**
         * <p>
         * The <code>KeyARN</code> of the certificate chain that signs the signing key certificate during TR-34 key
         * import.
         * </p>
         * 
         * @param certificateAuthorityPublicKeyIdentifier
         *        The <code>KeyARN</code> of the certificate chain that signs the signing key certificate during TR-34
         *        key import.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier);

        /**
         * <p>
         * The public key component in PEM certificate format of the private key that signs the KDH TR-34
         * WrappedKeyBlock.
         * </p>
         * 
         * @param signingKeyCertificate
         *        The public key component in PEM certificate format of the private key that signs the KDH TR-34
         *        WrappedKeyBlock.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingKeyCertificate(String signingKeyCertificate);

        /**
         * <p>
         * The import token that initiates key import using the asymmetric TR-34 key exchange method into Amazon Web
         * Services Payment Cryptography. It expires after 30 days. You can use the same import token to import multiple
         * keys to the same service account.
         * </p>
         * 
         * @param importToken
         *        The import token that initiates key import using the asymmetric TR-34 key exchange method into Amazon
         *        Web Services Payment Cryptography. It expires after 30 days. You can use the same import token to
         *        import multiple keys to the same service account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importToken(String importToken);

        /**
         * <p>
         * The TR-34 wrapped key block to import.
         * </p>
         * 
         * @param wrappedKeyBlock
         *        The TR-34 wrapped key block to import.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wrappedKeyBlock(String wrappedKeyBlock);

        /**
         * <p>
         * The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
         * </p>
         * 
         * @param keyBlockFormat
         *        The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
         * @see Tr34KeyBlockFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tr34KeyBlockFormat
         */
        Builder keyBlockFormat(String keyBlockFormat);

        /**
         * <p>
         * The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
         * </p>
         * 
         * @param keyBlockFormat
         *        The key block format to use during key import. The only value allowed is <code>X9_TR34_2012</code>.
         * @see Tr34KeyBlockFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tr34KeyBlockFormat
         */
        Builder keyBlockFormat(Tr34KeyBlockFormat keyBlockFormat);

        /**
         * <p>
         * A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will fail,
         * if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
         * </p>
         * 
         * @param randomNonce
         *        A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will
         *        fail, if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder randomNonce(String randomNonce);
    }

    static final class BuilderImpl implements Builder {
        private String certificateAuthorityPublicKeyIdentifier;

        private String signingKeyCertificate;

        private String importToken;

        private String wrappedKeyBlock;

        private String keyBlockFormat;

        private String randomNonce;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportTr34KeyBlock model) {
            certificateAuthorityPublicKeyIdentifier(model.certificateAuthorityPublicKeyIdentifier);
            signingKeyCertificate(model.signingKeyCertificate);
            importToken(model.importToken);
            wrappedKeyBlock(model.wrappedKeyBlock);
            keyBlockFormat(model.keyBlockFormat);
            randomNonce(model.randomNonce);
        }

        public final String getCertificateAuthorityPublicKeyIdentifier() {
            return certificateAuthorityPublicKeyIdentifier;
        }

        public final void setCertificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
        }

        @Override
        public final Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
            return this;
        }

        public final String getSigningKeyCertificate() {
            return signingKeyCertificate;
        }

        public final void setSigningKeyCertificate(String signingKeyCertificate) {
            this.signingKeyCertificate = signingKeyCertificate;
        }

        @Override
        public final Builder signingKeyCertificate(String signingKeyCertificate) {
            this.signingKeyCertificate = signingKeyCertificate;
            return this;
        }

        public final String getImportToken() {
            return importToken;
        }

        public final void setImportToken(String importToken) {
            this.importToken = importToken;
        }

        @Override
        public final Builder importToken(String importToken) {
            this.importToken = importToken;
            return this;
        }

        public final String getWrappedKeyBlock() {
            return wrappedKeyBlock;
        }

        public final void setWrappedKeyBlock(String wrappedKeyBlock) {
            this.wrappedKeyBlock = wrappedKeyBlock;
        }

        @Override
        public final Builder wrappedKeyBlock(String wrappedKeyBlock) {
            this.wrappedKeyBlock = wrappedKeyBlock;
            return this;
        }

        public final String getKeyBlockFormat() {
            return keyBlockFormat;
        }

        public final void setKeyBlockFormat(String keyBlockFormat) {
            this.keyBlockFormat = keyBlockFormat;
        }

        @Override
        public final Builder keyBlockFormat(String keyBlockFormat) {
            this.keyBlockFormat = keyBlockFormat;
            return this;
        }

        @Override
        public final Builder keyBlockFormat(Tr34KeyBlockFormat keyBlockFormat) {
            this.keyBlockFormat(keyBlockFormat == null ? null : keyBlockFormat.toString());
            return this;
        }

        public final String getRandomNonce() {
            return randomNonce;
        }

        public final void setRandomNonce(String randomNonce) {
            this.randomNonce = randomNonce;
        }

        @Override
        public final Builder randomNonce(String randomNonce) {
            this.randomNonce = randomNonce;
            return this;
        }

        @Override
        public ImportTr34KeyBlock build() {
            return new ImportTr34KeyBlock(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
