/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pcaconnectorad;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.pcaconnectorad.internal.PcaConnectorAdServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pcaconnectorad.model.AccessDeniedException;
import software.amazon.awssdk.services.pcaconnectorad.model.ConflictException;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateDirectoryRegistrationRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateDirectoryRegistrationResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateServicePrincipalNameRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateServicePrincipalNameResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateTemplateGroupAccessControlEntryRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateTemplateGroupAccessControlEntryResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateTemplateRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.CreateTemplateResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteDirectoryRegistrationRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteDirectoryRegistrationResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteServicePrincipalNameRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteServicePrincipalNameResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteTemplateGroupAccessControlEntryRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteTemplateGroupAccessControlEntryResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteTemplateRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.DeleteTemplateResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.GetConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.GetConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.GetDirectoryRegistrationRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.GetDirectoryRegistrationResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.GetServicePrincipalNameRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.GetServicePrincipalNameResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.GetTemplateGroupAccessControlEntryRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.GetTemplateGroupAccessControlEntryResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.GetTemplateRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.GetTemplateResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.InternalServerException;
import software.amazon.awssdk.services.pcaconnectorad.model.ListConnectorsRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.ListConnectorsResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.ListDirectoryRegistrationsRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.ListDirectoryRegistrationsResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.ListServicePrincipalNamesRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.ListServicePrincipalNamesResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.ListTemplateGroupAccessControlEntriesRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.ListTemplateGroupAccessControlEntriesResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.ListTemplatesRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.ListTemplatesResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.PcaConnectorAdException;
import software.amazon.awssdk.services.pcaconnectorad.model.ResourceNotFoundException;
import software.amazon.awssdk.services.pcaconnectorad.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.pcaconnectorad.model.TagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.TagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.ThrottlingException;
import software.amazon.awssdk.services.pcaconnectorad.model.UntagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.UntagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.UpdateTemplateGroupAccessControlEntryRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.UpdateTemplateGroupAccessControlEntryResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.UpdateTemplateRequest;
import software.amazon.awssdk.services.pcaconnectorad.model.UpdateTemplateResponse;
import software.amazon.awssdk.services.pcaconnectorad.model.ValidationException;
import software.amazon.awssdk.services.pcaconnectorad.transform.CreateConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.CreateDirectoryRegistrationRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.CreateServicePrincipalNameRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.CreateTemplateGroupAccessControlEntryRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.CreateTemplateRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.DeleteConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.DeleteDirectoryRegistrationRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.DeleteServicePrincipalNameRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.DeleteTemplateGroupAccessControlEntryRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.DeleteTemplateRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.GetConnectorRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.GetDirectoryRegistrationRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.GetServicePrincipalNameRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.GetTemplateGroupAccessControlEntryRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.GetTemplateRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.ListConnectorsRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.ListDirectoryRegistrationsRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.ListServicePrincipalNamesRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.ListTemplateGroupAccessControlEntriesRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.ListTemplatesRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.UpdateTemplateGroupAccessControlEntryRequestMarshaller;
import software.amazon.awssdk.services.pcaconnectorad.transform.UpdateTemplateRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PcaConnectorAdAsyncClient}.
 *
 * @see PcaConnectorAdAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPcaConnectorAdAsyncClient implements PcaConnectorAdAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPcaConnectorAdAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PcaConnectorAdServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPcaConnectorAdAsyncClient(PcaConnectorAdServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a connector between Amazon Web Services Private CA and an Active Directory. You must specify the private
     * CA, directory ID, and security groups.
     * </p>
     *
     * @param createConnectorRequest
     * @return A Java Future containing the result of the CreateConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.CreateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/CreateConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConnectorResponse> createConnector(CreateConnectorRequest createConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConnectorRequest, CreateConnectorResponse>()
                            .withOperationName("CreateConnector")
                            .withMarshaller(new CreateConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConnectorRequest));
            CompletableFuture<CreateConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a directory registration that authorizes communication between Amazon Web Services Private CA and an
     * Active Directory
     * </p>
     *
     * @param createDirectoryRegistrationRequest
     * @return A Java Future containing the result of the CreateDirectoryRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.CreateDirectoryRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/CreateDirectoryRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDirectoryRegistrationResponse> createDirectoryRegistration(
            CreateDirectoryRegistrationRequest createDirectoryRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDirectoryRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDirectoryRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDirectoryRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDirectoryRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDirectoryRegistrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDirectoryRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDirectoryRegistrationRequest, CreateDirectoryRegistrationResponse>()
                            .withOperationName("CreateDirectoryRegistration")
                            .withMarshaller(new CreateDirectoryRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDirectoryRegistrationRequest));
            CompletableFuture<CreateDirectoryRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a service principal name (SPN) for the service account in Active Directory. Kerberos authentication uses
     * SPNs to associate a service instance with a service sign-in account.
     * </p>
     *
     * @param createServicePrincipalNameRequest
     * @return A Java Future containing the result of the CreateServicePrincipalName operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.CreateServicePrincipalName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/CreateServicePrincipalName"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateServicePrincipalNameResponse> createServicePrincipalName(
            CreateServicePrincipalNameRequest createServicePrincipalNameRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createServicePrincipalNameRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServicePrincipalNameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateServicePrincipalName");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateServicePrincipalNameResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateServicePrincipalNameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateServicePrincipalNameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateServicePrincipalNameRequest, CreateServicePrincipalNameResponse>()
                            .withOperationName("CreateServicePrincipalName")
                            .withMarshaller(new CreateServicePrincipalNameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createServicePrincipalNameRequest));
            CompletableFuture<CreateServicePrincipalNameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Active Directory compatible certificate template. The connectors issues certificates using these
     * templates based on the requester’s Active Directory group membership.
     * </p>
     *
     * @param createTemplateRequest
     * @return A Java Future containing the result of the CreateTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.CreateTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/CreateTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTemplateResponse> createTemplate(CreateTemplateRequest createTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTemplateRequest, CreateTemplateResponse>()
                            .withOperationName("CreateTemplate")
                            .withMarshaller(new CreateTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTemplateRequest));
            CompletableFuture<CreateTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a group access control entry. Allow or deny Active Directory groups from enrolling and/or autoenrolling
     * with the template based on the group security identifiers (SIDs).
     * </p>
     *
     * @param createTemplateGroupAccessControlEntryRequest
     * @return A Java Future containing the result of the CreateTemplateGroupAccessControlEntry operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.CreateTemplateGroupAccessControlEntry
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/CreateTemplateGroupAccessControlEntry"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTemplateGroupAccessControlEntryResponse> createTemplateGroupAccessControlEntry(
            CreateTemplateGroupAccessControlEntryRequest createTemplateGroupAccessControlEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTemplateGroupAccessControlEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createTemplateGroupAccessControlEntryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTemplateGroupAccessControlEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTemplateGroupAccessControlEntryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateTemplateGroupAccessControlEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTemplateGroupAccessControlEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTemplateGroupAccessControlEntryRequest, CreateTemplateGroupAccessControlEntryResponse>()
                            .withOperationName("CreateTemplateGroupAccessControlEntry")
                            .withMarshaller(new CreateTemplateGroupAccessControlEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTemplateGroupAccessControlEntryRequest));
            CompletableFuture<CreateTemplateGroupAccessControlEntryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a connector for Active Directory. You must provide the Amazon Resource Name (ARN) of the connector that
     * you want to delete. You can find the ARN by calling the <a
     * href="https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListConnectors"
     * >https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListConnectors</a> action. Deleting a
     * connector does not deregister your directory with Amazon Web Services Private CA. You can deregister your
     * directory by calling the <a
     * href="https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DeleteDirectoryRegistration"
     * >https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DeleteDirectoryRegistration</a> action.
     * </p>
     *
     * @param deleteConnectorRequest
     * @return A Java Future containing the result of the DeleteConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.DeleteConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/DeleteConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConnectorResponse> deleteConnector(DeleteConnectorRequest deleteConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectorRequest, DeleteConnectorResponse>()
                            .withOperationName("DeleteConnector")
                            .withMarshaller(new DeleteConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConnectorRequest));
            CompletableFuture<DeleteConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a directory registration. Deleting a directory registration deauthorizes Amazon Web Services Private CA
     * with the directory.
     * </p>
     *
     * @param deleteDirectoryRegistrationRequest
     * @return A Java Future containing the result of the DeleteDirectoryRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.DeleteDirectoryRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/DeleteDirectoryRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDirectoryRegistrationResponse> deleteDirectoryRegistration(
            DeleteDirectoryRegistrationRequest deleteDirectoryRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDirectoryRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDirectoryRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDirectoryRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDirectoryRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDirectoryRegistrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDirectoryRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDirectoryRegistrationRequest, DeleteDirectoryRegistrationResponse>()
                            .withOperationName("DeleteDirectoryRegistration")
                            .withMarshaller(new DeleteDirectoryRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDirectoryRegistrationRequest));
            CompletableFuture<DeleteDirectoryRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the service principal name (SPN) used by a connector to authenticate with your Active Directory.
     * </p>
     *
     * @param deleteServicePrincipalNameRequest
     * @return A Java Future containing the result of the DeleteServicePrincipalName operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.DeleteServicePrincipalName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/DeleteServicePrincipalName"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServicePrincipalNameResponse> deleteServicePrincipalName(
            DeleteServicePrincipalNameRequest deleteServicePrincipalNameRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteServicePrincipalNameRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServicePrincipalNameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteServicePrincipalName");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServicePrincipalNameResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteServicePrincipalNameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServicePrincipalNameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServicePrincipalNameRequest, DeleteServicePrincipalNameResponse>()
                            .withOperationName("DeleteServicePrincipalName")
                            .withMarshaller(new DeleteServicePrincipalNameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteServicePrincipalNameRequest));
            CompletableFuture<DeleteServicePrincipalNameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a template. Certificates issued using the template are still valid until they are revoked or expired.
     * </p>
     *
     * @param deleteTemplateRequest
     * @return A Java Future containing the result of the DeleteTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.DeleteTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/DeleteTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTemplateResponse> deleteTemplate(DeleteTemplateRequest deleteTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTemplateRequest, DeleteTemplateResponse>()
                            .withOperationName("DeleteTemplate")
                            .withMarshaller(new DeleteTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTemplateRequest));
            CompletableFuture<DeleteTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a group access control entry.
     * </p>
     *
     * @param deleteTemplateGroupAccessControlEntryRequest
     * @return A Java Future containing the result of the DeleteTemplateGroupAccessControlEntry operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.DeleteTemplateGroupAccessControlEntry
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/DeleteTemplateGroupAccessControlEntry"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTemplateGroupAccessControlEntryResponse> deleteTemplateGroupAccessControlEntry(
            DeleteTemplateGroupAccessControlEntryRequest deleteTemplateGroupAccessControlEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTemplateGroupAccessControlEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteTemplateGroupAccessControlEntryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTemplateGroupAccessControlEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTemplateGroupAccessControlEntryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteTemplateGroupAccessControlEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTemplateGroupAccessControlEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTemplateGroupAccessControlEntryRequest, DeleteTemplateGroupAccessControlEntryResponse>()
                            .withOperationName("DeleteTemplateGroupAccessControlEntry")
                            .withMarshaller(new DeleteTemplateGroupAccessControlEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTemplateGroupAccessControlEntryRequest));
            CompletableFuture<DeleteTemplateGroupAccessControlEntryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about your connector. You specify the connector on input by its ARN (Amazon Resource Name).
     * </p>
     *
     * @param getConnectorRequest
     * @return A Java Future containing the result of the GetConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.GetConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/GetConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConnectorResponse> getConnector(GetConnectorRequest getConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConnectorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConnectorRequest, GetConnectorResponse>()
                            .withOperationName("GetConnector").withMarshaller(new GetConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConnectorRequest));
            CompletableFuture<GetConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A structure that contains information about your directory registration.
     * </p>
     *
     * @param getDirectoryRegistrationRequest
     * @return A Java Future containing the result of the GetDirectoryRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.GetDirectoryRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/GetDirectoryRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDirectoryRegistrationResponse> getDirectoryRegistration(
            GetDirectoryRegistrationRequest getDirectoryRegistrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDirectoryRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDirectoryRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDirectoryRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDirectoryRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDirectoryRegistrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDirectoryRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDirectoryRegistrationRequest, GetDirectoryRegistrationResponse>()
                            .withOperationName("GetDirectoryRegistration")
                            .withMarshaller(new GetDirectoryRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDirectoryRegistrationRequest));
            CompletableFuture<GetDirectoryRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the service principal name that the connector uses to authenticate with Active Directory.
     * </p>
     *
     * @param getServicePrincipalNameRequest
     * @return A Java Future containing the result of the GetServicePrincipalName operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.GetServicePrincipalName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/GetServicePrincipalName"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServicePrincipalNameResponse> getServicePrincipalName(
            GetServicePrincipalNameRequest getServicePrincipalNameRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServicePrincipalNameRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServicePrincipalNameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServicePrincipalName");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServicePrincipalNameResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServicePrincipalNameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServicePrincipalNameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServicePrincipalNameRequest, GetServicePrincipalNameResponse>()
                            .withOperationName("GetServicePrincipalName")
                            .withMarshaller(new GetServicePrincipalNameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServicePrincipalNameRequest));
            CompletableFuture<GetServicePrincipalNameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a certificate template that the connector uses to issue certificates from a private CA.
     * </p>
     *
     * @param getTemplateRequest
     * @return A Java Future containing the result of the GetTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.GetTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/GetTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTemplateResponse> getTemplate(GetTemplateRequest getTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTemplateRequest, GetTemplateResponse>()
                            .withOperationName("GetTemplate").withMarshaller(new GetTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTemplateRequest));
            CompletableFuture<GetTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the group access control entries for a template.
     * </p>
     *
     * @param getTemplateGroupAccessControlEntryRequest
     * @return A Java Future containing the result of the GetTemplateGroupAccessControlEntry operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.GetTemplateGroupAccessControlEntry
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/GetTemplateGroupAccessControlEntry"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTemplateGroupAccessControlEntryResponse> getTemplateGroupAccessControlEntry(
            GetTemplateGroupAccessControlEntryRequest getTemplateGroupAccessControlEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTemplateGroupAccessControlEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getTemplateGroupAccessControlEntryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTemplateGroupAccessControlEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTemplateGroupAccessControlEntryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetTemplateGroupAccessControlEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTemplateGroupAccessControlEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTemplateGroupAccessControlEntryRequest, GetTemplateGroupAccessControlEntryResponse>()
                            .withOperationName("GetTemplateGroupAccessControlEntry")
                            .withMarshaller(new GetTemplateGroupAccessControlEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTemplateGroupAccessControlEntryRequest));
            CompletableFuture<GetTemplateGroupAccessControlEntryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the connectors that you created by using the <a
     * href="https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateConnector"
     * >https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateConnector</a> action.
     * </p>
     *
     * @param listConnectorsRequest
     * @return A Java Future containing the result of the ListConnectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/ListConnectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectorsResponse> listConnectors(ListConnectorsRequest listConnectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConnectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectorsRequest, ListConnectorsResponse>()
                            .withOperationName("ListConnectors")
                            .withMarshaller(new ListConnectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConnectorsRequest));
            CompletableFuture<ListConnectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the directory registrations that you created by using the <a
     * href="https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateDirectoryRegistration"
     * >https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateDirectoryRegistration</a> action.
     * </p>
     *
     * @param listDirectoryRegistrationsRequest
     * @return A Java Future containing the result of the ListDirectoryRegistrations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.ListDirectoryRegistrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/ListDirectoryRegistrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDirectoryRegistrationsResponse> listDirectoryRegistrations(
            ListDirectoryRegistrationsRequest listDirectoryRegistrationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDirectoryRegistrationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDirectoryRegistrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDirectoryRegistrations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDirectoryRegistrationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDirectoryRegistrationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDirectoryRegistrationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDirectoryRegistrationsRequest, ListDirectoryRegistrationsResponse>()
                            .withOperationName("ListDirectoryRegistrations")
                            .withMarshaller(new ListDirectoryRegistrationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDirectoryRegistrationsRequest));
            CompletableFuture<ListDirectoryRegistrationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the service principal names that the connector uses to authenticate with Active Directory.
     * </p>
     *
     * @param listServicePrincipalNamesRequest
     * @return A Java Future containing the result of the ListServicePrincipalNames operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.ListServicePrincipalNames
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/ListServicePrincipalNames"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicePrincipalNamesResponse> listServicePrincipalNames(
            ListServicePrincipalNamesRequest listServicePrincipalNamesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicePrincipalNamesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicePrincipalNamesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServicePrincipalNames");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicePrincipalNamesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServicePrincipalNamesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicePrincipalNamesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicePrincipalNamesRequest, ListServicePrincipalNamesResponse>()
                            .withOperationName("ListServicePrincipalNames")
                            .withMarshaller(new ListServicePrincipalNamesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServicePrincipalNamesRequest));
            CompletableFuture<ListServicePrincipalNamesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags, if any, that are associated with your resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists group access control entries you created.
     * </p>
     *
     * @param listTemplateGroupAccessControlEntriesRequest
     * @return A Java Future containing the result of the ListTemplateGroupAccessControlEntries operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.ListTemplateGroupAccessControlEntries
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/ListTemplateGroupAccessControlEntries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTemplateGroupAccessControlEntriesResponse> listTemplateGroupAccessControlEntries(
            ListTemplateGroupAccessControlEntriesRequest listTemplateGroupAccessControlEntriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTemplateGroupAccessControlEntriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listTemplateGroupAccessControlEntriesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTemplateGroupAccessControlEntries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTemplateGroupAccessControlEntriesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListTemplateGroupAccessControlEntriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTemplateGroupAccessControlEntriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTemplateGroupAccessControlEntriesRequest, ListTemplateGroupAccessControlEntriesResponse>()
                            .withOperationName("ListTemplateGroupAccessControlEntries")
                            .withMarshaller(new ListTemplateGroupAccessControlEntriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTemplateGroupAccessControlEntriesRequest));
            CompletableFuture<ListTemplateGroupAccessControlEntriesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the templates, if any, that are associated with a connector.
     * </p>
     *
     * @param listTemplatesRequest
     * @return A Java Future containing the result of the ListTemplates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.ListTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/ListTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTemplatesResponse> listTemplates(ListTemplatesRequest listTemplatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTemplatesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTemplatesRequest, ListTemplatesResponse>()
                            .withOperationName("ListTemplates")
                            .withMarshaller(new ListTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTemplatesRequest));
            CompletableFuture<ListTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to your resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from your resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update template configuration to define the information included in certificates.
     * </p>
     *
     * @param updateTemplateRequest
     * @return A Java Future containing the result of the UpdateTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.UpdateTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/UpdateTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTemplateResponse> updateTemplate(UpdateTemplateRequest updateTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTemplateRequest, UpdateTemplateResponse>()
                            .withOperationName("UpdateTemplate")
                            .withMarshaller(new UpdateTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTemplateRequest));
            CompletableFuture<UpdateTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a group access control entry you created using <a href=
     * "https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateTemplateGroupAccessControlEntry.html"
     * >CreateTemplateGroupAccessControlEntry</a>.
     * </p>
     *
     * @param updateTemplateGroupAccessControlEntryRequest
     * @return A Java Future containing the result of the UpdateTemplateGroupAccessControlEntry operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You can receive this error if you attempt to create a resource share when you
     *         don't have the required permissions. This can be caused by insufficient permissions in policies attached
     *         to your Amazon Web Services Identity and Access Management (IAM) principal. It can also happen because of
     *         restrictions in place from an Amazon Web Services Organizations service control policy (SCP) that affects
     *         your Amazon Web Services account.</li>
     *         <li>ValidationException An input validation error occurred. For example, invalid characters in a template
     *         name, or if a pagination token is invalid.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent resource. The resource might
     *         not be specified correctly, or its status might not be ACTIVE.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>ConflictException This request cannot be completed for one of the following reasons because the
     *         requested resource was being concurrently modified by another request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcaConnectorAdException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PcaConnectorAdAsyncClient.UpdateTemplateGroupAccessControlEntry
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/UpdateTemplateGroupAccessControlEntry"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTemplateGroupAccessControlEntryResponse> updateTemplateGroupAccessControlEntry(
            UpdateTemplateGroupAccessControlEntryRequest updateTemplateGroupAccessControlEntryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTemplateGroupAccessControlEntryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateTemplateGroupAccessControlEntryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pca Connector Ad");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTemplateGroupAccessControlEntry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTemplateGroupAccessControlEntryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateTemplateGroupAccessControlEntryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTemplateGroupAccessControlEntryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTemplateGroupAccessControlEntryRequest, UpdateTemplateGroupAccessControlEntryResponse>()
                            .withOperationName("UpdateTemplateGroupAccessControlEntry")
                            .withMarshaller(new UpdateTemplateGroupAccessControlEntryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateTemplateGroupAccessControlEntryRequest));
            CompletableFuture<UpdateTemplateGroupAccessControlEntryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PcaConnectorAdServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PcaConnectorAdException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        PcaConnectorAdServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = PcaConnectorAdServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
