/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pcaconnectorscep;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.pcaconnectorscep.model.AccessDeniedException;
import software.amazon.awssdk.services.pcaconnectorscep.model.BadRequestException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ConflictException;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateChallengeRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateChallengeResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.CreateConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteChallengeRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteChallengeResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.DeleteConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengeMetadataRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengeMetadataResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengePasswordRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengePasswordResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetConnectorRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.GetConnectorResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.InternalServerException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.PcaConnectorScepException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ResourceNotFoundException;
import software.amazon.awssdk.services.pcaconnectorscep.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.pcaconnectorscep.model.TagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.TagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ThrottlingException;
import software.amazon.awssdk.services.pcaconnectorscep.model.UntagResourceRequest;
import software.amazon.awssdk.services.pcaconnectorscep.model.UntagResourceResponse;
import software.amazon.awssdk.services.pcaconnectorscep.model.ValidationException;
import software.amazon.awssdk.services.pcaconnectorscep.paginators.ListChallengeMetadataIterable;
import software.amazon.awssdk.services.pcaconnectorscep.paginators.ListConnectorsIterable;

/**
 * Service client for accessing Private CA Connector for SCEP. This can be created using the static {@link #builder()}
 * method.
 *
 * <note>
 * <p>
 * Connector for SCEP (Preview) is in preview release for Amazon Web Services Private Certificate Authority and is
 * subject to change.
 * </p>
 * </note>
 * <p>
 * Connector for SCEP (Preview) creates a connector between Amazon Web Services Private CA and your SCEP-enabled clients
 * and devices. For more information, see <a
 * href="https://docs.aws.amazon.com/privateca/latest/userguide/scep-connector.html">Connector for SCEP</a> in the
 * <i>Amazon Web Services Private CA User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface PcaConnectorScepClient extends AwsClient {
    String SERVICE_NAME = "pca-connector-scep";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "pca-connector-scep";

    /**
     * <p>
     * For general-purpose connectors. Creates a <i>challenge password</i> for the specified connector. The SCEP
     * protocol uses a challenge password to authenticate a request before issuing a certificate from a certificate
     * authority (CA). Your SCEP clients include the challenge password as part of their certificate request to
     * Connector for SCEP. To retrieve the connector Amazon Resource Names (ARNs) for the connectors in your account,
     * call <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_ListConnectors.html"
     * >ListConnectors</a>.
     * </p>
     * <p>
     * To create additional challenge passwords for the connector, call <code>CreateChallenge</code> again. We recommend
     * frequently rotating your challenge passwords.
     * </p>
     *
     * @param createChallengeRequest
     * @return Result of the CreateChallenge operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.CreateChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateChallengeResponse createChallenge(CreateChallengeRequest createChallengeRequest)
            throws ResourceNotFoundException, BadRequestException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * For general-purpose connectors. Creates a <i>challenge password</i> for the specified connector. The SCEP
     * protocol uses a challenge password to authenticate a request before issuing a certificate from a certificate
     * authority (CA). Your SCEP clients include the challenge password as part of their certificate request to
     * Connector for SCEP. To retrieve the connector Amazon Resource Names (ARNs) for the connectors in your account,
     * call <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_ListConnectors.html"
     * >ListConnectors</a>.
     * </p>
     * <p>
     * To create additional challenge passwords for the connector, call <code>CreateChallenge</code> again. We recommend
     * frequently rotating your challenge passwords.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateChallengeRequest.Builder} avoiding the need
     * to create one manually via {@link CreateChallengeRequest#builder()}
     * </p>
     *
     * @param createChallengeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.CreateChallengeRequest.Builder} to create a
     *        request.
     * @return Result of the CreateChallenge operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.CreateChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateChallengeResponse createChallenge(Consumer<CreateChallengeRequest.Builder> createChallengeRequest)
            throws ResourceNotFoundException, BadRequestException, InternalServerException, ValidationException,
            ThrottlingException, AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return createChallenge(CreateChallengeRequest.builder().applyMutation(createChallengeRequest).build());
    }

    /**
     * <p>
     * Creates a SCEP connector. A SCEP connector links Amazon Web Services Private Certificate Authority to your
     * SCEP-compatible devices and mobile device management (MDM) systems. Before you create a connector, you must
     * complete a set of prerequisites, including creation of a private certificate authority (CA) to use with this
     * connector. For more information, see <a href=
     * "https://docs.aws.amazon.com/privateca/latest/userguide/scep-connector.htmlconnector-for-scep-prerequisites.html"
     * >Connector for SCEP prerequisites</a>.
     * </p>
     *
     * @param createConnectorRequest
     * @return Result of the CreateConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.CreateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateConnector"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateConnectorResponse createConnector(CreateConnectorRequest createConnectorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a SCEP connector. A SCEP connector links Amazon Web Services Private Certificate Authority to your
     * SCEP-compatible devices and mobile device management (MDM) systems. Before you create a connector, you must
     * complete a set of prerequisites, including creation of a private certificate authority (CA) to use with this
     * connector. For more information, see <a href=
     * "https://docs.aws.amazon.com/privateca/latest/userguide/scep-connector.htmlconnector-for-scep-prerequisites.html"
     * >Connector for SCEP prerequisites</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateConnectorRequest.Builder} avoiding the need
     * to create one manually via {@link CreateConnectorRequest#builder()}
     * </p>
     *
     * @param createConnectorRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.CreateConnectorRequest.Builder} to create a
     *        request.
     * @return Result of the CreateConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.CreateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/CreateConnector"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateConnectorResponse createConnector(Consumer<CreateConnectorRequest.Builder> createConnectorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            PcaConnectorScepException {
        return createConnector(CreateConnectorRequest.builder().applyMutation(createConnectorRequest).build());
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param deleteChallengeRequest
     * @return Result of the DeleteChallenge operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.DeleteChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteChallengeResponse deleteChallenge(DeleteChallengeRequest deleteChallengeRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteChallengeRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteChallengeRequest#builder()}
     * </p>
     *
     * @param deleteChallengeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.DeleteChallengeRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteChallenge operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.DeleteChallenge
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteChallenge"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteChallengeResponse deleteChallenge(Consumer<DeleteChallengeRequest.Builder> deleteChallengeRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        return deleteChallenge(DeleteChallengeRequest.builder().applyMutation(deleteChallengeRequest).build());
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. This operation also deletes any challenges associated with the connector.
     * </p>
     *
     * @param deleteConnectorRequest
     * @return Result of the DeleteConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.DeleteConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteConnector"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteConnectorResponse deleteConnector(DeleteConnectorRequest deleteConnectorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. This operation also deletes any challenges associated with the connector.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteConnectorRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteConnectorRequest#builder()}
     * </p>
     *
     * @param deleteConnectorRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.DeleteConnectorRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws ConflictException
     *         This request can't be completed for one of the following reasons because the requested resource was being
     *         concurrently modified by another request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.DeleteConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/DeleteConnector"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteConnectorResponse deleteConnector(Consumer<DeleteConnectorRequest.Builder> deleteConnectorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        return deleteConnector(DeleteConnectorRequest.builder().applyMutation(deleteConnectorRequest).build());
    }

    /**
     * <p>
     * Retrieves the metadata for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param getChallengeMetadataRequest
     * @return Result of the GetChallengeMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default GetChallengeMetadataResponse getChallengeMetadata(GetChallengeMetadataRequest getChallengeMetadataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the metadata for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChallengeMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link GetChallengeMetadataRequest#builder()}
     * </p>
     *
     * @param getChallengeMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengeMetadataRequest.Builder} to
     *        create a request.
     * @return Result of the GetChallengeMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default GetChallengeMetadataResponse getChallengeMetadata(
            Consumer<GetChallengeMetadataRequest.Builder> getChallengeMetadataRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return getChallengeMetadata(GetChallengeMetadataRequest.builder().applyMutation(getChallengeMetadataRequest).build());
    }

    /**
     * <p>
     * Retrieves the challenge password for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     *
     * @param getChallengePasswordRequest
     * @return Result of the GetChallengePassword operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetChallengePassword
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengePassword"
     *      target="_top">AWS API Documentation</a>
     */
    default GetChallengePasswordResponse getChallengePassword(GetChallengePasswordRequest getChallengePasswordRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the challenge password for the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Challenge.html"
     * >Challenge</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChallengePasswordRequest.Builder} avoiding the
     * need to create one manually via {@link GetChallengePasswordRequest#builder()}
     * </p>
     *
     * @param getChallengePasswordRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.GetChallengePasswordRequest.Builder} to
     *        create a request.
     * @return Result of the GetChallengePassword operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetChallengePassword
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetChallengePassword"
     *      target="_top">AWS API Documentation</a>
     */
    default GetChallengePasswordResponse getChallengePassword(
            Consumer<GetChallengePasswordRequest.Builder> getChallengePasswordRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return getChallengePassword(GetChallengePasswordRequest.builder().applyMutation(getChallengePasswordRequest).build());
    }

    /**
     * <p>
     * Retrieves details about the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. Calling this action returns important details about the connector, such as the public SCEP URL
     * where your clients can request certificates.
     * </p>
     *
     * @param getConnectorRequest
     * @return Result of the GetConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetConnector"
     *      target="_top">AWS API Documentation</a>
     */
    default GetConnectorResponse getConnector(GetConnectorRequest getConnectorRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves details about the specified <a
     * href="https://docs.aws.amazon.com/C4SCEP_API/pca-connector-scep/latest/APIReference/API_Connector.html"
     * >Connector</a>. Calling this action returns important details about the connector, such as the public SCEP URL
     * where your clients can request certificates.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetConnectorRequest.Builder} avoiding the need to
     * create one manually via {@link GetConnectorRequest#builder()}
     * </p>
     *
     * @param getConnectorRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.GetConnectorRequest.Builder} to create a
     *        request.
     * @return Result of the GetConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.GetConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/GetConnector"
     *      target="_top">AWS API Documentation</a>
     */
    default GetConnectorResponse getConnector(Consumer<GetConnectorRequest.Builder> getConnectorRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        return getConnector(GetConnectorRequest.builder().applyMutation(getConnectorRequest).build());
    }

    /**
     * <p>
     * Retrieves the challenge metadata for the specified ARN.
     * </p>
     *
     * @param listChallengeMetadataRequest
     * @return Result of the ListChallengeMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ListChallengeMetadataResponse listChallengeMetadata(ListChallengeMetadataRequest listChallengeMetadataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the challenge metadata for the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChallengeMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link ListChallengeMetadataRequest#builder()}
     * </p>
     *
     * @param listChallengeMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest.Builder} to
     *        create a request.
     * @return Result of the ListChallengeMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ListChallengeMetadataResponse listChallengeMetadata(
            Consumer<ListChallengeMetadataRequest.Builder> listChallengeMetadataRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return listChallengeMetadata(ListChallengeMetadataRequest.builder().applyMutation(listChallengeMetadataRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listChallengeMetadata(software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListChallengeMetadataIterable responses = client.listChallengeMetadataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pcaconnectorscep.paginators.ListChallengeMetadataIterable responses = client
     *             .listChallengeMetadataPaginator(request);
     *     for (software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListChallengeMetadataIterable responses = client.listChallengeMetadataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChallengeMetadata(software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChallengeMetadataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ListChallengeMetadataIterable listChallengeMetadataPaginator(ListChallengeMetadataRequest listChallengeMetadataRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        return new ListChallengeMetadataIterable(this, listChallengeMetadataRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listChallengeMetadata(software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListChallengeMetadataIterable responses = client.listChallengeMetadataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pcaconnectorscep.paginators.ListChallengeMetadataIterable responses = client
     *             .listChallengeMetadataPaginator(request);
     *     for (software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListChallengeMetadataIterable responses = client.listChallengeMetadataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChallengeMetadata(software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChallengeMetadataRequest.Builder} avoiding the
     * need to create one manually via {@link ListChallengeMetadataRequest#builder()}
     * </p>
     *
     * @param listChallengeMetadataRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.ListChallengeMetadataRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListChallengeMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListChallengeMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    default ListChallengeMetadataIterable listChallengeMetadataPaginator(
            Consumer<ListChallengeMetadataRequest.Builder> listChallengeMetadataRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return listChallengeMetadataPaginator(ListChallengeMetadataRequest.builder().applyMutation(listChallengeMetadataRequest)
                .build());
    }

    /**
     * <p>
     * Lists the connectors belonging to your Amazon Web Services account.
     * </p>
     *
     * @param listConnectorsRequest
     * @return Result of the ListConnectors operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListConnectors"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectorsResponse listConnectors(ListConnectorsRequest listConnectorsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the connectors belonging to your Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListConnectorsRequest.Builder} avoiding the need to
     * create one manually via {@link ListConnectorsRequest#builder()}
     * </p>
     *
     * @param listConnectorsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest.Builder} to create a
     *        request.
     * @return Result of the ListConnectors operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListConnectors"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectorsResponse listConnectors(Consumer<ListConnectorsRequest.Builder> listConnectorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return listConnectors(ListConnectorsRequest.builder().applyMutation(listConnectorsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listConnectors(software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListConnectorsIterable responses = client.listConnectorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pcaconnectorscep.paginators.ListConnectorsIterable responses = client
     *             .listConnectorsPaginator(request);
     *     for (software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListConnectorsIterable responses = client.listConnectorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnectors(software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listConnectorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListConnectors"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectorsIterable listConnectorsPaginator(ListConnectorsRequest listConnectorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return new ListConnectorsIterable(this, listConnectorsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listConnectors(software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListConnectorsIterable responses = client.listConnectorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pcaconnectorscep.paginators.ListConnectorsIterable responses = client
     *             .listConnectorsPaginator(request);
     *     for (software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pcaconnectorscep.paginators.ListConnectorsIterable responses = client.listConnectorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnectors(software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListConnectorsRequest.Builder} avoiding the need to
     * create one manually via {@link ListConnectorsRequest#builder()}
     * </p>
     *
     * @param listConnectorsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.ListConnectorsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListConnectors"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectorsIterable listConnectorsPaginator(Consumer<ListConnectorsRequest.Builder> listConnectorsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return listConnectorsPaginator(ListConnectorsRequest.builder().applyMutation(listConnectorsRequest).build());
    }

    /**
     * <p>
     * Retrieves the tags associated with the specified resource. Tags are key-value pairs that you can use to
     * categorize and manage your resources, for purposes like billing. For example, you might set the tag key to
     * "customer" and the value to the customer name or ID. You can specify one or more tags to add to each Amazon Web
     * Services resource, up to 50 tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the tags associated with the specified resource. Tags are key-value pairs that you can use to
     * categorize and manage your resources, for purposes like billing. For example, you might set the tag key to
     * "customer" and the value to the customer name or ID. You can specify one or more tags to add to each Amazon Web
     * Services resource, up to 50 tags for a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.ListTagsForResourceRequest.Builder} to
     *        create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Adds one or more tags to your resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds one or more tags to your resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.TagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes one or more tags from your resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, PcaConnectorScepException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes one or more tags from your resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pcaconnectorscep.model.UntagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might be incorrectly specified, or it
     *         might have a status other than <code>ACTIVE</code>.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         An input validation error occurred. For example, invalid characters in a name tag, or an invalid
     *         pagination token.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws AccessDeniedException
     *         You can receive this error if you attempt to perform an operation and you don't have the required
     *         permissions. This can be caused by insufficient permissions in policies attached to your Amazon Web
     *         Services Identity and Access Management (IAM) principal. It can also happen because of restrictions in
     *         place from an Amazon Web Services Organizations service control policy (SCP) that affects your Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PcaConnectorScepException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PcaConnectorScepClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pca-connector-scep-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, PcaConnectorScepException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * Create a {@link PcaConnectorScepClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static PcaConnectorScepClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link PcaConnectorScepClient}.
     */
    static PcaConnectorScepClientBuilder builder() {
        return new DefaultPcaConnectorScepClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default PcaConnectorScepServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
