/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pcs;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.pcs.internal.PcsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pcs.model.AccessDeniedException;
import software.amazon.awssdk.services.pcs.model.ConflictException;
import software.amazon.awssdk.services.pcs.model.CreateClusterRequest;
import software.amazon.awssdk.services.pcs.model.CreateClusterResponse;
import software.amazon.awssdk.services.pcs.model.CreateComputeNodeGroupRequest;
import software.amazon.awssdk.services.pcs.model.CreateComputeNodeGroupResponse;
import software.amazon.awssdk.services.pcs.model.CreateQueueRequest;
import software.amazon.awssdk.services.pcs.model.CreateQueueResponse;
import software.amazon.awssdk.services.pcs.model.DeleteClusterRequest;
import software.amazon.awssdk.services.pcs.model.DeleteClusterResponse;
import software.amazon.awssdk.services.pcs.model.DeleteComputeNodeGroupRequest;
import software.amazon.awssdk.services.pcs.model.DeleteComputeNodeGroupResponse;
import software.amazon.awssdk.services.pcs.model.DeleteQueueRequest;
import software.amazon.awssdk.services.pcs.model.DeleteQueueResponse;
import software.amazon.awssdk.services.pcs.model.GetClusterRequest;
import software.amazon.awssdk.services.pcs.model.GetClusterResponse;
import software.amazon.awssdk.services.pcs.model.GetComputeNodeGroupRequest;
import software.amazon.awssdk.services.pcs.model.GetComputeNodeGroupResponse;
import software.amazon.awssdk.services.pcs.model.GetQueueRequest;
import software.amazon.awssdk.services.pcs.model.GetQueueResponse;
import software.amazon.awssdk.services.pcs.model.InternalServerException;
import software.amazon.awssdk.services.pcs.model.ListClustersRequest;
import software.amazon.awssdk.services.pcs.model.ListClustersResponse;
import software.amazon.awssdk.services.pcs.model.ListComputeNodeGroupsRequest;
import software.amazon.awssdk.services.pcs.model.ListComputeNodeGroupsResponse;
import software.amazon.awssdk.services.pcs.model.ListQueuesRequest;
import software.amazon.awssdk.services.pcs.model.ListQueuesResponse;
import software.amazon.awssdk.services.pcs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pcs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pcs.model.PcsException;
import software.amazon.awssdk.services.pcs.model.RegisterComputeNodeGroupInstanceRequest;
import software.amazon.awssdk.services.pcs.model.RegisterComputeNodeGroupInstanceResponse;
import software.amazon.awssdk.services.pcs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.pcs.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.pcs.model.TagResourceRequest;
import software.amazon.awssdk.services.pcs.model.TagResourceResponse;
import software.amazon.awssdk.services.pcs.model.ThrottlingException;
import software.amazon.awssdk.services.pcs.model.UntagResourceRequest;
import software.amazon.awssdk.services.pcs.model.UntagResourceResponse;
import software.amazon.awssdk.services.pcs.model.UpdateComputeNodeGroupRequest;
import software.amazon.awssdk.services.pcs.model.UpdateComputeNodeGroupResponse;
import software.amazon.awssdk.services.pcs.model.UpdateQueueRequest;
import software.amazon.awssdk.services.pcs.model.UpdateQueueResponse;
import software.amazon.awssdk.services.pcs.model.ValidationException;
import software.amazon.awssdk.services.pcs.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.CreateComputeNodeGroupRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.CreateQueueRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.DeleteComputeNodeGroupRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.DeleteQueueRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.GetClusterRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.GetComputeNodeGroupRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.GetQueueRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.ListComputeNodeGroupsRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.ListQueuesRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.RegisterComputeNodeGroupInstanceRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.UpdateComputeNodeGroupRequestMarshaller;
import software.amazon.awssdk.services.pcs.transform.UpdateQueueRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PcsAsyncClient}.
 *
 * @see PcsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPcsAsyncClient implements PcsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPcsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPcsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a cluster in your account. Amazon Web Services PCS creates the cluster controller in a service-owned
     * account. The cluster controller communicates with the cluster resources in your account. The subnets and security
     * groups for the cluster must already exist before you use this API action.
     * </p>
     * <note>
     * <p>
     * It takes time for Amazon Web Services PCS to create the cluster. The cluster is in a <code>Creating</code> state
     * until it is ready to use. There can only be 1 cluster in a <code>Creating</code> state per Amazon Web Services
     * Region per Amazon Web Services account. <code>CreateCluster</code> fails with a
     * <code>ServiceQuotaExceededException</code> if there is already a cluster in a <code>Creating</code> state.
     * </p>
     * </note>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.
     *         To learn how to increase your service quota, see <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html">Requesting
     *         a quota increase</a> in the <i>Service Quotas User Guide</i> </p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The max number of clusters or queues has been reached for the account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The max number of compute node groups has been reached for the associated cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The total of <code>maxInstances</code> across all compute node groups has been reached for associated
     *         cluster.
     *         </p>
     *         </li></li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a managed set of compute nodes. You associate a compute node group with a cluster through 1 or more
     * Amazon Web Services PCS queues or as part of the login fleet. A compute node group includes the definition of the
     * compute properties and lifecycle management. Amazon Web Services PCS uses the information you provide to this API
     * action to launch compute nodes in your account. You can only specify subnets in the same Amazon VPC as your
     * cluster. You receive billing charges for the compute nodes that Amazon Web Services PCS launches in your account.
     * You must already have a launch template before you call this API. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">Launch an instance from a
     * launch template</a> in the <i>Amazon Elastic Compute Cloud User Guide for Linux Instances</i>.
     * </p>
     *
     * @param createComputeNodeGroupRequest
     * @return A Java Future containing the result of the CreateComputeNodeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.
     *         To learn how to increase your service quota, see <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html">Requesting
     *         a quota increase</a> in the <i>Service Quotas User Guide</i> </p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The max number of clusters or queues has been reached for the account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The max number of compute node groups has been reached for the associated cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The total of <code>maxInstances</code> across all compute node groups has been reached for associated
     *         cluster.
     *         </p>
     *         </li></li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.CreateComputeNodeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/CreateComputeNodeGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComputeNodeGroupResponse> createComputeNodeGroup(
            CreateComputeNodeGroupRequest createComputeNodeGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createComputeNodeGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComputeNodeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComputeNodeGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComputeNodeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComputeNodeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComputeNodeGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComputeNodeGroupRequest, CreateComputeNodeGroupResponse>()
                            .withOperationName("CreateComputeNodeGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateComputeNodeGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createComputeNodeGroupRequest));
            CompletableFuture<CreateComputeNodeGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a job queue. You must associate 1 or more compute node groups with the queue. You can associate 1 compute
     * node group with multiple queues.
     * </p>
     *
     * @param createQueueRequest
     * @return A Java Future containing the result of the CreateQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.
     *         To learn how to increase your service quota, see <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html">Requesting
     *         a quota increase</a> in the <i>Service Quotas User Guide</i> </p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The max number of clusters or queues has been reached for the account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The max number of compute node groups has been reached for the associated cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The total of <code>maxInstances</code> across all compute node groups has been reached for associated
     *         cluster.
     *         </p>
     *         </li></li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.CreateQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/CreateQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateQueueResponse> createQueue(CreateQueueRequest createQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateQueueRequest, CreateQueueResponse>()
                            .withOperationName("CreateQueue").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createQueueRequest));
            CompletableFuture<CreateQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a cluster and all its linked resources. You must delete all queues and compute node groups associated
     * with the cluster before you can delete the cluster.
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                            .withOperationName("DeleteCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteClusterRequest));
            CompletableFuture<DeleteClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a compute node group. You must delete all queues associated with the compute node group first.
     * </p>
     *
     * @param deleteComputeNodeGroupRequest
     * @return A Java Future containing the result of the DeleteComputeNodeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.DeleteComputeNodeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/DeleteComputeNodeGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComputeNodeGroupResponse> deleteComputeNodeGroup(
            DeleteComputeNodeGroupRequest deleteComputeNodeGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteComputeNodeGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteComputeNodeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComputeNodeGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComputeNodeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComputeNodeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteComputeNodeGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComputeNodeGroupRequest, DeleteComputeNodeGroupResponse>()
                            .withOperationName("DeleteComputeNodeGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteComputeNodeGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteComputeNodeGroupRequest));
            CompletableFuture<DeleteComputeNodeGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a job queue. If the compute node group associated with this queue isn't associated with any other queues,
     * Amazon Web Services PCS terminates all the compute nodes for this queue.
     * </p>
     *
     * @param deleteQueueRequest
     * @return A Java Future containing the result of the DeleteQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.DeleteQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/DeleteQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteQueueResponse> deleteQueue(DeleteQueueRequest deleteQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteQueueRequest, DeleteQueueResponse>()
                            .withOperationName("DeleteQueue").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteQueueRequest));
            CompletableFuture<DeleteQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a running cluster in your account. This API action provides networking
     * information, endpoint information for communication with the scheduler, and provisioning status.
     * </p>
     *
     * @param getClusterRequest
     * @return A Java Future containing the result of the GetCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.GetCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/GetCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetClusterResponse> getCluster(GetClusterRequest getClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetClusterRequest, GetClusterResponse>().withOperationName("GetCluster")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getClusterRequest));
            CompletableFuture<GetClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a compute node group. This API action provides networking information, EC2
     * instance type, compute node group status, and scheduler (such as Slurm) configuration.
     * </p>
     *
     * @param getComputeNodeGroupRequest
     * @return A Java Future containing the result of the GetComputeNodeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.GetComputeNodeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/GetComputeNodeGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetComputeNodeGroupResponse> getComputeNodeGroup(
            GetComputeNodeGroupRequest getComputeNodeGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComputeNodeGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComputeNodeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComputeNodeGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComputeNodeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetComputeNodeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetComputeNodeGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComputeNodeGroupRequest, GetComputeNodeGroupResponse>()
                            .withOperationName("GetComputeNodeGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetComputeNodeGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getComputeNodeGroupRequest));
            CompletableFuture<GetComputeNodeGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about a queue. The information includes the compute node groups that the queue uses
     * to schedule jobs.
     * </p>
     *
     * @param getQueueRequest
     * @return A Java Future containing the result of the GetQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.GetQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/GetQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueueResponse> getQueue(GetQueueRequest getQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueueRequest, GetQueueResponse>().withOperationName("GetQueue")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQueueRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getQueueRequest));
            CompletableFuture<GetQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of running clusters in your account.
     * </p>
     *
     * @param listClustersRequest
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListClustersResponse> listClusters(ListClustersRequest listClustersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listClustersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                            .withOperationName("ListClusters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listClustersRequest));
            CompletableFuture<ListClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all compute node groups associated with a cluster.
     * </p>
     *
     * @param listComputeNodeGroupsRequest
     * @return A Java Future containing the result of the ListComputeNodeGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.ListComputeNodeGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/ListComputeNodeGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListComputeNodeGroupsResponse> listComputeNodeGroups(
            ListComputeNodeGroupsRequest listComputeNodeGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listComputeNodeGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComputeNodeGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComputeNodeGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComputeNodeGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComputeNodeGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComputeNodeGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComputeNodeGroupsRequest, ListComputeNodeGroupsResponse>()
                            .withOperationName("ListComputeNodeGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListComputeNodeGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listComputeNodeGroupsRequest));
            CompletableFuture<ListComputeNodeGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all queues associated with a cluster.
     * </p>
     *
     * @param listQueuesRequest
     * @return A Java Future containing the result of the ListQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.ListQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/ListQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListQueuesResponse> listQueues(ListQueuesRequest listQueuesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQueuesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueues");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQueuesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListQueuesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListQueuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQueuesRequest, ListQueuesResponse>().withOperationName("ListQueues")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQueuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQueuesRequest));
            CompletableFuture<ListQueuesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all tags on an Amazon Web Services PCS resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <important>
     * <p>
     * This API action isn't intended for you to use.
     * </p>
     * </important>
     * <p>
     * Amazon Web Services PCS uses this API action to register the compute nodes it launches in your account.
     * </p>
     *
     * @param registerComputeNodeGroupInstanceRequest
     * @return A Java Future containing the result of the RegisterComputeNodeGroupInstance operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.RegisterComputeNodeGroupInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/RegisterComputeNodeGroupInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterComputeNodeGroupInstanceResponse> registerComputeNodeGroupInstance(
            RegisterComputeNodeGroupInstanceRequest registerComputeNodeGroupInstanceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerComputeNodeGroupInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                registerComputeNodeGroupInstanceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterComputeNodeGroupInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterComputeNodeGroupInstanceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, RegisterComputeNodeGroupInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterComputeNodeGroupInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterComputeNodeGroupInstanceRequest, RegisterComputeNodeGroupInstanceResponse>()
                            .withOperationName("RegisterComputeNodeGroupInstance").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterComputeNodeGroupInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerComputeNodeGroupInstanceRequest));
            CompletableFuture<RegisterComputeNodeGroupInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or edits tags on an Amazon Web Services PCS resource. Each tag consists of a tag key and a tag value. The
     * tag key and tag value are case-sensitive strings. The tag value can be an empty (null) string. To add a tag,
     * specify a new tag key and a tag value. To edit a tag, specify an existing tag key and a new tag value.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes tags from an Amazon Web Services PCS resource. To delete a tag, specify the tag key and the Amazon
     * Resource Name (ARN) of the Amazon Web Services PCS resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a compute node group. You can update many of the fields related to your compute node group including the
     * configurations for networking, compute nodes, and settings specific to your scheduler (such as Slurm).
     * </p>
     *
     * @param updateComputeNodeGroupRequest
     * @return A Java Future containing the result of the UpdateComputeNodeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.
     *         To learn how to increase your service quota, see <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html">Requesting
     *         a quota increase</a> in the <i>Service Quotas User Guide</i> </p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The max number of clusters or queues has been reached for the account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The max number of compute node groups has been reached for the associated cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The total of <code>maxInstances</code> across all compute node groups has been reached for associated
     *         cluster.
     *         </p>
     *         </li></li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.UpdateComputeNodeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/UpdateComputeNodeGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateComputeNodeGroupResponse> updateComputeNodeGroup(
            UpdateComputeNodeGroupRequest updateComputeNodeGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateComputeNodeGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateComputeNodeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateComputeNodeGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateComputeNodeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateComputeNodeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateComputeNodeGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateComputeNodeGroupRequest, UpdateComputeNodeGroupResponse>()
                            .withOperationName("UpdateComputeNodeGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateComputeNodeGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateComputeNodeGroupRequest));
            CompletableFuture<UpdateComputeNodeGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the compute node group configuration of a queue. Use this API to change the compute node groups that the
     * queue can send jobs to.
     * </p>
     *
     * @param updateQueueRequest
     * @return A Java Future containing the result of the UpdateQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.
     *         To learn how to increase your service quota, see <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html">Requesting
     *         a quota increase</a> in the <i>Service Quotas User Guide</i> </p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The max number of clusters or queues has been reached for the account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The max number of compute node groups has been reached for the associated cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The total of <code>maxInstances</code> across all compute node groups has been reached for associated
     *         cluster.
     *         </p>
     *         </li></li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. Check the resource's request rate
     *         quota and try again.</li>
     *         <li>ValidationException The request isn't valid.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Your request contains malformed JSON or unsupported characters.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The scheduler version isn't supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are networking related errors, such as network validation failure.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AMI type is <code>CUSTOM</code> and the launch template doesn't define the AMI ID, or the AMI type is AL2
     *         and the launch template defines the AMI.
     *         </p>
     *         </li></li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than 1 operation on the same resource at the same time.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A cluster with the same name already exists.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster isn't in <code>ACTIVE</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A cluster to delete is in an unstable state. For example, because it still has <code>ACTIVE</code> node
     *         groups or queues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A queue already exists in a cluster.
     *         </p>
     *         </li></li>
     *         <li>ResourceNotFoundException The requested resource can't be found. The cluster, node group, or queue
     *         you're attempting to get, update, list, or delete doesn't exist.</p>
     *         <p>
     *         <u>Examples</u></li>
     *         <li>InternalServerException Amazon Web Services PCS can't process your request right now. Try again
     *         later.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</p>
     *         <p>
     *         <u>Examples</u>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The launch template instance profile doesn't pass <code>iam:PassRole</code> verification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a mismatch between the account ID and cluster ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The cluster ID doesn't exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The EC2 instance isn't present.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PcsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PcsAsyncClient.UpdateQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pcs-2023-02-10/UpdateQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateQueueResponse> updateQueue(UpdateQueueRequest updateQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PCS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateQueueRequest, UpdateQueueResponse>()
                            .withOperationName("UpdateQueue").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateQueueRequest));
            CompletableFuture<UpdateQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PcsServiceClientConfiguration serviceClientConfiguration() {
        return new PcsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PcsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        PcsServiceClientConfigurationBuilder serviceConfigBuilder = new PcsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
