/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pcs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The cluster resource and configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Cluster implements SdkPojo, Serializable, ToCopyableBuilder<Cluster.Builder, Cluster> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Cluster::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Cluster::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Cluster::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Cluster::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(Cluster::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt")
            .getter(getter(Cluster::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Scheduler> SCHEDULER_FIELD = SdkField.<Scheduler> builder(MarshallingType.SDK_POJO)
            .memberName("scheduler").getter(getter(Cluster::scheduler)).setter(setter(Builder::scheduler))
            .constructor(Scheduler::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scheduler").build()).build();

    private static final SdkField<String> SIZE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("size")
            .getter(getter(Cluster::sizeAsString)).setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("size").build()).build();

    private static final SdkField<ClusterSlurmConfiguration> SLURM_CONFIGURATION_FIELD = SdkField
            .<ClusterSlurmConfiguration> builder(MarshallingType.SDK_POJO).memberName("slurmConfiguration")
            .getter(getter(Cluster::slurmConfiguration)).setter(setter(Builder::slurmConfiguration))
            .constructor(ClusterSlurmConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slurmConfiguration").build())
            .build();

    private static final SdkField<Networking> NETWORKING_FIELD = SdkField.<Networking> builder(MarshallingType.SDK_POJO)
            .memberName("networking").getter(getter(Cluster::networking)).setter(setter(Builder::networking))
            .constructor(Networking::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networking").build()).build();

    private static final SdkField<List<Endpoint>> ENDPOINTS_FIELD = SdkField
            .<List<Endpoint>> builder(MarshallingType.LIST)
            .memberName("endpoints")
            .getter(getter(Cluster::endpoints))
            .setter(setter(Builder::endpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Endpoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(Endpoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ErrorInfo>> ERROR_INFO_FIELD = SdkField
            .<List<ErrorInfo>> builder(MarshallingType.LIST)
            .memberName("errorInfo")
            .getter(getter(Cluster::errorInfo))
            .setter(setter(Builder::errorInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("errorInfo").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ErrorInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(ErrorInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            ARN_FIELD, STATUS_FIELD, CREATED_AT_FIELD, MODIFIED_AT_FIELD, SCHEDULER_FIELD, SIZE_FIELD, SLURM_CONFIGURATION_FIELD,
            NETWORKING_FIELD, ENDPOINTS_FIELD, ERROR_INFO_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final String arn;

    private final String status;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private final Scheduler scheduler;

    private final String size;

    private final ClusterSlurmConfiguration slurmConfiguration;

    private final Networking networking;

    private final List<Endpoint> endpoints;

    private final List<ErrorInfo> errorInfo;

    private Cluster(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.arn = builder.arn;
        this.status = builder.status;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
        this.scheduler = builder.scheduler;
        this.size = builder.size;
        this.slurmConfiguration = builder.slurmConfiguration;
        this.networking = builder.networking;
        this.endpoints = builder.endpoints;
        this.errorInfo = builder.errorInfo;
    }

    /**
     * <p>
     * The name that identifies the cluster.
     * </p>
     * 
     * @return The name that identifies the cluster.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The generated unique ID of the cluster.
     * </p>
     * 
     * @return The generated unique ID of the cluster.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The unique Amazon Resource Name (ARN) of the cluster.
     * </p>
     * 
     * @return The unique Amazon Resource Name (ARN) of the cluster.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The provisioning status of the cluster.
     * </p>
     * <note>
     * <p>
     * The provisioning status doesn't indicate the overall health of the cluster.
     * </p>
     * </note> <important>
     * <p>
     * The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler is beyond
     * end of life and we have suspended the cluster. When in these states, you can't use the cluster. The cluster
     * controller is down and all compute instances are terminated. The resources still count toward your service
     * quotas. You can delete a resource if its status is <code>SUSPENDED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked questions about
     * Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The provisioning status of the cluster.</p> <note>
     *         <p>
     *         The provisioning status doesn't indicate the overall health of the cluster.
     *         </p>
     *         </note> <important>
     *         <p>
     *         The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler is
     *         beyond end of life and we have suspended the cluster. When in these states, you can't use the cluster.
     *         The cluster controller is down and all compute instances are terminated. The resources still count toward
     *         your service quotas. You can delete a resource if its status is <code>SUSPENDED</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked
     *         questions about Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
     *         </p>
     * @see ClusterStatus
     */
    public final ClusterStatus status() {
        return ClusterStatus.fromValue(status);
    }

    /**
     * <p>
     * The provisioning status of the cluster.
     * </p>
     * <note>
     * <p>
     * The provisioning status doesn't indicate the overall health of the cluster.
     * </p>
     * </note> <important>
     * <p>
     * The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler is beyond
     * end of life and we have suspended the cluster. When in these states, you can't use the cluster. The cluster
     * controller is down and all compute instances are terminated. The resources still count toward your service
     * quotas. You can delete a resource if its status is <code>SUSPENDED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked questions about
     * Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The provisioning status of the cluster.</p> <note>
     *         <p>
     *         The provisioning status doesn't indicate the overall health of the cluster.
     *         </p>
     *         </note> <important>
     *         <p>
     *         The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler is
     *         beyond end of life and we have suspended the cluster. When in these states, you can't use the cluster.
     *         The cluster controller is down and all compute instances are terminated. The resources still count toward
     *         your service quotas. You can delete a resource if its status is <code>SUSPENDED</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked
     *         questions about Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
     *         </p>
     * @see ClusterStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time the resource was created.
     * </p>
     * 
     * @return The date and time the resource was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The date and time the resource was modified.
     * </p>
     * 
     * @return The date and time the resource was modified.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * Returns the value of the Scheduler property for this object.
     * 
     * @return The value of the Scheduler property for this object.
     */
    public final Scheduler scheduler() {
        return scheduler;
    }

    /**
     * <p>
     * The size of the cluster.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SMALL</code>: 32 compute nodes and 256 jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #size} will return
     * {@link Size#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sizeAsString}.
     * </p>
     * 
     * @return The size of the cluster.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SMALL</code>: 32 compute nodes and 256 jobs
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
     *         </p>
     *         </li>
     * @see Size
     */
    public final Size size() {
        return Size.fromValue(size);
    }

    /**
     * <p>
     * The size of the cluster.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SMALL</code>: 32 compute nodes and 256 jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #size} will return
     * {@link Size#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sizeAsString}.
     * </p>
     * 
     * @return The size of the cluster.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SMALL</code>: 32 compute nodes and 256 jobs
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
     *         </p>
     *         </li>
     * @see Size
     */
    public final String sizeAsString() {
        return size;
    }

    /**
     * <p>
     * Additional options related to the Slurm scheduler.
     * </p>
     * 
     * @return Additional options related to the Slurm scheduler.
     */
    public final ClusterSlurmConfiguration slurmConfiguration() {
        return slurmConfiguration;
    }

    /**
     * Returns the value of the Networking property for this object.
     * 
     * @return The value of the Networking property for this object.
     */
    public final Networking networking() {
        return networking;
    }

    /**
     * For responses, this returns true if the service returned a value for the Endpoints property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEndpoints() {
        return endpoints != null && !(endpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of endpoints available for interaction with the scheduler.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndpoints} method.
     * </p>
     * 
     * @return The list of endpoints available for interaction with the scheduler.
     */
    public final List<Endpoint> endpoints() {
        return endpoints;
    }

    /**
     * For responses, this returns true if the service returned a value for the ErrorInfo property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasErrorInfo() {
        return errorInfo != null && !(errorInfo instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of errors that occurred during cluster provisioning.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasErrorInfo} method.
     * </p>
     * 
     * @return The list of errors that occurred during cluster provisioning.
     */
    public final List<ErrorInfo> errorInfo() {
        return errorInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(scheduler());
        hashCode = 31 * hashCode + Objects.hashCode(sizeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(slurmConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(networking());
        hashCode = 31 * hashCode + Objects.hashCode(hasEndpoints() ? endpoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasErrorInfo() ? errorInfo() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Cluster)) {
            return false;
        }
        Cluster other = (Cluster) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt()) && Objects.equals(scheduler(), other.scheduler())
                && Objects.equals(sizeAsString(), other.sizeAsString())
                && Objects.equals(slurmConfiguration(), other.slurmConfiguration())
                && Objects.equals(networking(), other.networking()) && hasEndpoints() == other.hasEndpoints()
                && Objects.equals(endpoints(), other.endpoints()) && hasErrorInfo() == other.hasErrorInfo()
                && Objects.equals(errorInfo(), other.errorInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Cluster").add("Name", name()).add("Id", id()).add("Arn", arn()).add("Status", statusAsString())
                .add("CreatedAt", createdAt()).add("ModifiedAt", modifiedAt()).add("Scheduler", scheduler())
                .add("Size", sizeAsString()).add("SlurmConfiguration", slurmConfiguration()).add("Networking", networking())
                .add("Endpoints", hasEndpoints() ? endpoints() : null).add("ErrorInfo", hasErrorInfo() ? errorInfo() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "scheduler":
            return Optional.ofNullable(clazz.cast(scheduler()));
        case "size":
            return Optional.ofNullable(clazz.cast(sizeAsString()));
        case "slurmConfiguration":
            return Optional.ofNullable(clazz.cast(slurmConfiguration()));
        case "networking":
            return Optional.ofNullable(clazz.cast(networking()));
        case "endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        case "errorInfo":
            return Optional.ofNullable(clazz.cast(errorInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("id", ID_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("modifiedAt", MODIFIED_AT_FIELD);
        map.put("scheduler", SCHEDULER_FIELD);
        map.put("size", SIZE_FIELD);
        map.put("slurmConfiguration", SLURM_CONFIGURATION_FIELD);
        map.put("networking", NETWORKING_FIELD);
        map.put("endpoints", ENDPOINTS_FIELD);
        map.put("errorInfo", ERROR_INFO_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Cluster, T> g) {
        return obj -> g.apply((Cluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Cluster> {
        /**
         * <p>
         * The name that identifies the cluster.
         * </p>
         * 
         * @param name
         *        The name that identifies the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The generated unique ID of the cluster.
         * </p>
         * 
         * @param id
         *        The generated unique ID of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The unique Amazon Resource Name (ARN) of the cluster.
         * </p>
         * 
         * @param arn
         *        The unique Amazon Resource Name (ARN) of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The provisioning status of the cluster.
         * </p>
         * <note>
         * <p>
         * The provisioning status doesn't indicate the overall health of the cluster.
         * </p>
         * </note> <important>
         * <p>
         * The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler is
         * beyond end of life and we have suspended the cluster. When in these states, you can't use the cluster. The
         * cluster controller is down and all compute instances are terminated. The resources still count toward your
         * service quotas. You can delete a resource if its status is <code>SUSPENDED</code>. For more information, see
         * <a href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked questions
         * about Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
         * </p>
         * </important>
         * 
         * @param status
         *        The provisioning status of the cluster.</p> <note>
         *        <p>
         *        The provisioning status doesn't indicate the overall health of the cluster.
         *        </p>
         *        </note> <important>
         *        <p>
         *        The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler
         *        is beyond end of life and we have suspended the cluster. When in these states, you can't use the
         *        cluster. The cluster controller is down and all compute instances are terminated. The resources still
         *        count toward your service quotas. You can delete a resource if its status is <code>SUSPENDED</code>.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked
         *        questions about Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
         *        </p>
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The provisioning status of the cluster.
         * </p>
         * <note>
         * <p>
         * The provisioning status doesn't indicate the overall health of the cluster.
         * </p>
         * </note> <important>
         * <p>
         * The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler is
         * beyond end of life and we have suspended the cluster. When in these states, you can't use the cluster. The
         * cluster controller is down and all compute instances are terminated. The resources still count toward your
         * service quotas. You can delete a resource if its status is <code>SUSPENDED</code>. For more information, see
         * <a href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked questions
         * about Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
         * </p>
         * </important>
         * 
         * @param status
         *        The provisioning status of the cluster.</p> <note>
         *        <p>
         *        The provisioning status doesn't indicate the overall health of the cluster.
         *        </p>
         *        </note> <important>
         *        <p>
         *        The resource enters the <code>SUSPENDING</code> and <code>SUSPENDED</code> states when the scheduler
         *        is beyond end of life and we have suspended the cluster. When in these states, you can't use the
         *        cluster. The cluster controller is down and all compute instances are terminated. The resources still
         *        count toward your service quotas. You can delete a resource if its status is <code>SUSPENDED</code>.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/pcs/latest/userguide/slurm-versions_faq.html">Frequently asked
         *        questions about Slurm versions in PCS</a> in the <i>PCS User Guide</i>.
         *        </p>
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(ClusterStatus status);

        /**
         * <p>
         * The date and time the resource was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time the resource was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The date and time the resource was modified.
         * </p>
         * 
         * @param modifiedAt
         *        The date and time the resource was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * Sets the value of the Scheduler property for this object.
         *
         * @param scheduler
         *        The new value for the Scheduler property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduler(Scheduler scheduler);

        /**
         * Sets the value of the Scheduler property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Scheduler.Builder} avoiding the need to
         * create one manually via {@link Scheduler#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Scheduler.Builder#build()} is called immediately and its result
         * is passed to {@link #scheduler(Scheduler)}.
         * 
         * @param scheduler
         *        a consumer that will call methods on {@link Scheduler.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scheduler(Scheduler)
         */
        default Builder scheduler(Consumer<Scheduler.Builder> scheduler) {
            return scheduler(Scheduler.builder().applyMutation(scheduler).build());
        }

        /**
         * <p>
         * The size of the cluster.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SMALL</code>: 32 compute nodes and 256 jobs
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
         * </p>
         * </li>
         * </ul>
         * 
         * @param size
         *        The size of the cluster.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SMALL</code>: 32 compute nodes and 256 jobs
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
         *        </p>
         *        </li>
         * @see Size
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Size
         */
        Builder size(String size);

        /**
         * <p>
         * The size of the cluster.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SMALL</code>: 32 compute nodes and 256 jobs
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
         * </p>
         * </li>
         * </ul>
         * 
         * @param size
         *        The size of the cluster.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SMALL</code>: 32 compute nodes and 256 jobs
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MEDIUM</code>: 512 compute nodes and 8192 jobs
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LARGE</code>: 2048 compute nodes and 16,384 jobs
         *        </p>
         *        </li>
         * @see Size
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Size
         */
        Builder size(Size size);

        /**
         * <p>
         * Additional options related to the Slurm scheduler.
         * </p>
         * 
         * @param slurmConfiguration
         *        Additional options related to the Slurm scheduler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slurmConfiguration(ClusterSlurmConfiguration slurmConfiguration);

        /**
         * <p>
         * Additional options related to the Slurm scheduler.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClusterSlurmConfiguration.Builder}
         * avoiding the need to create one manually via {@link ClusterSlurmConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClusterSlurmConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #slurmConfiguration(ClusterSlurmConfiguration)}.
         * 
         * @param slurmConfiguration
         *        a consumer that will call methods on {@link ClusterSlurmConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #slurmConfiguration(ClusterSlurmConfiguration)
         */
        default Builder slurmConfiguration(Consumer<ClusterSlurmConfiguration.Builder> slurmConfiguration) {
            return slurmConfiguration(ClusterSlurmConfiguration.builder().applyMutation(slurmConfiguration).build());
        }

        /**
         * Sets the value of the Networking property for this object.
         *
         * @param networking
         *        The new value for the Networking property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networking(Networking networking);

        /**
         * Sets the value of the Networking property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Networking.Builder} avoiding the need to
         * create one manually via {@link Networking#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Networking.Builder#build()} is called immediately and its result
         * is passed to {@link #networking(Networking)}.
         * 
         * @param networking
         *        a consumer that will call methods on {@link Networking.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networking(Networking)
         */
        default Builder networking(Consumer<Networking.Builder> networking) {
            return networking(Networking.builder().applyMutation(networking).build());
        }

        /**
         * <p>
         * The list of endpoints available for interaction with the scheduler.
         * </p>
         * 
         * @param endpoints
         *        The list of endpoints available for interaction with the scheduler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(Collection<Endpoint> endpoints);

        /**
         * <p>
         * The list of endpoints available for interaction with the scheduler.
         * </p>
         * 
         * @param endpoints
         *        The list of endpoints available for interaction with the scheduler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(Endpoint... endpoints);

        /**
         * <p>
         * The list of endpoints available for interaction with the scheduler.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pcs.model.Endpoint.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.pcs.model.Endpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.pcs.model.Endpoint.Builder#build()} is called immediately and its
         * result is passed to {@link #endpoints(List<Endpoint>)}.
         * 
         * @param endpoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.pcs.model.Endpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpoints(java.util.Collection<Endpoint>)
         */
        Builder endpoints(Consumer<Endpoint.Builder>... endpoints);

        /**
         * <p>
         * The list of errors that occurred during cluster provisioning.
         * </p>
         * 
         * @param errorInfo
         *        The list of errors that occurred during cluster provisioning.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorInfo(Collection<ErrorInfo> errorInfo);

        /**
         * <p>
         * The list of errors that occurred during cluster provisioning.
         * </p>
         * 
         * @param errorInfo
         *        The list of errors that occurred during cluster provisioning.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorInfo(ErrorInfo... errorInfo);

        /**
         * <p>
         * The list of errors that occurred during cluster provisioning.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pcs.model.ErrorInfo.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.pcs.model.ErrorInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.pcs.model.ErrorInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #errorInfo(List<ErrorInfo>)}.
         * 
         * @param errorInfo
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.pcs.model.ErrorInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #errorInfo(java.util.Collection<ErrorInfo>)
         */
        Builder errorInfo(Consumer<ErrorInfo.Builder>... errorInfo);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private String arn;

        private String status;

        private Instant createdAt;

        private Instant modifiedAt;

        private Scheduler scheduler;

        private String size;

        private ClusterSlurmConfiguration slurmConfiguration;

        private Networking networking;

        private List<Endpoint> endpoints = DefaultSdkAutoConstructList.getInstance();

        private List<ErrorInfo> errorInfo = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Cluster model) {
            name(model.name);
            id(model.id);
            arn(model.arn);
            status(model.status);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
            scheduler(model.scheduler);
            size(model.size);
            slurmConfiguration(model.slurmConfiguration);
            networking(model.networking);
            endpoints(model.endpoints);
            errorInfo(model.errorInfo);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ClusterStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final Scheduler.Builder getScheduler() {
            return scheduler != null ? scheduler.toBuilder() : null;
        }

        public final void setScheduler(Scheduler.BuilderImpl scheduler) {
            this.scheduler = scheduler != null ? scheduler.build() : null;
        }

        @Override
        public final Builder scheduler(Scheduler scheduler) {
            this.scheduler = scheduler;
            return this;
        }

        public final String getSize() {
            return size;
        }

        public final void setSize(String size) {
            this.size = size;
        }

        @Override
        public final Builder size(String size) {
            this.size = size;
            return this;
        }

        @Override
        public final Builder size(Size size) {
            this.size(size == null ? null : size.toString());
            return this;
        }

        public final ClusterSlurmConfiguration.Builder getSlurmConfiguration() {
            return slurmConfiguration != null ? slurmConfiguration.toBuilder() : null;
        }

        public final void setSlurmConfiguration(ClusterSlurmConfiguration.BuilderImpl slurmConfiguration) {
            this.slurmConfiguration = slurmConfiguration != null ? slurmConfiguration.build() : null;
        }

        @Override
        public final Builder slurmConfiguration(ClusterSlurmConfiguration slurmConfiguration) {
            this.slurmConfiguration = slurmConfiguration;
            return this;
        }

        public final Networking.Builder getNetworking() {
            return networking != null ? networking.toBuilder() : null;
        }

        public final void setNetworking(Networking.BuilderImpl networking) {
            this.networking = networking != null ? networking.build() : null;
        }

        @Override
        public final Builder networking(Networking networking) {
            this.networking = networking;
            return this;
        }

        public final List<Endpoint.Builder> getEndpoints() {
            List<Endpoint.Builder> result = EndpointsCopier.copyToBuilder(this.endpoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEndpoints(Collection<Endpoint.BuilderImpl> endpoints) {
            this.endpoints = EndpointsCopier.copyFromBuilder(endpoints);
        }

        @Override
        public final Builder endpoints(Collection<Endpoint> endpoints) {
            this.endpoints = EndpointsCopier.copy(endpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpoints(Endpoint... endpoints) {
            endpoints(Arrays.asList(endpoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpoints(Consumer<Endpoint.Builder>... endpoints) {
            endpoints(Stream.of(endpoints).map(c -> Endpoint.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<ErrorInfo.Builder> getErrorInfo() {
            List<ErrorInfo.Builder> result = ErrorInfoListCopier.copyToBuilder(this.errorInfo);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setErrorInfo(Collection<ErrorInfo.BuilderImpl> errorInfo) {
            this.errorInfo = ErrorInfoListCopier.copyFromBuilder(errorInfo);
        }

        @Override
        public final Builder errorInfo(Collection<ErrorInfo> errorInfo) {
            this.errorInfo = ErrorInfoListCopier.copy(errorInfo);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder errorInfo(ErrorInfo... errorInfo) {
            errorInfo(Arrays.asList(errorInfo));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder errorInfo(Consumer<ErrorInfo.Builder>... errorInfo) {
            errorInfo(Stream.of(errorInfo).map(c -> ErrorInfo.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Cluster build() {
            return new Cluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
