/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalize.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the name and default range of a categorical hyperparameter and whether the hyperparameter is tunable. A
 * tunable hyperparameter can have its value determined during hyperparameter optimization (HPO).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DefaultCategoricalHyperParameterRange implements SdkPojo, Serializable,
        ToCopyableBuilder<DefaultCategoricalHyperParameterRange.Builder, DefaultCategoricalHyperParameterRange> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(DefaultCategoricalHyperParameterRange::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("values")
            .getter(getter(DefaultCategoricalHyperParameterRange::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> IS_TUNABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isTunable").getter(getter(DefaultCategoricalHyperParameterRange::isTunable))
            .setter(setter(Builder::isTunable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isTunable").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VALUES_FIELD,
            IS_TUNABLE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> values;

    private final Boolean isTunable;

    private DefaultCategoricalHyperParameterRange(BuilderImpl builder) {
        this.name = builder.name;
        this.values = builder.values;
        this.isTunable = builder.isTunable;
    }

    /**
     * <p>
     * The name of the hyperparameter.
     * </p>
     * 
     * @return The name of the hyperparameter.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns true if the Values property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the categories for the hyperparameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the categories for the hyperparameter.
     */
    public final List<String> values() {
        return values;
    }

    /**
     * <p>
     * Whether the hyperparameter is tunable.
     * </p>
     * 
     * @return Whether the hyperparameter is tunable.
     */
    public final Boolean isTunable() {
        return isTunable;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        hashCode = 31 * hashCode + Objects.hashCode(isTunable());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DefaultCategoricalHyperParameterRange)) {
            return false;
        }
        DefaultCategoricalHyperParameterRange other = (DefaultCategoricalHyperParameterRange) obj;
        return Objects.equals(name(), other.name()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values()) && Objects.equals(isTunable(), other.isTunable());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DefaultCategoricalHyperParameterRange").add("Name", name())
                .add("Values", hasValues() ? values() : null).add("IsTunable", isTunable()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "values":
            return Optional.ofNullable(clazz.cast(values()));
        case "isTunable":
            return Optional.ofNullable(clazz.cast(isTunable()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DefaultCategoricalHyperParameterRange, T> g) {
        return obj -> g.apply((DefaultCategoricalHyperParameterRange) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DefaultCategoricalHyperParameterRange> {
        /**
         * <p>
         * The name of the hyperparameter.
         * </p>
         * 
         * @param name
         *        The name of the hyperparameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A list of the categories for the hyperparameter.
         * </p>
         * 
         * @param values
         *        A list of the categories for the hyperparameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * A list of the categories for the hyperparameter.
         * </p>
         * 
         * @param values
         *        A list of the categories for the hyperparameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);

        /**
         * <p>
         * Whether the hyperparameter is tunable.
         * </p>
         * 
         * @param isTunable
         *        Whether the hyperparameter is tunable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isTunable(Boolean isTunable);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private Boolean isTunable;

        private BuilderImpl() {
        }

        private BuilderImpl(DefaultCategoricalHyperParameterRange model) {
            name(model.name);
            values(model.values);
            isTunable(model.isTunable);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = CategoricalValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<String> values) {
            this.values = CategoricalValuesCopier.copy(values);
        }

        public final Boolean getIsTunable() {
            return isTunable;
        }

        @Override
        public final Builder isTunable(Boolean isTunable) {
            this.isTunable = isTunable;
            return this;
        }

        public final void setIsTunable(Boolean isTunable) {
            this.isTunable = isTunable;
        }

        @Override
        public DefaultCategoricalHyperParameterRange build() {
            return new DefaultCategoricalHyperParameterRange(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
