/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalize.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration properties for the solution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SolutionConfig implements SdkPojo, Serializable, ToCopyableBuilder<SolutionConfig.Builder, SolutionConfig> {
    private static final SdkField<String> EVENT_VALUE_THRESHOLD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventValueThreshold").getter(getter(SolutionConfig::eventValueThreshold))
            .setter(setter(Builder::eventValueThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventValueThreshold").build())
            .build();

    private static final SdkField<HPOConfig> HPO_CONFIG_FIELD = SdkField.<HPOConfig> builder(MarshallingType.SDK_POJO)
            .memberName("hpoConfig").getter(getter(SolutionConfig::hpoConfig)).setter(setter(Builder::hpoConfig))
            .constructor(HPOConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hpoConfig").build()).build();

    private static final SdkField<Map<String, String>> ALGORITHM_HYPER_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("algorithmHyperParameters")
            .getter(getter(SolutionConfig::algorithmHyperParameters))
            .setter(setter(Builder::algorithmHyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithmHyperParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> FEATURE_TRANSFORMATION_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("featureTransformationParameters")
            .getter(getter(SolutionConfig::featureTransformationParameters))
            .setter(setter(Builder::featureTransformationParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("featureTransformationParameters")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<AutoMLConfig> AUTO_ML_CONFIG_FIELD = SdkField.<AutoMLConfig> builder(MarshallingType.SDK_POJO)
            .memberName("autoMLConfig").getter(getter(SolutionConfig::autoMLConfig)).setter(setter(Builder::autoMLConfig))
            .constructor(AutoMLConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoMLConfig").build()).build();

    private static final SdkField<OptimizationObjective> OPTIMIZATION_OBJECTIVE_FIELD = SdkField
            .<OptimizationObjective> builder(MarshallingType.SDK_POJO).memberName("optimizationObjective")
            .getter(getter(SolutionConfig::optimizationObjective)).setter(setter(Builder::optimizationObjective))
            .constructor(OptimizationObjective::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("optimizationObjective").build())
            .build();

    private static final SdkField<TrainingDataConfig> TRAINING_DATA_CONFIG_FIELD = SdkField
            .<TrainingDataConfig> builder(MarshallingType.SDK_POJO).memberName("trainingDataConfig")
            .getter(getter(SolutionConfig::trainingDataConfig)).setter(setter(Builder::trainingDataConfig))
            .constructor(TrainingDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("trainingDataConfig").build())
            .build();

    private static final SdkField<AutoTrainingConfig> AUTO_TRAINING_CONFIG_FIELD = SdkField
            .<AutoTrainingConfig> builder(MarshallingType.SDK_POJO).memberName("autoTrainingConfig")
            .getter(getter(SolutionConfig::autoTrainingConfig)).setter(setter(Builder::autoTrainingConfig))
            .constructor(AutoTrainingConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoTrainingConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_VALUE_THRESHOLD_FIELD,
            HPO_CONFIG_FIELD, ALGORITHM_HYPER_PARAMETERS_FIELD, FEATURE_TRANSFORMATION_PARAMETERS_FIELD, AUTO_ML_CONFIG_FIELD,
            OPTIMIZATION_OBJECTIVE_FIELD, TRAINING_DATA_CONFIG_FIELD, AUTO_TRAINING_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("eventValueThreshold", EVENT_VALUE_THRESHOLD_FIELD);
                    put("hpoConfig", HPO_CONFIG_FIELD);
                    put("algorithmHyperParameters", ALGORITHM_HYPER_PARAMETERS_FIELD);
                    put("featureTransformationParameters", FEATURE_TRANSFORMATION_PARAMETERS_FIELD);
                    put("autoMLConfig", AUTO_ML_CONFIG_FIELD);
                    put("optimizationObjective", OPTIMIZATION_OBJECTIVE_FIELD);
                    put("trainingDataConfig", TRAINING_DATA_CONFIG_FIELD);
                    put("autoTrainingConfig", AUTO_TRAINING_CONFIG_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String eventValueThreshold;

    private final HPOConfig hpoConfig;

    private final Map<String, String> algorithmHyperParameters;

    private final Map<String, String> featureTransformationParameters;

    private final AutoMLConfig autoMLConfig;

    private final OptimizationObjective optimizationObjective;

    private final TrainingDataConfig trainingDataConfig;

    private final AutoTrainingConfig autoTrainingConfig;

    private SolutionConfig(BuilderImpl builder) {
        this.eventValueThreshold = builder.eventValueThreshold;
        this.hpoConfig = builder.hpoConfig;
        this.algorithmHyperParameters = builder.algorithmHyperParameters;
        this.featureTransformationParameters = builder.featureTransformationParameters;
        this.autoMLConfig = builder.autoMLConfig;
        this.optimizationObjective = builder.optimizationObjective;
        this.trainingDataConfig = builder.trainingDataConfig;
        this.autoTrainingConfig = builder.autoTrainingConfig;
    }

    /**
     * <p>
     * Only events with a value greater than or equal to this threshold are used for training a model.
     * </p>
     * 
     * @return Only events with a value greater than or equal to this threshold are used for training a model.
     */
    public final String eventValueThreshold() {
        return eventValueThreshold;
    }

    /**
     * <p>
     * Describes the properties for hyperparameter optimization (HPO).
     * </p>
     * 
     * @return Describes the properties for hyperparameter optimization (HPO).
     */
    public final HPOConfig hpoConfig() {
        return hpoConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the AlgorithmHyperParameters property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAlgorithmHyperParameters() {
        return algorithmHyperParameters != null && !(algorithmHyperParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Lists the algorithm hyperparameters and their values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAlgorithmHyperParameters} method.
     * </p>
     * 
     * @return Lists the algorithm hyperparameters and their values.
     */
    public final Map<String, String> algorithmHyperParameters() {
        return algorithmHyperParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the FeatureTransformationParameters
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasFeatureTransformationParameters() {
        return featureTransformationParameters != null && !(featureTransformationParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Lists the feature transformation parameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureTransformationParameters}
     * method.
     * </p>
     * 
     * @return Lists the feature transformation parameters.
     */
    public final Map<String, String> featureTransformationParameters() {
        return featureTransformationParameters;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/personalize/latest/dg/API_AutoMLConfig.html">AutoMLConfig</a> object
     * containing a list of recipes to search when AutoML is performed.
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/personalize/latest/dg/API_AutoMLConfig.html">AutoMLConfig</a>
     *         object containing a list of recipes to search when AutoML is performed.
     */
    public final AutoMLConfig autoMLConfig() {
        return autoMLConfig;
    }

    /**
     * <p>
     * Describes the additional objective for the solution, such as maximizing streaming minutes or increasing revenue.
     * For more information see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/optimizing-solution-for-objective.html">Optimizing a
     * solution</a>.
     * </p>
     * 
     * @return Describes the additional objective for the solution, such as maximizing streaming minutes or increasing
     *         revenue. For more information see <a
     *         href="https://docs.aws.amazon.com/personalize/latest/dg/optimizing-solution-for-objective.html"
     *         >Optimizing a solution</a>.
     */
    public final OptimizationObjective optimizationObjective() {
        return optimizationObjective;
    }

    /**
     * <p>
     * Specifies the training data configuration to use when creating a custom solution version (trained model).
     * </p>
     * 
     * @return Specifies the training data configuration to use when creating a custom solution version (trained model).
     */
    public final TrainingDataConfig trainingDataConfig() {
        return trainingDataConfig;
    }

    /**
     * <p>
     * Specifies the automatic training configuration to use.
     * </p>
     * 
     * @return Specifies the automatic training configuration to use.
     */
    public final AutoTrainingConfig autoTrainingConfig() {
        return autoTrainingConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventValueThreshold());
        hashCode = 31 * hashCode + Objects.hashCode(hpoConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlgorithmHyperParameters() ? algorithmHyperParameters() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasFeatureTransformationParameters() ? featureTransformationParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(autoMLConfig());
        hashCode = 31 * hashCode + Objects.hashCode(optimizationObjective());
        hashCode = 31 * hashCode + Objects.hashCode(trainingDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(autoTrainingConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SolutionConfig)) {
            return false;
        }
        SolutionConfig other = (SolutionConfig) obj;
        return Objects.equals(eventValueThreshold(), other.eventValueThreshold())
                && Objects.equals(hpoConfig(), other.hpoConfig())
                && hasAlgorithmHyperParameters() == other.hasAlgorithmHyperParameters()
                && Objects.equals(algorithmHyperParameters(), other.algorithmHyperParameters())
                && hasFeatureTransformationParameters() == other.hasFeatureTransformationParameters()
                && Objects.equals(featureTransformationParameters(), other.featureTransformationParameters())
                && Objects.equals(autoMLConfig(), other.autoMLConfig())
                && Objects.equals(optimizationObjective(), other.optimizationObjective())
                && Objects.equals(trainingDataConfig(), other.trainingDataConfig())
                && Objects.equals(autoTrainingConfig(), other.autoTrainingConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("SolutionConfig")
                .add("EventValueThreshold", eventValueThreshold())
                .add("HpoConfig", hpoConfig())
                .add("AlgorithmHyperParameters", hasAlgorithmHyperParameters() ? algorithmHyperParameters() : null)
                .add("FeatureTransformationParameters",
                        hasFeatureTransformationParameters() ? featureTransformationParameters() : null)
                .add("AutoMLConfig", autoMLConfig()).add("OptimizationObjective", optimizationObjective())
                .add("TrainingDataConfig", trainingDataConfig()).add("AutoTrainingConfig", autoTrainingConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eventValueThreshold":
            return Optional.ofNullable(clazz.cast(eventValueThreshold()));
        case "hpoConfig":
            return Optional.ofNullable(clazz.cast(hpoConfig()));
        case "algorithmHyperParameters":
            return Optional.ofNullable(clazz.cast(algorithmHyperParameters()));
        case "featureTransformationParameters":
            return Optional.ofNullable(clazz.cast(featureTransformationParameters()));
        case "autoMLConfig":
            return Optional.ofNullable(clazz.cast(autoMLConfig()));
        case "optimizationObjective":
            return Optional.ofNullable(clazz.cast(optimizationObjective()));
        case "trainingDataConfig":
            return Optional.ofNullable(clazz.cast(trainingDataConfig()));
        case "autoTrainingConfig":
            return Optional.ofNullable(clazz.cast(autoTrainingConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<SolutionConfig, T> g) {
        return obj -> g.apply((SolutionConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SolutionConfig> {
        /**
         * <p>
         * Only events with a value greater than or equal to this threshold are used for training a model.
         * </p>
         * 
         * @param eventValueThreshold
         *        Only events with a value greater than or equal to this threshold are used for training a model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventValueThreshold(String eventValueThreshold);

        /**
         * <p>
         * Describes the properties for hyperparameter optimization (HPO).
         * </p>
         * 
         * @param hpoConfig
         *        Describes the properties for hyperparameter optimization (HPO).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hpoConfig(HPOConfig hpoConfig);

        /**
         * <p>
         * Describes the properties for hyperparameter optimization (HPO).
         * </p>
         * This is a convenience method that creates an instance of the {@link HPOConfig.Builder} avoiding the need to
         * create one manually via {@link HPOConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HPOConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #hpoConfig(HPOConfig)}.
         * 
         * @param hpoConfig
         *        a consumer that will call methods on {@link HPOConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hpoConfig(HPOConfig)
         */
        default Builder hpoConfig(Consumer<HPOConfig.Builder> hpoConfig) {
            return hpoConfig(HPOConfig.builder().applyMutation(hpoConfig).build());
        }

        /**
         * <p>
         * Lists the algorithm hyperparameters and their values.
         * </p>
         * 
         * @param algorithmHyperParameters
         *        Lists the algorithm hyperparameters and their values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmHyperParameters(Map<String, String> algorithmHyperParameters);

        /**
         * <p>
         * Lists the feature transformation parameters.
         * </p>
         * 
         * @param featureTransformationParameters
         *        Lists the feature transformation parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureTransformationParameters(Map<String, String> featureTransformationParameters);

        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/personalize/latest/dg/API_AutoMLConfig.html">AutoMLConfig</a> object
         * containing a list of recipes to search when AutoML is performed.
         * </p>
         * 
         * @param autoMLConfig
         *        The <a href="https://docs.aws.amazon.com/personalize/latest/dg/API_AutoMLConfig.html">AutoMLConfig</a>
         *        object containing a list of recipes to search when AutoML is performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMLConfig(AutoMLConfig autoMLConfig);

        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/personalize/latest/dg/API_AutoMLConfig.html">AutoMLConfig</a> object
         * containing a list of recipes to search when AutoML is performed.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoMLConfig.Builder} avoiding the need
         * to create one manually via {@link AutoMLConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoMLConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #autoMLConfig(AutoMLConfig)}.
         * 
         * @param autoMLConfig
         *        a consumer that will call methods on {@link AutoMLConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoMLConfig(AutoMLConfig)
         */
        default Builder autoMLConfig(Consumer<AutoMLConfig.Builder> autoMLConfig) {
            return autoMLConfig(AutoMLConfig.builder().applyMutation(autoMLConfig).build());
        }

        /**
         * <p>
         * Describes the additional objective for the solution, such as maximizing streaming minutes or increasing
         * revenue. For more information see <a
         * href="https://docs.aws.amazon.com/personalize/latest/dg/optimizing-solution-for-objective.html">Optimizing a
         * solution</a>.
         * </p>
         * 
         * @param optimizationObjective
         *        Describes the additional objective for the solution, such as maximizing streaming minutes or
         *        increasing revenue. For more information see <a
         *        href="https://docs.aws.amazon.com/personalize/latest/dg/optimizing-solution-for-objective.html"
         *        >Optimizing a solution</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optimizationObjective(OptimizationObjective optimizationObjective);

        /**
         * <p>
         * Describes the additional objective for the solution, such as maximizing streaming minutes or increasing
         * revenue. For more information see <a
         * href="https://docs.aws.amazon.com/personalize/latest/dg/optimizing-solution-for-objective.html">Optimizing a
         * solution</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link OptimizationObjective.Builder} avoiding
         * the need to create one manually via {@link OptimizationObjective#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OptimizationObjective.Builder#build()} is called immediately and
         * its result is passed to {@link #optimizationObjective(OptimizationObjective)}.
         * 
         * @param optimizationObjective
         *        a consumer that will call methods on {@link OptimizationObjective.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #optimizationObjective(OptimizationObjective)
         */
        default Builder optimizationObjective(Consumer<OptimizationObjective.Builder> optimizationObjective) {
            return optimizationObjective(OptimizationObjective.builder().applyMutation(optimizationObjective).build());
        }

        /**
         * <p>
         * Specifies the training data configuration to use when creating a custom solution version (trained model).
         * </p>
         * 
         * @param trainingDataConfig
         *        Specifies the training data configuration to use when creating a custom solution version (trained
         *        model).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingDataConfig(TrainingDataConfig trainingDataConfig);

        /**
         * <p>
         * Specifies the training data configuration to use when creating a custom solution version (trained model).
         * </p>
         * This is a convenience method that creates an instance of the {@link TrainingDataConfig.Builder} avoiding the
         * need to create one manually via {@link TrainingDataConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TrainingDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #trainingDataConfig(TrainingDataConfig)}.
         * 
         * @param trainingDataConfig
         *        a consumer that will call methods on {@link TrainingDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trainingDataConfig(TrainingDataConfig)
         */
        default Builder trainingDataConfig(Consumer<TrainingDataConfig.Builder> trainingDataConfig) {
            return trainingDataConfig(TrainingDataConfig.builder().applyMutation(trainingDataConfig).build());
        }

        /**
         * <p>
         * Specifies the automatic training configuration to use.
         * </p>
         * 
         * @param autoTrainingConfig
         *        Specifies the automatic training configuration to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoTrainingConfig(AutoTrainingConfig autoTrainingConfig);

        /**
         * <p>
         * Specifies the automatic training configuration to use.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoTrainingConfig.Builder} avoiding the
         * need to create one manually via {@link AutoTrainingConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoTrainingConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #autoTrainingConfig(AutoTrainingConfig)}.
         * 
         * @param autoTrainingConfig
         *        a consumer that will call methods on {@link AutoTrainingConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoTrainingConfig(AutoTrainingConfig)
         */
        default Builder autoTrainingConfig(Consumer<AutoTrainingConfig.Builder> autoTrainingConfig) {
            return autoTrainingConfig(AutoTrainingConfig.builder().applyMutation(autoTrainingConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String eventValueThreshold;

        private HPOConfig hpoConfig;

        private Map<String, String> algorithmHyperParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> featureTransformationParameters = DefaultSdkAutoConstructMap.getInstance();

        private AutoMLConfig autoMLConfig;

        private OptimizationObjective optimizationObjective;

        private TrainingDataConfig trainingDataConfig;

        private AutoTrainingConfig autoTrainingConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(SolutionConfig model) {
            eventValueThreshold(model.eventValueThreshold);
            hpoConfig(model.hpoConfig);
            algorithmHyperParameters(model.algorithmHyperParameters);
            featureTransformationParameters(model.featureTransformationParameters);
            autoMLConfig(model.autoMLConfig);
            optimizationObjective(model.optimizationObjective);
            trainingDataConfig(model.trainingDataConfig);
            autoTrainingConfig(model.autoTrainingConfig);
        }

        public final String getEventValueThreshold() {
            return eventValueThreshold;
        }

        public final void setEventValueThreshold(String eventValueThreshold) {
            this.eventValueThreshold = eventValueThreshold;
        }

        @Override
        public final Builder eventValueThreshold(String eventValueThreshold) {
            this.eventValueThreshold = eventValueThreshold;
            return this;
        }

        public final HPOConfig.Builder getHpoConfig() {
            return hpoConfig != null ? hpoConfig.toBuilder() : null;
        }

        public final void setHpoConfig(HPOConfig.BuilderImpl hpoConfig) {
            this.hpoConfig = hpoConfig != null ? hpoConfig.build() : null;
        }

        @Override
        public final Builder hpoConfig(HPOConfig hpoConfig) {
            this.hpoConfig = hpoConfig;
            return this;
        }

        public final Map<String, String> getAlgorithmHyperParameters() {
            if (algorithmHyperParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return algorithmHyperParameters;
        }

        public final void setAlgorithmHyperParameters(Map<String, String> algorithmHyperParameters) {
            this.algorithmHyperParameters = HyperParametersCopier.copy(algorithmHyperParameters);
        }

        @Override
        public final Builder algorithmHyperParameters(Map<String, String> algorithmHyperParameters) {
            this.algorithmHyperParameters = HyperParametersCopier.copy(algorithmHyperParameters);
            return this;
        }

        public final Map<String, String> getFeatureTransformationParameters() {
            if (featureTransformationParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return featureTransformationParameters;
        }

        public final void setFeatureTransformationParameters(Map<String, String> featureTransformationParameters) {
            this.featureTransformationParameters = FeatureTransformationParametersCopier.copy(featureTransformationParameters);
        }

        @Override
        public final Builder featureTransformationParameters(Map<String, String> featureTransformationParameters) {
            this.featureTransformationParameters = FeatureTransformationParametersCopier.copy(featureTransformationParameters);
            return this;
        }

        public final AutoMLConfig.Builder getAutoMLConfig() {
            return autoMLConfig != null ? autoMLConfig.toBuilder() : null;
        }

        public final void setAutoMLConfig(AutoMLConfig.BuilderImpl autoMLConfig) {
            this.autoMLConfig = autoMLConfig != null ? autoMLConfig.build() : null;
        }

        @Override
        public final Builder autoMLConfig(AutoMLConfig autoMLConfig) {
            this.autoMLConfig = autoMLConfig;
            return this;
        }

        public final OptimizationObjective.Builder getOptimizationObjective() {
            return optimizationObjective != null ? optimizationObjective.toBuilder() : null;
        }

        public final void setOptimizationObjective(OptimizationObjective.BuilderImpl optimizationObjective) {
            this.optimizationObjective = optimizationObjective != null ? optimizationObjective.build() : null;
        }

        @Override
        public final Builder optimizationObjective(OptimizationObjective optimizationObjective) {
            this.optimizationObjective = optimizationObjective;
            return this;
        }

        public final TrainingDataConfig.Builder getTrainingDataConfig() {
            return trainingDataConfig != null ? trainingDataConfig.toBuilder() : null;
        }

        public final void setTrainingDataConfig(TrainingDataConfig.BuilderImpl trainingDataConfig) {
            this.trainingDataConfig = trainingDataConfig != null ? trainingDataConfig.build() : null;
        }

        @Override
        public final Builder trainingDataConfig(TrainingDataConfig trainingDataConfig) {
            this.trainingDataConfig = trainingDataConfig;
            return this;
        }

        public final AutoTrainingConfig.Builder getAutoTrainingConfig() {
            return autoTrainingConfig != null ? autoTrainingConfig.toBuilder() : null;
        }

        public final void setAutoTrainingConfig(AutoTrainingConfig.BuilderImpl autoTrainingConfig) {
            this.autoTrainingConfig = autoTrainingConfig != null ? autoTrainingConfig.build() : null;
        }

        @Override
        public final Builder autoTrainingConfig(AutoTrainingConfig autoTrainingConfig) {
            this.autoTrainingConfig = autoTrainingConfig;
            return this;
        }

        @Override
        public SolutionConfig build() {
            return new SolutionConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
