/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalize.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a job that imports training data from a data source (Amazon S3 bucket) to an Amazon Personalize dataset.
 * For more information, see <a
 * href="https://docs.aws.amazon.com/personalize/latest/dg/API_CreateDatasetImportJob.html">CreateDatasetImportJob</a>.
 * </p>
 * <p>
 * A dataset import job can be in one of the following states:
 * </p>
 * <ul>
 * <li>
 * <p>
 * CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DatasetImportJob implements SdkPojo, Serializable,
        ToCopyableBuilder<DatasetImportJob.Builder, DatasetImportJob> {
    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobName").getter(getter(DatasetImportJob::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobName").build()).build();

    private static final SdkField<String> DATASET_IMPORT_JOB_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("datasetImportJobArn").getter(getter(DatasetImportJob::datasetImportJobArn))
            .setter(setter(Builder::datasetImportJobArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("datasetImportJobArn").build())
            .build();

    private static final SdkField<String> DATASET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("datasetArn").getter(getter(DatasetImportJob::datasetArn)).setter(setter(Builder::datasetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("datasetArn").build()).build();

    private static final SdkField<DataSource> DATA_SOURCE_FIELD = SdkField.<DataSource> builder(MarshallingType.SDK_POJO)
            .memberName("dataSource").getter(getter(DatasetImportJob::dataSource)).setter(setter(Builder::dataSource))
            .constructor(DataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataSource").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(DatasetImportJob::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(DatasetImportJob::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATION_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationDateTime").getter(getter(DatasetImportJob::creationDateTime))
            .setter(setter(Builder::creationDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationDateTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedDateTime").getter(getter(DatasetImportJob::lastUpdatedDateTime))
            .setter(setter(Builder::lastUpdatedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedDateTime").build())
            .build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("failureReason").getter(getter(DatasetImportJob::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReason").build()).build();

    private static final SdkField<String> IMPORT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("importMode").getter(getter(DatasetImportJob::importModeAsString)).setter(setter(Builder::importMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("importMode").build()).build();

    private static final SdkField<Boolean> PUBLISH_ATTRIBUTION_METRICS_TO_S3_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("publishAttributionMetricsToS3")
            .getter(getter(DatasetImportJob::publishAttributionMetricsToS3))
            .setter(setter(Builder::publishAttributionMetricsToS3))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publishAttributionMetricsToS3")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_NAME_FIELD,
            DATASET_IMPORT_JOB_ARN_FIELD, DATASET_ARN_FIELD, DATA_SOURCE_FIELD, ROLE_ARN_FIELD, STATUS_FIELD,
            CREATION_DATE_TIME_FIELD, LAST_UPDATED_DATE_TIME_FIELD, FAILURE_REASON_FIELD, IMPORT_MODE_FIELD,
            PUBLISH_ATTRIBUTION_METRICS_TO_S3_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String jobName;

    private final String datasetImportJobArn;

    private final String datasetArn;

    private final DataSource dataSource;

    private final String roleArn;

    private final String status;

    private final Instant creationDateTime;

    private final Instant lastUpdatedDateTime;

    private final String failureReason;

    private final String importMode;

    private final Boolean publishAttributionMetricsToS3;

    private DatasetImportJob(BuilderImpl builder) {
        this.jobName = builder.jobName;
        this.datasetImportJobArn = builder.datasetImportJobArn;
        this.datasetArn = builder.datasetArn;
        this.dataSource = builder.dataSource;
        this.roleArn = builder.roleArn;
        this.status = builder.status;
        this.creationDateTime = builder.creationDateTime;
        this.lastUpdatedDateTime = builder.lastUpdatedDateTime;
        this.failureReason = builder.failureReason;
        this.importMode = builder.importMode;
        this.publishAttributionMetricsToS3 = builder.publishAttributionMetricsToS3;
    }

    /**
     * <p>
     * The name of the import job.
     * </p>
     * 
     * @return The name of the import job.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The ARN of the dataset import job.
     * </p>
     * 
     * @return The ARN of the dataset import job.
     */
    public final String datasetImportJobArn() {
        return datasetImportJobArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the dataset that receives the imported data.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the dataset that receives the imported data.
     */
    public final String datasetArn() {
        return datasetArn;
    }

    /**
     * <p>
     * The Amazon S3 bucket that contains the training data to import.
     * </p>
     * 
     * @return The Amazon S3 bucket that contains the training data to import.
     */
    public final DataSource dataSource() {
        return dataSource;
    }

    /**
     * <p>
     * The ARN of the IAM role that has permissions to read from the Amazon S3 data source.
     * </p>
     * 
     * @return The ARN of the IAM role that has permissions to read from the Amazon S3 data source.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The status of the dataset import job.
     * </p>
     * <p>
     * A dataset import job can be in one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
     * </p>
     * </li>
     * </ul>
     * 
     * @return The status of the dataset import job.</p>
     *         <p>
     *         A dataset import job can be in one of the following states:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
     *         </p>
     *         </li>
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The creation date and time (in Unix time) of the dataset import job.
     * </p>
     * 
     * @return The creation date and time (in Unix time) of the dataset import job.
     */
    public final Instant creationDateTime() {
        return creationDateTime;
    }

    /**
     * <p>
     * The date and time (in Unix time) the dataset was last updated.
     * </p>
     * 
     * @return The date and time (in Unix time) the dataset was last updated.
     */
    public final Instant lastUpdatedDateTime() {
        return lastUpdatedDateTime;
    }

    /**
     * <p>
     * If a dataset import job fails, provides the reason why.
     * </p>
     * 
     * @return If a dataset import job fails, provides the reason why.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * The import mode used by the dataset import job to import new records.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #importMode} will
     * return {@link ImportMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #importModeAsString}.
     * </p>
     * 
     * @return The import mode used by the dataset import job to import new records.
     * @see ImportMode
     */
    public final ImportMode importMode() {
        return ImportMode.fromValue(importMode);
    }

    /**
     * <p>
     * The import mode used by the dataset import job to import new records.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #importMode} will
     * return {@link ImportMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #importModeAsString}.
     * </p>
     * 
     * @return The import mode used by the dataset import job to import new records.
     * @see ImportMode
     */
    public final String importModeAsString() {
        return importMode;
    }

    /**
     * <p>
     * Whether the job publishes metrics to Amazon S3 for a metric attribution.
     * </p>
     * 
     * @return Whether the job publishes metrics to Amazon S3 for a metric attribution.
     */
    public final Boolean publishAttributionMetricsToS3() {
        return publishAttributionMetricsToS3;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(datasetImportJobArn());
        hashCode = 31 * hashCode + Objects.hashCode(datasetArn());
        hashCode = 31 * hashCode + Objects.hashCode(dataSource());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(creationDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(importModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(publishAttributionMetricsToS3());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DatasetImportJob)) {
            return false;
        }
        DatasetImportJob other = (DatasetImportJob) obj;
        return Objects.equals(jobName(), other.jobName()) && Objects.equals(datasetImportJobArn(), other.datasetImportJobArn())
                && Objects.equals(datasetArn(), other.datasetArn()) && Objects.equals(dataSource(), other.dataSource())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(status(), other.status())
                && Objects.equals(creationDateTime(), other.creationDateTime())
                && Objects.equals(lastUpdatedDateTime(), other.lastUpdatedDateTime())
                && Objects.equals(failureReason(), other.failureReason())
                && Objects.equals(importModeAsString(), other.importModeAsString())
                && Objects.equals(publishAttributionMetricsToS3(), other.publishAttributionMetricsToS3());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DatasetImportJob").add("JobName", jobName()).add("DatasetImportJobArn", datasetImportJobArn())
                .add("DatasetArn", datasetArn()).add("DataSource", dataSource()).add("RoleArn", roleArn())
                .add("Status", status()).add("CreationDateTime", creationDateTime())
                .add("LastUpdatedDateTime", lastUpdatedDateTime()).add("FailureReason", failureReason())
                .add("ImportMode", importModeAsString()).add("PublishAttributionMetricsToS3", publishAttributionMetricsToS3())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "jobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "datasetImportJobArn":
            return Optional.ofNullable(clazz.cast(datasetImportJobArn()));
        case "datasetArn":
            return Optional.ofNullable(clazz.cast(datasetArn()));
        case "dataSource":
            return Optional.ofNullable(clazz.cast(dataSource()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "creationDateTime":
            return Optional.ofNullable(clazz.cast(creationDateTime()));
        case "lastUpdatedDateTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedDateTime()));
        case "failureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "importMode":
            return Optional.ofNullable(clazz.cast(importModeAsString()));
        case "publishAttributionMetricsToS3":
            return Optional.ofNullable(clazz.cast(publishAttributionMetricsToS3()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("jobName", JOB_NAME_FIELD);
        map.put("datasetImportJobArn", DATASET_IMPORT_JOB_ARN_FIELD);
        map.put("datasetArn", DATASET_ARN_FIELD);
        map.put("dataSource", DATA_SOURCE_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("creationDateTime", CREATION_DATE_TIME_FIELD);
        map.put("lastUpdatedDateTime", LAST_UPDATED_DATE_TIME_FIELD);
        map.put("failureReason", FAILURE_REASON_FIELD);
        map.put("importMode", IMPORT_MODE_FIELD);
        map.put("publishAttributionMetricsToS3", PUBLISH_ATTRIBUTION_METRICS_TO_S3_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DatasetImportJob, T> g) {
        return obj -> g.apply((DatasetImportJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DatasetImportJob> {
        /**
         * <p>
         * The name of the import job.
         * </p>
         * 
         * @param jobName
         *        The name of the import job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The ARN of the dataset import job.
         * </p>
         * 
         * @param datasetImportJobArn
         *        The ARN of the dataset import job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasetImportJobArn(String datasetImportJobArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the dataset that receives the imported data.
         * </p>
         * 
         * @param datasetArn
         *        The Amazon Resource Name (ARN) of the dataset that receives the imported data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasetArn(String datasetArn);

        /**
         * <p>
         * The Amazon S3 bucket that contains the training data to import.
         * </p>
         * 
         * @param dataSource
         *        The Amazon S3 bucket that contains the training data to import.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSource(DataSource dataSource);

        /**
         * <p>
         * The Amazon S3 bucket that contains the training data to import.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataSource.Builder} avoiding the need to
         * create one manually via {@link DataSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataSource.Builder#build()} is called immediately and its result
         * is passed to {@link #dataSource(DataSource)}.
         * 
         * @param dataSource
         *        a consumer that will call methods on {@link DataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataSource(DataSource)
         */
        default Builder dataSource(Consumer<DataSource.Builder> dataSource) {
            return dataSource(DataSource.builder().applyMutation(dataSource).build());
        }

        /**
         * <p>
         * The ARN of the IAM role that has permissions to read from the Amazon S3 data source.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the IAM role that has permissions to read from the Amazon S3 data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The status of the dataset import job.
         * </p>
         * <p>
         * A dataset import job can be in one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the dataset import job.</p>
         *        <p>
         *        A dataset import job can be in one of the following states:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The creation date and time (in Unix time) of the dataset import job.
         * </p>
         * 
         * @param creationDateTime
         *        The creation date and time (in Unix time) of the dataset import job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateTime(Instant creationDateTime);

        /**
         * <p>
         * The date and time (in Unix time) the dataset was last updated.
         * </p>
         * 
         * @param lastUpdatedDateTime
         *        The date and time (in Unix time) the dataset was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedDateTime(Instant lastUpdatedDateTime);

        /**
         * <p>
         * If a dataset import job fails, provides the reason why.
         * </p>
         * 
         * @param failureReason
         *        If a dataset import job fails, provides the reason why.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The import mode used by the dataset import job to import new records.
         * </p>
         * 
         * @param importMode
         *        The import mode used by the dataset import job to import new records.
         * @see ImportMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImportMode
         */
        Builder importMode(String importMode);

        /**
         * <p>
         * The import mode used by the dataset import job to import new records.
         * </p>
         * 
         * @param importMode
         *        The import mode used by the dataset import job to import new records.
         * @see ImportMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImportMode
         */
        Builder importMode(ImportMode importMode);

        /**
         * <p>
         * Whether the job publishes metrics to Amazon S3 for a metric attribution.
         * </p>
         * 
         * @param publishAttributionMetricsToS3
         *        Whether the job publishes metrics to Amazon S3 for a metric attribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publishAttributionMetricsToS3(Boolean publishAttributionMetricsToS3);
    }

    static final class BuilderImpl implements Builder {
        private String jobName;

        private String datasetImportJobArn;

        private String datasetArn;

        private DataSource dataSource;

        private String roleArn;

        private String status;

        private Instant creationDateTime;

        private Instant lastUpdatedDateTime;

        private String failureReason;

        private String importMode;

        private Boolean publishAttributionMetricsToS3;

        private BuilderImpl() {
        }

        private BuilderImpl(DatasetImportJob model) {
            jobName(model.jobName);
            datasetImportJobArn(model.datasetImportJobArn);
            datasetArn(model.datasetArn);
            dataSource(model.dataSource);
            roleArn(model.roleArn);
            status(model.status);
            creationDateTime(model.creationDateTime);
            lastUpdatedDateTime(model.lastUpdatedDateTime);
            failureReason(model.failureReason);
            importMode(model.importMode);
            publishAttributionMetricsToS3(model.publishAttributionMetricsToS3);
        }

        public final String getJobName() {
            return jobName;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final String getDatasetImportJobArn() {
            return datasetImportJobArn;
        }

        public final void setDatasetImportJobArn(String datasetImportJobArn) {
            this.datasetImportJobArn = datasetImportJobArn;
        }

        @Override
        public final Builder datasetImportJobArn(String datasetImportJobArn) {
            this.datasetImportJobArn = datasetImportJobArn;
            return this;
        }

        public final String getDatasetArn() {
            return datasetArn;
        }

        public final void setDatasetArn(String datasetArn) {
            this.datasetArn = datasetArn;
        }

        @Override
        public final Builder datasetArn(String datasetArn) {
            this.datasetArn = datasetArn;
            return this;
        }

        public final DataSource.Builder getDataSource() {
            return dataSource != null ? dataSource.toBuilder() : null;
        }

        public final void setDataSource(DataSource.BuilderImpl dataSource) {
            this.dataSource = dataSource != null ? dataSource.build() : null;
        }

        @Override
        public final Builder dataSource(DataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Instant getCreationDateTime() {
            return creationDateTime;
        }

        public final void setCreationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
        }

        @Override
        public final Builder creationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
            return this;
        }

        public final Instant getLastUpdatedDateTime() {
            return lastUpdatedDateTime;
        }

        public final void setLastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
        }

        @Override
        public final Builder lastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final String getImportMode() {
            return importMode;
        }

        public final void setImportMode(String importMode) {
            this.importMode = importMode;
        }

        @Override
        public final Builder importMode(String importMode) {
            this.importMode = importMode;
            return this;
        }

        @Override
        public final Builder importMode(ImportMode importMode) {
            this.importMode(importMode == null ? null : importMode.toString());
            return this;
        }

        public final Boolean getPublishAttributionMetricsToS3() {
            return publishAttributionMetricsToS3;
        }

        public final void setPublishAttributionMetricsToS3(Boolean publishAttributionMetricsToS3) {
            this.publishAttributionMetricsToS3 = publishAttributionMetricsToS3;
        }

        @Override
        public final Builder publishAttributionMetricsToS3(Boolean publishAttributionMetricsToS3) {
            this.publishAttributionMetricsToS3 = publishAttributionMetricsToS3;
            return this;
        }

        @Override
        public DatasetImportJob build() {
            return new DatasetImportJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
