/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalizeevents;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.personalizeevents.internal.PersonalizeEventsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.personalizeevents.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.personalizeevents.model.InvalidInputException;
import software.amazon.awssdk.services.personalizeevents.model.PersonalizeEventsException;
import software.amazon.awssdk.services.personalizeevents.model.PutActionInteractionsRequest;
import software.amazon.awssdk.services.personalizeevents.model.PutActionInteractionsResponse;
import software.amazon.awssdk.services.personalizeevents.model.PutActionsRequest;
import software.amazon.awssdk.services.personalizeevents.model.PutActionsResponse;
import software.amazon.awssdk.services.personalizeevents.model.PutEventsRequest;
import software.amazon.awssdk.services.personalizeevents.model.PutEventsResponse;
import software.amazon.awssdk.services.personalizeevents.model.PutItemsRequest;
import software.amazon.awssdk.services.personalizeevents.model.PutItemsResponse;
import software.amazon.awssdk.services.personalizeevents.model.PutUsersRequest;
import software.amazon.awssdk.services.personalizeevents.model.PutUsersResponse;
import software.amazon.awssdk.services.personalizeevents.model.ResourceInUseException;
import software.amazon.awssdk.services.personalizeevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.personalizeevents.transform.PutActionInteractionsRequestMarshaller;
import software.amazon.awssdk.services.personalizeevents.transform.PutActionsRequestMarshaller;
import software.amazon.awssdk.services.personalizeevents.transform.PutEventsRequestMarshaller;
import software.amazon.awssdk.services.personalizeevents.transform.PutItemsRequestMarshaller;
import software.amazon.awssdk.services.personalizeevents.transform.PutUsersRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PersonalizeEventsClient}.
 *
 * @see PersonalizeEventsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPersonalizeEventsClient implements PersonalizeEventsClient {
    private static final Logger log = Logger.loggerFor(DefaultPersonalizeEventsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPersonalizeEventsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Personalize_Events" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Records action interaction event data. An <i>action interaction</i> event is an interaction between a user and an
     * <i>action</i>. For example, a user taking an action, such a enrolling in a membership program or downloading your
     * app.
     * </p>
     * <p>
     * For more information about recording action interactions, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/recording-action-interaction-events.html">Recording
     * action interaction events</a>. For more information about actions in an Actions dataset, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/actions-datasets.html">Actions dataset</a>.
     * </p>
     *
     * @param putActionInteractionsRequest
     * @return Result of the PutActionInteractions operation returned by the service.
     * @throws InvalidInputException
     *         Provide a valid value for the field or parameter.
     * @throws ResourceNotFoundException
     *         Could not find the specified resource.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PersonalizeEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PersonalizeEventsClient.PutActionInteractions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/personalize-events-2018-03-22/PutActionInteractions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutActionInteractionsResponse putActionInteractions(PutActionInteractionsRequest putActionInteractionsRequest)
            throws InvalidInputException, ResourceNotFoundException, ResourceInUseException, AwsServiceException,
            SdkClientException, PersonalizeEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutActionInteractionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutActionInteractionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putActionInteractionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putActionInteractionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Personalize Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutActionInteractions");

            return clientHandler.execute(new ClientExecutionParams<PutActionInteractionsRequest, PutActionInteractionsResponse>()
                    .withOperationName("PutActionInteractions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putActionInteractionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutActionInteractionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more actions to an Actions dataset. For more information see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/importing-actions.html">Importing actions
     * individually</a>.
     * </p>
     *
     * @param putActionsRequest
     * @return Result of the PutActions operation returned by the service.
     * @throws InvalidInputException
     *         Provide a valid value for the field or parameter.
     * @throws ResourceNotFoundException
     *         Could not find the specified resource.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PersonalizeEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PersonalizeEventsClient.PutActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/personalize-events-2018-03-22/PutActions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutActionsResponse putActions(PutActionsRequest putActionsRequest) throws InvalidInputException,
            ResourceNotFoundException, ResourceInUseException, AwsServiceException, SdkClientException,
            PersonalizeEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutActionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutActionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putActionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Personalize Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutActions");

            return clientHandler.execute(new ClientExecutionParams<PutActionsRequest, PutActionsResponse>()
                    .withOperationName("PutActions").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putActionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Records item interaction event data. For more information see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/recording-item-interaction-events.html">Recording item
     * interaction events</a>.
     * </p>
     *
     * @param putEventsRequest
     * @return Result of the PutEvents operation returned by the service.
     * @throws InvalidInputException
     *         Provide a valid value for the field or parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PersonalizeEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PersonalizeEventsClient.PutEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/personalize-events-2018-03-22/PutEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutEventsResponse putEvents(PutEventsRequest putEventsRequest) throws InvalidInputException, AwsServiceException,
            SdkClientException, PersonalizeEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutEventsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putEventsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Personalize Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEvents");

            return clientHandler.execute(new ClientExecutionParams<PutEventsRequest, PutEventsResponse>()
                    .withOperationName("PutEvents").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putEventsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more items to an Items dataset. For more information see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/importing-items.html">Importing items individually</a>.
     * </p>
     *
     * @param putItemsRequest
     * @return Result of the PutItems operation returned by the service.
     * @throws InvalidInputException
     *         Provide a valid value for the field or parameter.
     * @throws ResourceNotFoundException
     *         Could not find the specified resource.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PersonalizeEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PersonalizeEventsClient.PutItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/personalize-events-2018-03-22/PutItems" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutItemsResponse putItems(PutItemsRequest putItemsRequest) throws InvalidInputException, ResourceNotFoundException,
            ResourceInUseException, AwsServiceException, SdkClientException, PersonalizeEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutItemsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutItemsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putItemsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Personalize Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutItems");

            return clientHandler.execute(new ClientExecutionParams<PutItemsRequest, PutItemsResponse>()
                    .withOperationName("PutItems").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putItemsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more users to a Users dataset. For more information see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/importing-users.html">Importing users individually</a>.
     * </p>
     *
     * @param putUsersRequest
     * @return Result of the PutUsers operation returned by the service.
     * @throws InvalidInputException
     *         Provide a valid value for the field or parameter.
     * @throws ResourceNotFoundException
     *         Could not find the specified resource.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PersonalizeEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PersonalizeEventsClient.PutUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/personalize-events-2018-03-22/PutUsers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutUsersResponse putUsers(PutUsersRequest putUsersRequest) throws InvalidInputException, ResourceNotFoundException,
            ResourceInUseException, AwsServiceException, SdkClientException, PersonalizeEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutUsersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Personalize Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutUsers");

            return clientHandler.execute(new ClientExecutionParams<PutUsersRequest, PutUsersResponse>()
                    .withOperationName("PutUsers").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putUsersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        PersonalizeEventsServiceClientConfigurationBuilder serviceConfigBuilder = new PersonalizeEventsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PersonalizeEventsException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    @Override
    public final PersonalizeEventsServiceClientConfiguration serviceClientConfiguration() {
        return new PersonalizeEventsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
