/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalizeruntime;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.personalizeruntime.internal.PersonalizeRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.personalizeruntime.model.GetPersonalizedRankingRequest;
import software.amazon.awssdk.services.personalizeruntime.model.GetPersonalizedRankingResponse;
import software.amazon.awssdk.services.personalizeruntime.model.GetRecommendationsRequest;
import software.amazon.awssdk.services.personalizeruntime.model.GetRecommendationsResponse;
import software.amazon.awssdk.services.personalizeruntime.model.InvalidInputException;
import software.amazon.awssdk.services.personalizeruntime.model.PersonalizeRuntimeException;
import software.amazon.awssdk.services.personalizeruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.personalizeruntime.transform.GetPersonalizedRankingRequestMarshaller;
import software.amazon.awssdk.services.personalizeruntime.transform.GetRecommendationsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PersonalizeRuntimeClient}.
 *
 * @see PersonalizeRuntimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPersonalizeRuntimeClient implements PersonalizeRuntimeClient {
    private static final Logger log = Logger.loggerFor(DefaultPersonalizeRuntimeClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PersonalizeRuntimeServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPersonalizeRuntimeClient(PersonalizeRuntimeServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Re-ranks a list of recommended items for the given user. The first item in the list is deemed the most likely
     * item to be of interest to the user.
     * </p>
     * <note>
     * <p>
     * The solution backing the campaign must have been created using a recipe of type PERSONALIZED_RANKING.
     * </p>
     * </note>
     *
     * @param getPersonalizedRankingRequest
     * @return Result of the GetPersonalizedRanking operation returned by the service.
     * @throws InvalidInputException
     *         Provide a valid value for the field or parameter.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PersonalizeRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PersonalizeRuntimeClient.GetPersonalizedRanking
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/personalize-runtime-2018-05-22/GetPersonalizedRanking"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPersonalizedRankingResponse getPersonalizedRanking(GetPersonalizedRankingRequest getPersonalizedRankingRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            PersonalizeRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPersonalizedRankingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPersonalizedRankingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPersonalizedRankingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPersonalizedRankingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Personalize Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPersonalizedRanking");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPersonalizedRankingRequest, GetPersonalizedRankingResponse>()
                            .withOperationName("GetPersonalizedRanking").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getPersonalizedRankingRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPersonalizedRankingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of recommended items. For campaigns, the campaign's Amazon Resource Name (ARN) is required and the
     * required user and item input depends on the recipe type used to create the solution backing the campaign as
     * follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * USER_PERSONALIZATION - <code>userId</code> required, <code>itemId</code> not used
     * </p>
     * </li>
     * <li>
     * <p>
     * RELATED_ITEMS - <code>itemId</code> required, <code>userId</code> not used
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Campaigns that are backed by a solution created using a recipe of type PERSONALIZED_RANKING use the API.
     * </p>
     * </note>
     * <p>
     * For recommenders, the recommender's ARN is required and the required item and user input depends on the use case
     * (domain-based recipe) backing the recommender. For information on use case requirements see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/domain-use-cases.html">Choosing recommender use
     * cases</a>.
     * </p>
     *
     * @param getRecommendationsRequest
     * @return Result of the GetRecommendations operation returned by the service.
     * @throws InvalidInputException
     *         Provide a valid value for the field or parameter.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PersonalizeRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PersonalizeRuntimeClient.GetRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/personalize-runtime-2018-05-22/GetRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecommendationsResponse getRecommendations(GetRecommendationsRequest getRecommendationsRequest)
            throws InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            PersonalizeRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Personalize Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendations");

            return clientHandler.execute(new ClientExecutionParams<GetRecommendationsRequest, GetRecommendationsResponse>()
                    .withOperationName("GetRecommendations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRecommendationsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        PersonalizeRuntimeServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = PersonalizeRuntimeServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PersonalizeRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public final PersonalizeRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
