/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pi.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A single query to be processed. You must provide the metric to query. If no other parameters are specified,
 * Performance Insights returns all of the data points for that metric. You can optionally request that the data points
 * be aggregated by dimension group ( <code>GroupBy</code>), and return only those data points that match your criteria
 * (<code>Filter</code>).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricQuery implements SdkPojo, Serializable, ToCopyableBuilder<MetricQuery.Builder, MetricQuery> {
    private static final SdkField<String> METRIC_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Metric")
            .getter(getter(MetricQuery::metric)).setter(setter(Builder::metric))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metric").build()).build();

    private static final SdkField<DimensionGroup> GROUP_BY_FIELD = SdkField.<DimensionGroup> builder(MarshallingType.SDK_POJO)
            .memberName("GroupBy").getter(getter(MetricQuery::groupBy)).setter(setter(Builder::groupBy))
            .constructor(DimensionGroup::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupBy").build()).build();

    private static final SdkField<Map<String, String>> FILTER_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Filter")
            .getter(getter(MetricQuery::filter))
            .setter(setter(Builder::filter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filter").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_FIELD, GROUP_BY_FIELD,
            FILTER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metric;

    private final DimensionGroup groupBy;

    private final Map<String, String> filter;

    private MetricQuery(BuilderImpl builder) {
        this.metric = builder.metric;
        this.groupBy = builder.groupBy;
        this.filter = builder.filter;
    }

    /**
     * <p>
     * The name of a Performance Insights metric to be measured.
     * </p>
     * <p>
     * Valid values for <code>Metric</code> are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database engine.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of a Performance Insights metric to be measured.</p>
     *         <p>
     *         Valid values for <code>Metric</code> are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database
     *         engine.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
     *         </p>
     *         </li>
     */
    public String metric() {
        return metric;
    }

    /**
     * <p>
     * A specification for how to aggregate the data points from a query result. You must specify a valid dimension
     * group. Performance Insights will return all of the dimensions within that group, unless you provide the names of
     * specific dimensions within that group. You can also request that Performance Insights return a limited number of
     * values for a dimension.
     * </p>
     * 
     * @return A specification for how to aggregate the data points from a query result. You must specify a valid
     *         dimension group. Performance Insights will return all of the dimensions within that group, unless you
     *         provide the names of specific dimensions within that group. You can also request that Performance
     *         Insights return a limited number of values for a dimension.
     */
    public DimensionGroup groupBy() {
        return groupBy;
    }

    /**
     * Returns true if the Filter property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasFilter() {
        return filter != null && !(filter instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * One or more filters to apply in the request. Restrictions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * A single filter for any other dimension in this dimension group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFilter()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more filters to apply in the request. Restrictions:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A single filter for any other dimension in this dimension group.
     *         </p>
     *         </li>
     */
    public Map<String, String> filter() {
        return filter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metric());
        hashCode = 31 * hashCode + Objects.hashCode(groupBy());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilter() ? filter() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricQuery)) {
            return false;
        }
        MetricQuery other = (MetricQuery) obj;
        return Objects.equals(metric(), other.metric()) && Objects.equals(groupBy(), other.groupBy())
                && hasFilter() == other.hasFilter() && Objects.equals(filter(), other.filter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MetricQuery").add("Metric", metric()).add("GroupBy", groupBy())
                .add("Filter", hasFilter() ? filter() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Metric":
            return Optional.ofNullable(clazz.cast(metric()));
        case "GroupBy":
            return Optional.ofNullable(clazz.cast(groupBy()));
        case "Filter":
            return Optional.ofNullable(clazz.cast(filter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricQuery, T> g) {
        return obj -> g.apply((MetricQuery) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricQuery> {
        /**
         * <p>
         * The name of a Performance Insights metric to be measured.
         * </p>
         * <p>
         * Valid values for <code>Metric</code> are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database engine.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metric
         *        The name of a Performance Insights metric to be measured.</p>
         *        <p>
         *        Valid values for <code>Metric</code> are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database
         *        engine.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metric(String metric);

        /**
         * <p>
         * A specification for how to aggregate the data points from a query result. You must specify a valid dimension
         * group. Performance Insights will return all of the dimensions within that group, unless you provide the names
         * of specific dimensions within that group. You can also request that Performance Insights return a limited
         * number of values for a dimension.
         * </p>
         * 
         * @param groupBy
         *        A specification for how to aggregate the data points from a query result. You must specify a valid
         *        dimension group. Performance Insights will return all of the dimensions within that group, unless you
         *        provide the names of specific dimensions within that group. You can also request that Performance
         *        Insights return a limited number of values for a dimension.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupBy(DimensionGroup groupBy);

        /**
         * <p>
         * A specification for how to aggregate the data points from a query result. You must specify a valid dimension
         * group. Performance Insights will return all of the dimensions within that group, unless you provide the names
         * of specific dimensions within that group. You can also request that Performance Insights return a limited
         * number of values for a dimension.
         * </p>
         * This is a convenience that creates an instance of the {@link DimensionGroup.Builder} avoiding the need to
         * create one manually via {@link DimensionGroup#builder()}.
         *
         * When the {@link Consumer} completes, {@link DimensionGroup.Builder#build()} is called immediately and its
         * result is passed to {@link #groupBy(DimensionGroup)}.
         * 
         * @param groupBy
         *        a consumer that will call methods on {@link DimensionGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groupBy(DimensionGroup)
         */
        default Builder groupBy(Consumer<DimensionGroup.Builder> groupBy) {
            return groupBy(DimensionGroup.builder().applyMutation(groupBy).build());
        }

        /**
         * <p>
         * One or more filters to apply in the request. Restrictions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * A single filter for any other dimension in this dimension group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param filter
         *        One or more filters to apply in the request. Restrictions:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Any number of filters by the same dimension, as specified in the <code>GroupBy</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A single filter for any other dimension in this dimension group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filter(Map<String, String> filter);
    }

    static final class BuilderImpl implements Builder {
        private String metric;

        private DimensionGroup groupBy;

        private Map<String, String> filter = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MetricQuery model) {
            metric(model.metric);
            groupBy(model.groupBy);
            filter(model.filter);
        }

        public final String getMetric() {
            return metric;
        }

        @Override
        public final Builder metric(String metric) {
            this.metric = metric;
            return this;
        }

        public final void setMetric(String metric) {
            this.metric = metric;
        }

        public final DimensionGroup.Builder getGroupBy() {
            return groupBy != null ? groupBy.toBuilder() : null;
        }

        @Override
        public final Builder groupBy(DimensionGroup groupBy) {
            this.groupBy = groupBy;
            return this;
        }

        public final void setGroupBy(DimensionGroup.BuilderImpl groupBy) {
            this.groupBy = groupBy != null ? groupBy.build() : null;
        }

        public final Map<String, String> getFilter() {
            if (filter instanceof SdkAutoConstructMap) {
                return null;
            }
            return filter;
        }

        @Override
        public final Builder filter(Map<String, String> filter) {
            this.filter = MetricQueryFilterMapCopier.copy(filter);
            return this;
        }

        public final void setFilter(Map<String, String> filter) {
            this.filter = MetricQueryFilterMapCopier.copy(filter);
        }

        @Override
        public MetricQuery build() {
            return new MetricQuery(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
