/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pi.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object describing a Performance Insights metric and one or more dimensions for that metric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseResourceMetricKey implements SdkPojo, Serializable,
        ToCopyableBuilder<ResponseResourceMetricKey.Builder, ResponseResourceMetricKey> {
    private static final SdkField<String> METRIC_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Metric")
            .getter(getter(ResponseResourceMetricKey::metric)).setter(setter(Builder::metric))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metric").build()).build();

    private static final SdkField<Map<String, String>> DIMENSIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Dimensions")
            .getter(getter(ResponseResourceMetricKey::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(METRIC_FIELD, DIMENSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metric;

    private final Map<String, String> dimensions;

    private ResponseResourceMetricKey(BuilderImpl builder) {
        this.metric = builder.metric;
        this.dimensions = builder.dimensions;
    }

    /**
     * <p>
     * The name of a Performance Insights metric to be measured.
     * </p>
     * <p>
     * Valid values for <code>Metric</code> are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database engine.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the number of active sessions is less than an internal Performance Insights threshold,
     * <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of active sessions
     * is greater than the internal threshold, Performance Insights samples the active sessions, with
     * <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code> showing the raw values, and
     * <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For most use cases, you can query
     * <code>db.load.avg</code> only.
     * </p>
     * 
     * @return The name of a Performance Insights metric to be measured.</p>
     *         <p>
     *         Valid values for <code>Metric</code> are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database
     *         engine.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the number of active sessions is less than an internal Performance Insights threshold,
     *         <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of active
     *         sessions is greater than the internal threshold, Performance Insights samples the active sessions, with
     *         <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code> showing the raw
     *         values, and <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For most use cases, you
     *         can query <code>db.load.avg</code> only.
     */
    public String metric() {
        return metric;
    }

    /**
     * Returns true if the Dimensions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The valid dimensions for the metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDimensions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The valid dimensions for the metric.
     */
    public Map<String, String> dimensions() {
        return dimensions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metric());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseResourceMetricKey)) {
            return false;
        }
        ResponseResourceMetricKey other = (ResponseResourceMetricKey) obj;
        return Objects.equals(metric(), other.metric()) && hasDimensions() == other.hasDimensions()
                && Objects.equals(dimensions(), other.dimensions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ResponseResourceMetricKey").add("Metric", metric())
                .add("Dimensions", hasDimensions() ? dimensions() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Metric":
            return Optional.ofNullable(clazz.cast(metric()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResponseResourceMetricKey, T> g) {
        return obj -> g.apply((ResponseResourceMetricKey) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResponseResourceMetricKey> {
        /**
         * <p>
         * The name of a Performance Insights metric to be measured.
         * </p>
         * <p>
         * Valid values for <code>Metric</code> are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database engine.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the number of active sessions is less than an internal Performance Insights threshold,
         * <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of active
         * sessions is greater than the internal threshold, Performance Insights samples the active sessions, with
         * <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code> showing the raw values,
         * and <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For most use cases, you can query
         * <code>db.load.avg</code> only.
         * </p>
         * 
         * @param metric
         *        The name of a Performance Insights metric to be measured.</p>
         *        <p>
         *        Valid values for <code>Metric</code> are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>db.load.avg</code> - a scaled representation of the number of active sessions for the database
         *        engine.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sampledload.avg</code> - the raw number of active sessions for the database engine.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the number of active sessions is less than an internal Performance Insights threshold,
         *        <code>db.load.avg</code> and <code>db.sampledload.avg</code> are the same value. If the number of
         *        active sessions is greater than the internal threshold, Performance Insights samples the active
         *        sessions, with <code>db.load.avg</code> showing the scaled values, <code>db.sampledload.avg</code>
         *        showing the raw values, and <code>db.sampledload.avg</code> less than <code>db.load.avg</code>. For
         *        most use cases, you can query <code>db.load.avg</code> only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metric(String metric);

        /**
         * <p>
         * The valid dimensions for the metric.
         * </p>
         * 
         * @param dimensions
         *        The valid dimensions for the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Map<String, String> dimensions);
    }

    static final class BuilderImpl implements Builder {
        private String metric;

        private Map<String, String> dimensions = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseResourceMetricKey model) {
            metric(model.metric);
            dimensions(model.dimensions);
        }

        public final String getMetric() {
            return metric;
        }

        @Override
        public final Builder metric(String metric) {
            this.metric = metric;
            return this;
        }

        public final void setMetric(String metric) {
            this.metric = metric;
        }

        public final Map<String, String> getDimensions() {
            if (dimensions instanceof SdkAutoConstructMap) {
                return null;
            }
            return dimensions;
        }

        @Override
        public final Builder dimensions(Map<String, String> dimensions) {
            this.dimensions = DimensionMapCopier.copy(dimensions);
            return this;
        }

        public final void setDimensions(Map<String, String> dimensions) {
            this.dimensions = DimensionMapCopier.copy(dimensions);
        }

        @Override
        public ResponseResourceMetricKey build() {
            return new ResponseResourceMetricKey(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
