/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pi;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest;
import software.amazon.awssdk.services.pi.model.DescribeDimensionKeysResponse;
import software.amazon.awssdk.services.pi.model.GetDimensionKeyDetailsRequest;
import software.amazon.awssdk.services.pi.model.GetDimensionKeyDetailsResponse;
import software.amazon.awssdk.services.pi.model.GetResourceMetadataRequest;
import software.amazon.awssdk.services.pi.model.GetResourceMetadataResponse;
import software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest;
import software.amazon.awssdk.services.pi.model.GetResourceMetricsResponse;
import software.amazon.awssdk.services.pi.model.InternalServiceErrorException;
import software.amazon.awssdk.services.pi.model.InvalidArgumentException;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsResponse;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsResponse;
import software.amazon.awssdk.services.pi.model.NotAuthorizedException;
import software.amazon.awssdk.services.pi.model.PiException;
import software.amazon.awssdk.services.pi.model.PiRequest;
import software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable;
import software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable;
import software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable;
import software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable;
import software.amazon.awssdk.services.pi.transform.DescribeDimensionKeysRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.GetDimensionKeyDetailsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.GetResourceMetadataRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.GetResourceMetricsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.ListAvailableResourceDimensionsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.ListAvailableResourceMetricsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PiClient}.
 *
 * @see PiClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPiClient implements PiClient {
    private static final Logger log = Logger.loggerFor(DefaultPiClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPiClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * For a specific time period, retrieve the top <code>N</code> dimension keys for a metric.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note>
     *
     * @param describeDimensionKeysRequest
     * @return Result of the DescribeDimensionKeys operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DescribeDimensionKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DescribeDimensionKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDimensionKeysResponse describeDimensionKeys(DescribeDimensionKeysRequest describeDimensionKeysRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDimensionKeysResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDimensionKeysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDimensionKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDimensionKeys");

            return clientHandler.execute(new ClientExecutionParams<DescribeDimensionKeysRequest, DescribeDimensionKeysResponse>()
                    .withOperationName("DescribeDimensionKeys").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDimensionKeysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDimensionKeysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For a specific time period, retrieve the top <code>N</code> dimension keys for a metric.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #describeDimensionKeys(software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client.describeDimensionKeysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client
     *             .describeDimensionKeysPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.DescribeDimensionKeysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.DescribeDimensionKeysIterable responses = client.describeDimensionKeysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDimensionKeys(software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDimensionKeysRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.DescribeDimensionKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DescribeDimensionKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDimensionKeysIterable describeDimensionKeysPaginator(DescribeDimensionKeysRequest describeDimensionKeysRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return new DescribeDimensionKeysIterable(this, applyPaginatorUserAgent(describeDimensionKeysRequest));
    }

    /**
     * <p>
     * Get the attributes of the specified dimension group for a DB instance or data source. For example, if you specify
     * a SQL ID, <code>GetDimensionKeyDetails</code> retrieves the full text of the dimension
     * <code>db.sql.statement</code> associated with this ID. This operation is useful because
     * <code>GetResourceMetrics</code> and <code>DescribeDimensionKeys</code> don't support retrieval of large SQL
     * statement text.
     * </p>
     *
     * @param getDimensionKeyDetailsRequest
     * @return Result of the GetDimensionKeyDetails operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetDimensionKeyDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetDimensionKeyDetails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDimensionKeyDetailsResponse getDimensionKeyDetails(GetDimensionKeyDetailsRequest getDimensionKeyDetailsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDimensionKeyDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDimensionKeyDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDimensionKeyDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDimensionKeyDetails");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDimensionKeyDetailsRequest, GetDimensionKeyDetailsResponse>()
                            .withOperationName("GetDimensionKeyDetails").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getDimensionKeyDetailsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDimensionKeyDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve the metadata for different features. For example, the metadata might indicate that a feature is turned
     * on or off on a specific DB instance.
     * </p>
     *
     * @param getResourceMetadataRequest
     * @return Result of the GetResourceMetadata operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceMetadataResponse getResourceMetadata(GetResourceMetadataRequest getResourceMetadataRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceMetadata");

            return clientHandler.execute(new ClientExecutionParams<GetResourceMetadataRequest, GetResourceMetadataResponse>()
                    .withOperationName("GetResourceMetadata").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourceMetadataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve Performance Insights metrics for a set of data sources over a time period. You can provide specific
     * dimension groups and dimensions, and provide aggregation and filtering criteria for each group.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note>
     *
     * @param getResourceMetricsRequest
     * @return Result of the GetResourceMetrics operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceMetricsResponse getResourceMetrics(GetResourceMetricsRequest getResourceMetricsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceMetrics");

            return clientHandler.execute(new ClientExecutionParams<GetResourceMetricsRequest, GetResourceMetricsResponse>()
                    .withOperationName("GetResourceMetrics").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourceMetricsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve Performance Insights metrics for a set of data sources over a time period. You can provide specific
     * dimension groups and dimensions, and provide aggregation and filtering criteria for each group.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #getResourceMetrics(software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client.getResourceMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client
     *             .getResourceMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.GetResourceMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.GetResourceMetricsIterable responses = client.getResourceMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceMetrics(software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest)} operation.</b>
     * </p>
     *
     * @param getResourceMetricsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.GetResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceMetricsIterable getResourceMetricsPaginator(GetResourceMetricsRequest getResourceMetricsRequest)
            throws InvalidArgumentException, InternalServiceErrorException, NotAuthorizedException, AwsServiceException,
            SdkClientException, PiException {
        return new GetResourceMetricsIterable(this, applyPaginatorUserAgent(getResourceMetricsRequest));
    }

    /**
     * <p>
     * Retrieve the dimensions that can be queried for each specified metric type on a specified DB instance.
     * </p>
     *
     * @param listAvailableResourceDimensionsRequest
     * @return Result of the ListAvailableResourceDimensions operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceDimensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceDimensions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAvailableResourceDimensionsResponse listAvailableResourceDimensions(
            ListAvailableResourceDimensionsRequest listAvailableResourceDimensionsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAvailableResourceDimensionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAvailableResourceDimensionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAvailableResourceDimensionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAvailableResourceDimensions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAvailableResourceDimensionsRequest, ListAvailableResourceDimensionsResponse>()
                            .withOperationName("ListAvailableResourceDimensions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAvailableResourceDimensionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAvailableResourceDimensionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve the dimensions that can be queried for each specified metric type on a specified DB instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAvailableResourceDimensions(software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client.listAvailableResourceDimensionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client
     *             .listAvailableResourceDimensionsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceDimensionsIterable responses = client.listAvailableResourceDimensionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAvailableResourceDimensions(software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAvailableResourceDimensionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceDimensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceDimensions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAvailableResourceDimensionsIterable listAvailableResourceDimensionsPaginator(
            ListAvailableResourceDimensionsRequest listAvailableResourceDimensionsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return new ListAvailableResourceDimensionsIterable(this, applyPaginatorUserAgent(listAvailableResourceDimensionsRequest));
    }

    /**
     * <p>
     * Retrieve metrics of the specified types that can be queried for a specified DB instance.
     * </p>
     *
     * @param listAvailableResourceMetricsRequest
     * @return Result of the ListAvailableResourceMetrics operation returned by the service.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAvailableResourceMetricsResponse listAvailableResourceMetrics(
            ListAvailableResourceMetricsRequest listAvailableResourceMetricsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAvailableResourceMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAvailableResourceMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAvailableResourceMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAvailableResourceMetrics");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAvailableResourceMetricsRequest, ListAvailableResourceMetricsResponse>()
                            .withOperationName("ListAvailableResourceMetrics").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAvailableResourceMetricsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAvailableResourceMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve metrics of the specified types that can be queried for a specified DB instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAvailableResourceMetrics(software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client.listAvailableResourceMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client
     *             .listAvailableResourceMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pi.paginators.ListAvailableResourceMetricsIterable responses = client.listAvailableResourceMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAvailableResourceMetrics(software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAvailableResourceMetricsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidArgumentException
     *         One of the arguments provided is invalid for this request.
     * @throws InternalServiceErrorException
     *         The request failed due to an unknown error.
     * @throws NotAuthorizedException
     *         The user is not authorized to perform this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PiException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PiClient.ListAvailableResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAvailableResourceMetricsIterable listAvailableResourceMetricsPaginator(
            ListAvailableResourceMetricsRequest listAvailableResourceMetricsRequest) throws InvalidArgumentException,
            InternalServiceErrorException, NotAuthorizedException, AwsServiceException, SdkClientException, PiException {
        return new ListAvailableResourceMetricsIterable(this, applyPaginatorUserAgent(listAvailableResourceMetricsRequest));
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PiException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends PiRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
