/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pi;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.pi.internal.PiServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pi.model.CreatePerformanceAnalysisReportRequest;
import software.amazon.awssdk.services.pi.model.CreatePerformanceAnalysisReportResponse;
import software.amazon.awssdk.services.pi.model.DeletePerformanceAnalysisReportRequest;
import software.amazon.awssdk.services.pi.model.DeletePerformanceAnalysisReportResponse;
import software.amazon.awssdk.services.pi.model.DescribeDimensionKeysRequest;
import software.amazon.awssdk.services.pi.model.DescribeDimensionKeysResponse;
import software.amazon.awssdk.services.pi.model.GetDimensionKeyDetailsRequest;
import software.amazon.awssdk.services.pi.model.GetDimensionKeyDetailsResponse;
import software.amazon.awssdk.services.pi.model.GetPerformanceAnalysisReportRequest;
import software.amazon.awssdk.services.pi.model.GetPerformanceAnalysisReportResponse;
import software.amazon.awssdk.services.pi.model.GetResourceMetadataRequest;
import software.amazon.awssdk.services.pi.model.GetResourceMetadataResponse;
import software.amazon.awssdk.services.pi.model.GetResourceMetricsRequest;
import software.amazon.awssdk.services.pi.model.GetResourceMetricsResponse;
import software.amazon.awssdk.services.pi.model.InternalServiceErrorException;
import software.amazon.awssdk.services.pi.model.InvalidArgumentException;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsRequest;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceDimensionsResponse;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsRequest;
import software.amazon.awssdk.services.pi.model.ListAvailableResourceMetricsResponse;
import software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsRequest;
import software.amazon.awssdk.services.pi.model.ListPerformanceAnalysisReportsResponse;
import software.amazon.awssdk.services.pi.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pi.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pi.model.NotAuthorizedException;
import software.amazon.awssdk.services.pi.model.PiException;
import software.amazon.awssdk.services.pi.model.TagResourceRequest;
import software.amazon.awssdk.services.pi.model.TagResourceResponse;
import software.amazon.awssdk.services.pi.model.UntagResourceRequest;
import software.amazon.awssdk.services.pi.model.UntagResourceResponse;
import software.amazon.awssdk.services.pi.transform.CreatePerformanceAnalysisReportRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.DeletePerformanceAnalysisReportRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.DescribeDimensionKeysRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.GetDimensionKeyDetailsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.GetPerformanceAnalysisReportRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.GetResourceMetadataRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.GetResourceMetricsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.ListAvailableResourceDimensionsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.ListAvailableResourceMetricsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.ListPerformanceAnalysisReportsRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pi.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PiAsyncClient}.
 *
 * @see PiAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPiAsyncClient implements PiAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPiAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PiServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPiAsyncClient(PiServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new performance analysis report for a specific time period for the DB instance.
     * </p>
     *
     * @param createPerformanceAnalysisReportRequest
     * @return A Java Future containing the result of the CreatePerformanceAnalysisReport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.CreatePerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/CreatePerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePerformanceAnalysisReportResponse> createPerformanceAnalysisReport(
            CreatePerformanceAnalysisReportRequest createPerformanceAnalysisReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPerformanceAnalysisReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createPerformanceAnalysisReportRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePerformanceAnalysisReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePerformanceAnalysisReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePerformanceAnalysisReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePerformanceAnalysisReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePerformanceAnalysisReportRequest, CreatePerformanceAnalysisReportResponse>()
                            .withOperationName("CreatePerformanceAnalysisReport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePerformanceAnalysisReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPerformanceAnalysisReportRequest));
            CompletableFuture<CreatePerformanceAnalysisReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a performance analysis report.
     * </p>
     *
     * @param deletePerformanceAnalysisReportRequest
     * @return A Java Future containing the result of the DeletePerformanceAnalysisReport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.DeletePerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DeletePerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePerformanceAnalysisReportResponse> deletePerformanceAnalysisReport(
            DeletePerformanceAnalysisReportRequest deletePerformanceAnalysisReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePerformanceAnalysisReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deletePerformanceAnalysisReportRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePerformanceAnalysisReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePerformanceAnalysisReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePerformanceAnalysisReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePerformanceAnalysisReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePerformanceAnalysisReportRequest, DeletePerformanceAnalysisReportResponse>()
                            .withOperationName("DeletePerformanceAnalysisReport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePerformanceAnalysisReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePerformanceAnalysisReportRequest));
            CompletableFuture<DeletePerformanceAnalysisReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For a specific time period, retrieve the top <code>N</code> dimension keys for a metric.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note>
     *
     * @param describeDimensionKeysRequest
     * @return A Java Future containing the result of the DescribeDimensionKeys operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.DescribeDimensionKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/DescribeDimensionKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDimensionKeysResponse> describeDimensionKeys(
            DescribeDimensionKeysRequest describeDimensionKeysRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDimensionKeysRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDimensionKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDimensionKeys");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDimensionKeysResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDimensionKeysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDimensionKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDimensionKeysRequest, DescribeDimensionKeysResponse>()
                            .withOperationName("DescribeDimensionKeys").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDimensionKeysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDimensionKeysRequest));
            CompletableFuture<DescribeDimensionKeysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the attributes of the specified dimension group for a DB instance or data source. For example, if you specify
     * a SQL ID, <code>GetDimensionKeyDetails</code> retrieves the full text of the dimension
     * <code>db.sql.statement</code> associated with this ID. This operation is useful because
     * <code>GetResourceMetrics</code> and <code>DescribeDimensionKeys</code> don't support retrieval of large SQL
     * statement text.
     * </p>
     *
     * @param getDimensionKeyDetailsRequest
     * @return A Java Future containing the result of the GetDimensionKeyDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.GetDimensionKeyDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetDimensionKeyDetails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDimensionKeyDetailsResponse> getDimensionKeyDetails(
            GetDimensionKeyDetailsRequest getDimensionKeyDetailsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDimensionKeyDetailsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDimensionKeyDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDimensionKeyDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDimensionKeyDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDimensionKeyDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDimensionKeyDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDimensionKeyDetailsRequest, GetDimensionKeyDetailsResponse>()
                            .withOperationName("GetDimensionKeyDetails").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDimensionKeyDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDimensionKeyDetailsRequest));
            CompletableFuture<GetDimensionKeyDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the report including the report ID, status, time details, and the insights with recommendations. The
     * report status can be <code>RUNNING</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>. The insights include
     * the <code>description</code> and <code>recommendation</code> fields.
     * </p>
     *
     * @param getPerformanceAnalysisReportRequest
     * @return A Java Future containing the result of the GetPerformanceAnalysisReport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.GetPerformanceAnalysisReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetPerformanceAnalysisReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPerformanceAnalysisReportResponse> getPerformanceAnalysisReport(
            GetPerformanceAnalysisReportRequest getPerformanceAnalysisReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPerformanceAnalysisReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPerformanceAnalysisReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPerformanceAnalysisReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPerformanceAnalysisReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPerformanceAnalysisReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPerformanceAnalysisReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPerformanceAnalysisReportRequest, GetPerformanceAnalysisReportResponse>()
                            .withOperationName("GetPerformanceAnalysisReport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPerformanceAnalysisReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPerformanceAnalysisReportRequest));
            CompletableFuture<GetPerformanceAnalysisReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the metadata for different features. For example, the metadata might indicate that a feature is turned
     * on or off on a specific DB instance.
     * </p>
     *
     * @param getResourceMetadataRequest
     * @return A Java Future containing the result of the GetResourceMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.GetResourceMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceMetadataResponse> getResourceMetadata(
            GetResourceMetadataRequest getResourceMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceMetadataRequest, GetResourceMetadataResponse>()
                            .withOperationName("GetResourceMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceMetadataRequest));
            CompletableFuture<GetResourceMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve Performance Insights metrics for a set of data sources over a time period. You can provide specific
     * dimension groups and dimensions, and provide aggregation and filtering criteria for each group.
     * </p>
     * <note>
     * <p>
     * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first
     * 500 bytes are returned.
     * </p>
     * </note>
     *
     * @param getResourceMetricsRequest
     * @return A Java Future containing the result of the GetResourceMetrics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.GetResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/GetResourceMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceMetricsResponse> getResourceMetrics(GetResourceMetricsRequest getResourceMetricsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceMetricsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceMetricsRequest, GetResourceMetricsResponse>()
                            .withOperationName("GetResourceMetrics").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceMetricsRequest));
            CompletableFuture<GetResourceMetricsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the dimensions that can be queried for each specified metric type on a specified DB instance.
     * </p>
     *
     * @param listAvailableResourceDimensionsRequest
     * @return A Java Future containing the result of the ListAvailableResourceDimensions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.ListAvailableResourceDimensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceDimensions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAvailableResourceDimensionsResponse> listAvailableResourceDimensions(
            ListAvailableResourceDimensionsRequest listAvailableResourceDimensionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAvailableResourceDimensionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAvailableResourceDimensionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAvailableResourceDimensions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAvailableResourceDimensionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAvailableResourceDimensionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAvailableResourceDimensionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAvailableResourceDimensionsRequest, ListAvailableResourceDimensionsResponse>()
                            .withOperationName("ListAvailableResourceDimensions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAvailableResourceDimensionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAvailableResourceDimensionsRequest));
            CompletableFuture<ListAvailableResourceDimensionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve metrics of the specified types that can be queried for a specified DB instance.
     * </p>
     *
     * @param listAvailableResourceMetricsRequest
     * @return A Java Future containing the result of the ListAvailableResourceMetrics operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.ListAvailableResourceMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListAvailableResourceMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAvailableResourceMetricsResponse> listAvailableResourceMetrics(
            ListAvailableResourceMetricsRequest listAvailableResourceMetricsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAvailableResourceMetricsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAvailableResourceMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAvailableResourceMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAvailableResourceMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAvailableResourceMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAvailableResourceMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAvailableResourceMetricsRequest, ListAvailableResourceMetricsResponse>()
                            .withOperationName("ListAvailableResourceMetrics").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAvailableResourceMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAvailableResourceMetricsRequest));
            CompletableFuture<ListAvailableResourceMetricsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the analysis reports created for the DB instance. The reports are sorted based on the start time of
     * each report.
     * </p>
     *
     * @param listPerformanceAnalysisReportsRequest
     * @return A Java Future containing the result of the ListPerformanceAnalysisReports operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.ListPerformanceAnalysisReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListPerformanceAnalysisReports"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPerformanceAnalysisReportsResponse> listPerformanceAnalysisReports(
            ListPerformanceAnalysisReportsRequest listPerformanceAnalysisReportsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPerformanceAnalysisReportsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPerformanceAnalysisReportsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPerformanceAnalysisReports");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPerformanceAnalysisReportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPerformanceAnalysisReportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPerformanceAnalysisReportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPerformanceAnalysisReportsRequest, ListPerformanceAnalysisReportsResponse>()
                            .withOperationName("ListPerformanceAnalysisReports").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPerformanceAnalysisReportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPerformanceAnalysisReportsRequest));
            CompletableFuture<ListPerformanceAnalysisReportsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all the metadata tags associated with Amazon RDS Performance Insights resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds metadata tags to the Amazon RDS Performance Insights resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the metadata tags from the Amazon RDS Performance Insights resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgumentException One of the arguments provided is invalid for this request.</li>
     *         <li>InternalServiceErrorException The request failed due to an unknown error.</li>
     *         <li>NotAuthorizedException The user is not authorized to perform this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PiException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample PiAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pi-2018-02-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PI");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PiServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PiException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        PiServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = PiServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
