/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pi.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A logical grouping of Performance Insights metrics for a related subject area. For example, the <code>db.sql</code>
 * dimension group consists of the following dimensions:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>db.sql.id</code> - The hash of a running SQL statement, generated by Performance Insights.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>db.sql.db_id</code> - Either the SQL ID generated by the database engine, or a value generated by Performance
 * Insights that begins with <code>pi-</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>db.sql.statement</code> - The full text of the SQL statement that is running, for example,
 * <code>SELECT * FROM employees</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>db.sql_tokenized.id</code> - The hash of the SQL digest generated by Performance Insights.
 * </p>
 * </li>
 * </ul>
 * <note>
 * <p>
 * Each response element returns a maximum of 500 bytes. For larger elements, such as SQL statements, only the first 500
 * bytes are returned.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DimensionGroup implements SdkPojo, Serializable, ToCopyableBuilder<DimensionGroup.Builder, DimensionGroup> {
    private static final SdkField<String> GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Group")
            .getter(getter(DimensionGroup::group)).setter(setter(Builder::group))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Group").build()).build();

    private static final SdkField<List<String>> DIMENSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(DimensionGroup::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> LIMIT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Limit")
            .getter(getter(DimensionGroup::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Limit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GROUP_FIELD, DIMENSIONS_FIELD,
            LIMIT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String group;

    private final List<String> dimensions;

    private final Integer limit;

    private DimensionGroup(BuilderImpl builder) {
        this.group = builder.group;
        this.dimensions = builder.dimensions;
        this.limit = builder.limit;
    }

    /**
     * <p>
     * The name of the dimension group. Valid values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>db</code> - The name of the database to which the client is connected. The following values are permitted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Aurora PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora MySQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS MySQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS MariaDB
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon DocumentDB
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>db.application</code> - The name of the application that is connected to the database. The following values
     * are permitted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Aurora PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon DocumentDB
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>db.host</code> - The host name of the connected client (all engines).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query</code> - The query that is currently running (only Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query_tokenized</code> - The digest query (only Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.session_type</code> - The type of the current session (only Aurora PostgreSQL and RDS PostgreSQL).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql</code> - The text of the SQL statement that is currently running (all engines except Amazon
     * DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql_tokenized</code> - The SQL digest (all engines except Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.user</code> - The user logged in to the database (all engines except Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.wait_event</code> - The event for which the database backend is waiting (all engines except Amazon
     * DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.wait_event_type</code> - The type of event for which the database backend is waiting (all engines except
     * Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.wait_state</code> - The event for which the database backend is waiting (only Amazon DocumentDB).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the dimension group. Valid values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>db</code> - The name of the database to which the client is connected. The following values are
     *         permitted:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Aurora PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora MySQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS MySQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS MariaDB
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon DocumentDB
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.application</code> - The name of the application that is connected to the database. The
     *         following values are permitted:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Aurora PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon DocumentDB
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.host</code> - The host name of the connected client (all engines).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query</code> - The query that is currently running (only Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query_tokenized</code> - The digest query (only Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.session_type</code> - The type of the current session (only Aurora PostgreSQL and RDS
     *         PostgreSQL).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql</code> - The text of the SQL statement that is currently running (all engines except Amazon
     *         DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql_tokenized</code> - The SQL digest (all engines except Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.user</code> - The user logged in to the database (all engines except Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.wait_event</code> - The event for which the database backend is waiting (all engines except
     *         Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.wait_event_type</code> - The type of event for which the database backend is waiting (all
     *         engines except Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.wait_state</code> - The event for which the database backend is waiting (only Amazon
     *         DocumentDB).
     *         </p>
     *         </li>
     */
    public final String group() {
        return group;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of specific dimensions from a dimension group. If this parameter is not present, then it signifies that
     * all of the dimensions in the group were requested, or are present in the response.
     * </p>
     * <p>
     * Valid values for elements in the <code>Dimensions</code> array are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>db.application.name</code> - The name of the application that is connected to the database. Valid values
     * are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Aurora PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon DocumentDB
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>db.host.id</code> - The host ID of the connected client (all engines).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.host.name</code> - The host name of the connected client (all engines).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.name</code> - The name of the database to which the client is connected. Valid values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Aurora PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS PostgreSQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora MySQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS MySQL
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon RDS MariaDB
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon DocumentDB
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>db.query.id</code> - The query ID generated by Performance Insights (only Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query.db_id</code> - The query ID generated by the database (only Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query.statement</code> - The text of the query that is being run (only Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query.tokenized_id</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query.tokenized.id</code> - The query digest ID generated by Performance Insights (only Amazon
     * DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query.tokenized.db_id</code> - The query digest ID generated by Performance Insights (only Amazon
     * DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.query.tokenized.statement</code> - The text of the query digest (only Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.session_type.name</code> - The type of the current session (only Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql.id</code> - The hash of the full, non-tokenized SQL statement generated by Performance Insights (all
     * engines except Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql.db_id</code> - Either the SQL ID generated by the database engine, or a value generated by
     * Performance Insights that begins with <code>pi-</code> (all engines except Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql.statement</code> - The full text of the SQL statement that is running, as in
     * <code>SELECT * FROM employees</code> (all engines except Amazon DocumentDB)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql.tokenized_id</code> - The hash of the SQL digest generated by Performance Insights (all engines
     * except Amazon DocumentDB). The <code>db.sql.tokenized_id</code> dimension fetches the value of the
     * <code>db.sql_tokenized.id</code> dimension. Amazon RDS returns <code>db.sql.tokenized_id</code> from the
     * <code>db.sql</code> dimension group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql_tokenized.id</code> - The hash of the SQL digest generated by Performance Insights (all engines
     * except Amazon DocumentDB). In the console, <code>db.sql_tokenized.id</code> is called the Support ID because
     * Amazon Web Services Support can look at this data to help you troubleshoot database issues.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql_tokenized.db_id</code> - Either the native database ID used to refer to the SQL statement, or a
     * synthetic ID such as <code>pi-2372568224</code> that Performance Insights generates if the native database ID
     * isn't available (all engines except Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.sql_tokenized.statement</code> - The text of the SQL digest, as in
     * <code>SELECT * FROM employees WHERE employee_id = ?</code> (all engines except Amazon DocumentDB)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.user.id</code> - The ID of the user logged in to the database (all engines except Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.user.name</code> - The name of the user logged in to the database (all engines except Amazon
     * DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.wait_event.name</code> - The event for which the backend is waiting (all engines except Amazon
     * DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.wait_event.type</code> - The type of event for which the backend is waiting (all engines except Amazon
     * DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.wait_event_type.name</code> - The name of the event type for which the backend is waiting (all engines
     * except Amazon DocumentDB).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>db.wait_state.name</code> - The event for which the backend is waiting (only Amazon DocumentDB).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return A list of specific dimensions from a dimension group. If this parameter is not present, then it signifies
     *         that all of the dimensions in the group were requested, or are present in the response.</p>
     *         <p>
     *         Valid values for elements in the <code>Dimensions</code> array are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>db.application.name</code> - The name of the application that is connected to the database. Valid
     *         values are as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Aurora PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon DocumentDB
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.host.id</code> - The host ID of the connected client (all engines).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.host.name</code> - The host name of the connected client (all engines).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.name</code> - The name of the database to which the client is connected. Valid values are as
     *         follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Aurora PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS PostgreSQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora MySQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS MySQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon RDS MariaDB
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon DocumentDB
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query.id</code> - The query ID generated by Performance Insights (only Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query.db_id</code> - The query ID generated by the database (only Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query.statement</code> - The text of the query that is being run (only Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query.tokenized_id</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query.tokenized.id</code> - The query digest ID generated by Performance Insights (only Amazon
     *         DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query.tokenized.db_id</code> - The query digest ID generated by Performance Insights (only
     *         Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.query.tokenized.statement</code> - The text of the query digest (only Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.session_type.name</code> - The type of the current session (only Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql.id</code> - The hash of the full, non-tokenized SQL statement generated by Performance
     *         Insights (all engines except Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql.db_id</code> - Either the SQL ID generated by the database engine, or a value generated by
     *         Performance Insights that begins with <code>pi-</code> (all engines except Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql.statement</code> - The full text of the SQL statement that is running, as in
     *         <code>SELECT * FROM employees</code> (all engines except Amazon DocumentDB)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql.tokenized_id</code> - The hash of the SQL digest generated by Performance Insights (all
     *         engines except Amazon DocumentDB). The <code>db.sql.tokenized_id</code> dimension fetches the value of
     *         the <code>db.sql_tokenized.id</code> dimension. Amazon RDS returns <code>db.sql.tokenized_id</code> from
     *         the <code>db.sql</code> dimension group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql_tokenized.id</code> - The hash of the SQL digest generated by Performance Insights (all
     *         engines except Amazon DocumentDB). In the console, <code>db.sql_tokenized.id</code> is called the Support
     *         ID because Amazon Web Services Support can look at this data to help you troubleshoot database issues.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql_tokenized.db_id</code> - Either the native database ID used to refer to the SQL statement,
     *         or a synthetic ID such as <code>pi-2372568224</code> that Performance Insights generates if the native
     *         database ID isn't available (all engines except Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.sql_tokenized.statement</code> - The text of the SQL digest, as in
     *         <code>SELECT * FROM employees WHERE employee_id = ?</code> (all engines except Amazon DocumentDB)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.user.id</code> - The ID of the user logged in to the database (all engines except Amazon
     *         DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.user.name</code> - The name of the user logged in to the database (all engines except Amazon
     *         DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.wait_event.name</code> - The event for which the backend is waiting (all engines except Amazon
     *         DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.wait_event.type</code> - The type of event for which the backend is waiting (all engines except
     *         Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.wait_event_type.name</code> - The name of the event type for which the backend is waiting (all
     *         engines except Amazon DocumentDB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>db.wait_state.name</code> - The event for which the backend is waiting (only Amazon DocumentDB).
     *         </p>
     *         </li>
     */
    public final List<String> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The maximum number of items to fetch for this dimension group.
     * </p>
     * 
     * @return The maximum number of items to fetch for this dimension group.
     */
    public final Integer limit() {
        return limit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(group());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DimensionGroup)) {
            return false;
        }
        DimensionGroup other = (DimensionGroup) obj;
        return Objects.equals(group(), other.group()) && hasDimensions() == other.hasDimensions()
                && Objects.equals(dimensions(), other.dimensions()) && Objects.equals(limit(), other.limit());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DimensionGroup").add("Group", group()).add("Dimensions", hasDimensions() ? dimensions() : null)
                .add("Limit", limit()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Group":
            return Optional.ofNullable(clazz.cast(group()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "Limit":
            return Optional.ofNullable(clazz.cast(limit()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Group", GROUP_FIELD);
        map.put("Dimensions", DIMENSIONS_FIELD);
        map.put("Limit", LIMIT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DimensionGroup, T> g) {
        return obj -> g.apply((DimensionGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DimensionGroup> {
        /**
         * <p>
         * The name of the dimension group. Valid values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>db</code> - The name of the database to which the client is connected. The following values are
         * permitted:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Aurora PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS MariaDB
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon DocumentDB
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>db.application</code> - The name of the application that is connected to the database. The following
         * values are permitted:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Aurora PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon DocumentDB
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>db.host</code> - The host name of the connected client (all engines).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query</code> - The query that is currently running (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query_tokenized</code> - The digest query (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.session_type</code> - The type of the current session (only Aurora PostgreSQL and RDS PostgreSQL).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql</code> - The text of the SQL statement that is currently running (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql_tokenized</code> - The SQL digest (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.user</code> - The user logged in to the database (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event</code> - The event for which the database backend is waiting (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event_type</code> - The type of event for which the database backend is waiting (all engines
         * except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_state</code> - The event for which the database backend is waiting (only Amazon DocumentDB).
         * </p>
         * </li>
         * </ul>
         * 
         * @param group
         *        The name of the dimension group. Valid values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>db</code> - The name of the database to which the client is connected. The following values are
         *        permitted:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Aurora PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS MariaDB
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon DocumentDB
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.application</code> - The name of the application that is connected to the database. The
         *        following values are permitted:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Aurora PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon DocumentDB
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.host</code> - The host name of the connected client (all engines).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query</code> - The query that is currently running (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query_tokenized</code> - The digest query (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.session_type</code> - The type of the current session (only Aurora PostgreSQL and RDS
         *        PostgreSQL).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql</code> - The text of the SQL statement that is currently running (all engines except
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql_tokenized</code> - The SQL digest (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.user</code> - The user logged in to the database (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event</code> - The event for which the database backend is waiting (all engines except
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event_type</code> - The type of event for which the database backend is waiting (all
         *        engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_state</code> - The event for which the database backend is waiting (only Amazon
         *        DocumentDB).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder group(String group);

        /**
         * <p>
         * A list of specific dimensions from a dimension group. If this parameter is not present, then it signifies
         * that all of the dimensions in the group were requested, or are present in the response.
         * </p>
         * <p>
         * Valid values for elements in the <code>Dimensions</code> array are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>db.application.name</code> - The name of the application that is connected to the database. Valid
         * values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Aurora PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon DocumentDB
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>db.host.id</code> - The host ID of the connected client (all engines).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.host.name</code> - The host name of the connected client (all engines).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.name</code> - The name of the database to which the client is connected. Valid values are as
         * follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Aurora PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS MariaDB
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon DocumentDB
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>db.query.id</code> - The query ID generated by Performance Insights (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.db_id</code> - The query ID generated by the database (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.statement</code> - The text of the query that is being run (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized_id</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized.id</code> - The query digest ID generated by Performance Insights (only Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized.db_id</code> - The query digest ID generated by Performance Insights (only Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized.statement</code> - The text of the query digest (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.session_type.name</code> - The type of the current session (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.id</code> - The hash of the full, non-tokenized SQL statement generated by Performance Insights
         * (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.db_id</code> - Either the SQL ID generated by the database engine, or a value generated by
         * Performance Insights that begins with <code>pi-</code> (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.statement</code> - The full text of the SQL statement that is running, as in
         * <code>SELECT * FROM employees</code> (all engines except Amazon DocumentDB)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.tokenized_id</code> - The hash of the SQL digest generated by Performance Insights (all engines
         * except Amazon DocumentDB). The <code>db.sql.tokenized_id</code> dimension fetches the value of the
         * <code>db.sql_tokenized.id</code> dimension. Amazon RDS returns <code>db.sql.tokenized_id</code> from the
         * <code>db.sql</code> dimension group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql_tokenized.id</code> - The hash of the SQL digest generated by Performance Insights (all engines
         * except Amazon DocumentDB). In the console, <code>db.sql_tokenized.id</code> is called the Support ID because
         * Amazon Web Services Support can look at this data to help you troubleshoot database issues.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql_tokenized.db_id</code> - Either the native database ID used to refer to the SQL statement, or a
         * synthetic ID such as <code>pi-2372568224</code> that Performance Insights generates if the native database ID
         * isn't available (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql_tokenized.statement</code> - The text of the SQL digest, as in
         * <code>SELECT * FROM employees WHERE employee_id = ?</code> (all engines except Amazon DocumentDB)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.user.id</code> - The ID of the user logged in to the database (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.user.name</code> - The name of the user logged in to the database (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event.name</code> - The event for which the backend is waiting (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event.type</code> - The type of event for which the backend is waiting (all engines except
         * Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event_type.name</code> - The name of the event type for which the backend is waiting (all
         * engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_state.name</code> - The event for which the backend is waiting (only Amazon DocumentDB).
         * </p>
         * </li>
         * </ul>
         * 
         * @param dimensions
         *        A list of specific dimensions from a dimension group. If this parameter is not present, then it
         *        signifies that all of the dimensions in the group were requested, or are present in the response.</p>
         *        <p>
         *        Valid values for elements in the <code>Dimensions</code> array are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>db.application.name</code> - The name of the application that is connected to the database.
         *        Valid values are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Aurora PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon DocumentDB
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.host.id</code> - The host ID of the connected client (all engines).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.host.name</code> - The host name of the connected client (all engines).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.name</code> - The name of the database to which the client is connected. Valid values are as
         *        follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Aurora PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS MariaDB
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon DocumentDB
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.id</code> - The query ID generated by Performance Insights (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.db_id</code> - The query ID generated by the database (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.statement</code> - The text of the query that is being run (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized_id</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized.id</code> - The query digest ID generated by Performance Insights (only
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized.db_id</code> - The query digest ID generated by Performance Insights (only
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized.statement</code> - The text of the query digest (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.session_type.name</code> - The type of the current session (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.id</code> - The hash of the full, non-tokenized SQL statement generated by Performance
         *        Insights (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.db_id</code> - Either the SQL ID generated by the database engine, or a value generated
         *        by Performance Insights that begins with <code>pi-</code> (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.statement</code> - The full text of the SQL statement that is running, as in
         *        <code>SELECT * FROM employees</code> (all engines except Amazon DocumentDB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.tokenized_id</code> - The hash of the SQL digest generated by Performance Insights (all
         *        engines except Amazon DocumentDB). The <code>db.sql.tokenized_id</code> dimension fetches the value of
         *        the <code>db.sql_tokenized.id</code> dimension. Amazon RDS returns <code>db.sql.tokenized_id</code>
         *        from the <code>db.sql</code> dimension group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql_tokenized.id</code> - The hash of the SQL digest generated by Performance Insights (all
         *        engines except Amazon DocumentDB). In the console, <code>db.sql_tokenized.id</code> is called the
         *        Support ID because Amazon Web Services Support can look at this data to help you troubleshoot database
         *        issues.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql_tokenized.db_id</code> - Either the native database ID used to refer to the SQL
         *        statement, or a synthetic ID such as <code>pi-2372568224</code> that Performance Insights generates if
         *        the native database ID isn't available (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql_tokenized.statement</code> - The text of the SQL digest, as in
         *        <code>SELECT * FROM employees WHERE employee_id = ?</code> (all engines except Amazon DocumentDB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.user.id</code> - The ID of the user logged in to the database (all engines except Amazon
         *        DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.user.name</code> - The name of the user logged in to the database (all engines except Amazon
         *        DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event.name</code> - The event for which the backend is waiting (all engines except
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event.type</code> - The type of event for which the backend is waiting (all engines
         *        except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event_type.name</code> - The name of the event type for which the backend is waiting
         *        (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_state.name</code> - The event for which the backend is waiting (only Amazon DocumentDB).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<String> dimensions);

        /**
         * <p>
         * A list of specific dimensions from a dimension group. If this parameter is not present, then it signifies
         * that all of the dimensions in the group were requested, or are present in the response.
         * </p>
         * <p>
         * Valid values for elements in the <code>Dimensions</code> array are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>db.application.name</code> - The name of the application that is connected to the database. Valid
         * values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Aurora PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon DocumentDB
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>db.host.id</code> - The host ID of the connected client (all engines).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.host.name</code> - The host name of the connected client (all engines).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.name</code> - The name of the database to which the client is connected. Valid values are as
         * follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Aurora PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS PostgreSQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon RDS MariaDB
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon DocumentDB
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>db.query.id</code> - The query ID generated by Performance Insights (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.db_id</code> - The query ID generated by the database (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.statement</code> - The text of the query that is being run (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized_id</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized.id</code> - The query digest ID generated by Performance Insights (only Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized.db_id</code> - The query digest ID generated by Performance Insights (only Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.query.tokenized.statement</code> - The text of the query digest (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.session_type.name</code> - The type of the current session (only Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.id</code> - The hash of the full, non-tokenized SQL statement generated by Performance Insights
         * (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.db_id</code> - Either the SQL ID generated by the database engine, or a value generated by
         * Performance Insights that begins with <code>pi-</code> (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.statement</code> - The full text of the SQL statement that is running, as in
         * <code>SELECT * FROM employees</code> (all engines except Amazon DocumentDB)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql.tokenized_id</code> - The hash of the SQL digest generated by Performance Insights (all engines
         * except Amazon DocumentDB). The <code>db.sql.tokenized_id</code> dimension fetches the value of the
         * <code>db.sql_tokenized.id</code> dimension. Amazon RDS returns <code>db.sql.tokenized_id</code> from the
         * <code>db.sql</code> dimension group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql_tokenized.id</code> - The hash of the SQL digest generated by Performance Insights (all engines
         * except Amazon DocumentDB). In the console, <code>db.sql_tokenized.id</code> is called the Support ID because
         * Amazon Web Services Support can look at this data to help you troubleshoot database issues.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql_tokenized.db_id</code> - Either the native database ID used to refer to the SQL statement, or a
         * synthetic ID such as <code>pi-2372568224</code> that Performance Insights generates if the native database ID
         * isn't available (all engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.sql_tokenized.statement</code> - The text of the SQL digest, as in
         * <code>SELECT * FROM employees WHERE employee_id = ?</code> (all engines except Amazon DocumentDB)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.user.id</code> - The ID of the user logged in to the database (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.user.name</code> - The name of the user logged in to the database (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event.name</code> - The event for which the backend is waiting (all engines except Amazon
         * DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event.type</code> - The type of event for which the backend is waiting (all engines except
         * Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_event_type.name</code> - The name of the event type for which the backend is waiting (all
         * engines except Amazon DocumentDB).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>db.wait_state.name</code> - The event for which the backend is waiting (only Amazon DocumentDB).
         * </p>
         * </li>
         * </ul>
         * 
         * @param dimensions
         *        A list of specific dimensions from a dimension group. If this parameter is not present, then it
         *        signifies that all of the dimensions in the group were requested, or are present in the response.</p>
         *        <p>
         *        Valid values for elements in the <code>Dimensions</code> array are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>db.application.name</code> - The name of the application that is connected to the database.
         *        Valid values are as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Aurora PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon DocumentDB
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.host.id</code> - The host ID of the connected client (all engines).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.host.name</code> - The host name of the connected client (all engines).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.name</code> - The name of the database to which the client is connected. Valid values are as
         *        follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Aurora PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS PostgreSQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon RDS MariaDB
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon DocumentDB
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.id</code> - The query ID generated by Performance Insights (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.db_id</code> - The query ID generated by the database (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.statement</code> - The text of the query that is being run (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized_id</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized.id</code> - The query digest ID generated by Performance Insights (only
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized.db_id</code> - The query digest ID generated by Performance Insights (only
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.query.tokenized.statement</code> - The text of the query digest (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.session_type.name</code> - The type of the current session (only Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.id</code> - The hash of the full, non-tokenized SQL statement generated by Performance
         *        Insights (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.db_id</code> - Either the SQL ID generated by the database engine, or a value generated
         *        by Performance Insights that begins with <code>pi-</code> (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.statement</code> - The full text of the SQL statement that is running, as in
         *        <code>SELECT * FROM employees</code> (all engines except Amazon DocumentDB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql.tokenized_id</code> - The hash of the SQL digest generated by Performance Insights (all
         *        engines except Amazon DocumentDB). The <code>db.sql.tokenized_id</code> dimension fetches the value of
         *        the <code>db.sql_tokenized.id</code> dimension. Amazon RDS returns <code>db.sql.tokenized_id</code>
         *        from the <code>db.sql</code> dimension group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql_tokenized.id</code> - The hash of the SQL digest generated by Performance Insights (all
         *        engines except Amazon DocumentDB). In the console, <code>db.sql_tokenized.id</code> is called the
         *        Support ID because Amazon Web Services Support can look at this data to help you troubleshoot database
         *        issues.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql_tokenized.db_id</code> - Either the native database ID used to refer to the SQL
         *        statement, or a synthetic ID such as <code>pi-2372568224</code> that Performance Insights generates if
         *        the native database ID isn't available (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.sql_tokenized.statement</code> - The text of the SQL digest, as in
         *        <code>SELECT * FROM employees WHERE employee_id = ?</code> (all engines except Amazon DocumentDB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.user.id</code> - The ID of the user logged in to the database (all engines except Amazon
         *        DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.user.name</code> - The name of the user logged in to the database (all engines except Amazon
         *        DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event.name</code> - The event for which the backend is waiting (all engines except
         *        Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event.type</code> - The type of event for which the backend is waiting (all engines
         *        except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_event_type.name</code> - The name of the event type for which the backend is waiting
         *        (all engines except Amazon DocumentDB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>db.wait_state.name</code> - The event for which the backend is waiting (only Amazon DocumentDB).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(String... dimensions);

        /**
         * <p>
         * The maximum number of items to fetch for this dimension group.
         * </p>
         * 
         * @param limit
         *        The maximum number of items to fetch for this dimension group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Integer limit);
    }

    static final class BuilderImpl implements Builder {
        private String group;

        private List<String> dimensions = DefaultSdkAutoConstructList.getInstance();

        private Integer limit;

        private BuilderImpl() {
        }

        private BuilderImpl(DimensionGroup model) {
            group(model.group);
            dimensions(model.dimensions);
            limit(model.limit);
        }

        public final String getGroup() {
            return group;
        }

        public final void setGroup(String group) {
            this.group = group;
        }

        @Override
        public final Builder group(String group) {
            this.group = group;
            return this;
        }

        public final Collection<String> getDimensions() {
            if (dimensions instanceof SdkAutoConstructList) {
                return null;
            }
            return dimensions;
        }

        public final void setDimensions(Collection<String> dimensions) {
            this.dimensions = SanitizedStringListCopier.copy(dimensions);
        }

        @Override
        public final Builder dimensions(Collection<String> dimensions) {
            this.dimensions = SanitizedStringListCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(String... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        public final Integer getLimit() {
            return limit;
        }

        public final void setLimit(Integer limit) {
            this.limit = limit;
        }

        @Override
        public final Builder limit(Integer limit) {
            this.limit = limit;
            return this;
        }

        @Override
        public DimensionGroup build() {
            return new DimensionGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
