/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pi.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Retrieves the list of performance issues which are identified.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Insight implements SdkPojo, Serializable, ToCopyableBuilder<Insight.Builder, Insight> {
    private static final SdkField<String> INSIGHT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InsightId").getter(getter(Insight::insightId)).setter(setter(Builder::insightId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InsightId").build()).build();

    private static final SdkField<String> INSIGHT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InsightType").getter(getter(Insight::insightType)).setter(setter(Builder::insightType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InsightType").build()).build();

    private static final SdkField<String> CONTEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Context")
            .getter(getter(Insight::contextAsString)).setter(setter(Builder::context))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Context").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(Insight::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(Insight::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(Insight::severityAsString)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<List<Insight>> SUPPORTING_INSIGHTS_FIELD = SdkField
            .<List<Insight>> builder(MarshallingType.LIST)
            .memberName("SupportingInsights")
            .getter(getter(Insight::supportingInsights))
            .setter(setter(Builder::supportingInsights))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportingInsights").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Insight> builder(MarshallingType.SDK_POJO)
                                            .constructor(Insight::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Insight::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<Recommendation>> RECOMMENDATIONS_FIELD = SdkField
            .<List<Recommendation>> builder(MarshallingType.LIST)
            .memberName("Recommendations")
            .getter(getter(Insight::recommendations))
            .setter(setter(Builder::recommendations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recommendations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Recommendation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Recommendation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Data>> INSIGHT_DATA_FIELD = SdkField
            .<List<Data>> builder(MarshallingType.LIST)
            .memberName("InsightData")
            .getter(getter(Insight::insightData))
            .setter(setter(Builder::insightData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InsightData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Data> builder(MarshallingType.SDK_POJO)
                                            .constructor(Data::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Data>> BASELINE_DATA_FIELD = SdkField
            .<List<Data>> builder(MarshallingType.LIST)
            .memberName("BaselineData")
            .getter(getter(Insight::baselineData))
            .setter(setter(Builder::baselineData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaselineData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Data> builder(MarshallingType.SDK_POJO)
                                            .constructor(Data::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSIGHT_ID_FIELD,
            INSIGHT_TYPE_FIELD, CONTEXT_FIELD, START_TIME_FIELD, END_TIME_FIELD, SEVERITY_FIELD, SUPPORTING_INSIGHTS_FIELD,
            DESCRIPTION_FIELD, RECOMMENDATIONS_FIELD, INSIGHT_DATA_FIELD, BASELINE_DATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String insightId;

    private final String insightType;

    private final String context;

    private final Instant startTime;

    private final Instant endTime;

    private final String severity;

    private final List<Insight> supportingInsights;

    private final String description;

    private final List<Recommendation> recommendations;

    private final List<Data> insightData;

    private final List<Data> baselineData;

    private Insight(BuilderImpl builder) {
        this.insightId = builder.insightId;
        this.insightType = builder.insightType;
        this.context = builder.context;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.severity = builder.severity;
        this.supportingInsights = builder.supportingInsights;
        this.description = builder.description;
        this.recommendations = builder.recommendations;
        this.insightData = builder.insightData;
        this.baselineData = builder.baselineData;
    }

    /**
     * <p>
     * The unique identifier for the insight. For example, <code>insight-12345678901234567</code>.
     * </p>
     * 
     * @return The unique identifier for the insight. For example, <code>insight-12345678901234567</code>.
     */
    public final String insightId() {
        return insightId;
    }

    /**
     * <p>
     * The type of insight. For example, <code>HighDBLoad</code>, <code>HighCPU</code>, or <code>DominatingSQLs</code>.
     * </p>
     * 
     * @return The type of insight. For example, <code>HighDBLoad</code>, <code>HighCPU</code>, or
     *         <code>DominatingSQLs</code>.
     */
    public final String insightType() {
        return insightType;
    }

    /**
     * <p>
     * Indicates if the insight is causal or correlated insight.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #context} will
     * return {@link ContextType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contextAsString}.
     * </p>
     * 
     * @return Indicates if the insight is causal or correlated insight.
     * @see ContextType
     */
    public final ContextType context() {
        return ContextType.fromValue(context);
    }

    /**
     * <p>
     * Indicates if the insight is causal or correlated insight.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #context} will
     * return {@link ContextType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contextAsString}.
     * </p>
     * 
     * @return Indicates if the insight is causal or correlated insight.
     * @see ContextType
     */
    public final String contextAsString() {
        return context;
    }

    /**
     * <p>
     * The start time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
     * </p>
     * 
     * @return The start time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The end time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
     * </p>
     * 
     * @return The end time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or <code>High</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link Severity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or <code>High</code>.
     * @see Severity
     */
    public final Severity severity() {
        return Severity.fromValue(severity);
    }

    /**
     * <p>
     * The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or <code>High</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link Severity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or <code>High</code>.
     * @see Severity
     */
    public final String severityAsString() {
        return severity;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportingInsights property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportingInsights() {
        return supportingInsights != null && !(supportingInsights instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of supporting insights that provide additional factors for the insight.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportingInsights} method.
     * </p>
     * 
     * @return List of supporting insights that provide additional factors for the insight.
     */
    public final List<Insight> supportingInsights() {
        return supportingInsights;
    }

    /**
     * <p>
     * Description of the insight. For example:
     * <code>A high severity Insight found between 02:00 to 02:30, where there was an unusually high DB load 600x above baseline. Likely performance impact</code>
     * .
     * </p>
     * 
     * @return Description of the insight. For example:
     *         <code>A high severity Insight found between 02:00 to 02:30, where there was an unusually high DB load 600x above baseline. Likely performance impact</code>
     *         .
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Recommendations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendations() {
        return recommendations != null && !(recommendations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of recommendations for the insight. For example,
     * <code>Investigate the following SQLs that contributed to 100% of the total DBLoad during that time period: sql-id</code>
     * .
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendations} method.
     * </p>
     * 
     * @return List of recommendations for the insight. For example,
     *         <code>Investigate the following SQLs that contributed to 100% of the total DBLoad during that time period: sql-id</code>
     *         .
     */
    public final List<Recommendation> recommendations() {
        return recommendations;
    }

    /**
     * For responses, this returns true if the service returned a value for the InsightData property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInsightData() {
        return insightData != null && !(insightData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of data objects containing metrics and references from the time range while generating the insight.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInsightData} method.
     * </p>
     * 
     * @return List of data objects containing metrics and references from the time range while generating the insight.
     */
    public final List<Data> insightData() {
        return insightData;
    }

    /**
     * For responses, this returns true if the service returned a value for the BaselineData property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBaselineData() {
        return baselineData != null && !(baselineData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Metric names and values from the timeframe used as baseline to generate the insight.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBaselineData} method.
     * </p>
     * 
     * @return Metric names and values from the timeframe used as baseline to generate the insight.
     */
    public final List<Data> baselineData() {
        return baselineData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(insightId());
        hashCode = 31 * hashCode + Objects.hashCode(insightType());
        hashCode = 31 * hashCode + Objects.hashCode(contextAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportingInsights() ? supportingInsights() : null);
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendations() ? recommendations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInsightData() ? insightData() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasBaselineData() ? baselineData() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Insight)) {
            return false;
        }
        Insight other = (Insight) obj;
        return Objects.equals(insightId(), other.insightId()) && Objects.equals(insightType(), other.insightType())
                && Objects.equals(contextAsString(), other.contextAsString()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(severityAsString(), other.severityAsString())
                && hasSupportingInsights() == other.hasSupportingInsights()
                && Objects.equals(supportingInsights(), other.supportingInsights())
                && Objects.equals(description(), other.description()) && hasRecommendations() == other.hasRecommendations()
                && Objects.equals(recommendations(), other.recommendations()) && hasInsightData() == other.hasInsightData()
                && Objects.equals(insightData(), other.insightData()) && hasBaselineData() == other.hasBaselineData()
                && Objects.equals(baselineData(), other.baselineData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Insight").add("InsightId", insightId()).add("InsightType", insightType())
                .add("Context", contextAsString()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("Severity", severityAsString())
                .add("SupportingInsights", hasSupportingInsights() ? supportingInsights() : null)
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Recommendations", hasRecommendations() ? recommendations() : null)
                .add("InsightData", hasInsightData() ? insightData() : null)
                .add("BaselineData", hasBaselineData() ? baselineData() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InsightId":
            return Optional.ofNullable(clazz.cast(insightId()));
        case "InsightType":
            return Optional.ofNullable(clazz.cast(insightType()));
        case "Context":
            return Optional.ofNullable(clazz.cast(contextAsString()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "SupportingInsights":
            return Optional.ofNullable(clazz.cast(supportingInsights()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Recommendations":
            return Optional.ofNullable(clazz.cast(recommendations()));
        case "InsightData":
            return Optional.ofNullable(clazz.cast(insightData()));
        case "BaselineData":
            return Optional.ofNullable(clazz.cast(baselineData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InsightId", INSIGHT_ID_FIELD);
        map.put("InsightType", INSIGHT_TYPE_FIELD);
        map.put("Context", CONTEXT_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("EndTime", END_TIME_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("SupportingInsights", SUPPORTING_INSIGHTS_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Recommendations", RECOMMENDATIONS_FIELD);
        map.put("InsightData", INSIGHT_DATA_FIELD);
        map.put("BaselineData", BASELINE_DATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Insight, T> g) {
        return obj -> g.apply((Insight) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Insight> {
        /**
         * <p>
         * The unique identifier for the insight. For example, <code>insight-12345678901234567</code>.
         * </p>
         * 
         * @param insightId
         *        The unique identifier for the insight. For example, <code>insight-12345678901234567</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insightId(String insightId);

        /**
         * <p>
         * The type of insight. For example, <code>HighDBLoad</code>, <code>HighCPU</code>, or
         * <code>DominatingSQLs</code>.
         * </p>
         * 
         * @param insightType
         *        The type of insight. For example, <code>HighDBLoad</code>, <code>HighCPU</code>, or
         *        <code>DominatingSQLs</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insightType(String insightType);

        /**
         * <p>
         * Indicates if the insight is causal or correlated insight.
         * </p>
         * 
         * @param context
         *        Indicates if the insight is causal or correlated insight.
         * @see ContextType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContextType
         */
        Builder context(String context);

        /**
         * <p>
         * Indicates if the insight is causal or correlated insight.
         * </p>
         * 
         * @param context
         *        Indicates if the insight is causal or correlated insight.
         * @see ContextType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContextType
         */
        Builder context(ContextType context);

        /**
         * <p>
         * The start time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
         * </p>
         * 
         * @param startTime
         *        The start time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The end time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
         * </p>
         * 
         * @param endTime
         *        The end time of the insight. For example, <code>2018-10-30T00:00:00Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or <code>High</code>.
         * </p>
         * 
         * @param severity
         *        The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or
         *        <code>High</code>.
         * @see Severity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Severity
         */
        Builder severity(String severity);

        /**
         * <p>
         * The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or <code>High</code>.
         * </p>
         * 
         * @param severity
         *        The severity of the insight. The values are: <code>Low</code>, <code>Medium</code>, or
         *        <code>High</code>.
         * @see Severity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Severity
         */
        Builder severity(Severity severity);

        /**
         * <p>
         * List of supporting insights that provide additional factors for the insight.
         * </p>
         * 
         * @param supportingInsights
         *        List of supporting insights that provide additional factors for the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportingInsights(Collection<Insight> supportingInsights);

        /**
         * <p>
         * List of supporting insights that provide additional factors for the insight.
         * </p>
         * 
         * @param supportingInsights
         *        List of supporting insights that provide additional factors for the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportingInsights(Insight... supportingInsights);

        /**
         * <p>
         * List of supporting insights that provide additional factors for the insight.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pi.model.Insight.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.pi.model.Insight#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.pi.model.Insight.Builder#build()}
         * is called immediately and its result is passed to {@link #supportingInsights(List<Insight>)}.
         * 
         * @param supportingInsights
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.pi.model.Insight.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportingInsights(java.util.Collection<Insight>)
         */
        Builder supportingInsights(Consumer<Builder>... supportingInsights);

        /**
         * <p>
         * Description of the insight. For example:
         * <code>A high severity Insight found between 02:00 to 02:30, where there was an unusually high DB load 600x above baseline. Likely performance impact</code>
         * .
         * </p>
         * 
         * @param description
         *        Description of the insight. For example:
         *        <code>A high severity Insight found between 02:00 to 02:30, where there was an unusually high DB load 600x above baseline. Likely performance impact</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * List of recommendations for the insight. For example,
         * <code>Investigate the following SQLs that contributed to 100% of the total DBLoad during that time period: sql-id</code>
         * .
         * </p>
         * 
         * @param recommendations
         *        List of recommendations for the insight. For example,
         *        <code>Investigate the following SQLs that contributed to 100% of the total DBLoad during that time period: sql-id</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Collection<Recommendation> recommendations);

        /**
         * <p>
         * List of recommendations for the insight. For example,
         * <code>Investigate the following SQLs that contributed to 100% of the total DBLoad during that time period: sql-id</code>
         * .
         * </p>
         * 
         * @param recommendations
         *        List of recommendations for the insight. For example,
         *        <code>Investigate the following SQLs that contributed to 100% of the total DBLoad during that time period: sql-id</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Recommendation... recommendations);

        /**
         * <p>
         * List of recommendations for the insight. For example,
         * <code>Investigate the following SQLs that contributed to 100% of the total DBLoad during that time period: sql-id</code>
         * .
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pi.model.Recommendation.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.pi.model.Recommendation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.pi.model.Recommendation.Builder#build()} is called immediately and its
         * result is passed to {@link #recommendations(List<Recommendation>)}.
         * 
         * @param recommendations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.pi.model.Recommendation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendations(java.util.Collection<Recommendation>)
         */
        Builder recommendations(Consumer<Recommendation.Builder>... recommendations);

        /**
         * <p>
         * List of data objects containing metrics and references from the time range while generating the insight.
         * </p>
         * 
         * @param insightData
         *        List of data objects containing metrics and references from the time range while generating the
         *        insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insightData(Collection<Data> insightData);

        /**
         * <p>
         * List of data objects containing metrics and references from the time range while generating the insight.
         * </p>
         * 
         * @param insightData
         *        List of data objects containing metrics and references from the time range while generating the
         *        insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insightData(Data... insightData);

        /**
         * <p>
         * List of data objects containing metrics and references from the time range while generating the insight.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pi.model.Data.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.pi.model.Data#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.pi.model.Data.Builder#build()} is
         * called immediately and its result is passed to {@link #insightData(List<Data>)}.
         * 
         * @param insightData
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.pi.model.Data.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #insightData(java.util.Collection<Data>)
         */
        Builder insightData(Consumer<Data.Builder>... insightData);

        /**
         * <p>
         * Metric names and values from the timeframe used as baseline to generate the insight.
         * </p>
         * 
         * @param baselineData
         *        Metric names and values from the timeframe used as baseline to generate the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineData(Collection<Data> baselineData);

        /**
         * <p>
         * Metric names and values from the timeframe used as baseline to generate the insight.
         * </p>
         * 
         * @param baselineData
         *        Metric names and values from the timeframe used as baseline to generate the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineData(Data... baselineData);

        /**
         * <p>
         * Metric names and values from the timeframe used as baseline to generate the insight.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pi.model.Data.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.pi.model.Data#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.pi.model.Data.Builder#build()} is
         * called immediately and its result is passed to {@link #baselineData(List<Data>)}.
         * 
         * @param baselineData
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.pi.model.Data.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #baselineData(java.util.Collection<Data>)
         */
        Builder baselineData(Consumer<Data.Builder>... baselineData);
    }

    static final class BuilderImpl implements Builder {
        private String insightId;

        private String insightType;

        private String context;

        private Instant startTime;

        private Instant endTime;

        private String severity;

        private List<Insight> supportingInsights = DefaultSdkAutoConstructList.getInstance();

        private String description;

        private List<Recommendation> recommendations = DefaultSdkAutoConstructList.getInstance();

        private List<Data> insightData = DefaultSdkAutoConstructList.getInstance();

        private List<Data> baselineData = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Insight model) {
            insightId(model.insightId);
            insightType(model.insightType);
            context(model.context);
            startTime(model.startTime);
            endTime(model.endTime);
            severity(model.severity);
            supportingInsights(model.supportingInsights);
            description(model.description);
            recommendations(model.recommendations);
            insightData(model.insightData);
            baselineData(model.baselineData);
        }

        public final String getInsightId() {
            return insightId;
        }

        public final void setInsightId(String insightId) {
            this.insightId = insightId;
        }

        @Override
        public final Builder insightId(String insightId) {
            this.insightId = insightId;
            return this;
        }

        public final String getInsightType() {
            return insightType;
        }

        public final void setInsightType(String insightType) {
            this.insightType = insightType;
        }

        @Override
        public final Builder insightType(String insightType) {
            this.insightType = insightType;
            return this;
        }

        public final String getContext() {
            return context;
        }

        public final void setContext(String context) {
            this.context = context;
        }

        @Override
        public final Builder context(String context) {
            this.context = context;
            return this;
        }

        @Override
        public final Builder context(ContextType context) {
            this.context(context == null ? null : context.toString());
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        public final Builder severity(Severity severity) {
            this.severity(severity == null ? null : severity.toString());
            return this;
        }

        public final List<Builder> getSupportingInsights() {
            List<Builder> result = InsightListCopier.copyToBuilder(this.supportingInsights);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSupportingInsights(Collection<BuilderImpl> supportingInsights) {
            this.supportingInsights = InsightListCopier.copyFromBuilder(supportingInsights);
        }

        @Override
        public final Builder supportingInsights(Collection<Insight> supportingInsights) {
            this.supportingInsights = InsightListCopier.copy(supportingInsights);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportingInsights(Insight... supportingInsights) {
            supportingInsights(Arrays.asList(supportingInsights));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportingInsights(Consumer<Builder>... supportingInsights) {
            supportingInsights(Stream.of(supportingInsights).map(c -> Insight.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<Recommendation.Builder> getRecommendations() {
            List<Recommendation.Builder> result = RecommendationListCopier.copyToBuilder(this.recommendations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecommendations(Collection<Recommendation.BuilderImpl> recommendations) {
            this.recommendations = RecommendationListCopier.copyFromBuilder(recommendations);
        }

        @Override
        public final Builder recommendations(Collection<Recommendation> recommendations) {
            this.recommendations = RecommendationListCopier.copy(recommendations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Recommendation... recommendations) {
            recommendations(Arrays.asList(recommendations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Consumer<Recommendation.Builder>... recommendations) {
            recommendations(Stream.of(recommendations).map(c -> Recommendation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Data.Builder> getInsightData() {
            List<Data.Builder> result = DataListCopier.copyToBuilder(this.insightData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInsightData(Collection<Data.BuilderImpl> insightData) {
            this.insightData = DataListCopier.copyFromBuilder(insightData);
        }

        @Override
        public final Builder insightData(Collection<Data> insightData) {
            this.insightData = DataListCopier.copy(insightData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder insightData(Data... insightData) {
            insightData(Arrays.asList(insightData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder insightData(Consumer<Data.Builder>... insightData) {
            insightData(Stream.of(insightData).map(c -> Data.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Data.Builder> getBaselineData() {
            List<Data.Builder> result = DataListCopier.copyToBuilder(this.baselineData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBaselineData(Collection<Data.BuilderImpl> baselineData) {
            this.baselineData = DataListCopier.copyFromBuilder(baselineData);
        }

        @Override
        public final Builder baselineData(Collection<Data> baselineData) {
            this.baselineData = DataListCopier.copy(baselineData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder baselineData(Data... baselineData) {
            baselineData(Arrays.asList(baselineData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder baselineData(Consumer<Data.Builder>... baselineData) {
            baselineData(Stream.of(baselineData).map(c -> Data.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Insight build() {
            return new Insight(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
