/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Send message request.
 */
@Generated("software.amazon.awssdk:codegen")
public final class MessageRequest implements SdkPojo, Serializable, ToCopyableBuilder<MessageRequest.Builder, MessageRequest> {
    private static final SdkField<Map<String, AddressConfiguration>> ADDRESSES_FIELD = SdkField
            .<Map<String, AddressConfiguration>> builder(MarshallingType.MAP)
            .getter(getter(MessageRequest::addresses))
            .setter(setter(Builder::addresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Addresses").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AddressConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(AddressConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(MessageRequest::context))
            .setter(setter(Builder::context))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Context").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, EndpointSendConfiguration>> ENDPOINTS_FIELD = SdkField
            .<Map<String, EndpointSendConfiguration>> builder(MarshallingType.MAP)
            .getter(getter(MessageRequest::endpoints))
            .setter(setter(Builder::endpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoints").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<EndpointSendConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(EndpointSendConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<DirectMessageConfiguration> MESSAGE_CONFIGURATION_FIELD = SdkField
            .<DirectMessageConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(MessageRequest::messageConfiguration))
            .setter(setter(Builder::messageConfiguration)).constructor(DirectMessageConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageConfiguration").build())
            .build();

    private static final SdkField<String> TRACE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MessageRequest::traceId)).setter(setter(Builder::traceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TraceId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADDRESSES_FIELD,
            CONTEXT_FIELD, ENDPOINTS_FIELD, MESSAGE_CONFIGURATION_FIELD, TRACE_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, AddressConfiguration> addresses;

    private final Map<String, String> context;

    private final Map<String, EndpointSendConfiguration> endpoints;

    private final DirectMessageConfiguration messageConfiguration;

    private final String traceId;

    private MessageRequest(BuilderImpl builder) {
        this.addresses = builder.addresses;
        this.context = builder.context;
        this.endpoints = builder.endpoints;
        this.messageConfiguration = builder.messageConfiguration;
        this.traceId = builder.traceId;
    }

    /**
     * A map of key-value pairs, where each key is an address and each value is an AddressConfiguration object. An
     * address can be a push notification token, a phone number, or an email address.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A map of key-value pairs, where each key is an address and each value is an AddressConfiguration object.
     *         An address can be a push notification token, a phone number, or an email address.
     */
    public Map<String, AddressConfiguration> addresses() {
        return addresses;
    }

    /**
     * A map of custom attributes to attributes to be attached to the message. This payload is added to the push
     * notification's 'data.pinpoint' object or added to the email/sms delivery receipt event attributes.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A map of custom attributes to attributes to be attached to the message. This payload is added to the push
     *         notification's 'data.pinpoint' object or added to the email/sms delivery receipt event attributes.
     */
    public Map<String, String> context() {
        return context;
    }

    /**
     * A map of key-value pairs, where each key is an endpoint ID and each value is an EndpointSendConfiguration object.
     * Within an EndpointSendConfiguration object, you can tailor the message for an endpoint by specifying message
     * overrides or substitutions.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A map of key-value pairs, where each key is an endpoint ID and each value is an EndpointSendConfiguration
     *         object. Within an EndpointSendConfiguration object, you can tailor the message for an endpoint by
     *         specifying message overrides or substitutions.
     */
    public Map<String, EndpointSendConfiguration> endpoints() {
        return endpoints;
    }

    /**
     * Message configuration.
     * 
     * @return Message configuration.
     */
    public DirectMessageConfiguration messageConfiguration() {
        return messageConfiguration;
    }

    /**
     * A unique ID that you can use to trace a message. This ID is visible to recipients.
     * 
     * @return A unique ID that you can use to trace a message. This ID is visible to recipients.
     */
    public String traceId() {
        return traceId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(addresses());
        hashCode = 31 * hashCode + Objects.hashCode(context());
        hashCode = 31 * hashCode + Objects.hashCode(endpoints());
        hashCode = 31 * hashCode + Objects.hashCode(messageConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(traceId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MessageRequest)) {
            return false;
        }
        MessageRequest other = (MessageRequest) obj;
        return Objects.equals(addresses(), other.addresses()) && Objects.equals(context(), other.context())
                && Objects.equals(endpoints(), other.endpoints())
                && Objects.equals(messageConfiguration(), other.messageConfiguration())
                && Objects.equals(traceId(), other.traceId());
    }

    @Override
    public String toString() {
        return ToString.builder("MessageRequest").add("Addresses", addresses()).add("Context", context())
                .add("Endpoints", endpoints()).add("MessageConfiguration", messageConfiguration()).add("TraceId", traceId())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Addresses":
            return Optional.ofNullable(clazz.cast(addresses()));
        case "Context":
            return Optional.ofNullable(clazz.cast(context()));
        case "Endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        case "MessageConfiguration":
            return Optional.ofNullable(clazz.cast(messageConfiguration()));
        case "TraceId":
            return Optional.ofNullable(clazz.cast(traceId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MessageRequest, T> g) {
        return obj -> g.apply((MessageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MessageRequest> {
        /**
         * A map of key-value pairs, where each key is an address and each value is an AddressConfiguration object. An
         * address can be a push notification token, a phone number, or an email address.
         * 
         * @param addresses
         *        A map of key-value pairs, where each key is an address and each value is an AddressConfiguration
         *        object. An address can be a push notification token, a phone number, or an email address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addresses(Map<String, AddressConfiguration> addresses);

        /**
         * A map of custom attributes to attributes to be attached to the message. This payload is added to the push
         * notification's 'data.pinpoint' object or added to the email/sms delivery receipt event attributes.
         * 
         * @param context
         *        A map of custom attributes to attributes to be attached to the message. This payload is added to the
         *        push notification's 'data.pinpoint' object or added to the email/sms delivery receipt event
         *        attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder context(Map<String, String> context);

        /**
         * A map of key-value pairs, where each key is an endpoint ID and each value is an EndpointSendConfiguration
         * object. Within an EndpointSendConfiguration object, you can tailor the message for an endpoint by specifying
         * message overrides or substitutions.
         * 
         * @param endpoints
         *        A map of key-value pairs, where each key is an endpoint ID and each value is an
         *        EndpointSendConfiguration object. Within an EndpointSendConfiguration object, you can tailor the
         *        message for an endpoint by specifying message overrides or substitutions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(Map<String, EndpointSendConfiguration> endpoints);

        /**
         * Message configuration.
         * 
         * @param messageConfiguration
         *        Message configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageConfiguration(DirectMessageConfiguration messageConfiguration);

        /**
         * Message configuration. This is a convenience that creates an instance of the
         * {@link DirectMessageConfiguration.Builder} avoiding the need to create one manually via
         * {@link DirectMessageConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DirectMessageConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #messageConfiguration(DirectMessageConfiguration)}.
         * 
         * @param messageConfiguration
         *        a consumer that will call methods on {@link DirectMessageConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messageConfiguration(DirectMessageConfiguration)
         */
        default Builder messageConfiguration(Consumer<DirectMessageConfiguration.Builder> messageConfiguration) {
            return messageConfiguration(DirectMessageConfiguration.builder().applyMutation(messageConfiguration).build());
        }

        /**
         * A unique ID that you can use to trace a message. This ID is visible to recipients.
         * 
         * @param traceId
         *        A unique ID that you can use to trace a message. This ID is visible to recipients.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder traceId(String traceId);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, AddressConfiguration> addresses = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> context = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, EndpointSendConfiguration> endpoints = DefaultSdkAutoConstructMap.getInstance();

        private DirectMessageConfiguration messageConfiguration;

        private String traceId;

        private BuilderImpl() {
        }

        private BuilderImpl(MessageRequest model) {
            addresses(model.addresses);
            context(model.context);
            endpoints(model.endpoints);
            messageConfiguration(model.messageConfiguration);
            traceId(model.traceId);
        }

        public final Map<String, AddressConfiguration.Builder> getAddresses() {
            return addresses != null ? CollectionUtils.mapValues(addresses, AddressConfiguration::toBuilder) : null;
        }

        @Override
        public final Builder addresses(Map<String, AddressConfiguration> addresses) {
            this.addresses = MapOfAddressConfigurationCopier.copy(addresses);
            return this;
        }

        public final void setAddresses(Map<String, AddressConfiguration.BuilderImpl> addresses) {
            this.addresses = MapOfAddressConfigurationCopier.copyFromBuilder(addresses);
        }

        public final Map<String, String> getContext() {
            return context;
        }

        @Override
        public final Builder context(Map<String, String> context) {
            this.context = MapOf__stringCopier.copy(context);
            return this;
        }

        public final void setContext(Map<String, String> context) {
            this.context = MapOf__stringCopier.copy(context);
        }

        public final Map<String, EndpointSendConfiguration.Builder> getEndpoints() {
            return endpoints != null ? CollectionUtils.mapValues(endpoints, EndpointSendConfiguration::toBuilder) : null;
        }

        @Override
        public final Builder endpoints(Map<String, EndpointSendConfiguration> endpoints) {
            this.endpoints = MapOfEndpointSendConfigurationCopier.copy(endpoints);
            return this;
        }

        public final void setEndpoints(Map<String, EndpointSendConfiguration.BuilderImpl> endpoints) {
            this.endpoints = MapOfEndpointSendConfigurationCopier.copyFromBuilder(endpoints);
        }

        public final DirectMessageConfiguration.Builder getMessageConfiguration() {
            return messageConfiguration != null ? messageConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder messageConfiguration(DirectMessageConfiguration messageConfiguration) {
            this.messageConfiguration = messageConfiguration;
            return this;
        }

        public final void setMessageConfiguration(DirectMessageConfiguration.BuilderImpl messageConfiguration) {
            this.messageConfiguration = messageConfiguration != null ? messageConfiguration.build() : null;
        }

        public final String getTraceId() {
            return traceId;
        }

        @Override
        public final Builder traceId(String traceId) {
            this.traceId = traceId;
            return this;
        }

        public final void setTraceId(String traceId) {
            this.traceId = traceId;
        }

        @Override
        public MessageRequest build() {
            return new MessageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
