/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the settings for a one-time message that's sent directly to an endpoint through the GCM channel. The GCM
 * channel enables Amazon Pinpoint to send messages to the Firebase Cloud Messaging (FCM), formerly Google Cloud
 * Messaging (GCM), service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GCMMessage implements SdkPojo, Serializable, ToCopyableBuilder<GCMMessage.Builder, GCMMessage> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<String> BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::body)).setter(setter(Builder::body))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Body").build()).build();

    private static final SdkField<String> COLLAPSE_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::collapseKey)).setter(setter(Builder::collapseKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CollapseKey").build()).build();

    private static final SdkField<Map<String, String>> DATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(GCMMessage::data))
            .setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Data").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ICON_REFERENCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::iconReference)).setter(setter(Builder::iconReference))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IconReference").build()).build();

    private static final SdkField<String> IMAGE_ICON_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::imageIconUrl)).setter(setter(Builder::imageIconUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageIconUrl").build()).build();

    private static final SdkField<String> IMAGE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::imageUrl)).setter(setter(Builder::imageUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUrl").build()).build();

    private static final SdkField<String> PRIORITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<String> RAW_CONTENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::rawContent)).setter(setter(Builder::rawContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RawContent").build()).build();

    private static final SdkField<String> RESTRICTED_PACKAGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::restrictedPackageName)).setter(setter(Builder::restrictedPackageName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestrictedPackageName").build())
            .build();

    private static final SdkField<Boolean> SILENT_PUSH_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(GCMMessage::silentPush)).setter(setter(Builder::silentPush))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SilentPush").build()).build();

    private static final SdkField<String> SMALL_IMAGE_ICON_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::smallImageIconUrl)).setter(setter(Builder::smallImageIconUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SmallImageIconUrl").build()).build();

    private static final SdkField<String> SOUND_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::sound)).setter(setter(Builder::sound))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sound").build()).build();

    private static final SdkField<Map<String, List<String>>> SUBSTITUTIONS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .getter(getter(GCMMessage::substitutions))
            .setter(setter(Builder::substitutions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Substitutions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Integer> TIME_TO_LIVE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(GCMMessage::timeToLive)).setter(setter(Builder::timeToLive))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeToLive").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GCMMessage::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Url").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_FIELD, BODY_FIELD,
            COLLAPSE_KEY_FIELD, DATA_FIELD, ICON_REFERENCE_FIELD, IMAGE_ICON_URL_FIELD, IMAGE_URL_FIELD, PRIORITY_FIELD,
            RAW_CONTENT_FIELD, RESTRICTED_PACKAGE_NAME_FIELD, SILENT_PUSH_FIELD, SMALL_IMAGE_ICON_URL_FIELD, SOUND_FIELD,
            SUBSTITUTIONS_FIELD, TIME_TO_LIVE_FIELD, TITLE_FIELD, URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String action;

    private final String body;

    private final String collapseKey;

    private final Map<String, String> data;

    private final String iconReference;

    private final String imageIconUrl;

    private final String imageUrl;

    private final String priority;

    private final String rawContent;

    private final String restrictedPackageName;

    private final Boolean silentPush;

    private final String smallImageIconUrl;

    private final String sound;

    private final Map<String, List<String>> substitutions;

    private final Integer timeToLive;

    private final String title;

    private final String url;

    private GCMMessage(BuilderImpl builder) {
        this.action = builder.action;
        this.body = builder.body;
        this.collapseKey = builder.collapseKey;
        this.data = builder.data;
        this.iconReference = builder.iconReference;
        this.imageIconUrl = builder.imageIconUrl;
        this.imageUrl = builder.imageUrl;
        this.priority = builder.priority;
        this.rawContent = builder.rawContent;
        this.restrictedPackageName = builder.restrictedPackageName;
        this.silentPush = builder.silentPush;
        this.smallImageIconUrl = builder.smallImageIconUrl;
        this.sound = builder.sound;
        this.substitutions = builder.substitutions;
        this.timeToLive = builder.timeToLive;
        this.title = builder.title;
        this.url = builder.url;
    }

    /**
     * <p>
     * The action to occur if the recipient taps the push notification. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the default
     * action.
     * </p>
     * </li>
     * <li>
     * <p>
     * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the deep-linking
     * features of the Android platform.
     * </p>
     * </li>
     * <li>
     * <p>
     * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     * specify.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link Action#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to occur if the recipient taps the push notification. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
     *         default action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
     *         deep-linking features of the Android platform.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     *         specify.
     *         </p>
     *         </li>
     * @see Action
     */
    public Action action() {
        return Action.fromValue(action);
    }

    /**
     * <p>
     * The action to occur if the recipient taps the push notification. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the default
     * action.
     * </p>
     * </li>
     * <li>
     * <p>
     * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the deep-linking
     * features of the Android platform.
     * </p>
     * </li>
     * <li>
     * <p>
     * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     * specify.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link Action#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to occur if the recipient taps the push notification. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
     *         default action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
     *         deep-linking features of the Android platform.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     *         specify.
     *         </p>
     *         </li>
     * @see Action
     */
    public String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The body of the notification message.
     * </p>
     * 
     * @return The body of the notification message.
     */
    public String body() {
        return body;
    }

    /**
     * <p>
     * An arbitrary string that identifies a group of messages that can be collapsed to ensure that only the last
     * message is sent when delivery can resume. This helps avoid sending too many instances of the same messages when
     * the recipient's device comes online again or becomes active.
     * </p>
     * <p>
     * Amazon Pinpoint specifies this value in the Firebase Cloud Messaging (FCM) collapse_key parameter when it sends
     * the notification message to FCM.
     * </p>
     * 
     * @return An arbitrary string that identifies a group of messages that can be collapsed to ensure that only the
     *         last message is sent when delivery can resume. This helps avoid sending too many instances of the same
     *         messages when the recipient's device comes online again or becomes active.</p>
     *         <p>
     *         Amazon Pinpoint specifies this value in the Firebase Cloud Messaging (FCM) collapse_key parameter when it
     *         sends the notification message to FCM.
     */
    public String collapseKey() {
        return collapseKey;
    }

    /**
     * <p>
     * The JSON data payload to use for the push notification, if the notification is a silent push notification. This
     * payload is added to the data.pinpoint.jsonBody object of the notification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The JSON data payload to use for the push notification, if the notification is a silent push
     *         notification. This payload is added to the data.pinpoint.jsonBody object of the notification.
     */
    public Map<String, String> data() {
        return data;
    }

    /**
     * <p>
     * The icon image name of the asset saved in your app.
     * </p>
     * 
     * @return The icon image name of the asset saved in your app.
     */
    public String iconReference() {
        return iconReference;
    }

    /**
     * <p>
     * The URL of the large icon image to display in the content view of the push notification.
     * </p>
     * 
     * @return The URL of the large icon image to display in the content view of the push notification.
     */
    public String imageIconUrl() {
        return imageIconUrl;
    }

    /**
     * <p>
     * The URL of an image to display in the push notification.
     * </p>
     * 
     * @return The URL of an image to display in the push notification.
     */
    public String imageUrl() {
        return imageUrl;
    }

    /**
     * <p>
     * para>normal - The notification might be delayed. Delivery is optimized for battery usage on the recipient's
     * device. Use this value unless immediate delivery is required.
     * </p>
     * /listitem> <li>
     * <p>
     * high - The notification is sent immediately and might wake a sleeping device.
     * </p>
     * </li>/para>
     * <p>
     * Amazon Pinpoint specifies this value in the FCM priority parameter when it sends the notification message to FCM.
     * </p>
     * <p>
     * The equivalent values for Apple Push Notification service (APNs) are 5, for normal, and 10, for high. If you
     * specify an APNs value for this property, Amazon Pinpoint accepts and converts the value to the corresponding FCM
     * value.
     * </p>
     * 
     * @return para>normal - The notification might be delayed. Delivery is optimized for battery usage on the
     *         recipient's device. Use this value unless immediate delivery is required.</p>/listitem> <li>
     *         <p>
     *         high - The notification is sent immediately and might wake a sleeping device.
     *         </p>
     *         </li>/para>
     *         <p>
     *         Amazon Pinpoint specifies this value in the FCM priority parameter when it sends the notification message
     *         to FCM.
     *         </p>
     *         <p>
     *         The equivalent values for Apple Push Notification service (APNs) are 5, for normal, and 10, for high. If
     *         you specify an APNs value for this property, Amazon Pinpoint accepts and converts the value to the
     *         corresponding FCM value.
     */
    public String priority() {
        return priority;
    }

    /**
     * <p>
     * The raw, JSON-formatted string to use as the payload for the notification message. If specified, this value
     * overrides all other content for the message.
     * </p>
     * 
     * @return The raw, JSON-formatted string to use as the payload for the notification message. If specified, this
     *         value overrides all other content for the message.
     */
    public String rawContent() {
        return rawContent;
    }

    /**
     * <p>
     * The package name of the application where registration tokens must match in order for the recipient to receive
     * the message.
     * </p>
     * 
     * @return The package name of the application where registration tokens must match in order for the recipient to
     *         receive the message.
     */
    public String restrictedPackageName() {
        return restrictedPackageName;
    }

    /**
     * <p>
     * Specifies whether the notification is a silent push notification, which is a push notification that doesn't
     * display on a recipient's device. Silent push notifications can be used for cases such as updating an app's
     * configuration or supporting phone home functionality.
     * </p>
     * 
     * @return Specifies whether the notification is a silent push notification, which is a push notification that
     *         doesn't display on a recipient's device. Silent push notifications can be used for cases such as updating
     *         an app's configuration or supporting phone home functionality.
     */
    public Boolean silentPush() {
        return silentPush;
    }

    /**
     * <p>
     * The URL of the small icon image to display in the status bar and the content view of the push notification.
     * </p>
     * 
     * @return The URL of the small icon image to display in the status bar and the content view of the push
     *         notification.
     */
    public String smallImageIconUrl() {
        return smallImageIconUrl;
    }

    /**
     * <p>
     * The sound to play when the recipient receives the push notification. You can use the default stream or specify
     * the file name of a sound resource that's bundled in your app. On an Android platform, the sound file must reside
     * in /res/raw/.
     * </p>
     * 
     * @return The sound to play when the recipient receives the push notification. You can use the default stream or
     *         specify the file name of a sound resource that's bundled in your app. On an Android platform, the sound
     *         file must reside in /res/raw/.
     */
    public String sound() {
        return sound;
    }

    /**
     * <p>
     * The default message variables to use in the notification message. You can override the default variables with
     * individual address variables.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The default message variables to use in the notification message. You can override the default variables
     *         with individual address variables.
     */
    public Map<String, List<String>> substitutions() {
        return substitutions;
    }

    /**
     * <p>
     * The amount of time, in seconds, that FCM should store and attempt to deliver the push notification, if the
     * service is unable to deliver the notification the first time. If you don't specify this value, FCM defaults to
     * the maximum value, which is 2,419,200 seconds (28 days).
     * </p>
     * <p>
     * Amazon Pinpoint specifies this value in the FCM time_to_live parameter when it sends the notification message to
     * FCM.
     * </p>
     * 
     * @return The amount of time, in seconds, that FCM should store and attempt to deliver the push notification, if
     *         the service is unable to deliver the notification the first time. If you don't specify this value, FCM
     *         defaults to the maximum value, which is 2,419,200 seconds (28 days).</p>
     *         <p>
     *         Amazon Pinpoint specifies this value in the FCM time_to_live parameter when it sends the notification
     *         message to FCM.
     */
    public Integer timeToLive() {
        return timeToLive;
    }

    /**
     * <p>
     * The title to display above the notification message on the recipient's device.
     * </p>
     * 
     * @return The title to display above the notification message on the recipient's device.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and the
     * value of the Action property is URL.
     * </p>
     * 
     * @return The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and
     *         the value of the Action property is URL.
     */
    public String url() {
        return url;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(body());
        hashCode = 31 * hashCode + Objects.hashCode(collapseKey());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(iconReference());
        hashCode = 31 * hashCode + Objects.hashCode(imageIconUrl());
        hashCode = 31 * hashCode + Objects.hashCode(imageUrl());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(rawContent());
        hashCode = 31 * hashCode + Objects.hashCode(restrictedPackageName());
        hashCode = 31 * hashCode + Objects.hashCode(silentPush());
        hashCode = 31 * hashCode + Objects.hashCode(smallImageIconUrl());
        hashCode = 31 * hashCode + Objects.hashCode(sound());
        hashCode = 31 * hashCode + Objects.hashCode(substitutions());
        hashCode = 31 * hashCode + Objects.hashCode(timeToLive());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(url());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GCMMessage)) {
            return false;
        }
        GCMMessage other = (GCMMessage) obj;
        return Objects.equals(actionAsString(), other.actionAsString()) && Objects.equals(body(), other.body())
                && Objects.equals(collapseKey(), other.collapseKey()) && Objects.equals(data(), other.data())
                && Objects.equals(iconReference(), other.iconReference()) && Objects.equals(imageIconUrl(), other.imageIconUrl())
                && Objects.equals(imageUrl(), other.imageUrl()) && Objects.equals(priority(), other.priority())
                && Objects.equals(rawContent(), other.rawContent())
                && Objects.equals(restrictedPackageName(), other.restrictedPackageName())
                && Objects.equals(silentPush(), other.silentPush())
                && Objects.equals(smallImageIconUrl(), other.smallImageIconUrl()) && Objects.equals(sound(), other.sound())
                && Objects.equals(substitutions(), other.substitutions()) && Objects.equals(timeToLive(), other.timeToLive())
                && Objects.equals(title(), other.title()) && Objects.equals(url(), other.url());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GCMMessage").add("Action", actionAsString()).add("Body", body())
                .add("CollapseKey", collapseKey()).add("Data", data()).add("IconReference", iconReference())
                .add("ImageIconUrl", imageIconUrl()).add("ImageUrl", imageUrl()).add("Priority", priority())
                .add("RawContent", rawContent()).add("RestrictedPackageName", restrictedPackageName())
                .add("SilentPush", silentPush()).add("SmallImageIconUrl", smallImageIconUrl()).add("Sound", sound())
                .add("Substitutions", substitutions()).add("TimeToLive", timeToLive()).add("Title", title()).add("Url", url())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Body":
            return Optional.ofNullable(clazz.cast(body()));
        case "CollapseKey":
            return Optional.ofNullable(clazz.cast(collapseKey()));
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        case "IconReference":
            return Optional.ofNullable(clazz.cast(iconReference()));
        case "ImageIconUrl":
            return Optional.ofNullable(clazz.cast(imageIconUrl()));
        case "ImageUrl":
            return Optional.ofNullable(clazz.cast(imageUrl()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RawContent":
            return Optional.ofNullable(clazz.cast(rawContent()));
        case "RestrictedPackageName":
            return Optional.ofNullable(clazz.cast(restrictedPackageName()));
        case "SilentPush":
            return Optional.ofNullable(clazz.cast(silentPush()));
        case "SmallImageIconUrl":
            return Optional.ofNullable(clazz.cast(smallImageIconUrl()));
        case "Sound":
            return Optional.ofNullable(clazz.cast(sound()));
        case "Substitutions":
            return Optional.ofNullable(clazz.cast(substitutions()));
        case "TimeToLive":
            return Optional.ofNullable(clazz.cast(timeToLive()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GCMMessage, T> g) {
        return obj -> g.apply((GCMMessage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GCMMessage> {
        /**
         * <p>
         * The action to occur if the recipient taps the push notification. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
         * default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         * deep-linking features of the Android platform.
         * </p>
         * </li>
         * <li>
         * <p>
         * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
         * specify.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to occur if the recipient taps the push notification. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is
         *        the default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         *        deep-linking features of the Android platform.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that
         *        you specify.
         *        </p>
         *        </li>
         * @see Action
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Action
         */
        Builder action(String action);

        /**
         * <p>
         * The action to occur if the recipient taps the push notification. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
         * default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         * deep-linking features of the Android platform.
         * </p>
         * </li>
         * <li>
         * <p>
         * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
         * specify.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to occur if the recipient taps the push notification. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is
         *        the default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DEEP_LINK - Your app opens and displays a designated user interface in the app. This action uses the
         *        deep-linking features of the Android platform.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that
         *        you specify.
         *        </p>
         *        </li>
         * @see Action
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Action
         */
        Builder action(Action action);

        /**
         * <p>
         * The body of the notification message.
         * </p>
         * 
         * @param body
         *        The body of the notification message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(String body);

        /**
         * <p>
         * An arbitrary string that identifies a group of messages that can be collapsed to ensure that only the last
         * message is sent when delivery can resume. This helps avoid sending too many instances of the same messages
         * when the recipient's device comes online again or becomes active.
         * </p>
         * <p>
         * Amazon Pinpoint specifies this value in the Firebase Cloud Messaging (FCM) collapse_key parameter when it
         * sends the notification message to FCM.
         * </p>
         * 
         * @param collapseKey
         *        An arbitrary string that identifies a group of messages that can be collapsed to ensure that only the
         *        last message is sent when delivery can resume. This helps avoid sending too many instances of the same
         *        messages when the recipient's device comes online again or becomes active.</p>
         *        <p>
         *        Amazon Pinpoint specifies this value in the Firebase Cloud Messaging (FCM) collapse_key parameter when
         *        it sends the notification message to FCM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collapseKey(String collapseKey);

        /**
         * <p>
         * The JSON data payload to use for the push notification, if the notification is a silent push notification.
         * This payload is added to the data.pinpoint.jsonBody object of the notification.
         * </p>
         * 
         * @param data
         *        The JSON data payload to use for the push notification, if the notification is a silent push
         *        notification. This payload is added to the data.pinpoint.jsonBody object of the notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(Map<String, String> data);

        /**
         * <p>
         * The icon image name of the asset saved in your app.
         * </p>
         * 
         * @param iconReference
         *        The icon image name of the asset saved in your app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iconReference(String iconReference);

        /**
         * <p>
         * The URL of the large icon image to display in the content view of the push notification.
         * </p>
         * 
         * @param imageIconUrl
         *        The URL of the large icon image to display in the content view of the push notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageIconUrl(String imageIconUrl);

        /**
         * <p>
         * The URL of an image to display in the push notification.
         * </p>
         * 
         * @param imageUrl
         *        The URL of an image to display in the push notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUrl(String imageUrl);

        /**
         * <p>
         * para>normal - The notification might be delayed. Delivery is optimized for battery usage on the recipient's
         * device. Use this value unless immediate delivery is required.
         * </p>
         * /listitem> <li>
         * <p>
         * high - The notification is sent immediately and might wake a sleeping device.
         * </p>
         * </li>/para>
         * <p>
         * Amazon Pinpoint specifies this value in the FCM priority parameter when it sends the notification message to
         * FCM.
         * </p>
         * <p>
         * The equivalent values for Apple Push Notification service (APNs) are 5, for normal, and 10, for high. If you
         * specify an APNs value for this property, Amazon Pinpoint accepts and converts the value to the corresponding
         * FCM value.
         * </p>
         * 
         * @param priority
         *        para>normal - The notification might be delayed. Delivery is optimized for battery usage on the
         *        recipient's device. Use this value unless immediate delivery is required.</p>/listitem> <li>
         *        <p>
         *        high - The notification is sent immediately and might wake a sleeping device.
         *        </p>
         *        </li>/para>
         *        <p>
         *        Amazon Pinpoint specifies this value in the FCM priority parameter when it sends the notification
         *        message to FCM.
         *        </p>
         *        <p>
         *        The equivalent values for Apple Push Notification service (APNs) are 5, for normal, and 10, for high.
         *        If you specify an APNs value for this property, Amazon Pinpoint accepts and converts the value to the
         *        corresponding FCM value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(String priority);

        /**
         * <p>
         * The raw, JSON-formatted string to use as the payload for the notification message. If specified, this value
         * overrides all other content for the message.
         * </p>
         * 
         * @param rawContent
         *        The raw, JSON-formatted string to use as the payload for the notification message. If specified, this
         *        value overrides all other content for the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rawContent(String rawContent);

        /**
         * <p>
         * The package name of the application where registration tokens must match in order for the recipient to
         * receive the message.
         * </p>
         * 
         * @param restrictedPackageName
         *        The package name of the application where registration tokens must match in order for the recipient to
         *        receive the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restrictedPackageName(String restrictedPackageName);

        /**
         * <p>
         * Specifies whether the notification is a silent push notification, which is a push notification that doesn't
         * display on a recipient's device. Silent push notifications can be used for cases such as updating an app's
         * configuration or supporting phone home functionality.
         * </p>
         * 
         * @param silentPush
         *        Specifies whether the notification is a silent push notification, which is a push notification that
         *        doesn't display on a recipient's device. Silent push notifications can be used for cases such as
         *        updating an app's configuration or supporting phone home functionality.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder silentPush(Boolean silentPush);

        /**
         * <p>
         * The URL of the small icon image to display in the status bar and the content view of the push notification.
         * </p>
         * 
         * @param smallImageIconUrl
         *        The URL of the small icon image to display in the status bar and the content view of the push
         *        notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smallImageIconUrl(String smallImageIconUrl);

        /**
         * <p>
         * The sound to play when the recipient receives the push notification. You can use the default stream or
         * specify the file name of a sound resource that's bundled in your app. On an Android platform, the sound file
         * must reside in /res/raw/.
         * </p>
         * 
         * @param sound
         *        The sound to play when the recipient receives the push notification. You can use the default stream or
         *        specify the file name of a sound resource that's bundled in your app. On an Android platform, the
         *        sound file must reside in /res/raw/.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sound(String sound);

        /**
         * <p>
         * The default message variables to use in the notification message. You can override the default variables with
         * individual address variables.
         * </p>
         * 
         * @param substitutions
         *        The default message variables to use in the notification message. You can override the default
         *        variables with individual address variables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder substitutions(Map<String, ? extends Collection<String>> substitutions);

        /**
         * <p>
         * The amount of time, in seconds, that FCM should store and attempt to deliver the push notification, if the
         * service is unable to deliver the notification the first time. If you don't specify this value, FCM defaults
         * to the maximum value, which is 2,419,200 seconds (28 days).
         * </p>
         * <p>
         * Amazon Pinpoint specifies this value in the FCM time_to_live parameter when it sends the notification message
         * to FCM.
         * </p>
         * 
         * @param timeToLive
         *        The amount of time, in seconds, that FCM should store and attempt to deliver the push notification, if
         *        the service is unable to deliver the notification the first time. If you don't specify this value, FCM
         *        defaults to the maximum value, which is 2,419,200 seconds (28 days).</p>
         *        <p>
         *        Amazon Pinpoint specifies this value in the FCM time_to_live parameter when it sends the notification
         *        message to FCM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeToLive(Integer timeToLive);

        /**
         * <p>
         * The title to display above the notification message on the recipient's device.
         * </p>
         * 
         * @param title
         *        The title to display above the notification message on the recipient's device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and the
         * value of the Action property is URL.
         * </p>
         * 
         * @param url
         *        The URL to open in the recipient's default mobile browser, if a recipient taps the push notification
         *        and the value of the Action property is URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private String body;

        private String collapseKey;

        private Map<String, String> data = DefaultSdkAutoConstructMap.getInstance();

        private String iconReference;

        private String imageIconUrl;

        private String imageUrl;

        private String priority;

        private String rawContent;

        private String restrictedPackageName;

        private Boolean silentPush;

        private String smallImageIconUrl;

        private String sound;

        private Map<String, List<String>> substitutions = DefaultSdkAutoConstructMap.getInstance();

        private Integer timeToLive;

        private String title;

        private String url;

        private BuilderImpl() {
        }

        private BuilderImpl(GCMMessage model) {
            action(model.action);
            body(model.body);
            collapseKey(model.collapseKey);
            data(model.data);
            iconReference(model.iconReference);
            imageIconUrl(model.imageIconUrl);
            imageUrl(model.imageUrl);
            priority(model.priority);
            rawContent(model.rawContent);
            restrictedPackageName(model.restrictedPackageName);
            silentPush(model.silentPush);
            smallImageIconUrl(model.smallImageIconUrl);
            sound(model.sound);
            substitutions(model.substitutions);
            timeToLive(model.timeToLive);
            title(model.title);
            url(model.url);
        }

        public final String getActionAsString() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(Action action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final String getBody() {
            return body;
        }

        @Override
        public final Builder body(String body) {
            this.body = body;
            return this;
        }

        public final void setBody(String body) {
            this.body = body;
        }

        public final String getCollapseKey() {
            return collapseKey;
        }

        @Override
        public final Builder collapseKey(String collapseKey) {
            this.collapseKey = collapseKey;
            return this;
        }

        public final void setCollapseKey(String collapseKey) {
            this.collapseKey = collapseKey;
        }

        public final Map<String, String> getData() {
            return data;
        }

        @Override
        public final Builder data(Map<String, String> data) {
            this.data = MapOf__stringCopier.copy(data);
            return this;
        }

        public final void setData(Map<String, String> data) {
            this.data = MapOf__stringCopier.copy(data);
        }

        public final String getIconReference() {
            return iconReference;
        }

        @Override
        public final Builder iconReference(String iconReference) {
            this.iconReference = iconReference;
            return this;
        }

        public final void setIconReference(String iconReference) {
            this.iconReference = iconReference;
        }

        public final String getImageIconUrl() {
            return imageIconUrl;
        }

        @Override
        public final Builder imageIconUrl(String imageIconUrl) {
            this.imageIconUrl = imageIconUrl;
            return this;
        }

        public final void setImageIconUrl(String imageIconUrl) {
            this.imageIconUrl = imageIconUrl;
        }

        public final String getImageUrl() {
            return imageUrl;
        }

        @Override
        public final Builder imageUrl(String imageUrl) {
            this.imageUrl = imageUrl;
            return this;
        }

        public final void setImageUrl(String imageUrl) {
            this.imageUrl = imageUrl;
        }

        public final String getPriority() {
            return priority;
        }

        @Override
        public final Builder priority(String priority) {
            this.priority = priority;
            return this;
        }

        public final void setPriority(String priority) {
            this.priority = priority;
        }

        public final String getRawContent() {
            return rawContent;
        }

        @Override
        public final Builder rawContent(String rawContent) {
            this.rawContent = rawContent;
            return this;
        }

        public final void setRawContent(String rawContent) {
            this.rawContent = rawContent;
        }

        public final String getRestrictedPackageName() {
            return restrictedPackageName;
        }

        @Override
        public final Builder restrictedPackageName(String restrictedPackageName) {
            this.restrictedPackageName = restrictedPackageName;
            return this;
        }

        public final void setRestrictedPackageName(String restrictedPackageName) {
            this.restrictedPackageName = restrictedPackageName;
        }

        public final Boolean getSilentPush() {
            return silentPush;
        }

        @Override
        public final Builder silentPush(Boolean silentPush) {
            this.silentPush = silentPush;
            return this;
        }

        public final void setSilentPush(Boolean silentPush) {
            this.silentPush = silentPush;
        }

        public final String getSmallImageIconUrl() {
            return smallImageIconUrl;
        }

        @Override
        public final Builder smallImageIconUrl(String smallImageIconUrl) {
            this.smallImageIconUrl = smallImageIconUrl;
            return this;
        }

        public final void setSmallImageIconUrl(String smallImageIconUrl) {
            this.smallImageIconUrl = smallImageIconUrl;
        }

        public final String getSound() {
            return sound;
        }

        @Override
        public final Builder sound(String sound) {
            this.sound = sound;
            return this;
        }

        public final void setSound(String sound) {
            this.sound = sound;
        }

        public final Map<String, ? extends Collection<String>> getSubstitutions() {
            return substitutions;
        }

        @Override
        public final Builder substitutions(Map<String, ? extends Collection<String>> substitutions) {
            this.substitutions = MapOfListOf__stringCopier.copy(substitutions);
            return this;
        }

        public final void setSubstitutions(Map<String, ? extends Collection<String>> substitutions) {
            this.substitutions = MapOfListOf__stringCopier.copy(substitutions);
        }

        public final Integer getTimeToLive() {
            return timeToLive;
        }

        @Override
        public final Builder timeToLive(Integer timeToLive) {
            this.timeToLive = timeToLive;
            return this;
        }

        public final void setTimeToLive(Integer timeToLive) {
            this.timeToLive = timeToLive;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getUrl() {
            return url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public GCMMessage build() {
            return new GCMMessage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
