/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the settings for a one-time message that's sent directly to an endpoint through the APNs (Apple Push
 * Notification service) channel.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class APNSMessage implements SdkPojo, Serializable, ToCopyableBuilder<APNSMessage.Builder, APNSMessage> {
    private static final SdkField<String> APNS_PUSH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("APNSPushType").getter(getter(APNSMessage::apnsPushType)).setter(setter(Builder::apnsPushType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("APNSPushType").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(APNSMessage::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<Integer> BADGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Badge")
            .getter(getter(APNSMessage::badge)).setter(setter(Builder::badge))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Badge").build()).build();

    private static final SdkField<String> BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Body")
            .getter(getter(APNSMessage::body)).setter(setter(Builder::body))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Body").build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Category").getter(getter(APNSMessage::category)).setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Category").build()).build();

    private static final SdkField<String> COLLAPSE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CollapseId").getter(getter(APNSMessage::collapseId)).setter(setter(Builder::collapseId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CollapseId").build()).build();

    private static final SdkField<Map<String, String>> DATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Data")
            .getter(getter(APNSMessage::data))
            .setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Data").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> MEDIA_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaUrl").getter(getter(APNSMessage::mediaUrl)).setter(setter(Builder::mediaUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaUrl").build()).build();

    private static final SdkField<String> PREFERRED_AUTHENTICATION_METHOD_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PreferredAuthenticationMethod")
            .getter(getter(APNSMessage::preferredAuthenticationMethod))
            .setter(setter(Builder::preferredAuthenticationMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredAuthenticationMethod")
                    .build()).build();

    private static final SdkField<String> PRIORITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Priority").getter(getter(APNSMessage::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<String> RAW_CONTENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RawContent").getter(getter(APNSMessage::rawContent)).setter(setter(Builder::rawContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RawContent").build()).build();

    private static final SdkField<Boolean> SILENT_PUSH_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SilentPush").getter(getter(APNSMessage::silentPush)).setter(setter(Builder::silentPush))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SilentPush").build()).build();

    private static final SdkField<String> SOUND_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Sound")
            .getter(getter(APNSMessage::sound)).setter(setter(Builder::sound))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sound").build()).build();

    private static final SdkField<Map<String, List<String>>> SUBSTITUTIONS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Substitutions")
            .getter(getter(APNSMessage::substitutions))
            .setter(setter(Builder::substitutions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Substitutions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> THREAD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ThreadId").getter(getter(APNSMessage::threadId)).setter(setter(Builder::threadId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThreadId").build()).build();

    private static final SdkField<Integer> TIME_TO_LIVE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TimeToLive").getter(getter(APNSMessage::timeToLive)).setter(setter(Builder::timeToLive))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeToLive").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(APNSMessage::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Url")
            .getter(getter(APNSMessage::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Url").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APNS_PUSH_TYPE_FIELD,
            ACTION_FIELD, BADGE_FIELD, BODY_FIELD, CATEGORY_FIELD, COLLAPSE_ID_FIELD, DATA_FIELD, MEDIA_URL_FIELD,
            PREFERRED_AUTHENTICATION_METHOD_FIELD, PRIORITY_FIELD, RAW_CONTENT_FIELD, SILENT_PUSH_FIELD, SOUND_FIELD,
            SUBSTITUTIONS_FIELD, THREAD_ID_FIELD, TIME_TO_LIVE_FIELD, TITLE_FIELD, URL_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String apnsPushType;

    private final String action;

    private final Integer badge;

    private final String body;

    private final String category;

    private final String collapseId;

    private final Map<String, String> data;

    private final String mediaUrl;

    private final String preferredAuthenticationMethod;

    private final String priority;

    private final String rawContent;

    private final Boolean silentPush;

    private final String sound;

    private final Map<String, List<String>> substitutions;

    private final String threadId;

    private final Integer timeToLive;

    private final String title;

    private final String url;

    private APNSMessage(BuilderImpl builder) {
        this.apnsPushType = builder.apnsPushType;
        this.action = builder.action;
        this.badge = builder.badge;
        this.body = builder.body;
        this.category = builder.category;
        this.collapseId = builder.collapseId;
        this.data = builder.data;
        this.mediaUrl = builder.mediaUrl;
        this.preferredAuthenticationMethod = builder.preferredAuthenticationMethod;
        this.priority = builder.priority;
        this.rawContent = builder.rawContent;
        this.silentPush = builder.silentPush;
        this.sound = builder.sound;
        this.substitutions = builder.substitutions;
        this.threadId = builder.threadId;
        this.timeToLive = builder.timeToLive;
        this.title = builder.title;
        this.url = builder.url;
    }

    /**
     * <p>
     * The type of push notification to send. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * alert - For a standard notification that's displayed on recipients' devices and prompts a recipient to interact
     * with the notification.
     * </p>
     * </li>
     * <li>
     * <p>
     * background - For a silent notification that delivers content in the background and isn't displayed on recipients'
     * devices.
     * </p>
     * </li>
     * <li>
     * <p>
     * complication - For a notification that contains update information for an app’s complication timeline.
     * </p>
     * </li>
     * <li>
     * <p>
     * fileprovider - For a notification that signals changes to a File Provider extension.
     * </p>
     * </li>
     * <li>
     * <p>
     * mdm - For a notification that tells managed devices to contact the MDM server.
     * </p>
     * </li>
     * <li>
     * <p>
     * voip - For a notification that provides information about an incoming VoIP call.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Amazon Pinpoint specifies this value in the apns-push-type request header when it sends the notification message
     * to APNs. If you don't specify a value for this property, Amazon Pinpoint sets the value to alert or background
     * automatically, based on the value that you specify for the SilentPush or RawContent property of the message.
     * </p>
     * <p>
     * For more information about the apns-push-type request header, see <a href=
     * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/sending_notification_requests_to_apns"
     * >Sending Notification Requests to APNs</a> on the Apple Developer website.
     * </p>
     * 
     * @return The type of push notification to send. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         alert - For a standard notification that's displayed on recipients' devices and prompts a recipient to
     *         interact with the notification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         background - For a silent notification that delivers content in the background and isn't displayed on
     *         recipients' devices.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         complication - For a notification that contains update information for an app’s complication timeline.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         fileprovider - For a notification that signals changes to a File Provider extension.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         mdm - For a notification that tells managed devices to contact the MDM server.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         voip - For a notification that provides information about an incoming VoIP call.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Amazon Pinpoint specifies this value in the apns-push-type request header when it sends the notification
     *         message to APNs. If you don't specify a value for this property, Amazon Pinpoint sets the value to alert
     *         or background automatically, based on the value that you specify for the SilentPush or RawContent
     *         property of the message.
     *         </p>
     *         <p>
     *         For more information about the apns-push-type request header, see <a href=
     *         "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/sending_notification_requests_to_apns"
     *         >Sending Notification Requests to APNs</a> on the Apple Developer website.
     */
    public final String apnsPushType() {
        return apnsPushType;
    }

    /**
     * <p>
     * The action to occur if the recipient taps the push notification. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the default
     * action.
     * </p>
     * </li>
     * <li>
     * <p>
     * DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
     * deep-linking features of the iOS platform.
     * </p>
     * </li>
     * <li>
     * <p>
     * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     * specify.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link Action#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to occur if the recipient taps the push notification. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
     *         default action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
     *         deep-linking features of the iOS platform.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     *         specify.
     *         </p>
     *         </li>
     * @see Action
     */
    public final Action action() {
        return Action.fromValue(action);
    }

    /**
     * <p>
     * The action to occur if the recipient taps the push notification. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the default
     * action.
     * </p>
     * </li>
     * <li>
     * <p>
     * DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
     * deep-linking features of the iOS platform.
     * </p>
     * </li>
     * <li>
     * <p>
     * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     * specify.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link Action#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action to occur if the recipient taps the push notification. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
     *         default action.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
     *         deep-linking features of the iOS platform.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
     *         specify.
     *         </p>
     *         </li>
     * @see Action
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The key that indicates whether and how to modify the badge of your app's icon when the recipient receives the
     * push notification. If this key isn't included in the dictionary, the badge doesn't change. To remove the badge,
     * set this value to 0.
     * </p>
     * 
     * @return The key that indicates whether and how to modify the badge of your app's icon when the recipient receives
     *         the push notification. If this key isn't included in the dictionary, the badge doesn't change. To remove
     *         the badge, set this value to 0.
     */
    public final Integer badge() {
        return badge;
    }

    /**
     * <p>
     * The body of the notification message.
     * </p>
     * 
     * @return The body of the notification message.
     */
    public final String body() {
        return body;
    }

    /**
     * <p>
     * The key that indicates the notification type for the push notification. This key is a value that's defined by the
     * identifier property of one of your app's registered categories.
     * </p>
     * 
     * @return The key that indicates the notification type for the push notification. This key is a value that's
     *         defined by the identifier property of one of your app's registered categories.
     */
    public final String category() {
        return category;
    }

    /**
     * <p>
     * An arbitrary identifier that, if assigned to multiple messages, APNs uses to coalesce the messages into a single
     * push notification instead of delivering each message individually. This value can't exceed 64 bytes.
     * </p>
     * <p>
     * Amazon Pinpoint specifies this value in the apns-collapse-id request header when it sends the notification
     * message to APNs.
     * </p>
     * 
     * @return An arbitrary identifier that, if assigned to multiple messages, APNs uses to coalesce the messages into a
     *         single push notification instead of delivering each message individually. This value can't exceed 64
     *         bytes.</p>
     *         <p>
     *         Amazon Pinpoint specifies this value in the apns-collapse-id request header when it sends the
     *         notification message to APNs.
     */
    public final String collapseId() {
        return collapseId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Data property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasData() {
        return data != null && !(data instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The JSON payload to use for a silent push notification. This payload is added to the data.pinpoint.jsonBody
     * object of the notification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasData} method.
     * </p>
     * 
     * @return The JSON payload to use for a silent push notification. This payload is added to the
     *         data.pinpoint.jsonBody object of the notification.
     */
    public final Map<String, String> data() {
        return data;
    }

    /**
     * <p>
     * The URL of an image or video to display in the push notification.
     * </p>
     * 
     * @return The URL of an image or video to display in the push notification.
     */
    public final String mediaUrl() {
        return mediaUrl;
    }

    /**
     * <p>
     * The authentication method that you want Amazon Pinpoint to use when authenticating with APNs, CERTIFICATE or
     * TOKEN.
     * </p>
     * 
     * @return The authentication method that you want Amazon Pinpoint to use when authenticating with APNs, CERTIFICATE
     *         or TOKEN.
     */
    public final String preferredAuthenticationMethod() {
        return preferredAuthenticationMethod;
    }

    /**
     * <p>
     * para>5 - Low priority, the notification might be delayed, delivered as part of a group, or throttled.
     * </p>
     * /listitem> <li>
     * <p>
     * 10 - High priority, the notification is sent immediately. This is the default value. A high priority notification
     * should trigger an alert, play a sound, or badge your app's icon on the recipient's device.
     * </p>
     * </li>/para>
     * <p>
     * Amazon Pinpoint specifies this value in the apns-priority request header when it sends the notification message
     * to APNs.
     * </p>
     * <p>
     * The equivalent values for Firebase Cloud Messaging (FCM), formerly Google Cloud Messaging (GCM), are normal, for
     * 5, and high, for 10. If you specify an FCM value for this property, Amazon Pinpoint accepts and converts the
     * value to the corresponding APNs value.
     * </p>
     * 
     * @return para>5 - Low priority, the notification might be delayed, delivered as part of a group, or
     *         throttled.</p>/listitem> <li>
     *         <p>
     *         10 - High priority, the notification is sent immediately. This is the default value. A high priority
     *         notification should trigger an alert, play a sound, or badge your app's icon on the recipient's device.
     *         </p>
     *         </li>/para>
     *         <p>
     *         Amazon Pinpoint specifies this value in the apns-priority request header when it sends the notification
     *         message to APNs.
     *         </p>
     *         <p>
     *         The equivalent values for Firebase Cloud Messaging (FCM), formerly Google Cloud Messaging (GCM), are
     *         normal, for 5, and high, for 10. If you specify an FCM value for this property, Amazon Pinpoint accepts
     *         and converts the value to the corresponding APNs value.
     */
    public final String priority() {
        return priority;
    }

    /**
     * <p>
     * The raw, JSON-formatted string to use as the payload for the notification message. If specified, this value
     * overrides all other content for the message.
     * </p>
     * <note>
     * <p>
     * If you specify the raw content of an APNs push notification, the message payload has to include the
     * content-available key. The value of the content-available key has to be an integer, and can only be 0 or 1. If
     * you're sending a standard notification, set the value of content-available to 0. If you're sending a silent
     * (background) notification, set the value of content-available to 1. Additionally, silent notification payloads
     * can't include the alert, badge, or sound keys. For more information, see <a href=
     * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
     * >Generating a Remote Notification</a> and <a href=
     * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
     * >Pushing Background Updates to Your App</a> on the Apple Developer website.
     * </p>
     * </note>
     * 
     * @return The raw, JSON-formatted string to use as the payload for the notification message. If specified, this
     *         value overrides all other content for the message.</p> <note>
     *         <p>
     *         If you specify the raw content of an APNs push notification, the message payload has to include the
     *         content-available key. The value of the content-available key has to be an integer, and can only be 0 or
     *         1. If you're sending a standard notification, set the value of content-available to 0. If you're sending
     *         a silent (background) notification, set the value of content-available to 1. Additionally, silent
     *         notification payloads can't include the alert, badge, or sound keys. For more information, see <a href=
     *         "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
     *         >Generating a Remote Notification</a> and <a href=
     *         "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
     *         >Pushing Background Updates to Your App</a> on the Apple Developer website.
     *         </p>
     */
    public final String rawContent() {
        return rawContent;
    }

    /**
     * <p>
     * Specifies whether the notification is a silent push notification. A silent (or background) push notification
     * isn't displayed on recipients' devices. You can use silent push notifications to make small updates to your app,
     * or to display messages in an in-app message center.
     * </p>
     * <p>
     * Amazon Pinpoint uses this property to determine the correct value for the apns-push-type request header when it
     * sends the notification message to APNs. If you specify a value of true for this property, Amazon Pinpoint sets
     * the value for the apns-push-type header field to background.
     * </p>
     * <note>
     * <p>
     * If you specify the raw content of an APNs push notification, the message payload has to include the
     * content-available key. For silent (background) notifications, set the value of content-available to 1.
     * Additionally, the message payload for a silent notification can't include the alert, badge, or sound keys. For
     * more information, see <a href=
     * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
     * >Generating a Remote Notification</a> and <a href=
     * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
     * >Pushing Background Updates to Your App</a> on the Apple Developer website.
     * </p>
     * <p>
     * Apple has indicated that they will throttle "excessive" background notifications based on current traffic
     * volumes. To prevent your notifications being throttled, Apple recommends that you send no more than 3 silent push
     * notifications to each recipient per hour.
     * </p>
     * </note>
     * 
     * @return Specifies whether the notification is a silent push notification. A silent (or background) push
     *         notification isn't displayed on recipients' devices. You can use silent push notifications to make small
     *         updates to your app, or to display messages in an in-app message center.</p>
     *         <p>
     *         Amazon Pinpoint uses this property to determine the correct value for the apns-push-type request header
     *         when it sends the notification message to APNs. If you specify a value of true for this property, Amazon
     *         Pinpoint sets the value for the apns-push-type header field to background.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify the raw content of an APNs push notification, the message payload has to include the
     *         content-available key. For silent (background) notifications, set the value of content-available to 1.
     *         Additionally, the message payload for a silent notification can't include the alert, badge, or sound
     *         keys. For more information, see <a href=
     *         "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
     *         >Generating a Remote Notification</a> and <a href=
     *         "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
     *         >Pushing Background Updates to Your App</a> on the Apple Developer website.
     *         </p>
     *         <p>
     *         Apple has indicated that they will throttle "excessive" background notifications based on current traffic
     *         volumes. To prevent your notifications being throttled, Apple recommends that you send no more than 3
     *         silent push notifications to each recipient per hour.
     *         </p>
     */
    public final Boolean silentPush() {
        return silentPush;
    }

    /**
     * <p>
     * The key for the sound to play when the recipient receives the push notification. The value for this key is the
     * name of a sound file in your app's main bundle or the Library/Sounds folder in your app's data container. If the
     * sound file can't be found or you specify default for the value, the system plays the default alert sound.
     * </p>
     * 
     * @return The key for the sound to play when the recipient receives the push notification. The value for this key
     *         is the name of a sound file in your app's main bundle or the Library/Sounds folder in your app's data
     *         container. If the sound file can't be found or you specify default for the value, the system plays the
     *         default alert sound.
     */
    public final String sound() {
        return sound;
    }

    /**
     * For responses, this returns true if the service returned a value for the Substitutions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSubstitutions() {
        return substitutions != null && !(substitutions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The default message variables to use in the notification message. You can override these default variables with
     * individual address variables.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubstitutions} method.
     * </p>
     * 
     * @return The default message variables to use in the notification message. You can override these default
     *         variables with individual address variables.
     */
    public final Map<String, List<String>> substitutions() {
        return substitutions;
    }

    /**
     * <p>
     * The key that represents your app-specific identifier for grouping notifications. If you provide a Notification
     * Content app extension, you can use this value to group your notifications together.
     * </p>
     * 
     * @return The key that represents your app-specific identifier for grouping notifications. If you provide a
     *         Notification Content app extension, you can use this value to group your notifications together.
     */
    public final String threadId() {
        return threadId;
    }

    /**
     * <p>
     * The amount of time, in seconds, that APNs should store and attempt to deliver the push notification, if the
     * service is unable to deliver the notification the first time. If this value is 0, APNs treats the notification as
     * if it expires immediately and the service doesn't store or try to deliver the notification again.
     * </p>
     * <p>
     * Amazon Pinpoint specifies this value in the apns-expiration request header when it sends the notification message
     * to APNs.
     * </p>
     * 
     * @return The amount of time, in seconds, that APNs should store and attempt to deliver the push notification, if
     *         the service is unable to deliver the notification the first time. If this value is 0, APNs treats the
     *         notification as if it expires immediately and the service doesn't store or try to deliver the
     *         notification again.</p>
     *         <p>
     *         Amazon Pinpoint specifies this value in the apns-expiration request header when it sends the notification
     *         message to APNs.
     */
    public final Integer timeToLive() {
        return timeToLive;
    }

    /**
     * <p>
     * The title to display above the notification message on the recipient's device.
     * </p>
     * 
     * @return The title to display above the notification message on the recipient's device.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and the
     * value of the Action property is URL.
     * </p>
     * 
     * @return The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and
     *         the value of the Action property is URL.
     */
    public final String url() {
        return url;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(apnsPushType());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(badge());
        hashCode = 31 * hashCode + Objects.hashCode(body());
        hashCode = 31 * hashCode + Objects.hashCode(category());
        hashCode = 31 * hashCode + Objects.hashCode(collapseId());
        hashCode = 31 * hashCode + Objects.hashCode(hasData() ? data() : null);
        hashCode = 31 * hashCode + Objects.hashCode(mediaUrl());
        hashCode = 31 * hashCode + Objects.hashCode(preferredAuthenticationMethod());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(rawContent());
        hashCode = 31 * hashCode + Objects.hashCode(silentPush());
        hashCode = 31 * hashCode + Objects.hashCode(sound());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubstitutions() ? substitutions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(threadId());
        hashCode = 31 * hashCode + Objects.hashCode(timeToLive());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(url());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof APNSMessage)) {
            return false;
        }
        APNSMessage other = (APNSMessage) obj;
        return Objects.equals(apnsPushType(), other.apnsPushType()) && Objects.equals(actionAsString(), other.actionAsString())
                && Objects.equals(badge(), other.badge()) && Objects.equals(body(), other.body())
                && Objects.equals(category(), other.category()) && Objects.equals(collapseId(), other.collapseId())
                && hasData() == other.hasData() && Objects.equals(data(), other.data())
                && Objects.equals(mediaUrl(), other.mediaUrl())
                && Objects.equals(preferredAuthenticationMethod(), other.preferredAuthenticationMethod())
                && Objects.equals(priority(), other.priority()) && Objects.equals(rawContent(), other.rawContent())
                && Objects.equals(silentPush(), other.silentPush()) && Objects.equals(sound(), other.sound())
                && hasSubstitutions() == other.hasSubstitutions() && Objects.equals(substitutions(), other.substitutions())
                && Objects.equals(threadId(), other.threadId()) && Objects.equals(timeToLive(), other.timeToLive())
                && Objects.equals(title(), other.title()) && Objects.equals(url(), other.url());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("APNSMessage").add("APNSPushType", apnsPushType()).add("Action", actionAsString())
                .add("Badge", badge()).add("Body", body()).add("Category", category()).add("CollapseId", collapseId())
                .add("Data", hasData() ? data() : null).add("MediaUrl", mediaUrl())
                .add("PreferredAuthenticationMethod", preferredAuthenticationMethod()).add("Priority", priority())
                .add("RawContent", rawContent()).add("SilentPush", silentPush()).add("Sound", sound())
                .add("Substitutions", hasSubstitutions() ? substitutions() : null).add("ThreadId", threadId())
                .add("TimeToLive", timeToLive()).add("Title", title()).add("Url", url()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "APNSPushType":
            return Optional.ofNullable(clazz.cast(apnsPushType()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Badge":
            return Optional.ofNullable(clazz.cast(badge()));
        case "Body":
            return Optional.ofNullable(clazz.cast(body()));
        case "Category":
            return Optional.ofNullable(clazz.cast(category()));
        case "CollapseId":
            return Optional.ofNullable(clazz.cast(collapseId()));
        case "Data":
            return Optional.ofNullable(clazz.cast(data()));
        case "MediaUrl":
            return Optional.ofNullable(clazz.cast(mediaUrl()));
        case "PreferredAuthenticationMethod":
            return Optional.ofNullable(clazz.cast(preferredAuthenticationMethod()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RawContent":
            return Optional.ofNullable(clazz.cast(rawContent()));
        case "SilentPush":
            return Optional.ofNullable(clazz.cast(silentPush()));
        case "Sound":
            return Optional.ofNullable(clazz.cast(sound()));
        case "Substitutions":
            return Optional.ofNullable(clazz.cast(substitutions()));
        case "ThreadId":
            return Optional.ofNullable(clazz.cast(threadId()));
        case "TimeToLive":
            return Optional.ofNullable(clazz.cast(timeToLive()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("APNSPushType", APNS_PUSH_TYPE_FIELD);
        map.put("Action", ACTION_FIELD);
        map.put("Badge", BADGE_FIELD);
        map.put("Body", BODY_FIELD);
        map.put("Category", CATEGORY_FIELD);
        map.put("CollapseId", COLLAPSE_ID_FIELD);
        map.put("Data", DATA_FIELD);
        map.put("MediaUrl", MEDIA_URL_FIELD);
        map.put("PreferredAuthenticationMethod", PREFERRED_AUTHENTICATION_METHOD_FIELD);
        map.put("Priority", PRIORITY_FIELD);
        map.put("RawContent", RAW_CONTENT_FIELD);
        map.put("SilentPush", SILENT_PUSH_FIELD);
        map.put("Sound", SOUND_FIELD);
        map.put("Substitutions", SUBSTITUTIONS_FIELD);
        map.put("ThreadId", THREAD_ID_FIELD);
        map.put("TimeToLive", TIME_TO_LIVE_FIELD);
        map.put("Title", TITLE_FIELD);
        map.put("Url", URL_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<APNSMessage, T> g) {
        return obj -> g.apply((APNSMessage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, APNSMessage> {
        /**
         * <p>
         * The type of push notification to send. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * alert - For a standard notification that's displayed on recipients' devices and prompts a recipient to
         * interact with the notification.
         * </p>
         * </li>
         * <li>
         * <p>
         * background - For a silent notification that delivers content in the background and isn't displayed on
         * recipients' devices.
         * </p>
         * </li>
         * <li>
         * <p>
         * complication - For a notification that contains update information for an app’s complication timeline.
         * </p>
         * </li>
         * <li>
         * <p>
         * fileprovider - For a notification that signals changes to a File Provider extension.
         * </p>
         * </li>
         * <li>
         * <p>
         * mdm - For a notification that tells managed devices to contact the MDM server.
         * </p>
         * </li>
         * <li>
         * <p>
         * voip - For a notification that provides information about an incoming VoIP call.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Amazon Pinpoint specifies this value in the apns-push-type request header when it sends the notification
         * message to APNs. If you don't specify a value for this property, Amazon Pinpoint sets the value to alert or
         * background automatically, based on the value that you specify for the SilentPush or RawContent property of
         * the message.
         * </p>
         * <p>
         * For more information about the apns-push-type request header, see <a href=
         * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/sending_notification_requests_to_apns"
         * >Sending Notification Requests to APNs</a> on the Apple Developer website.
         * </p>
         * 
         * @param apnsPushType
         *        The type of push notification to send. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        alert - For a standard notification that's displayed on recipients' devices and prompts a recipient to
         *        interact with the notification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        background - For a silent notification that delivers content in the background and isn't displayed on
         *        recipients' devices.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        complication - For a notification that contains update information for an app’s complication timeline.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        fileprovider - For a notification that signals changes to a File Provider extension.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        mdm - For a notification that tells managed devices to contact the MDM server.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        voip - For a notification that provides information about an incoming VoIP call.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Amazon Pinpoint specifies this value in the apns-push-type request header when it sends the
         *        notification message to APNs. If you don't specify a value for this property, Amazon Pinpoint sets the
         *        value to alert or background automatically, based on the value that you specify for the SilentPush or
         *        RawContent property of the message.
         *        </p>
         *        <p>
         *        For more information about the apns-push-type request header, see <a href=
         *        "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/sending_notification_requests_to_apns"
         *        >Sending Notification Requests to APNs</a> on the Apple Developer website.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apnsPushType(String apnsPushType);

        /**
         * <p>
         * The action to occur if the recipient taps the push notification. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
         * default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
         * deep-linking features of the iOS platform.
         * </p>
         * </li>
         * <li>
         * <p>
         * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
         * specify.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to occur if the recipient taps the push notification. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is
         *        the default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
         *        deep-linking features of the iOS platform.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that
         *        you specify.
         *        </p>
         *        </li>
         * @see Action
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Action
         */
        Builder action(String action);

        /**
         * <p>
         * The action to occur if the recipient taps the push notification. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is the
         * default action.
         * </p>
         * </li>
         * <li>
         * <p>
         * DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
         * deep-linking features of the iOS platform.
         * </p>
         * </li>
         * <li>
         * <p>
         * URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that you
         * specify.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        The action to occur if the recipient taps the push notification. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OPEN_APP - Your app opens or it becomes the foreground app if it was sent to the background. This is
         *        the default action.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DEEP_LINK - Your app opens and displays a designated user interface in the app. This setting uses the
         *        deep-linking features of the iOS platform.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        URL - The default mobile browser on the recipient's device opens and loads the web page at a URL that
         *        you specify.
         *        </p>
         *        </li>
         * @see Action
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Action
         */
        Builder action(Action action);

        /**
         * <p>
         * The key that indicates whether and how to modify the badge of your app's icon when the recipient receives the
         * push notification. If this key isn't included in the dictionary, the badge doesn't change. To remove the
         * badge, set this value to 0.
         * </p>
         * 
         * @param badge
         *        The key that indicates whether and how to modify the badge of your app's icon when the recipient
         *        receives the push notification. If this key isn't included in the dictionary, the badge doesn't
         *        change. To remove the badge, set this value to 0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder badge(Integer badge);

        /**
         * <p>
         * The body of the notification message.
         * </p>
         * 
         * @param body
         *        The body of the notification message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(String body);

        /**
         * <p>
         * The key that indicates the notification type for the push notification. This key is a value that's defined by
         * the identifier property of one of your app's registered categories.
         * </p>
         * 
         * @param category
         *        The key that indicates the notification type for the push notification. This key is a value that's
         *        defined by the identifier property of one of your app's registered categories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder category(String category);

        /**
         * <p>
         * An arbitrary identifier that, if assigned to multiple messages, APNs uses to coalesce the messages into a
         * single push notification instead of delivering each message individually. This value can't exceed 64 bytes.
         * </p>
         * <p>
         * Amazon Pinpoint specifies this value in the apns-collapse-id request header when it sends the notification
         * message to APNs.
         * </p>
         * 
         * @param collapseId
         *        An arbitrary identifier that, if assigned to multiple messages, APNs uses to coalesce the messages
         *        into a single push notification instead of delivering each message individually. This value can't
         *        exceed 64 bytes.</p>
         *        <p>
         *        Amazon Pinpoint specifies this value in the apns-collapse-id request header when it sends the
         *        notification message to APNs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collapseId(String collapseId);

        /**
         * <p>
         * The JSON payload to use for a silent push notification. This payload is added to the data.pinpoint.jsonBody
         * object of the notification.
         * </p>
         * 
         * @param data
         *        The JSON payload to use for a silent push notification. This payload is added to the
         *        data.pinpoint.jsonBody object of the notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(Map<String, String> data);

        /**
         * <p>
         * The URL of an image or video to display in the push notification.
         * </p>
         * 
         * @param mediaUrl
         *        The URL of an image or video to display in the push notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaUrl(String mediaUrl);

        /**
         * <p>
         * The authentication method that you want Amazon Pinpoint to use when authenticating with APNs, CERTIFICATE or
         * TOKEN.
         * </p>
         * 
         * @param preferredAuthenticationMethod
         *        The authentication method that you want Amazon Pinpoint to use when authenticating with APNs,
         *        CERTIFICATE or TOKEN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAuthenticationMethod(String preferredAuthenticationMethod);

        /**
         * <p>
         * para>5 - Low priority, the notification might be delayed, delivered as part of a group, or throttled.
         * </p>
         * /listitem> <li>
         * <p>
         * 10 - High priority, the notification is sent immediately. This is the default value. A high priority
         * notification should trigger an alert, play a sound, or badge your app's icon on the recipient's device.
         * </p>
         * </li>/para>
         * <p>
         * Amazon Pinpoint specifies this value in the apns-priority request header when it sends the notification
         * message to APNs.
         * </p>
         * <p>
         * The equivalent values for Firebase Cloud Messaging (FCM), formerly Google Cloud Messaging (GCM), are normal,
         * for 5, and high, for 10. If you specify an FCM value for this property, Amazon Pinpoint accepts and converts
         * the value to the corresponding APNs value.
         * </p>
         * 
         * @param priority
         *        para>5 - Low priority, the notification might be delayed, delivered as part of a group, or
         *        throttled.</p>/listitem> <li>
         *        <p>
         *        10 - High priority, the notification is sent immediately. This is the default value. A high priority
         *        notification should trigger an alert, play a sound, or badge your app's icon on the recipient's
         *        device.
         *        </p>
         *        </li>/para>
         *        <p>
         *        Amazon Pinpoint specifies this value in the apns-priority request header when it sends the
         *        notification message to APNs.
         *        </p>
         *        <p>
         *        The equivalent values for Firebase Cloud Messaging (FCM), formerly Google Cloud Messaging (GCM), are
         *        normal, for 5, and high, for 10. If you specify an FCM value for this property, Amazon Pinpoint
         *        accepts and converts the value to the corresponding APNs value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(String priority);

        /**
         * <p>
         * The raw, JSON-formatted string to use as the payload for the notification message. If specified, this value
         * overrides all other content for the message.
         * </p>
         * <note>
         * <p>
         * If you specify the raw content of an APNs push notification, the message payload has to include the
         * content-available key. The value of the content-available key has to be an integer, and can only be 0 or 1.
         * If you're sending a standard notification, set the value of content-available to 0. If you're sending a
         * silent (background) notification, set the value of content-available to 1. Additionally, silent notification
         * payloads can't include the alert, badge, or sound keys. For more information, see <a href=
         * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
         * >Generating a Remote Notification</a> and <a href=
         * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
         * >Pushing Background Updates to Your App</a> on the Apple Developer website.
         * </p>
         * </note>
         * 
         * @param rawContent
         *        The raw, JSON-formatted string to use as the payload for the notification message. If specified, this
         *        value overrides all other content for the message.</p> <note>
         *        <p>
         *        If you specify the raw content of an APNs push notification, the message payload has to include the
         *        content-available key. The value of the content-available key has to be an integer, and can only be 0
         *        or 1. If you're sending a standard notification, set the value of content-available to 0. If you're
         *        sending a silent (background) notification, set the value of content-available to 1. Additionally,
         *        silent notification payloads can't include the alert, badge, or sound keys. For more information, see
         *        <a href=
         *        "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
         *        >Generating a Remote Notification</a> and <a href=
         *        "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
         *        >Pushing Background Updates to Your App</a> on the Apple Developer website.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rawContent(String rawContent);

        /**
         * <p>
         * Specifies whether the notification is a silent push notification. A silent (or background) push notification
         * isn't displayed on recipients' devices. You can use silent push notifications to make small updates to your
         * app, or to display messages in an in-app message center.
         * </p>
         * <p>
         * Amazon Pinpoint uses this property to determine the correct value for the apns-push-type request header when
         * it sends the notification message to APNs. If you specify a value of true for this property, Amazon Pinpoint
         * sets the value for the apns-push-type header field to background.
         * </p>
         * <note>
         * <p>
         * If you specify the raw content of an APNs push notification, the message payload has to include the
         * content-available key. For silent (background) notifications, set the value of content-available to 1.
         * Additionally, the message payload for a silent notification can't include the alert, badge, or sound keys.
         * For more information, see <a href=
         * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
         * >Generating a Remote Notification</a> and <a href=
         * "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
         * >Pushing Background Updates to Your App</a> on the Apple Developer website.
         * </p>
         * <p>
         * Apple has indicated that they will throttle "excessive" background notifications based on current traffic
         * volumes. To prevent your notifications being throttled, Apple recommends that you send no more than 3 silent
         * push notifications to each recipient per hour.
         * </p>
         * </note>
         * 
         * @param silentPush
         *        Specifies whether the notification is a silent push notification. A silent (or background) push
         *        notification isn't displayed on recipients' devices. You can use silent push notifications to make
         *        small updates to your app, or to display messages in an in-app message center.</p>
         *        <p>
         *        Amazon Pinpoint uses this property to determine the correct value for the apns-push-type request
         *        header when it sends the notification message to APNs. If you specify a value of true for this
         *        property, Amazon Pinpoint sets the value for the apns-push-type header field to background.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify the raw content of an APNs push notification, the message payload has to include the
         *        content-available key. For silent (background) notifications, set the value of content-available to 1.
         *        Additionally, the message payload for a silent notification can't include the alert, badge, or sound
         *        keys. For more information, see <a href=
         *        "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/generating_a_remote_notification"
         *        >Generating a Remote Notification</a> and <a href=
         *        "https://developer.apple.com/documentation/usernotifications/setting_up_a_remote_notification_server/pushing_background_updates_to_your_app"
         *        >Pushing Background Updates to Your App</a> on the Apple Developer website.
         *        </p>
         *        <p>
         *        Apple has indicated that they will throttle "excessive" background notifications based on current
         *        traffic volumes. To prevent your notifications being throttled, Apple recommends that you send no more
         *        than 3 silent push notifications to each recipient per hour.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder silentPush(Boolean silentPush);

        /**
         * <p>
         * The key for the sound to play when the recipient receives the push notification. The value for this key is
         * the name of a sound file in your app's main bundle or the Library/Sounds folder in your app's data container.
         * If the sound file can't be found or you specify default for the value, the system plays the default alert
         * sound.
         * </p>
         * 
         * @param sound
         *        The key for the sound to play when the recipient receives the push notification. The value for this
         *        key is the name of a sound file in your app's main bundle or the Library/Sounds folder in your app's
         *        data container. If the sound file can't be found or you specify default for the value, the system
         *        plays the default alert sound.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sound(String sound);

        /**
         * <p>
         * The default message variables to use in the notification message. You can override these default variables
         * with individual address variables.
         * </p>
         * 
         * @param substitutions
         *        The default message variables to use in the notification message. You can override these default
         *        variables with individual address variables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder substitutions(Map<String, ? extends Collection<String>> substitutions);

        /**
         * <p>
         * The key that represents your app-specific identifier for grouping notifications. If you provide a
         * Notification Content app extension, you can use this value to group your notifications together.
         * </p>
         * 
         * @param threadId
         *        The key that represents your app-specific identifier for grouping notifications. If you provide a
         *        Notification Content app extension, you can use this value to group your notifications together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadId(String threadId);

        /**
         * <p>
         * The amount of time, in seconds, that APNs should store and attempt to deliver the push notification, if the
         * service is unable to deliver the notification the first time. If this value is 0, APNs treats the
         * notification as if it expires immediately and the service doesn't store or try to deliver the notification
         * again.
         * </p>
         * <p>
         * Amazon Pinpoint specifies this value in the apns-expiration request header when it sends the notification
         * message to APNs.
         * </p>
         * 
         * @param timeToLive
         *        The amount of time, in seconds, that APNs should store and attempt to deliver the push notification,
         *        if the service is unable to deliver the notification the first time. If this value is 0, APNs treats
         *        the notification as if it expires immediately and the service doesn't store or try to deliver the
         *        notification again.</p>
         *        <p>
         *        Amazon Pinpoint specifies this value in the apns-expiration request header when it sends the
         *        notification message to APNs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeToLive(Integer timeToLive);

        /**
         * <p>
         * The title to display above the notification message on the recipient's device.
         * </p>
         * 
         * @param title
         *        The title to display above the notification message on the recipient's device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The URL to open in the recipient's default mobile browser, if a recipient taps the push notification and the
         * value of the Action property is URL.
         * </p>
         * 
         * @param url
         *        The URL to open in the recipient's default mobile browser, if a recipient taps the push notification
         *        and the value of the Action property is URL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);
    }

    static final class BuilderImpl implements Builder {
        private String apnsPushType;

        private String action;

        private Integer badge;

        private String body;

        private String category;

        private String collapseId;

        private Map<String, String> data = DefaultSdkAutoConstructMap.getInstance();

        private String mediaUrl;

        private String preferredAuthenticationMethod;

        private String priority;

        private String rawContent;

        private Boolean silentPush;

        private String sound;

        private Map<String, List<String>> substitutions = DefaultSdkAutoConstructMap.getInstance();

        private String threadId;

        private Integer timeToLive;

        private String title;

        private String url;

        private BuilderImpl() {
        }

        private BuilderImpl(APNSMessage model) {
            apnsPushType(model.apnsPushType);
            action(model.action);
            badge(model.badge);
            body(model.body);
            category(model.category);
            collapseId(model.collapseId);
            data(model.data);
            mediaUrl(model.mediaUrl);
            preferredAuthenticationMethod(model.preferredAuthenticationMethod);
            priority(model.priority);
            rawContent(model.rawContent);
            silentPush(model.silentPush);
            sound(model.sound);
            substitutions(model.substitutions);
            threadId(model.threadId);
            timeToLive(model.timeToLive);
            title(model.title);
            url(model.url);
        }

        public final String getApnsPushType() {
            return apnsPushType;
        }

        public final void setApnsPushType(String apnsPushType) {
            this.apnsPushType = apnsPushType;
        }

        @Override
        public final Builder apnsPushType(String apnsPushType) {
            this.apnsPushType = apnsPushType;
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(Action action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final Integer getBadge() {
            return badge;
        }

        public final void setBadge(Integer badge) {
            this.badge = badge;
        }

        @Override
        public final Builder badge(Integer badge) {
            this.badge = badge;
            return this;
        }

        public final String getBody() {
            return body;
        }

        public final void setBody(String body) {
            this.body = body;
        }

        @Override
        public final Builder body(String body) {
            this.body = body;
            return this;
        }

        public final String getCategory() {
            return category;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        public final String getCollapseId() {
            return collapseId;
        }

        public final void setCollapseId(String collapseId) {
            this.collapseId = collapseId;
        }

        @Override
        public final Builder collapseId(String collapseId) {
            this.collapseId = collapseId;
            return this;
        }

        public final Map<String, String> getData() {
            if (data instanceof SdkAutoConstructMap) {
                return null;
            }
            return data;
        }

        public final void setData(Map<String, String> data) {
            this.data = MapOf__stringCopier.copy(data);
        }

        @Override
        public final Builder data(Map<String, String> data) {
            this.data = MapOf__stringCopier.copy(data);
            return this;
        }

        public final String getMediaUrl() {
            return mediaUrl;
        }

        public final void setMediaUrl(String mediaUrl) {
            this.mediaUrl = mediaUrl;
        }

        @Override
        public final Builder mediaUrl(String mediaUrl) {
            this.mediaUrl = mediaUrl;
            return this;
        }

        public final String getPreferredAuthenticationMethod() {
            return preferredAuthenticationMethod;
        }

        public final void setPreferredAuthenticationMethod(String preferredAuthenticationMethod) {
            this.preferredAuthenticationMethod = preferredAuthenticationMethod;
        }

        @Override
        public final Builder preferredAuthenticationMethod(String preferredAuthenticationMethod) {
            this.preferredAuthenticationMethod = preferredAuthenticationMethod;
            return this;
        }

        public final String getPriority() {
            return priority;
        }

        public final void setPriority(String priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(String priority) {
            this.priority = priority;
            return this;
        }

        public final String getRawContent() {
            return rawContent;
        }

        public final void setRawContent(String rawContent) {
            this.rawContent = rawContent;
        }

        @Override
        public final Builder rawContent(String rawContent) {
            this.rawContent = rawContent;
            return this;
        }

        public final Boolean getSilentPush() {
            return silentPush;
        }

        public final void setSilentPush(Boolean silentPush) {
            this.silentPush = silentPush;
        }

        @Override
        public final Builder silentPush(Boolean silentPush) {
            this.silentPush = silentPush;
            return this;
        }

        public final String getSound() {
            return sound;
        }

        public final void setSound(String sound) {
            this.sound = sound;
        }

        @Override
        public final Builder sound(String sound) {
            this.sound = sound;
            return this;
        }

        public final Map<String, ? extends Collection<String>> getSubstitutions() {
            if (substitutions instanceof SdkAutoConstructMap) {
                return null;
            }
            return substitutions;
        }

        public final void setSubstitutions(Map<String, ? extends Collection<String>> substitutions) {
            this.substitutions = MapOfListOf__stringCopier.copy(substitutions);
        }

        @Override
        public final Builder substitutions(Map<String, ? extends Collection<String>> substitutions) {
            this.substitutions = MapOfListOf__stringCopier.copy(substitutions);
            return this;
        }

        public final String getThreadId() {
            return threadId;
        }

        public final void setThreadId(String threadId) {
            this.threadId = threadId;
        }

        @Override
        public final Builder threadId(String threadId) {
            this.threadId = threadId;
            return this;
        }

        public final Integer getTimeToLive() {
            return timeToLive;
        }

        public final void setTimeToLive(Integer timeToLive) {
            this.timeToLive = timeToLive;
        }

        @Override
        public final Builder timeToLive(Integer timeToLive) {
            this.timeToLive = timeToLive;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        @Override
        public APNSMessage build() {
            return new APNSMessage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
