/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies information about an event that reports data to Amazon Pinpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> APP_PACKAGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppPackageName").getter(getter(Event::appPackageName)).setter(setter(Builder::appPackageName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppPackageName").build()).build();

    private static final SdkField<String> APP_TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppTitle").getter(getter(Event::appTitle)).setter(setter(Builder::appTitle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppTitle").build()).build();

    private static final SdkField<String> APP_VERSION_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppVersionCode").getter(getter(Event::appVersionCode)).setter(setter(Builder::appVersionCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppVersionCode").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Attributes")
            .getter(getter(Event::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CLIENT_SDK_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientSdkVersion").getter(getter(Event::clientSdkVersion)).setter(setter(Builder::clientSdkVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientSdkVersion").build()).build();

    private static final SdkField<String> EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventType").getter(getter(Event::eventType)).setter(setter(Builder::eventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventType").build()).build();

    private static final SdkField<Map<String, Double>> METRICS_FIELD = SdkField
            .<Map<String, Double>> builder(MarshallingType.MAP)
            .memberName("Metrics")
            .getter(getter(Event::metrics))
            .setter(setter(Builder::metrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metrics").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> SDK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SdkName").getter(getter(Event::sdkName)).setter(setter(Builder::sdkName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SdkName").build()).build();

    private static final SdkField<Session> SESSION_FIELD = SdkField.<Session> builder(MarshallingType.SDK_POJO)
            .memberName("Session").getter(getter(Event::session)).setter(setter(Builder::session)).constructor(Session::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Session").build()).build();

    private static final SdkField<String> TIMESTAMP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Timestamp").getter(getter(Event::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APP_PACKAGE_NAME_FIELD,
            APP_TITLE_FIELD, APP_VERSION_CODE_FIELD, ATTRIBUTES_FIELD, CLIENT_SDK_VERSION_FIELD, EVENT_TYPE_FIELD, METRICS_FIELD,
            SDK_NAME_FIELD, SESSION_FIELD, TIMESTAMP_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String appPackageName;

    private final String appTitle;

    private final String appVersionCode;

    private final Map<String, String> attributes;

    private final String clientSdkVersion;

    private final String eventType;

    private final Map<String, Double> metrics;

    private final String sdkName;

    private final Session session;

    private final String timestamp;

    private Event(BuilderImpl builder) {
        this.appPackageName = builder.appPackageName;
        this.appTitle = builder.appTitle;
        this.appVersionCode = builder.appVersionCode;
        this.attributes = builder.attributes;
        this.clientSdkVersion = builder.clientSdkVersion;
        this.eventType = builder.eventType;
        this.metrics = builder.metrics;
        this.sdkName = builder.sdkName;
        this.session = builder.session;
        this.timestamp = builder.timestamp;
    }

    /**
     * <p>
     * The package name of the app that's recording the event.
     * </p>
     * 
     * @return The package name of the app that's recording the event.
     */
    public final String appPackageName() {
        return appPackageName;
    }

    /**
     * <p>
     * The title of the app that's recording the event.
     * </p>
     * 
     * @return The title of the app that's recording the event.
     */
    public final String appTitle() {
        return appTitle;
    }

    /**
     * <p>
     * The version number of the app that's recording the event.
     * </p>
     * 
     * @return The version number of the app that's recording the event.
     */
    public final String appVersionCode() {
        return appVersionCode;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * One or more custom attributes that are associated with the event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return One or more custom attributes that are associated with the event.
     */
    public final Map<String, String> attributes() {
        return attributes;
    }

    /**
     * <p>
     * The version of the SDK that's running on the client device.
     * </p>
     * 
     * @return The version of the SDK that's running on the client device.
     */
    public final String clientSdkVersion() {
        return clientSdkVersion;
    }

    /**
     * <p>
     * The name of the event.
     * </p>
     * 
     * @return The name of the event.
     */
    public final String eventType() {
        return eventType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metrics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetrics() {
        return metrics != null && !(metrics instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * One or more custom metrics that are associated with the event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetrics} method.
     * </p>
     * 
     * @return One or more custom metrics that are associated with the event.
     */
    public final Map<String, Double> metrics() {
        return metrics;
    }

    /**
     * <p>
     * The name of the SDK that's being used to record the event.
     * </p>
     * 
     * @return The name of the SDK that's being used to record the event.
     */
    public final String sdkName() {
        return sdkName;
    }

    /**
     * <p>
     * Information about the session in which the event occurred.
     * </p>
     * 
     * @return Information about the session in which the event occurred.
     */
    public final Session session() {
        return session;
    }

    /**
     * <p>
     * The date and time, in ISO 8601 format, when the event occurred.
     * </p>
     * 
     * @return The date and time, in ISO 8601 format, when the event occurred.
     */
    public final String timestamp() {
        return timestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(appPackageName());
        hashCode = 31 * hashCode + Objects.hashCode(appTitle());
        hashCode = 31 * hashCode + Objects.hashCode(appVersionCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientSdkVersion());
        hashCode = 31 * hashCode + Objects.hashCode(eventType());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetrics() ? metrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sdkName());
        hashCode = 31 * hashCode + Objects.hashCode(session());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(appPackageName(), other.appPackageName()) && Objects.equals(appTitle(), other.appTitle())
                && Objects.equals(appVersionCode(), other.appVersionCode()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes())
                && Objects.equals(clientSdkVersion(), other.clientSdkVersion()) && Objects.equals(eventType(), other.eventType())
                && hasMetrics() == other.hasMetrics() && Objects.equals(metrics(), other.metrics())
                && Objects.equals(sdkName(), other.sdkName()) && Objects.equals(session(), other.session())
                && Objects.equals(timestamp(), other.timestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Event").add("AppPackageName", appPackageName()).add("AppTitle", appTitle())
                .add("AppVersionCode", appVersionCode()).add("Attributes", hasAttributes() ? attributes() : null)
                .add("ClientSdkVersion", clientSdkVersion()).add("EventType", eventType())
                .add("Metrics", hasMetrics() ? metrics() : null).add("SdkName", sdkName()).add("Session", session())
                .add("Timestamp", timestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AppPackageName":
            return Optional.ofNullable(clazz.cast(appPackageName()));
        case "AppTitle":
            return Optional.ofNullable(clazz.cast(appTitle()));
        case "AppVersionCode":
            return Optional.ofNullable(clazz.cast(appVersionCode()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "ClientSdkVersion":
            return Optional.ofNullable(clazz.cast(clientSdkVersion()));
        case "EventType":
            return Optional.ofNullable(clazz.cast(eventType()));
        case "Metrics":
            return Optional.ofNullable(clazz.cast(metrics()));
        case "SdkName":
            return Optional.ofNullable(clazz.cast(sdkName()));
        case "Session":
            return Optional.ofNullable(clazz.cast(session()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AppPackageName", APP_PACKAGE_NAME_FIELD);
        map.put("AppTitle", APP_TITLE_FIELD);
        map.put("AppVersionCode", APP_VERSION_CODE_FIELD);
        map.put("Attributes", ATTRIBUTES_FIELD);
        map.put("ClientSdkVersion", CLIENT_SDK_VERSION_FIELD);
        map.put("EventType", EVENT_TYPE_FIELD);
        map.put("Metrics", METRICS_FIELD);
        map.put("SdkName", SDK_NAME_FIELD);
        map.put("Session", SESSION_FIELD);
        map.put("Timestamp", TIMESTAMP_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The package name of the app that's recording the event.
         * </p>
         * 
         * @param appPackageName
         *        The package name of the app that's recording the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appPackageName(String appPackageName);

        /**
         * <p>
         * The title of the app that's recording the event.
         * </p>
         * 
         * @param appTitle
         *        The title of the app that's recording the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appTitle(String appTitle);

        /**
         * <p>
         * The version number of the app that's recording the event.
         * </p>
         * 
         * @param appVersionCode
         *        The version number of the app that's recording the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appVersionCode(String appVersionCode);

        /**
         * <p>
         * One or more custom attributes that are associated with the event.
         * </p>
         * 
         * @param attributes
         *        One or more custom attributes that are associated with the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);

        /**
         * <p>
         * The version of the SDK that's running on the client device.
         * </p>
         * 
         * @param clientSdkVersion
         *        The version of the SDK that's running on the client device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientSdkVersion(String clientSdkVersion);

        /**
         * <p>
         * The name of the event.
         * </p>
         * 
         * @param eventType
         *        The name of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventType(String eventType);

        /**
         * <p>
         * One or more custom metrics that are associated with the event.
         * </p>
         * 
         * @param metrics
         *        One or more custom metrics that are associated with the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(Map<String, Double> metrics);

        /**
         * <p>
         * The name of the SDK that's being used to record the event.
         * </p>
         * 
         * @param sdkName
         *        The name of the SDK that's being used to record the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sdkName(String sdkName);

        /**
         * <p>
         * Information about the session in which the event occurred.
         * </p>
         * 
         * @param session
         *        Information about the session in which the event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder session(Session session);

        /**
         * <p>
         * Information about the session in which the event occurred.
         * </p>
         * This is a convenience method that creates an instance of the {@link Session.Builder} avoiding the need to
         * create one manually via {@link Session#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Session.Builder#build()} is called immediately and its result is
         * passed to {@link #session(Session)}.
         * 
         * @param session
         *        a consumer that will call methods on {@link Session.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #session(Session)
         */
        default Builder session(Consumer<Session.Builder> session) {
            return session(Session.builder().applyMutation(session).build());
        }

        /**
         * <p>
         * The date and time, in ISO 8601 format, when the event occurred.
         * </p>
         * 
         * @param timestamp
         *        The date and time, in ISO 8601 format, when the event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(String timestamp);
    }

    static final class BuilderImpl implements Builder {
        private String appPackageName;

        private String appTitle;

        private String appVersionCode;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private String clientSdkVersion;

        private String eventType;

        private Map<String, Double> metrics = DefaultSdkAutoConstructMap.getInstance();

        private String sdkName;

        private Session session;

        private String timestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            appPackageName(model.appPackageName);
            appTitle(model.appTitle);
            appVersionCode(model.appVersionCode);
            attributes(model.attributes);
            clientSdkVersion(model.clientSdkVersion);
            eventType(model.eventType);
            metrics(model.metrics);
            sdkName(model.sdkName);
            session(model.session);
            timestamp(model.timestamp);
        }

        public final String getAppPackageName() {
            return appPackageName;
        }

        public final void setAppPackageName(String appPackageName) {
            this.appPackageName = appPackageName;
        }

        @Override
        public final Builder appPackageName(String appPackageName) {
            this.appPackageName = appPackageName;
            return this;
        }

        public final String getAppTitle() {
            return appTitle;
        }

        public final void setAppTitle(String appTitle) {
            this.appTitle = appTitle;
        }

        @Override
        public final Builder appTitle(String appTitle) {
            this.appTitle = appTitle;
            return this;
        }

        public final String getAppVersionCode() {
            return appVersionCode;
        }

        public final void setAppVersionCode(String appVersionCode) {
            this.appVersionCode = appVersionCode;
        }

        @Override
        public final Builder appVersionCode(String appVersionCode) {
            this.appVersionCode = appVersionCode;
            return this;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = MapOf__stringCopier.copy(attributes);
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = MapOf__stringCopier.copy(attributes);
            return this;
        }

        public final String getClientSdkVersion() {
            return clientSdkVersion;
        }

        public final void setClientSdkVersion(String clientSdkVersion) {
            this.clientSdkVersion = clientSdkVersion;
        }

        @Override
        public final Builder clientSdkVersion(String clientSdkVersion) {
            this.clientSdkVersion = clientSdkVersion;
            return this;
        }

        public final String getEventType() {
            return eventType;
        }

        public final void setEventType(String eventType) {
            this.eventType = eventType;
        }

        @Override
        public final Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        public final Map<String, Double> getMetrics() {
            if (metrics instanceof SdkAutoConstructMap) {
                return null;
            }
            return metrics;
        }

        public final void setMetrics(Map<String, Double> metrics) {
            this.metrics = MapOf__doubleCopier.copy(metrics);
        }

        @Override
        public final Builder metrics(Map<String, Double> metrics) {
            this.metrics = MapOf__doubleCopier.copy(metrics);
            return this;
        }

        public final String getSdkName() {
            return sdkName;
        }

        public final void setSdkName(String sdkName) {
            this.sdkName = sdkName;
        }

        @Override
        public final Builder sdkName(String sdkName) {
            this.sdkName = sdkName;
            return this;
        }

        public final Session.Builder getSession() {
            return session != null ? session.toBuilder() : null;
        }

        public final void setSession(Session.BuilderImpl session) {
            this.session = session != null ? session.build() : null;
        }

        @Override
        public final Builder session(Session session) {
            this.session = session;
            return this;
        }

        public final String getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(String timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(String timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
