/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpoint.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the base segments and dimensions for a segment, and the relationships between these base segments and
 * dimensions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SegmentGroup implements SdkPojo, Serializable, ToCopyableBuilder<SegmentGroup.Builder, SegmentGroup> {
    private static final SdkField<List<SegmentDimensions>> DIMENSIONS_FIELD = SdkField
            .<List<SegmentDimensions>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(SegmentGroup::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SegmentDimensions> builder(MarshallingType.SDK_POJO)
                                            .constructor(SegmentDimensions::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<SegmentReference>> SOURCE_SEGMENTS_FIELD = SdkField
            .<List<SegmentReference>> builder(MarshallingType.LIST)
            .memberName("SourceSegments")
            .getter(getter(SegmentGroup::sourceSegments))
            .setter(setter(Builder::sourceSegments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceSegments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SegmentReference> builder(MarshallingType.SDK_POJO)
                                            .constructor(SegmentReference::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceType").getter(getter(SegmentGroup::sourceTypeAsString)).setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceType").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(SegmentGroup::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIMENSIONS_FIELD,
            SOURCE_SEGMENTS_FIELD, SOURCE_TYPE_FIELD, TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<SegmentDimensions> dimensions;

    private final List<SegmentReference> sourceSegments;

    private final String sourceType;

    private final String type;

    private SegmentGroup(BuilderImpl builder) {
        this.dimensions = builder.dimensions;
        this.sourceSegments = builder.sourceSegments;
        this.sourceType = builder.sourceType;
        this.type = builder.type;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array that defines the dimensions for the segment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return An array that defines the dimensions for the segment.
     */
    public final List<SegmentDimensions> dimensions() {
        return dimensions;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceSegments property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceSegments() {
        return sourceSegments != null && !(sourceSegments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The base segment to build the segment on. A base segment, also referred to as a <i>source segment</i>, defines
     * the initial population of endpoints for a segment. When you add dimensions to a segment, Amazon Pinpoint filters
     * the base segment by using the dimensions that you specify.
     * </p>
     * <p>
     * You can specify more than one dimensional segment or only one imported segment. If you specify an imported
     * segment, the Amazon Pinpoint console displays a segment size estimate that indicates the size of the imported
     * segment without any filters applied to it.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceSegments} method.
     * </p>
     * 
     * @return The base segment to build the segment on. A base segment, also referred to as a <i>source segment</i>,
     *         defines the initial population of endpoints for a segment. When you add dimensions to a segment, Amazon
     *         Pinpoint filters the base segment by using the dimensions that you specify.</p>
     *         <p>
     *         You can specify more than one dimensional segment or only one imported segment. If you specify an
     *         imported segment, the Amazon Pinpoint console displays a segment size estimate that indicates the size of
     *         the imported segment without any filters applied to it.
     */
    public final List<SegmentReference> sourceSegments() {
        return sourceSegments;
    }

    /**
     * <p>
     * Specifies how to handle multiple base segments for the segment. For example, if you specify three base segments
     * for the segment, whether the resulting segment is based on all, any, or none of the base segments.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return Specifies how to handle multiple base segments for the segment. For example, if you specify three base
     *         segments for the segment, whether the resulting segment is based on all, any, or none of the base
     *         segments.
     * @see SourceType
     */
    public final SourceType sourceType() {
        return SourceType.fromValue(sourceType);
    }

    /**
     * <p>
     * Specifies how to handle multiple base segments for the segment. For example, if you specify three base segments
     * for the segment, whether the resulting segment is based on all, any, or none of the base segments.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return Specifies how to handle multiple base segments for the segment. For example, if you specify three base
     *         segments for the segment, whether the resulting segment is based on all, any, or none of the base
     *         segments.
     * @see SourceType
     */
    public final String sourceTypeAsString() {
        return sourceType;
    }

    /**
     * <p>
     * Specifies how to handle multiple dimensions for the segment. For example, if you specify three dimensions for the
     * segment, whether the resulting segment includes endpoints that match all, any, or none of the dimensions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link Type#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies how to handle multiple dimensions for the segment. For example, if you specify three dimensions
     *         for the segment, whether the resulting segment includes endpoints that match all, any, or none of the
     *         dimensions.
     * @see Type
     */
    public final Type type() {
        return Type.fromValue(type);
    }

    /**
     * <p>
     * Specifies how to handle multiple dimensions for the segment. For example, if you specify three dimensions for the
     * segment, whether the resulting segment includes endpoints that match all, any, or none of the dimensions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link Type#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies how to handle multiple dimensions for the segment. For example, if you specify three dimensions
     *         for the segment, whether the resulting segment includes endpoints that match all, any, or none of the
     *         dimensions.
     * @see Type
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceSegments() ? sourceSegments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SegmentGroup)) {
            return false;
        }
        SegmentGroup other = (SegmentGroup) obj;
        return hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && hasSourceSegments() == other.hasSourceSegments() && Objects.equals(sourceSegments(), other.sourceSegments())
                && Objects.equals(sourceTypeAsString(), other.sourceTypeAsString())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SegmentGroup").add("Dimensions", hasDimensions() ? dimensions() : null)
                .add("SourceSegments", hasSourceSegments() ? sourceSegments() : null).add("SourceType", sourceTypeAsString())
                .add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "SourceSegments":
            return Optional.ofNullable(clazz.cast(sourceSegments()));
        case "SourceType":
            return Optional.ofNullable(clazz.cast(sourceTypeAsString()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Dimensions", DIMENSIONS_FIELD);
        map.put("SourceSegments", SOURCE_SEGMENTS_FIELD);
        map.put("SourceType", SOURCE_TYPE_FIELD);
        map.put("Type", TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SegmentGroup, T> g) {
        return obj -> g.apply((SegmentGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SegmentGroup> {
        /**
         * <p>
         * An array that defines the dimensions for the segment.
         * </p>
         * 
         * @param dimensions
         *        An array that defines the dimensions for the segment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<SegmentDimensions> dimensions);

        /**
         * <p>
         * An array that defines the dimensions for the segment.
         * </p>
         * 
         * @param dimensions
         *        An array that defines the dimensions for the segment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(SegmentDimensions... dimensions);

        /**
         * <p>
         * An array that defines the dimensions for the segment.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pinpoint.model.SegmentDimensions.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.pinpoint.model.SegmentDimensions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.pinpoint.model.SegmentDimensions.Builder#build()} is called
         * immediately and its result is passed to {@link #dimensions(List<SegmentDimensions>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.pinpoint.model.SegmentDimensions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(java.util.Collection<SegmentDimensions>)
         */
        Builder dimensions(Consumer<SegmentDimensions.Builder>... dimensions);

        /**
         * <p>
         * The base segment to build the segment on. A base segment, also referred to as a <i>source segment</i>,
         * defines the initial population of endpoints for a segment. When you add dimensions to a segment, Amazon
         * Pinpoint filters the base segment by using the dimensions that you specify.
         * </p>
         * <p>
         * You can specify more than one dimensional segment or only one imported segment. If you specify an imported
         * segment, the Amazon Pinpoint console displays a segment size estimate that indicates the size of the imported
         * segment without any filters applied to it.
         * </p>
         * 
         * @param sourceSegments
         *        The base segment to build the segment on. A base segment, also referred to as a <i>source segment</i>,
         *        defines the initial population of endpoints for a segment. When you add dimensions to a segment,
         *        Amazon Pinpoint filters the base segment by using the dimensions that you specify.</p>
         *        <p>
         *        You can specify more than one dimensional segment or only one imported segment. If you specify an
         *        imported segment, the Amazon Pinpoint console displays a segment size estimate that indicates the size
         *        of the imported segment without any filters applied to it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceSegments(Collection<SegmentReference> sourceSegments);

        /**
         * <p>
         * The base segment to build the segment on. A base segment, also referred to as a <i>source segment</i>,
         * defines the initial population of endpoints for a segment. When you add dimensions to a segment, Amazon
         * Pinpoint filters the base segment by using the dimensions that you specify.
         * </p>
         * <p>
         * You can specify more than one dimensional segment or only one imported segment. If you specify an imported
         * segment, the Amazon Pinpoint console displays a segment size estimate that indicates the size of the imported
         * segment without any filters applied to it.
         * </p>
         * 
         * @param sourceSegments
         *        The base segment to build the segment on. A base segment, also referred to as a <i>source segment</i>,
         *        defines the initial population of endpoints for a segment. When you add dimensions to a segment,
         *        Amazon Pinpoint filters the base segment by using the dimensions that you specify.</p>
         *        <p>
         *        You can specify more than one dimensional segment or only one imported segment. If you specify an
         *        imported segment, the Amazon Pinpoint console displays a segment size estimate that indicates the size
         *        of the imported segment without any filters applied to it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceSegments(SegmentReference... sourceSegments);

        /**
         * <p>
         * The base segment to build the segment on. A base segment, also referred to as a <i>source segment</i>,
         * defines the initial population of endpoints for a segment. When you add dimensions to a segment, Amazon
         * Pinpoint filters the base segment by using the dimensions that you specify.
         * </p>
         * <p>
         * You can specify more than one dimensional segment or only one imported segment. If you specify an imported
         * segment, the Amazon Pinpoint console displays a segment size estimate that indicates the size of the imported
         * segment without any filters applied to it.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.pinpoint.model.SegmentReference.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.pinpoint.model.SegmentReference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.pinpoint.model.SegmentReference.Builder#build()} is called immediately
         * and its result is passed to {@link #sourceSegments(List<SegmentReference>)}.
         * 
         * @param sourceSegments
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.pinpoint.model.SegmentReference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceSegments(java.util.Collection<SegmentReference>)
         */
        Builder sourceSegments(Consumer<SegmentReference.Builder>... sourceSegments);

        /**
         * <p>
         * Specifies how to handle multiple base segments for the segment. For example, if you specify three base
         * segments for the segment, whether the resulting segment is based on all, any, or none of the base segments.
         * </p>
         * 
         * @param sourceType
         *        Specifies how to handle multiple base segments for the segment. For example, if you specify three base
         *        segments for the segment, whether the resulting segment is based on all, any, or none of the base
         *        segments.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(String sourceType);

        /**
         * <p>
         * Specifies how to handle multiple base segments for the segment. For example, if you specify three base
         * segments for the segment, whether the resulting segment is based on all, any, or none of the base segments.
         * </p>
         * 
         * @param sourceType
         *        Specifies how to handle multiple base segments for the segment. For example, if you specify three base
         *        segments for the segment, whether the resulting segment is based on all, any, or none of the base
         *        segments.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(SourceType sourceType);

        /**
         * <p>
         * Specifies how to handle multiple dimensions for the segment. For example, if you specify three dimensions for
         * the segment, whether the resulting segment includes endpoints that match all, any, or none of the dimensions.
         * </p>
         * 
         * @param type
         *        Specifies how to handle multiple dimensions for the segment. For example, if you specify three
         *        dimensions for the segment, whether the resulting segment includes endpoints that match all, any, or
         *        none of the dimensions.
         * @see Type
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Type
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies how to handle multiple dimensions for the segment. For example, if you specify three dimensions for
         * the segment, whether the resulting segment includes endpoints that match all, any, or none of the dimensions.
         * </p>
         * 
         * @param type
         *        Specifies how to handle multiple dimensions for the segment. For example, if you specify three
         *        dimensions for the segment, whether the resulting segment includes endpoints that match all, any, or
         *        none of the dimensions.
         * @see Type
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Type
         */
        Builder type(Type type);
    }

    static final class BuilderImpl implements Builder {
        private List<SegmentDimensions> dimensions = DefaultSdkAutoConstructList.getInstance();

        private List<SegmentReference> sourceSegments = DefaultSdkAutoConstructList.getInstance();

        private String sourceType;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(SegmentGroup model) {
            dimensions(model.dimensions);
            sourceSegments(model.sourceSegments);
            sourceType(model.sourceType);
            type(model.type);
        }

        public final List<SegmentDimensions.Builder> getDimensions() {
            List<SegmentDimensions.Builder> result = ListOfSegmentDimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDimensions(Collection<SegmentDimensions.BuilderImpl> dimensions) {
            this.dimensions = ListOfSegmentDimensionsCopier.copyFromBuilder(dimensions);
        }

        @Override
        public final Builder dimensions(Collection<SegmentDimensions> dimensions) {
            this.dimensions = ListOfSegmentDimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(SegmentDimensions... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<SegmentDimensions.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> SegmentDimensions.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<SegmentReference.Builder> getSourceSegments() {
            List<SegmentReference.Builder> result = ListOfSegmentReferenceCopier.copyToBuilder(this.sourceSegments);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceSegments(Collection<SegmentReference.BuilderImpl> sourceSegments) {
            this.sourceSegments = ListOfSegmentReferenceCopier.copyFromBuilder(sourceSegments);
        }

        @Override
        public final Builder sourceSegments(Collection<SegmentReference> sourceSegments) {
            this.sourceSegments = ListOfSegmentReferenceCopier.copy(sourceSegments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceSegments(SegmentReference... sourceSegments) {
            sourceSegments(Arrays.asList(sourceSegments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceSegments(Consumer<SegmentReference.Builder>... sourceSegments) {
            sourceSegments(Stream.of(sourceSegments).map(c -> SegmentReference.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getSourceType() {
            return sourceType;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        @Override
        public final Builder sourceType(SourceType sourceType) {
            this.sourceType(sourceType == null ? null : sourceType.toString());
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(Type type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public SegmentGroup build() {
            return new SegmentGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
