/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pinpointemail;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.pinpointemail.internal.PinpointEmailServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pinpointemail.model.AccountSuspendedException;
import software.amazon.awssdk.services.pinpointemail.model.AlreadyExistsException;
import software.amazon.awssdk.services.pinpointemail.model.BadRequestException;
import software.amazon.awssdk.services.pinpointemail.model.ConcurrentModificationException;
import software.amazon.awssdk.services.pinpointemail.model.CreateConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.pinpointemail.model.CreateConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.pinpointemail.model.CreateConfigurationSetRequest;
import software.amazon.awssdk.services.pinpointemail.model.CreateConfigurationSetResponse;
import software.amazon.awssdk.services.pinpointemail.model.CreateDedicatedIpPoolRequest;
import software.amazon.awssdk.services.pinpointemail.model.CreateDedicatedIpPoolResponse;
import software.amazon.awssdk.services.pinpointemail.model.CreateDeliverabilityTestReportRequest;
import software.amazon.awssdk.services.pinpointemail.model.CreateDeliverabilityTestReportResponse;
import software.amazon.awssdk.services.pinpointemail.model.CreateEmailIdentityRequest;
import software.amazon.awssdk.services.pinpointemail.model.CreateEmailIdentityResponse;
import software.amazon.awssdk.services.pinpointemail.model.DeleteConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.pinpointemail.model.DeleteConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.pinpointemail.model.DeleteConfigurationSetRequest;
import software.amazon.awssdk.services.pinpointemail.model.DeleteConfigurationSetResponse;
import software.amazon.awssdk.services.pinpointemail.model.DeleteDedicatedIpPoolRequest;
import software.amazon.awssdk.services.pinpointemail.model.DeleteDedicatedIpPoolResponse;
import software.amazon.awssdk.services.pinpointemail.model.DeleteEmailIdentityRequest;
import software.amazon.awssdk.services.pinpointemail.model.DeleteEmailIdentityResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetAccountRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetAccountResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetBlacklistReportsRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetBlacklistReportsResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetConfigurationSetEventDestinationsRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetConfigurationSetEventDestinationsResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetConfigurationSetRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetConfigurationSetResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetDedicatedIpRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetDedicatedIpResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetDedicatedIpsRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetDedicatedIpsResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetDeliverabilityDashboardOptionsRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetDeliverabilityDashboardOptionsResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetDeliverabilityTestReportRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetDeliverabilityTestReportResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetDomainDeliverabilityCampaignRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetDomainDeliverabilityCampaignResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetDomainStatisticsReportRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetDomainStatisticsReportResponse;
import software.amazon.awssdk.services.pinpointemail.model.GetEmailIdentityRequest;
import software.amazon.awssdk.services.pinpointemail.model.GetEmailIdentityResponse;
import software.amazon.awssdk.services.pinpointemail.model.LimitExceededException;
import software.amazon.awssdk.services.pinpointemail.model.ListConfigurationSetsRequest;
import software.amazon.awssdk.services.pinpointemail.model.ListConfigurationSetsResponse;
import software.amazon.awssdk.services.pinpointemail.model.ListDedicatedIpPoolsRequest;
import software.amazon.awssdk.services.pinpointemail.model.ListDedicatedIpPoolsResponse;
import software.amazon.awssdk.services.pinpointemail.model.ListDeliverabilityTestReportsRequest;
import software.amazon.awssdk.services.pinpointemail.model.ListDeliverabilityTestReportsResponse;
import software.amazon.awssdk.services.pinpointemail.model.ListDomainDeliverabilityCampaignsRequest;
import software.amazon.awssdk.services.pinpointemail.model.ListDomainDeliverabilityCampaignsResponse;
import software.amazon.awssdk.services.pinpointemail.model.ListEmailIdentitiesRequest;
import software.amazon.awssdk.services.pinpointemail.model.ListEmailIdentitiesResponse;
import software.amazon.awssdk.services.pinpointemail.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pinpointemail.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pinpointemail.model.MailFromDomainNotVerifiedException;
import software.amazon.awssdk.services.pinpointemail.model.MessageRejectedException;
import software.amazon.awssdk.services.pinpointemail.model.NotFoundException;
import software.amazon.awssdk.services.pinpointemail.model.PinpointEmailException;
import software.amazon.awssdk.services.pinpointemail.model.PutAccountDedicatedIpWarmupAttributesRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutAccountDedicatedIpWarmupAttributesResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutAccountSendingAttributesRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutAccountSendingAttributesResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetDeliveryOptionsRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetDeliveryOptionsResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetReputationOptionsRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetReputationOptionsResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetSendingOptionsRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetSendingOptionsResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetTrackingOptionsRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutConfigurationSetTrackingOptionsResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutDedicatedIpInPoolRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutDedicatedIpInPoolResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutDedicatedIpWarmupAttributesRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutDedicatedIpWarmupAttributesResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutDeliverabilityDashboardOptionRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutDeliverabilityDashboardOptionResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutEmailIdentityDkimAttributesRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutEmailIdentityDkimAttributesResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutEmailIdentityFeedbackAttributesRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutEmailIdentityFeedbackAttributesResponse;
import software.amazon.awssdk.services.pinpointemail.model.PutEmailIdentityMailFromAttributesRequest;
import software.amazon.awssdk.services.pinpointemail.model.PutEmailIdentityMailFromAttributesResponse;
import software.amazon.awssdk.services.pinpointemail.model.SendEmailRequest;
import software.amazon.awssdk.services.pinpointemail.model.SendEmailResponse;
import software.amazon.awssdk.services.pinpointemail.model.SendingPausedException;
import software.amazon.awssdk.services.pinpointemail.model.TagResourceRequest;
import software.amazon.awssdk.services.pinpointemail.model.TagResourceResponse;
import software.amazon.awssdk.services.pinpointemail.model.TooManyRequestsException;
import software.amazon.awssdk.services.pinpointemail.model.UntagResourceRequest;
import software.amazon.awssdk.services.pinpointemail.model.UntagResourceResponse;
import software.amazon.awssdk.services.pinpointemail.model.UpdateConfigurationSetEventDestinationRequest;
import software.amazon.awssdk.services.pinpointemail.model.UpdateConfigurationSetEventDestinationResponse;
import software.amazon.awssdk.services.pinpointemail.transform.CreateConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.CreateConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.CreateDedicatedIpPoolRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.CreateDeliverabilityTestReportRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.CreateEmailIdentityRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.DeleteConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.DeleteConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.DeleteDedicatedIpPoolRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.DeleteEmailIdentityRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetAccountRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetBlacklistReportsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetConfigurationSetEventDestinationsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetConfigurationSetRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetDedicatedIpRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetDedicatedIpsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetDeliverabilityDashboardOptionsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetDeliverabilityTestReportRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetDomainDeliverabilityCampaignRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetDomainStatisticsReportRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.GetEmailIdentityRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.ListConfigurationSetsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.ListDedicatedIpPoolsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.ListDeliverabilityTestReportsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.ListDomainDeliverabilityCampaignsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.ListEmailIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutAccountDedicatedIpWarmupAttributesRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutAccountSendingAttributesRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutConfigurationSetDeliveryOptionsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutConfigurationSetReputationOptionsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutConfigurationSetSendingOptionsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutConfigurationSetTrackingOptionsRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutDedicatedIpInPoolRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutDedicatedIpWarmupAttributesRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutDeliverabilityDashboardOptionRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutEmailIdentityDkimAttributesRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutEmailIdentityFeedbackAttributesRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.PutEmailIdentityMailFromAttributesRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.SendEmailRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.pinpointemail.transform.UpdateConfigurationSetEventDestinationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PinpointEmailAsyncClient}.
 *
 * @see PinpointEmailAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPinpointEmailAsyncClient implements PinpointEmailAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPinpointEmailAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPinpointEmailAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a configuration set. <i>Configuration sets</i> are groups of rules that you can apply to the emails you
     * send using Amazon Pinpoint. You apply a configuration set to an email by including a reference to the
     * configuration set in the headers of the email. When you apply a configuration set to an email, all of the rules
     * in that configuration set are applied to the email.
     * </p>
     *
     * @param createConfigurationSetRequest
     *        A request to create a configuration set.
     * @return A Java Future containing the result of the CreateConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.CreateConfigurationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/CreateConfigurationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationSetResponse> createConfigurationSet(
            CreateConfigurationSetRequest createConfigurationSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationSetRequest, CreateConfigurationSetResponse>()
                            .withOperationName("CreateConfigurationSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigurationSetRequest));
            CompletableFuture<CreateConfigurationSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an event destination. In Amazon Pinpoint, <i>events</i> include message sends, deliveries, opens, clicks,
     * bounces, and complaints. <i>Event destinations</i> are places that you can send information about these events
     * to. For example, you can send event data to Amazon SNS to receive notifications when you receive bounces or
     * complaints, or you can use Amazon Kinesis Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     * <p>
     * A single configuration set can include more than one event destination.
     * </p>
     *
     * @param createConfigurationSetEventDestinationRequest
     *        A request to add an event destination to a configuration set.
     * @return A Java Future containing the result of the CreateConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.CreateConfigurationSetEventDestination
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/CreateConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationSetEventDestinationResponse> createConfigurationSetEventDestination(
            CreateConfigurationSetEventDestinationRequest createConfigurationSetEventDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationSetEventDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createConfigurationSetEventDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationSetEventDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationSetEventDestinationRequest, CreateConfigurationSetEventDestinationResponse>()
                            .withOperationName("CreateConfigurationSetEventDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigurationSetEventDestinationRequest));
            CompletableFuture<CreateConfigurationSetEventDestinationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new pool of dedicated IP addresses. A pool can include one or more dedicated IP addresses that are
     * associated with your Amazon Pinpoint account. You can associate a pool with a configuration set. When you send an
     * email that uses that configuration set, Amazon Pinpoint sends it using only the IP addresses in the associated
     * pool.
     * </p>
     *
     * @param createDedicatedIpPoolRequest
     *        A request to create a new dedicated IP pool.
     * @return A Java Future containing the result of the CreateDedicatedIpPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.CreateDedicatedIpPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/CreateDedicatedIpPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDedicatedIpPoolResponse> createDedicatedIpPool(
            CreateDedicatedIpPoolRequest createDedicatedIpPoolRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDedicatedIpPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDedicatedIpPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDedicatedIpPool");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDedicatedIpPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDedicatedIpPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDedicatedIpPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDedicatedIpPoolRequest, CreateDedicatedIpPoolResponse>()
                            .withOperationName("CreateDedicatedIpPool").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDedicatedIpPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDedicatedIpPoolRequest));
            CompletableFuture<CreateDedicatedIpPoolResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new predictive inbox placement test. Predictive inbox placement tests can help you predict how your
     * messages will be handled by various email providers around the world. When you perform a predictive inbox
     * placement test, you provide a sample message that contains the content that you plan to send to your customers.
     * Amazon Pinpoint then sends that message to special email addresses spread across several major email providers.
     * After about 24 hours, the test is complete, and you can use the <code>GetDeliverabilityTestReport</code>
     * operation to view the results of the test.
     * </p>
     *
     * @param createDeliverabilityTestReportRequest
     *        A request to perform a predictive inbox placement test. Predictive inbox placement tests can help you
     *        predict how your messages will be handled by various email providers around the world. When you perform a
     *        predictive inbox placement test, you provide a sample message that contains the content that you plan to
     *        send to your customers. Amazon Pinpoint then sends that message to special email addresses spread across
     *        several major email providers. After about 24 hours, the test is complete, and you can use the
     *        <code>GetDeliverabilityTestReport</code> operation to view the results of the test.
     * @return A Java Future containing the result of the CreateDeliverabilityTestReport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccountSuspendedException The message can't be sent because the account's ability to send email has
     *         been permanently restricted.</li>
     *         <li>SendingPausedException The message can't be sent because the account's ability to send email is
     *         currently paused.</li>
     *         <li>MessageRejectedException The message can't be sent because it contains invalid content.</li>
     *         <li>MailFromDomainNotVerifiedException The message can't be sent because the sending domain isn't
     *         verified.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.CreateDeliverabilityTestReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/CreateDeliverabilityTestReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeliverabilityTestReportResponse> createDeliverabilityTestReport(
            CreateDeliverabilityTestReportRequest createDeliverabilityTestReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDeliverabilityTestReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDeliverabilityTestReportRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeliverabilityTestReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeliverabilityTestReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeliverabilityTestReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDeliverabilityTestReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeliverabilityTestReportRequest, CreateDeliverabilityTestReportResponse>()
                            .withOperationName("CreateDeliverabilityTestReport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDeliverabilityTestReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDeliverabilityTestReportRequest));
            CompletableFuture<CreateDeliverabilityTestReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Verifies an email identity for use with Amazon Pinpoint. In Amazon Pinpoint, an identity is an email address or
     * domain that you use when you send email. Before you can use an identity to send email with Amazon Pinpoint, you
     * first have to verify it. By verifying an address, you demonstrate that you're the owner of the address, and that
     * you've given Amazon Pinpoint permission to send email from the address.
     * </p>
     * <p>
     * When you verify an email address, Amazon Pinpoint sends an email to the address. Your email address is verified
     * as soon as you follow the link in the verification email.
     * </p>
     * <p>
     * When you verify a domain, this operation provides a set of DKIM tokens, which you can convert into CNAME tokens.
     * You add these CNAME tokens to the DNS configuration for your domain. Your domain is verified when Amazon Pinpoint
     * detects these records in the DNS configuration for your domain. It usually takes around 72 hours to complete the
     * domain verification process.
     * </p>
     *
     * @param createEmailIdentityRequest
     *        A request to begin the verification process for an email identity (an email address or domain).
     * @return A Java Future containing the result of the CreateEmailIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.CreateEmailIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/CreateEmailIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEmailIdentityResponse> createEmailIdentity(
            CreateEmailIdentityRequest createEmailIdentityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEmailIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEmailIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEmailIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEmailIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEmailIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEmailIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEmailIdentityRequest, CreateEmailIdentityResponse>()
                            .withOperationName("CreateEmailIdentity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEmailIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEmailIdentityRequest));
            CompletableFuture<CreateEmailIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an existing configuration set.
     * </p>
     * <p>
     * In Amazon Pinpoint, <i>configuration sets</i> are groups of rules that you can apply to the emails you send. You
     * apply a configuration set to an email by including a reference to the configuration set in the headers of the
     * email. When you apply a configuration set to an email, all of the rules in that configuration set are applied to
     * the email.
     * </p>
     *
     * @param deleteConfigurationSetRequest
     *        A request to delete a configuration set.
     * @return A Java Future containing the result of the DeleteConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.DeleteConfigurationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/DeleteConfigurationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationSetResponse> deleteConfigurationSet(
            DeleteConfigurationSetRequest deleteConfigurationSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigurationSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigurationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationSetRequest, DeleteConfigurationSetResponse>()
                            .withOperationName("DeleteConfigurationSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConfigurationSetRequest));
            CompletableFuture<DeleteConfigurationSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an event destination.
     * </p>
     * <p>
     * In Amazon Pinpoint, <i>events</i> include message sends, deliveries, opens, clicks, bounces, and complaints.
     * <i>Event destinations</i> are places that you can send information about these events to. For example, you can
     * send event data to Amazon SNS to receive notifications when you receive bounces or complaints, or you can use
     * Amazon Kinesis Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     *
     * @param deleteConfigurationSetEventDestinationRequest
     *        A request to delete an event destination from a configuration set.
     * @return A Java Future containing the result of the DeleteConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.DeleteConfigurationSetEventDestination
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/DeleteConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationSetEventDestinationResponse> deleteConfigurationSetEventDestination(
            DeleteConfigurationSetEventDestinationRequest deleteConfigurationSetEventDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigurationSetEventDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteConfigurationSetEventDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationSetEventDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationSetEventDestinationRequest, DeleteConfigurationSetEventDestinationResponse>()
                            .withOperationName("DeleteConfigurationSetEventDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConfigurationSetEventDestinationRequest));
            CompletableFuture<DeleteConfigurationSetEventDestinationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a dedicated IP pool.
     * </p>
     *
     * @param deleteDedicatedIpPoolRequest
     *        A request to delete a dedicated IP pool.
     * @return A Java Future containing the result of the DeleteDedicatedIpPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.DeleteDedicatedIpPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/DeleteDedicatedIpPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDedicatedIpPoolResponse> deleteDedicatedIpPool(
            DeleteDedicatedIpPoolRequest deleteDedicatedIpPoolRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDedicatedIpPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDedicatedIpPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDedicatedIpPool");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDedicatedIpPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDedicatedIpPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDedicatedIpPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDedicatedIpPoolRequest, DeleteDedicatedIpPoolResponse>()
                            .withOperationName("DeleteDedicatedIpPool").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDedicatedIpPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDedicatedIpPoolRequest));
            CompletableFuture<DeleteDedicatedIpPoolResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an email identity that you previously verified for use with Amazon Pinpoint. An identity can be either an
     * email address or a domain name.
     * </p>
     *
     * @param deleteEmailIdentityRequest
     *        A request to delete an existing email identity. When you delete an identity, you lose the ability to use
     *        Amazon Pinpoint to send email from that identity. You can restore your ability to send email by completing
     *        the verification process for the identity again.
     * @return A Java Future containing the result of the DeleteEmailIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.DeleteEmailIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/DeleteEmailIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEmailIdentityResponse> deleteEmailIdentity(
            DeleteEmailIdentityRequest deleteEmailIdentityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEmailIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEmailIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEmailIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEmailIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEmailIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEmailIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEmailIdentityRequest, DeleteEmailIdentityResponse>()
                            .withOperationName("DeleteEmailIdentity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEmailIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEmailIdentityRequest));
            CompletableFuture<DeleteEmailIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtain information about the email-sending status and capabilities of your Amazon Pinpoint account in the current
     * AWS Region.
     * </p>
     *
     * @param getAccountRequest
     *        A request to obtain information about the email-sending capabilities of your Amazon Pinpoint account.
     * @return A Java Future containing the result of the GetAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountResponse> getAccount(GetAccountRequest getAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountRequest, GetAccountResponse>().withOperationName("GetAccount")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountRequest));
            CompletableFuture<GetAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve a list of the blacklists that your dedicated IP addresses appear on.
     * </p>
     *
     * @param getBlacklistReportsRequest
     *        A request to retrieve a list of the blacklists that your dedicated IP addresses appear on.
     * @return A Java Future containing the result of the GetBlacklistReports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetBlacklistReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetBlacklistReports"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBlacklistReportsResponse> getBlacklistReports(
            GetBlacklistReportsRequest getBlacklistReportsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBlacklistReportsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBlacklistReportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBlacklistReports");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBlacklistReportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBlacklistReportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBlacklistReportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBlacklistReportsRequest, GetBlacklistReportsResponse>()
                            .withOperationName("GetBlacklistReports").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBlacklistReportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBlacklistReportsRequest));
            CompletableFuture<GetBlacklistReportsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about an existing configuration set, including the dedicated IP pool that it's associated with,
     * whether or not it's enabled for sending email, and more.
     * </p>
     * <p>
     * In Amazon Pinpoint, <i>configuration sets</i> are groups of rules that you can apply to the emails you send. You
     * apply a configuration set to an email by including a reference to the configuration set in the headers of the
     * email. When you apply a configuration set to an email, all of the rules in that configuration set are applied to
     * the email.
     * </p>
     *
     * @param getConfigurationSetRequest
     *        A request to obtain information about a configuration set.
     * @return A Java Future containing the result of the GetConfigurationSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetConfigurationSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetConfigurationSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationSetResponse> getConfigurationSet(
            GetConfigurationSetRequest getConfigurationSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfigurationSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationSetRequest, GetConfigurationSetResponse>()
                            .withOperationName("GetConfigurationSet").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConfigurationSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConfigurationSetRequest));
            CompletableFuture<GetConfigurationSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve a list of event destinations that are associated with a configuration set.
     * </p>
     * <p>
     * In Amazon Pinpoint, <i>events</i> include message sends, deliveries, opens, clicks, bounces, and complaints.
     * <i>Event destinations</i> are places that you can send information about these events to. For example, you can
     * send event data to Amazon SNS to receive notifications when you receive bounces or complaints, or you can use
     * Amazon Kinesis Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     *
     * @param getConfigurationSetEventDestinationsRequest
     *        A request to obtain information about the event destinations for a configuration set.
     * @return A Java Future containing the result of the GetConfigurationSetEventDestinations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetConfigurationSetEventDestinations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetConfigurationSetEventDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationSetEventDestinationsResponse> getConfigurationSetEventDestinations(
            GetConfigurationSetEventDestinationsRequest getConfigurationSetEventDestinationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationSetEventDestinationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getConfigurationSetEventDestinationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationSetEventDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationSetEventDestinationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetConfigurationSetEventDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationSetEventDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationSetEventDestinationsRequest, GetConfigurationSetEventDestinationsResponse>()
                            .withOperationName("GetConfigurationSetEventDestinations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConfigurationSetEventDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConfigurationSetEventDestinationsRequest));
            CompletableFuture<GetConfigurationSetEventDestinationsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about a dedicated IP address, including the name of the dedicated IP pool that it's associated
     * with, as well information about the automatic warm-up process for the address.
     * </p>
     *
     * @param getDedicatedIpRequest
     *        A request to obtain more information about a dedicated IP address.
     * @return A Java Future containing the result of the GetDedicatedIp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetDedicatedIp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetDedicatedIp" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDedicatedIpResponse> getDedicatedIp(GetDedicatedIpRequest getDedicatedIpRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDedicatedIpRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDedicatedIpRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDedicatedIp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDedicatedIpResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDedicatedIpResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDedicatedIpResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDedicatedIpRequest, GetDedicatedIpResponse>()
                            .withOperationName("GetDedicatedIp").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDedicatedIpRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDedicatedIpRequest));
            CompletableFuture<GetDedicatedIpResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the dedicated IP addresses that are associated with your Amazon Pinpoint account.
     * </p>
     *
     * @param getDedicatedIpsRequest
     *        A request to obtain more information about dedicated IP pools.
     * @return A Java Future containing the result of the GetDedicatedIps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetDedicatedIps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetDedicatedIps"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDedicatedIpsResponse> getDedicatedIps(GetDedicatedIpsRequest getDedicatedIpsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDedicatedIpsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDedicatedIpsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDedicatedIps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDedicatedIpsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDedicatedIpsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDedicatedIpsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDedicatedIpsRequest, GetDedicatedIpsResponse>()
                            .withOperationName("GetDedicatedIps").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDedicatedIpsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDedicatedIpsRequest));
            CompletableFuture<GetDedicatedIpsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve information about the status of the Deliverability dashboard for your Amazon Pinpoint account. When the
     * Deliverability dashboard is enabled, you gain access to reputation, deliverability, and other metrics for the
     * domains that you use to send email using Amazon Pinpoint. You also gain the ability to perform predictive inbox
     * placement tests.
     * </p>
     * <p>
     * When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other fees
     * that you accrue by using Amazon Pinpoint. For more information about the features and cost of a Deliverability
     * dashboard subscription, see <a href="http://aws.amazon.com/pinpoint/pricing/">Amazon Pinpoint Pricing</a>.
     * </p>
     *
     * @param getDeliverabilityDashboardOptionsRequest
     *        Retrieve information about the status of the Deliverability dashboard for your Amazon Pinpoint account.
     *        When the Deliverability dashboard is enabled, you gain access to reputation, deliverability, and other
     *        metrics for the domains that you use to send email using Amazon Pinpoint. You also gain the ability to
     *        perform predictive inbox placement tests.</p>
     *        <p>
     *        When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other
     *        fees that you accrue by using Amazon Pinpoint. For more information about the features and cost of a
     *        Deliverability dashboard subscription, see <a href="http://aws.amazon.com/pinpoint/pricing/">Amazon
     *        Pinpoint Pricing</a>.
     * @return A Java Future containing the result of the GetDeliverabilityDashboardOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li> <li>
     *         LimitExceededException There are too many instances of the specified resource type.</li> <li>
     *         BadRequestException The input you provided is invalid.</li> <li>SdkException Base class for all
     *         exceptions that can be thrown by the SDK (both service and client). Can be used for catch all scenarios.
     *         </li> <li>SdkClientException If any client side error occurs such as an IO related failure, failure to
     *         get credentials, etc.</li> <li>PinpointEmailException Base class for all service exceptions. Unknown
     *         exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetDeliverabilityDashboardOptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetDeliverabilityDashboardOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeliverabilityDashboardOptionsResponse> getDeliverabilityDashboardOptions(
            GetDeliverabilityDashboardOptionsRequest getDeliverabilityDashboardOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeliverabilityDashboardOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDeliverabilityDashboardOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeliverabilityDashboardOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeliverabilityDashboardOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetDeliverabilityDashboardOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeliverabilityDashboardOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeliverabilityDashboardOptionsRequest, GetDeliverabilityDashboardOptionsResponse>()
                            .withOperationName("GetDeliverabilityDashboardOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDeliverabilityDashboardOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDeliverabilityDashboardOptionsRequest));
            CompletableFuture<GetDeliverabilityDashboardOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the results of a predictive inbox placement test.
     * </p>
     *
     * @param getDeliverabilityTestReportRequest
     *        A request to retrieve the results of a predictive inbox placement test.
     * @return A Java Future containing the result of the GetDeliverabilityTestReport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetDeliverabilityTestReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetDeliverabilityTestReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeliverabilityTestReportResponse> getDeliverabilityTestReport(
            GetDeliverabilityTestReportRequest getDeliverabilityTestReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeliverabilityTestReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeliverabilityTestReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeliverabilityTestReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeliverabilityTestReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDeliverabilityTestReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeliverabilityTestReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeliverabilityTestReportRequest, GetDeliverabilityTestReportResponse>()
                            .withOperationName("GetDeliverabilityTestReport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDeliverabilityTestReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDeliverabilityTestReportRequest));
            CompletableFuture<GetDeliverabilityTestReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve all the deliverability data for a specific campaign. This data is available for a campaign only if the
     * campaign sent email by using a domain that the Deliverability dashboard is enabled for (
     * <code>PutDeliverabilityDashboardOption</code> operation).
     * </p>
     *
     * @param getDomainDeliverabilityCampaignRequest
     *        Retrieve all the deliverability data for a specific campaign. This data is available for a campaign only
     *        if the campaign sent email by using a domain that the Deliverability dashboard is enabled for (
     *        <code>PutDeliverabilityDashboardOption</code> operation).
     * @return A Java Future containing the result of the GetDomainDeliverabilityCampaign operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetDomainDeliverabilityCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetDomainDeliverabilityCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainDeliverabilityCampaignResponse> getDomainDeliverabilityCampaign(
            GetDomainDeliverabilityCampaignRequest getDomainDeliverabilityCampaignRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDomainDeliverabilityCampaignRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDomainDeliverabilityCampaignRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainDeliverabilityCampaign");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainDeliverabilityCampaignResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainDeliverabilityCampaignResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainDeliverabilityCampaignResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainDeliverabilityCampaignRequest, GetDomainDeliverabilityCampaignResponse>()
                            .withOperationName("GetDomainDeliverabilityCampaign").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDomainDeliverabilityCampaignRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDomainDeliverabilityCampaignRequest));
            CompletableFuture<GetDomainDeliverabilityCampaignResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve inbox placement and engagement rates for the domains that you use to send email.
     * </p>
     *
     * @param getDomainStatisticsReportRequest
     *        A request to obtain deliverability metrics for a domain.
     * @return A Java Future containing the result of the GetDomainStatisticsReport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetDomainStatisticsReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetDomainStatisticsReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainStatisticsReportResponse> getDomainStatisticsReport(
            GetDomainStatisticsReportRequest getDomainStatisticsReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDomainStatisticsReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainStatisticsReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainStatisticsReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainStatisticsReportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainStatisticsReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainStatisticsReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainStatisticsReportRequest, GetDomainStatisticsReportResponse>()
                            .withOperationName("GetDomainStatisticsReport").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDomainStatisticsReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDomainStatisticsReportRequest));
            CompletableFuture<GetDomainStatisticsReportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about a specific identity associated with your Amazon Pinpoint account, including the
     * identity's verification status, its DKIM authentication status, and its custom Mail-From settings.
     * </p>
     *
     * @param getEmailIdentityRequest
     *        A request to return details about an email identity.
     * @return A Java Future containing the result of the GetEmailIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.GetEmailIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/GetEmailIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEmailIdentityResponse> getEmailIdentity(GetEmailIdentityRequest getEmailIdentityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEmailIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEmailIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEmailIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEmailIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEmailIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEmailIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEmailIdentityRequest, GetEmailIdentityResponse>()
                            .withOperationName("GetEmailIdentity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEmailIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEmailIdentityRequest));
            CompletableFuture<GetEmailIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all of the configuration sets associated with your Amazon Pinpoint account in the current region.
     * </p>
     * <p>
     * In Amazon Pinpoint, <i>configuration sets</i> are groups of rules that you can apply to the emails you send. You
     * apply a configuration set to an email by including a reference to the configuration set in the headers of the
     * email. When you apply a configuration set to an email, all of the rules in that configuration set are applied to
     * the email.
     * </p>
     *
     * @param listConfigurationSetsRequest
     *        A request to obtain a list of configuration sets for your Amazon Pinpoint account in the current AWS
     *        Region.
     * @return A Java Future containing the result of the ListConfigurationSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.ListConfigurationSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/ListConfigurationSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationSetsResponse> listConfigurationSets(
            ListConfigurationSetsRequest listConfigurationSetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationSetsRequest, ListConfigurationSetsResponse>()
                            .withOperationName("ListConfigurationSets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConfigurationSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConfigurationSetsRequest));
            CompletableFuture<ListConfigurationSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all of the dedicated IP pools that exist in your Amazon Pinpoint account in the current AWS Region.
     * </p>
     *
     * @param listDedicatedIpPoolsRequest
     *        A request to obtain a list of dedicated IP pools.
     * @return A Java Future containing the result of the ListDedicatedIpPools operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.ListDedicatedIpPools
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/ListDedicatedIpPools"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDedicatedIpPoolsResponse> listDedicatedIpPools(
            ListDedicatedIpPoolsRequest listDedicatedIpPoolsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDedicatedIpPoolsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDedicatedIpPoolsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDedicatedIpPools");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDedicatedIpPoolsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDedicatedIpPoolsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDedicatedIpPoolsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDedicatedIpPoolsRequest, ListDedicatedIpPoolsResponse>()
                            .withOperationName("ListDedicatedIpPools").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDedicatedIpPoolsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDedicatedIpPoolsRequest));
            CompletableFuture<ListDedicatedIpPoolsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Show a list of the predictive inbox placement tests that you've performed, regardless of their statuses. For
     * predictive inbox placement tests that are complete, you can use the <code>GetDeliverabilityTestReport</code>
     * operation to view the results.
     * </p>
     *
     * @param listDeliverabilityTestReportsRequest
     *        A request to list all of the predictive inbox placement tests that you've performed.
     * @return A Java Future containing the result of the ListDeliverabilityTestReports operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.ListDeliverabilityTestReports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/ListDeliverabilityTestReports"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeliverabilityTestReportsResponse> listDeliverabilityTestReports(
            ListDeliverabilityTestReportsRequest listDeliverabilityTestReportsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeliverabilityTestReportsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listDeliverabilityTestReportsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeliverabilityTestReports");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeliverabilityTestReportsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeliverabilityTestReportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeliverabilityTestReportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeliverabilityTestReportsRequest, ListDeliverabilityTestReportsResponse>()
                            .withOperationName("ListDeliverabilityTestReports").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDeliverabilityTestReportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDeliverabilityTestReportsRequest));
            CompletableFuture<ListDeliverabilityTestReportsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve deliverability data for all the campaigns that used a specific domain to send email during a specified
     * time range. This data is available for a domain only if you enabled the Deliverability dashboard (
     * <code>PutDeliverabilityDashboardOption</code> operation) for the domain.
     * </p>
     *
     * @param listDomainDeliverabilityCampaignsRequest
     *        Retrieve deliverability data for all the campaigns that used a specific domain to send email during a
     *        specified time range. This data is available for a domain only if you enabled the Deliverability dashboard
     *        (<code>PutDeliverabilityDashboardOption</code> operation) for the domain.
     * @return A Java Future containing the result of the ListDomainDeliverabilityCampaigns operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.ListDomainDeliverabilityCampaigns
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/ListDomainDeliverabilityCampaigns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainDeliverabilityCampaignsResponse> listDomainDeliverabilityCampaigns(
            ListDomainDeliverabilityCampaignsRequest listDomainDeliverabilityCampaignsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDomainDeliverabilityCampaignsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listDomainDeliverabilityCampaignsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomainDeliverabilityCampaigns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainDeliverabilityCampaignsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListDomainDeliverabilityCampaignsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainDeliverabilityCampaignsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainDeliverabilityCampaignsRequest, ListDomainDeliverabilityCampaignsResponse>()
                            .withOperationName("ListDomainDeliverabilityCampaigns").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDomainDeliverabilityCampaignsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDomainDeliverabilityCampaignsRequest));
            CompletableFuture<ListDomainDeliverabilityCampaignsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all of the email identities that are associated with your Amazon Pinpoint account. An identity
     * can be either an email address or a domain. This operation returns identities that are verified as well as those
     * that aren't.
     * </p>
     *
     * @param listEmailIdentitiesRequest
     *        A request to list all of the email identities associated with your Amazon Pinpoint account. This list
     *        includes identities that you've already verified, identities that are unverified, and identities that were
     *        verified in the past, but are no longer verified.
     * @return A Java Future containing the result of the ListEmailIdentities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.ListEmailIdentities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/ListEmailIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEmailIdentitiesResponse> listEmailIdentities(
            ListEmailIdentitiesRequest listEmailIdentitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEmailIdentitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEmailIdentitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEmailIdentities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEmailIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEmailIdentitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEmailIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEmailIdentitiesRequest, ListEmailIdentitiesResponse>()
                            .withOperationName("ListEmailIdentities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEmailIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEmailIdentitiesRequest));
            CompletableFuture<ListEmailIdentitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve a list of the tags (keys and values) that are associated with a specified resource. A <i>tag</i> is a
     * label that you optionally define and associate with a resource in Amazon Pinpoint. Each tag consists of a
     * required <i>tag key</i> and an optional associated <i>tag value</i>. A tag key is a general label that acts as a
     * category for more specific tag values. A tag value acts as a descriptor within a tag key.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable the automatic warm-up feature for dedicated IP addresses.
     * </p>
     *
     * @param putAccountDedicatedIpWarmupAttributesRequest
     *        A request to enable or disable the automatic IP address warm-up feature.
     * @return A Java Future containing the result of the PutAccountDedicatedIpWarmupAttributes operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutAccountDedicatedIpWarmupAttributes
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutAccountDedicatedIpWarmupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccountDedicatedIpWarmupAttributesResponse> putAccountDedicatedIpWarmupAttributes(
            PutAccountDedicatedIpWarmupAttributesRequest putAccountDedicatedIpWarmupAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAccountDedicatedIpWarmupAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putAccountDedicatedIpWarmupAttributesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountDedicatedIpWarmupAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAccountDedicatedIpWarmupAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutAccountDedicatedIpWarmupAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAccountDedicatedIpWarmupAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccountDedicatedIpWarmupAttributesRequest, PutAccountDedicatedIpWarmupAttributesResponse>()
                            .withOperationName("PutAccountDedicatedIpWarmupAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutAccountDedicatedIpWarmupAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAccountDedicatedIpWarmupAttributesRequest));
            CompletableFuture<PutAccountDedicatedIpWarmupAttributesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable the ability of your account to send email.
     * </p>
     *
     * @param putAccountSendingAttributesRequest
     *        A request to change the ability of your account to send email.
     * @return A Java Future containing the result of the PutAccountSendingAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutAccountSendingAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutAccountSendingAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccountSendingAttributesResponse> putAccountSendingAttributes(
            PutAccountSendingAttributesRequest putAccountSendingAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAccountSendingAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccountSendingAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountSendingAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAccountSendingAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAccountSendingAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAccountSendingAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccountSendingAttributesRequest, PutAccountSendingAttributesResponse>()
                            .withOperationName("PutAccountSendingAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutAccountSendingAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAccountSendingAttributesRequest));
            CompletableFuture<PutAccountSendingAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associate a configuration set with a dedicated IP pool. You can use dedicated IP pools to create groups of
     * dedicated IP addresses for sending specific types of email.
     * </p>
     *
     * @param putConfigurationSetDeliveryOptionsRequest
     *        A request to associate a configuration set with a dedicated IP pool.
     * @return A Java Future containing the result of the PutConfigurationSetDeliveryOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutConfigurationSetDeliveryOptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutConfigurationSetDeliveryOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetDeliveryOptionsResponse> putConfigurationSetDeliveryOptions(
            PutConfigurationSetDeliveryOptionsRequest putConfigurationSetDeliveryOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putConfigurationSetDeliveryOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putConfigurationSetDeliveryOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutConfigurationSetDeliveryOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetDeliveryOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetDeliveryOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetDeliveryOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetDeliveryOptionsRequest, PutConfigurationSetDeliveryOptionsResponse>()
                            .withOperationName("PutConfigurationSetDeliveryOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutConfigurationSetDeliveryOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putConfigurationSetDeliveryOptionsRequest));
            CompletableFuture<PutConfigurationSetDeliveryOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable collection of reputation metrics for emails that you send using a particular configuration set
     * in a specific AWS Region.
     * </p>
     *
     * @param putConfigurationSetReputationOptionsRequest
     *        A request to enable or disable tracking of reputation metrics for a configuration set.
     * @return A Java Future containing the result of the PutConfigurationSetReputationOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutConfigurationSetReputationOptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutConfigurationSetReputationOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetReputationOptionsResponse> putConfigurationSetReputationOptions(
            PutConfigurationSetReputationOptionsRequest putConfigurationSetReputationOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putConfigurationSetReputationOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putConfigurationSetReputationOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutConfigurationSetReputationOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetReputationOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetReputationOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetReputationOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetReputationOptionsRequest, PutConfigurationSetReputationOptionsResponse>()
                            .withOperationName("PutConfigurationSetReputationOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutConfigurationSetReputationOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putConfigurationSetReputationOptionsRequest));
            CompletableFuture<PutConfigurationSetReputationOptionsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable email sending for messages that use a particular configuration set in a specific AWS Region.
     * </p>
     *
     * @param putConfigurationSetSendingOptionsRequest
     *        A request to enable or disable the ability of Amazon Pinpoint to send emails that use a specific
     *        configuration set.
     * @return A Java Future containing the result of the PutConfigurationSetSendingOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutConfigurationSetSendingOptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutConfigurationSetSendingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetSendingOptionsResponse> putConfigurationSetSendingOptions(
            PutConfigurationSetSendingOptionsRequest putConfigurationSetSendingOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putConfigurationSetSendingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putConfigurationSetSendingOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutConfigurationSetSendingOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetSendingOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetSendingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetSendingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetSendingOptionsRequest, PutConfigurationSetSendingOptionsResponse>()
                            .withOperationName("PutConfigurationSetSendingOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutConfigurationSetSendingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putConfigurationSetSendingOptionsRequest));
            CompletableFuture<PutConfigurationSetSendingOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Specify a custom domain to use for open and click tracking elements in email that you send using Amazon Pinpoint.
     * </p>
     *
     * @param putConfigurationSetTrackingOptionsRequest
     *        A request to add a custom domain for tracking open and click events to a configuration set.
     * @return A Java Future containing the result of the PutConfigurationSetTrackingOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutConfigurationSetTrackingOptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutConfigurationSetTrackingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutConfigurationSetTrackingOptionsResponse> putConfigurationSetTrackingOptions(
            PutConfigurationSetTrackingOptionsRequest putConfigurationSetTrackingOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putConfigurationSetTrackingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putConfigurationSetTrackingOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutConfigurationSetTrackingOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutConfigurationSetTrackingOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutConfigurationSetTrackingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutConfigurationSetTrackingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutConfigurationSetTrackingOptionsRequest, PutConfigurationSetTrackingOptionsResponse>()
                            .withOperationName("PutConfigurationSetTrackingOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutConfigurationSetTrackingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putConfigurationSetTrackingOptionsRequest));
            CompletableFuture<PutConfigurationSetTrackingOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Move a dedicated IP address to an existing dedicated IP pool.
     * </p>
     * <note>
     * <p>
     * The dedicated IP address that you specify must already exist, and must be associated with your Amazon Pinpoint
     * account.
     * </p>
     * <p>
     * The dedicated IP pool you specify must already exist. You can create a new pool by using the
     * <code>CreateDedicatedIpPool</code> operation.
     * </p>
     * </note>
     *
     * @param putDedicatedIpInPoolRequest
     *        A request to move a dedicated IP address to a dedicated IP pool.
     * @return A Java Future containing the result of the PutDedicatedIpInPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutDedicatedIpInPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutDedicatedIpInPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDedicatedIpInPoolResponse> putDedicatedIpInPool(
            PutDedicatedIpInPoolRequest putDedicatedIpInPoolRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putDedicatedIpInPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDedicatedIpInPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDedicatedIpInPool");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDedicatedIpInPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDedicatedIpInPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDedicatedIpInPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDedicatedIpInPoolRequest, PutDedicatedIpInPoolResponse>()
                            .withOperationName("PutDedicatedIpInPool").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutDedicatedIpInPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putDedicatedIpInPoolRequest));
            CompletableFuture<PutDedicatedIpInPoolResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p/>
     *
     * @param putDedicatedIpWarmupAttributesRequest
     *        A request to change the warm-up attributes for a dedicated IP address. This operation is useful when you
     *        want to resume the warm-up process for an existing IP address.
     * @return A Java Future containing the result of the PutDedicatedIpWarmupAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li> <li>
     *         TooManyRequestsException Too many requests have been made to the operation.</li> <li>BadRequestException
     *         The input you provided is invalid.</li> <li>SdkException Base class for all exceptions that can be thrown
     *         by the SDK (both service and client). Can be used for catch all scenarios.</li> <li>SdkClientException If
     *         any client side error occurs such as an IO related failure, failure to get credentials, etc.</li> <li>
     *         PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutDedicatedIpWarmupAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutDedicatedIpWarmupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDedicatedIpWarmupAttributesResponse> putDedicatedIpWarmupAttributes(
            PutDedicatedIpWarmupAttributesRequest putDedicatedIpWarmupAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putDedicatedIpWarmupAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putDedicatedIpWarmupAttributesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDedicatedIpWarmupAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDedicatedIpWarmupAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDedicatedIpWarmupAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDedicatedIpWarmupAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDedicatedIpWarmupAttributesRequest, PutDedicatedIpWarmupAttributesResponse>()
                            .withOperationName("PutDedicatedIpWarmupAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutDedicatedIpWarmupAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putDedicatedIpWarmupAttributesRequest));
            CompletableFuture<PutDedicatedIpWarmupAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable or disable the Deliverability dashboard for your Amazon Pinpoint account. When you enable the
     * Deliverability dashboard, you gain access to reputation, deliverability, and other metrics for the domains that
     * you use to send email using Amazon Pinpoint. You also gain the ability to perform predictive inbox placement
     * tests.
     * </p>
     * <p>
     * When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other fees
     * that you accrue by using Amazon Pinpoint. For more information about the features and cost of a Deliverability
     * dashboard subscription, see <a href="http://aws.amazon.com/pinpoint/pricing/">Amazon Pinpoint Pricing</a>.
     * </p>
     *
     * @param putDeliverabilityDashboardOptionRequest
     *        Enable or disable the Deliverability dashboard for your Amazon Pinpoint account. When you enable the
     *        Deliverability dashboard, you gain access to reputation, deliverability, and other metrics for the domains
     *        that you use to send email using Amazon Pinpoint. You also gain the ability to perform predictive inbox
     *        placement tests.</p>
     *        <p>
     *        When you use the Deliverability dashboard, you pay a monthly subscription charge, in addition to any other
     *        fees that you accrue by using Amazon Pinpoint. For more information about the features and cost of a
     *        Deliverability dashboard subscription, see <a href="http://aws.amazon.com/pinpoint/pricing/">Amazon
     *        Pinpoint Pricing</a>.
     * @return A Java Future containing the result of the PutDeliverabilityDashboardOption operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AlreadyExistsException The resource specified in your request already exists.</li> <li>
     *         NotFoundException The resource you attempted to access doesn't exist.</li> <li>TooManyRequestsException
     *         Too many requests have been made to the operation.</li> <li>LimitExceededException There are too many
     *         instances of the specified resource type.</li> <li>BadRequestException The input you provided is invalid.
     *         </li> <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li> <li>SdkClientException If any client side error occurs
     *         such as an IO related failure, failure to get credentials, etc.</li> <li>PinpointEmailException Base
     *         class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutDeliverabilityDashboardOption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutDeliverabilityDashboardOption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDeliverabilityDashboardOptionResponse> putDeliverabilityDashboardOption(
            PutDeliverabilityDashboardOptionRequest putDeliverabilityDashboardOptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putDeliverabilityDashboardOptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putDeliverabilityDashboardOptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDeliverabilityDashboardOption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDeliverabilityDashboardOptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutDeliverabilityDashboardOptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDeliverabilityDashboardOptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDeliverabilityDashboardOptionRequest, PutDeliverabilityDashboardOptionResponse>()
                            .withOperationName("PutDeliverabilityDashboardOption").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutDeliverabilityDashboardOptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putDeliverabilityDashboardOptionRequest));
            CompletableFuture<PutDeliverabilityDashboardOptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to enable or disable DKIM authentication for an email identity.
     * </p>
     *
     * @param putEmailIdentityDkimAttributesRequest
     *        A request to enable or disable DKIM signing of email that you send from an email identity.
     * @return A Java Future containing the result of the PutEmailIdentityDkimAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutEmailIdentityDkimAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutEmailIdentityDkimAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEmailIdentityDkimAttributesResponse> putEmailIdentityDkimAttributes(
            PutEmailIdentityDkimAttributesRequest putEmailIdentityDkimAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putEmailIdentityDkimAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putEmailIdentityDkimAttributesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEmailIdentityDkimAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEmailIdentityDkimAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutEmailIdentityDkimAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEmailIdentityDkimAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEmailIdentityDkimAttributesRequest, PutEmailIdentityDkimAttributesResponse>()
                            .withOperationName("PutEmailIdentityDkimAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutEmailIdentityDkimAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putEmailIdentityDkimAttributesRequest));
            CompletableFuture<PutEmailIdentityDkimAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to enable or disable feedback forwarding for an identity. This setting determines what happens when an
     * identity is used to send an email that results in a bounce or complaint event.
     * </p>
     * <p>
     * When you enable feedback forwarding, Amazon Pinpoint sends you email notifications when bounce or complaint
     * events occur. Amazon Pinpoint sends this notification to the address that you specified in the Return-Path header
     * of the original email.
     * </p>
     * <p>
     * When you disable feedback forwarding, Amazon Pinpoint sends notifications through other mechanisms, such as by
     * notifying an Amazon SNS topic. You're required to have a method of tracking bounces and complaints. If you
     * haven't set up another mechanism for receiving bounce or complaint notifications, Amazon Pinpoint sends an email
     * notification when these events occur (even if this setting is disabled).
     * </p>
     *
     * @param putEmailIdentityFeedbackAttributesRequest
     *        A request to set the attributes that control how bounce and complaint events are processed.
     * @return A Java Future containing the result of the PutEmailIdentityFeedbackAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutEmailIdentityFeedbackAttributes
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutEmailIdentityFeedbackAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEmailIdentityFeedbackAttributesResponse> putEmailIdentityFeedbackAttributes(
            PutEmailIdentityFeedbackAttributesRequest putEmailIdentityFeedbackAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putEmailIdentityFeedbackAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putEmailIdentityFeedbackAttributesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEmailIdentityFeedbackAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEmailIdentityFeedbackAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutEmailIdentityFeedbackAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEmailIdentityFeedbackAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEmailIdentityFeedbackAttributesRequest, PutEmailIdentityFeedbackAttributesResponse>()
                            .withOperationName("PutEmailIdentityFeedbackAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutEmailIdentityFeedbackAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putEmailIdentityFeedbackAttributesRequest));
            CompletableFuture<PutEmailIdentityFeedbackAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to enable or disable the custom Mail-From domain configuration for an email identity.
     * </p>
     *
     * @param putEmailIdentityMailFromAttributesRequest
     *        A request to configure the custom MAIL FROM domain for a verified identity.
     * @return A Java Future containing the result of the PutEmailIdentityMailFromAttributes operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.PutEmailIdentityMailFromAttributes
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/PutEmailIdentityMailFromAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEmailIdentityMailFromAttributesResponse> putEmailIdentityMailFromAttributes(
            PutEmailIdentityMailFromAttributesRequest putEmailIdentityMailFromAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putEmailIdentityMailFromAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putEmailIdentityMailFromAttributesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEmailIdentityMailFromAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEmailIdentityMailFromAttributesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutEmailIdentityMailFromAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEmailIdentityMailFromAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEmailIdentityMailFromAttributesRequest, PutEmailIdentityMailFromAttributesResponse>()
                            .withOperationName("PutEmailIdentityMailFromAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutEmailIdentityMailFromAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putEmailIdentityMailFromAttributesRequest));
            CompletableFuture<PutEmailIdentityMailFromAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends an email message. You can use the Amazon Pinpoint Email API to send two types of messages:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Simple</b> – A standard email message. When you create this type of message, you specify the sender, the
     * recipient, and the message body, and Amazon Pinpoint assembles the message for you.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Raw</b> – A raw, MIME-formatted email message. When you send this type of email, you have to specify all of
     * the message headers, as well as the message body. You can use this message type to send messages that contain
     * attachments. The message that you specify has to be a valid MIME message.
     * </p>
     * </li>
     * </ul>
     *
     * @param sendEmailRequest
     *        A request to send an email message.
     * @return A Java Future containing the result of the SendEmail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>LimitExceededException There are too many instances of the specified resource type.</li>
     *         <li>AccountSuspendedException The message can't be sent because the account's ability to send email has
     *         been permanently restricted.</li>
     *         <li>SendingPausedException The message can't be sent because the account's ability to send email is
     *         currently paused.</li>
     *         <li>MessageRejectedException The message can't be sent because it contains invalid content.</li>
     *         <li>MailFromDomainNotVerifiedException The message can't be sent because the sending domain isn't
     *         verified.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.SendEmail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/SendEmail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SendEmailResponse> sendEmail(SendEmailRequest sendEmailRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendEmailRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendEmailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendEmail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendEmailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendEmailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendEmailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendEmailRequest, SendEmailResponse>().withOperationName("SendEmail")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SendEmailRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendEmailRequest));
            CompletableFuture<SendEmailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add one or more tags (keys and values) to a specified resource. A <i>tag</i> is a label that you optionally
     * define and associate with a resource in Amazon Pinpoint. Tags can help you categorize and manage resources in
     * different ways, such as by purpose, owner, environment, or other criteria. A resource can have as many as 50
     * tags.
     * </p>
     * <p>
     * Each tag consists of a required <i>tag key</i> and an associated <i>tag value</i>, both of which you define. A
     * tag key is a general label that acts as a category for more specific tag values. A tag value acts as a descriptor
     * within a tag key.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove one or more tags (keys and values) from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ConcurrentModificationException The resource is being modified by another operation or thread.</li>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the configuration of an event destination for a configuration set.
     * </p>
     * <p>
     * In Amazon Pinpoint, <i>events</i> include message sends, deliveries, opens, clicks, bounces, and complaints.
     * <i>Event destinations</i> are places that you can send information about these events to. For example, you can
     * send event data to Amazon SNS to receive notifications when you receive bounces or complaints, or you can use
     * Amazon Kinesis Data Firehose to stream data to Amazon S3 for long-term storage.
     * </p>
     *
     * @param updateConfigurationSetEventDestinationRequest
     *        A request to change the settings for an event destination for a configuration set.
     * @return A Java Future containing the result of the UpdateConfigurationSetEventDestination operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The resource you attempted to access doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been made to the operation.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PinpointEmailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PinpointEmailAsyncClient.UpdateConfigurationSetEventDestination
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/pinpoint-email-2018-07-26/UpdateConfigurationSetEventDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationSetEventDestinationResponse> updateConfigurationSetEventDestination(
            UpdateConfigurationSetEventDestinationRequest updateConfigurationSetEventDestinationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationSetEventDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateConfigurationSetEventDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pinpoint Email");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationSetEventDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationSetEventDestinationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateConfigurationSetEventDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationSetEventDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationSetEventDestinationRequest, UpdateConfigurationSetEventDestinationResponse>()
                            .withOperationName("UpdateConfigurationSetEventDestination").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateConfigurationSetEventDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConfigurationSetEventDestinationRequest));
            CompletableFuture<UpdateConfigurationSetEventDestinationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PinpointEmailServiceClientConfiguration serviceClientConfiguration() {
        return new PinpointEmailServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PinpointEmailException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MessageRejected")
                                .exceptionBuilderSupplier(MessageRejectedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SendingPausedException")
                                .exceptionBuilderSupplier(SendingPausedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountSuspendedException")
                                .exceptionBuilderSupplier(AccountSuspendedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MailFromDomainNotVerifiedException")
                                .exceptionBuilderSupplier(MailFromDomainNotVerifiedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        PinpointEmailServiceClientConfigurationBuilder serviceConfigBuilder = new PinpointEmailServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
