/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pipes.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the logging configuration settings for the pipe.
 * </p>
 * <p>
 * When you call <code>UpdatePipe</code>, EventBridge updates the fields in the
 * <code>PipeLogConfigurationParameters</code> object atomically as one and overrides existing values. This is by
 * design. If you don't specify an optional field in any of the Amazon Web Services service parameters objects (
 * <code>CloudwatchLogsLogDestinationParameters</code>, <code>FirehoseLogDestinationParameters</code>, or
 * <code>S3LogDestinationParameters</code>), EventBridge sets that field to its system-default value during the update.
 * </p>
 * <p>
 * For example, suppose when you created the pipe you specified a Firehose stream log destination. You then update the
 * pipe to add an Amazon S3 log destination. In addition to specifying the <code>S3LogDestinationParameters</code> for
 * the new log destination, you must also specify the fields in the <code>FirehoseLogDestinationParameters</code> object
 * in order to retain the Firehose stream log destination.
 * </p>
 * <p>
 * For more information on generating pipe log records, see <a
 * href="eventbridge/latest/userguide/eb-pipes-logs.html">Log EventBridge Pipes</a> in the <i>Amazon EventBridge User
 * Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PipeLogConfigurationParameters implements SdkPojo, Serializable,
        ToCopyableBuilder<PipeLogConfigurationParameters.Builder, PipeLogConfigurationParameters> {
    private static final SdkField<S3LogDestinationParameters> S3_LOG_DESTINATION_FIELD = SdkField
            .<S3LogDestinationParameters> builder(MarshallingType.SDK_POJO).memberName("S3LogDestination")
            .getter(getter(PipeLogConfigurationParameters::s3LogDestination)).setter(setter(Builder::s3LogDestination))
            .constructor(S3LogDestinationParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3LogDestination").build()).build();

    private static final SdkField<FirehoseLogDestinationParameters> FIREHOSE_LOG_DESTINATION_FIELD = SdkField
            .<FirehoseLogDestinationParameters> builder(MarshallingType.SDK_POJO).memberName("FirehoseLogDestination")
            .getter(getter(PipeLogConfigurationParameters::firehoseLogDestination))
            .setter(setter(Builder::firehoseLogDestination)).constructor(FirehoseLogDestinationParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FirehoseLogDestination").build())
            .build();

    private static final SdkField<CloudwatchLogsLogDestinationParameters> CLOUDWATCH_LOGS_LOG_DESTINATION_FIELD = SdkField
            .<CloudwatchLogsLogDestinationParameters> builder(MarshallingType.SDK_POJO)
            .memberName("CloudwatchLogsLogDestination")
            .getter(getter(PipeLogConfigurationParameters::cloudwatchLogsLogDestination))
            .setter(setter(Builder::cloudwatchLogsLogDestination))
            .constructor(CloudwatchLogsLogDestinationParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudwatchLogsLogDestination")
                    .build()).build();

    private static final SdkField<String> LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Level")
            .getter(getter(PipeLogConfigurationParameters::levelAsString)).setter(setter(Builder::level))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Level").build()).build();

    private static final SdkField<List<String>> INCLUDE_EXECUTION_DATA_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IncludeExecutionData")
            .getter(getter(PipeLogConfigurationParameters::includeExecutionDataAsStrings))
            .setter(setter(Builder::includeExecutionDataWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeExecutionData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_LOG_DESTINATION_FIELD,
            FIREHOSE_LOG_DESTINATION_FIELD, CLOUDWATCH_LOGS_LOG_DESTINATION_FIELD, LEVEL_FIELD, INCLUDE_EXECUTION_DATA_FIELD));

    private static final long serialVersionUID = 1L;

    private final S3LogDestinationParameters s3LogDestination;

    private final FirehoseLogDestinationParameters firehoseLogDestination;

    private final CloudwatchLogsLogDestinationParameters cloudwatchLogsLogDestination;

    private final String level;

    private final List<String> includeExecutionData;

    private PipeLogConfigurationParameters(BuilderImpl builder) {
        this.s3LogDestination = builder.s3LogDestination;
        this.firehoseLogDestination = builder.firehoseLogDestination;
        this.cloudwatchLogsLogDestination = builder.cloudwatchLogsLogDestination;
        this.level = builder.level;
        this.includeExecutionData = builder.includeExecutionData;
    }

    /**
     * <p>
     * The Amazon S3 logging configuration settings for the pipe.
     * </p>
     * 
     * @return The Amazon S3 logging configuration settings for the pipe.
     */
    public final S3LogDestinationParameters s3LogDestination() {
        return s3LogDestination;
    }

    /**
     * <p>
     * The Amazon Data Firehose logging configuration settings for the pipe.
     * </p>
     * 
     * @return The Amazon Data Firehose logging configuration settings for the pipe.
     */
    public final FirehoseLogDestinationParameters firehoseLogDestination() {
        return firehoseLogDestination;
    }

    /**
     * <p>
     * The Amazon CloudWatch Logs logging configuration settings for the pipe.
     * </p>
     * 
     * @return The Amazon CloudWatch Logs logging configuration settings for the pipe.
     */
    public final CloudwatchLogsLogDestinationParameters cloudwatchLogsLogDestination() {
        return cloudwatchLogsLogDestination;
    }

    /**
     * <p>
     * The level of logging detail to include. This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level">Specifying
     * EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #level} will return
     * {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #levelAsString}.
     * </p>
     * 
     * @return The level of logging detail to include. This applies to all log destinations for the pipe.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level"
     *         >Specifying EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
     * @see LogLevel
     */
    public final LogLevel level() {
        return LogLevel.fromValue(level);
    }

    /**
     * <p>
     * The level of logging detail to include. This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level">Specifying
     * EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #level} will return
     * {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #levelAsString}.
     * </p>
     * 
     * @return The level of logging detail to include. This applies to all log destinations for the pipe.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level"
     *         >Specifying EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
     * @see LogLevel
     */
    public final String levelAsString() {
        return level;
    }

    /**
     * <p>
     * Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
     * <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.
     * </p>
     * <p>
     * This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * By default, execution data is not included.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeExecutionData} method.
     * </p>
     * 
     * @return Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
     *         <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.</p>
     *         <p>
     *         This applies to all log destinations for the pipe.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     *         >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     *         </p>
     *         <p>
     *         By default, execution data is not included.
     */
    public final List<IncludeExecutionDataOption> includeExecutionData() {
        return IncludeExecutionDataCopier.copyStringToEnum(includeExecutionData);
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludeExecutionData property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludeExecutionData() {
        return includeExecutionData != null && !(includeExecutionData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
     * <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.
     * </p>
     * <p>
     * This applies to all log destinations for the pipe.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * By default, execution data is not included.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeExecutionData} method.
     * </p>
     * 
     * @return Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
     *         <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.</p>
     *         <p>
     *         This applies to all log destinations for the pipe.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
     *         >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
     *         </p>
     *         <p>
     *         By default, execution data is not included.
     */
    public final List<String> includeExecutionDataAsStrings() {
        return includeExecutionData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3LogDestination());
        hashCode = 31 * hashCode + Objects.hashCode(firehoseLogDestination());
        hashCode = 31 * hashCode + Objects.hashCode(cloudwatchLogsLogDestination());
        hashCode = 31 * hashCode + Objects.hashCode(levelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeExecutionData() ? includeExecutionDataAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PipeLogConfigurationParameters)) {
            return false;
        }
        PipeLogConfigurationParameters other = (PipeLogConfigurationParameters) obj;
        return Objects.equals(s3LogDestination(), other.s3LogDestination())
                && Objects.equals(firehoseLogDestination(), other.firehoseLogDestination())
                && Objects.equals(cloudwatchLogsLogDestination(), other.cloudwatchLogsLogDestination())
                && Objects.equals(levelAsString(), other.levelAsString())
                && hasIncludeExecutionData() == other.hasIncludeExecutionData()
                && Objects.equals(includeExecutionDataAsStrings(), other.includeExecutionDataAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PipeLogConfigurationParameters").add("S3LogDestination", s3LogDestination())
                .add("FirehoseLogDestination", firehoseLogDestination())
                .add("CloudwatchLogsLogDestination", cloudwatchLogsLogDestination()).add("Level", levelAsString())
                .add("IncludeExecutionData", hasIncludeExecutionData() ? includeExecutionDataAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3LogDestination":
            return Optional.ofNullable(clazz.cast(s3LogDestination()));
        case "FirehoseLogDestination":
            return Optional.ofNullable(clazz.cast(firehoseLogDestination()));
        case "CloudwatchLogsLogDestination":
            return Optional.ofNullable(clazz.cast(cloudwatchLogsLogDestination()));
        case "Level":
            return Optional.ofNullable(clazz.cast(levelAsString()));
        case "IncludeExecutionData":
            return Optional.ofNullable(clazz.cast(includeExecutionDataAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PipeLogConfigurationParameters, T> g) {
        return obj -> g.apply((PipeLogConfigurationParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PipeLogConfigurationParameters> {
        /**
         * <p>
         * The Amazon S3 logging configuration settings for the pipe.
         * </p>
         * 
         * @param s3LogDestination
         *        The Amazon S3 logging configuration settings for the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3LogDestination(S3LogDestinationParameters s3LogDestination);

        /**
         * <p>
         * The Amazon S3 logging configuration settings for the pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3LogDestinationParameters.Builder}
         * avoiding the need to create one manually via {@link S3LogDestinationParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3LogDestinationParameters.Builder#build()} is called immediately
         * and its result is passed to {@link #s3LogDestination(S3LogDestinationParameters)}.
         * 
         * @param s3LogDestination
         *        a consumer that will call methods on {@link S3LogDestinationParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3LogDestination(S3LogDestinationParameters)
         */
        default Builder s3LogDestination(Consumer<S3LogDestinationParameters.Builder> s3LogDestination) {
            return s3LogDestination(S3LogDestinationParameters.builder().applyMutation(s3LogDestination).build());
        }

        /**
         * <p>
         * The Amazon Data Firehose logging configuration settings for the pipe.
         * </p>
         * 
         * @param firehoseLogDestination
         *        The Amazon Data Firehose logging configuration settings for the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firehoseLogDestination(FirehoseLogDestinationParameters firehoseLogDestination);

        /**
         * <p>
         * The Amazon Data Firehose logging configuration settings for the pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link FirehoseLogDestinationParameters.Builder}
         * avoiding the need to create one manually via {@link FirehoseLogDestinationParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FirehoseLogDestinationParameters.Builder#build()} is called
         * immediately and its result is passed to {@link #firehoseLogDestination(FirehoseLogDestinationParameters)}.
         * 
         * @param firehoseLogDestination
         *        a consumer that will call methods on {@link FirehoseLogDestinationParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #firehoseLogDestination(FirehoseLogDestinationParameters)
         */
        default Builder firehoseLogDestination(Consumer<FirehoseLogDestinationParameters.Builder> firehoseLogDestination) {
            return firehoseLogDestination(FirehoseLogDestinationParameters.builder().applyMutation(firehoseLogDestination)
                    .build());
        }

        /**
         * <p>
         * The Amazon CloudWatch Logs logging configuration settings for the pipe.
         * </p>
         * 
         * @param cloudwatchLogsLogDestination
         *        The Amazon CloudWatch Logs logging configuration settings for the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudwatchLogsLogDestination(CloudwatchLogsLogDestinationParameters cloudwatchLogsLogDestination);

        /**
         * <p>
         * The Amazon CloudWatch Logs logging configuration settings for the pipe.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CloudwatchLogsLogDestinationParameters.Builder} avoiding the need to create one manually via
         * {@link CloudwatchLogsLogDestinationParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CloudwatchLogsLogDestinationParameters.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #cloudwatchLogsLogDestination(CloudwatchLogsLogDestinationParameters)}.
         * 
         * @param cloudwatchLogsLogDestination
         *        a consumer that will call methods on {@link CloudwatchLogsLogDestinationParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudwatchLogsLogDestination(CloudwatchLogsLogDestinationParameters)
         */
        default Builder cloudwatchLogsLogDestination(
                Consumer<CloudwatchLogsLogDestinationParameters.Builder> cloudwatchLogsLogDestination) {
            return cloudwatchLogsLogDestination(CloudwatchLogsLogDestinationParameters.builder()
                    .applyMutation(cloudwatchLogsLogDestination).build());
        }

        /**
         * <p>
         * The level of logging detail to include. This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level"
         * >Specifying EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param level
         *        The level of logging detail to include. This applies to all log destinations for the pipe.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level"
         *        >Specifying EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder level(String level);

        /**
         * <p>
         * The level of logging detail to include. This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level"
         * >Specifying EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param level
         *        The level of logging detail to include. This applies to all log destinations for the pipe.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-level"
         *        >Specifying EventBridge Pipes log level</a> in the <i>Amazon EventBridge User Guide</i>.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder level(LogLevel level);

        /**
         * <p>
         * Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         * <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * <p>
         * By default, execution data is not included.
         * </p>
         * 
         * @param includeExecutionData
         *        Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         *        <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         *        </p>
         *        <p>
         *        By default, execution data is not included.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionDataWithStrings(Collection<String> includeExecutionData);

        /**
         * <p>
         * Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         * <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * <p>
         * By default, execution data is not included.
         * </p>
         * 
         * @param includeExecutionData
         *        Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         *        <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         *        </p>
         *        <p>
         *        By default, execution data is not included.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionDataWithStrings(String... includeExecutionData);

        /**
         * <p>
         * Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         * <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * <p>
         * By default, execution data is not included.
         * </p>
         * 
         * @param includeExecutionData
         *        Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         *        <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         *        </p>
         *        <p>
         *        By default, execution data is not included.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionData(Collection<IncludeExecutionDataOption> includeExecutionData);

        /**
         * <p>
         * Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         * <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.
         * </p>
         * <p>
         * This applies to all log destinations for the pipe.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         * >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * <p>
         * By default, execution data is not included.
         * </p>
         * 
         * @param includeExecutionData
         *        Specify <code>ALL</code> to include the execution data (specifically, the <code>payload</code>,
         *        <code>awsRequest</code>, and <code>awsResponse</code> fields) in the log messages for this pipe.</p>
         *        <p>
         *        This applies to all log destinations for the pipe.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-logs.html#eb-pipes-logs-execution-data"
         *        >Including execution data in logs</a> in the <i>Amazon EventBridge User Guide</i>.
         *        </p>
         *        <p>
         *        By default, execution data is not included.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeExecutionData(IncludeExecutionDataOption... includeExecutionData);
    }

    static final class BuilderImpl implements Builder {
        private S3LogDestinationParameters s3LogDestination;

        private FirehoseLogDestinationParameters firehoseLogDestination;

        private CloudwatchLogsLogDestinationParameters cloudwatchLogsLogDestination;

        private String level;

        private List<String> includeExecutionData = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PipeLogConfigurationParameters model) {
            s3LogDestination(model.s3LogDestination);
            firehoseLogDestination(model.firehoseLogDestination);
            cloudwatchLogsLogDestination(model.cloudwatchLogsLogDestination);
            level(model.level);
            includeExecutionDataWithStrings(model.includeExecutionData);
        }

        public final S3LogDestinationParameters.Builder getS3LogDestination() {
            return s3LogDestination != null ? s3LogDestination.toBuilder() : null;
        }

        public final void setS3LogDestination(S3LogDestinationParameters.BuilderImpl s3LogDestination) {
            this.s3LogDestination = s3LogDestination != null ? s3LogDestination.build() : null;
        }

        @Override
        public final Builder s3LogDestination(S3LogDestinationParameters s3LogDestination) {
            this.s3LogDestination = s3LogDestination;
            return this;
        }

        public final FirehoseLogDestinationParameters.Builder getFirehoseLogDestination() {
            return firehoseLogDestination != null ? firehoseLogDestination.toBuilder() : null;
        }

        public final void setFirehoseLogDestination(FirehoseLogDestinationParameters.BuilderImpl firehoseLogDestination) {
            this.firehoseLogDestination = firehoseLogDestination != null ? firehoseLogDestination.build() : null;
        }

        @Override
        public final Builder firehoseLogDestination(FirehoseLogDestinationParameters firehoseLogDestination) {
            this.firehoseLogDestination = firehoseLogDestination;
            return this;
        }

        public final CloudwatchLogsLogDestinationParameters.Builder getCloudwatchLogsLogDestination() {
            return cloudwatchLogsLogDestination != null ? cloudwatchLogsLogDestination.toBuilder() : null;
        }

        public final void setCloudwatchLogsLogDestination(
                CloudwatchLogsLogDestinationParameters.BuilderImpl cloudwatchLogsLogDestination) {
            this.cloudwatchLogsLogDestination = cloudwatchLogsLogDestination != null ? cloudwatchLogsLogDestination.build()
                    : null;
        }

        @Override
        public final Builder cloudwatchLogsLogDestination(CloudwatchLogsLogDestinationParameters cloudwatchLogsLogDestination) {
            this.cloudwatchLogsLogDestination = cloudwatchLogsLogDestination;
            return this;
        }

        public final String getLevel() {
            return level;
        }

        public final void setLevel(String level) {
            this.level = level;
        }

        @Override
        public final Builder level(String level) {
            this.level = level;
            return this;
        }

        @Override
        public final Builder level(LogLevel level) {
            this.level(level == null ? null : level.toString());
            return this;
        }

        public final Collection<String> getIncludeExecutionData() {
            if (includeExecutionData instanceof SdkAutoConstructList) {
                return null;
            }
            return includeExecutionData;
        }

        public final void setIncludeExecutionData(Collection<String> includeExecutionData) {
            this.includeExecutionData = IncludeExecutionDataCopier.copy(includeExecutionData);
        }

        @Override
        public final Builder includeExecutionDataWithStrings(Collection<String> includeExecutionData) {
            this.includeExecutionData = IncludeExecutionDataCopier.copy(includeExecutionData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeExecutionDataWithStrings(String... includeExecutionData) {
            includeExecutionDataWithStrings(Arrays.asList(includeExecutionData));
            return this;
        }

        @Override
        public final Builder includeExecutionData(Collection<IncludeExecutionDataOption> includeExecutionData) {
            this.includeExecutionData = IncludeExecutionDataCopier.copyEnumToString(includeExecutionData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeExecutionData(IncludeExecutionDataOption... includeExecutionData) {
            includeExecutionData(Arrays.asList(includeExecutionData));
            return this;
        }

        @Override
        public PipeLogConfigurationParameters build() {
            return new PipeLogConfigurationParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
