/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pipes;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.pipes.internal.PipesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.pipes.model.ConflictException;
import software.amazon.awssdk.services.pipes.model.CreatePipeRequest;
import software.amazon.awssdk.services.pipes.model.CreatePipeResponse;
import software.amazon.awssdk.services.pipes.model.DeletePipeRequest;
import software.amazon.awssdk.services.pipes.model.DeletePipeResponse;
import software.amazon.awssdk.services.pipes.model.DescribePipeRequest;
import software.amazon.awssdk.services.pipes.model.DescribePipeResponse;
import software.amazon.awssdk.services.pipes.model.InternalException;
import software.amazon.awssdk.services.pipes.model.ListPipesRequest;
import software.amazon.awssdk.services.pipes.model.ListPipesResponse;
import software.amazon.awssdk.services.pipes.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pipes.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pipes.model.NotFoundException;
import software.amazon.awssdk.services.pipes.model.PipesException;
import software.amazon.awssdk.services.pipes.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.pipes.model.StartPipeRequest;
import software.amazon.awssdk.services.pipes.model.StartPipeResponse;
import software.amazon.awssdk.services.pipes.model.StopPipeRequest;
import software.amazon.awssdk.services.pipes.model.StopPipeResponse;
import software.amazon.awssdk.services.pipes.model.TagResourceRequest;
import software.amazon.awssdk.services.pipes.model.TagResourceResponse;
import software.amazon.awssdk.services.pipes.model.ThrottlingException;
import software.amazon.awssdk.services.pipes.model.UntagResourceRequest;
import software.amazon.awssdk.services.pipes.model.UntagResourceResponse;
import software.amazon.awssdk.services.pipes.model.UpdatePipeRequest;
import software.amazon.awssdk.services.pipes.model.UpdatePipeResponse;
import software.amazon.awssdk.services.pipes.model.ValidationException;
import software.amazon.awssdk.services.pipes.transform.CreatePipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.DeletePipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.DescribePipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.ListPipesRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.StartPipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.StopPipeRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.pipes.transform.UpdatePipeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PipesAsyncClient}.
 *
 * @see PipesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPipesAsyncClient implements PipesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPipesAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPipesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a pipe. Amazon EventBridge Pipes connect event sources to targets and reduces the need for specialized
     * knowledge and integration code.
     * </p>
     *
     * @param createPipeRequest
     * @return A Java Future containing the result of the CreatePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>ServiceQuotaExceededException A quota has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.CreatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/CreatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePipeResponse> createPipe(CreatePipeRequest createPipeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreatePipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePipeRequest, CreatePipeResponse>().withOperationName("CreatePipe")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPipeRequest));
            CompletableFuture<CreatePipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param deletePipeRequest
     * @return A Java Future containing the result of the DeletePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.DeletePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DeletePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePipeResponse> deletePipe(DeletePipeRequest deletePipeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePipeRequest, DeletePipeResponse>().withOperationName("DeletePipe")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePipeRequest));
            CompletableFuture<DeletePipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the information about an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param describePipeRequest
     * @return A Java Future containing the result of the DescribePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.DescribePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DescribePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePipeResponse> describePipe(DescribePipeRequest describePipeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribePipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePipeRequest, DescribePipeResponse>()
                            .withOperationName("DescribePipe").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePipeRequest));
            CompletableFuture<DescribePipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the pipes associated with this account. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param listPipesRequest
     * @return A Java Future containing the result of the ListPipes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListPipes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListPipes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPipesResponse> listPipes(ListPipesRequest listPipesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPipesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipesRequest, ListPipesResponse>().withOperationName("ListPipes")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPipesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPipesRequest));
            CompletableFuture<ListPipesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the tags associated with a pipe.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Start an existing pipe.
     * </p>
     *
     * @param startPipeRequest
     * @return A Java Future containing the result of the StartPipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.StartPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StartPipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartPipeResponse> startPipe(StartPipeRequest startPipeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartPipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartPipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPipeRequest, StartPipeResponse>().withOperationName("StartPipe")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartPipeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(startPipeRequest));
            CompletableFuture<StartPipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stop an existing pipe.
     * </p>
     *
     * @param stopPipeRequest
     * @return A Java Future containing the result of the StopPipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.StopPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StopPipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopPipeResponse> stopPipe(StopPipeRequest stopPipeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopPipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopPipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopPipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopPipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopPipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopPipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopPipeRequest, StopPipeResponse>().withOperationName("StopPipe")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopPipeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopPipeRequest));
            CompletableFuture<StopPipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified pipe. Tags can help you organize and categorize your
     * resources. You can also use them to scope user permissions by granting a user permission to access or change only
     * resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a pipe that already has tags. If you specify a new tag key,
     * this tag is appended to the list of tags associated with the pipe. If you specify a tag key that is already
     * associated with the pipe, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a pipe.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified pipes.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an existing pipe. When you call <code>UpdatePipe</code>, EventBridge only the updates fields you have
     * specified in the request; the rest remain unchanged. The exception to this is if you modify any Amazon Web
     * Services-service specific fields in the <code>SourceParameters</code>, <code>EnrichmentParameters</code>, or
     * <code>TargetParameters</code> objects. For example, <code>DynamoDBStreamParameters</code> or
     * <code>EventBridgeEventBusParameters</code>. EventBridge updates the fields in these objects atomically as one and
     * overrides existing values. This is by design, and means that if you don't specify an optional field in one of
     * these <code>Parameters</code> objects, EventBridge sets that field to its system-default value during the update.
     * </p>
     * <p>
     * For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html"> Amazon EventBridge Pipes</a> in
     * the Amazon EventBridge User Guide.
     * </p>
     *
     * @param updatePipeRequest
     * @return A Java Future containing the result of the UpdatePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.UpdatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UpdatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePipeResponse> updatePipe(UpdatePipeRequest updatePipeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Pipes");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdatePipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePipeRequest, UpdatePipeResponse>().withOperationName("UpdatePipe")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePipeRequest));
            CompletableFuture<UpdatePipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PipesServiceClientConfiguration serviceClientConfiguration() {
        return new PipesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PipesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        PipesServiceClientConfigurationBuilder serviceConfigBuilder = new PipesServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
