/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pipes.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreatePipeRequest extends PipesRequest implements
        ToCopyableBuilder<CreatePipeRequest.Builder, CreatePipeRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreatePipeRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreatePipeRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> DESIRED_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DesiredState").getter(getter(CreatePipeRequest::desiredStateAsString))
            .setter(setter(Builder::desiredState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DesiredState").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Source")
            .getter(getter(CreatePipeRequest::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<PipeSourceParameters> SOURCE_PARAMETERS_FIELD = SdkField
            .<PipeSourceParameters> builder(MarshallingType.SDK_POJO).memberName("SourceParameters")
            .getter(getter(CreatePipeRequest::sourceParameters)).setter(setter(Builder::sourceParameters))
            .constructor(PipeSourceParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceParameters").build()).build();

    private static final SdkField<String> ENRICHMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Enrichment").getter(getter(CreatePipeRequest::enrichment)).setter(setter(Builder::enrichment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enrichment").build()).build();

    private static final SdkField<PipeEnrichmentParameters> ENRICHMENT_PARAMETERS_FIELD = SdkField
            .<PipeEnrichmentParameters> builder(MarshallingType.SDK_POJO).memberName("EnrichmentParameters")
            .getter(getter(CreatePipeRequest::enrichmentParameters)).setter(setter(Builder::enrichmentParameters))
            .constructor(PipeEnrichmentParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnrichmentParameters").build())
            .build();

    private static final SdkField<String> TARGET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Target")
            .getter(getter(CreatePipeRequest::target)).setter(setter(Builder::target))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Target").build()).build();

    private static final SdkField<PipeTargetParameters> TARGET_PARAMETERS_FIELD = SdkField
            .<PipeTargetParameters> builder(MarshallingType.SDK_POJO).memberName("TargetParameters")
            .getter(getter(CreatePipeRequest::targetParameters)).setter(setter(Builder::targetParameters))
            .constructor(PipeTargetParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetParameters").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(CreatePipeRequest::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreatePipeRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<PipeLogConfigurationParameters> LOG_CONFIGURATION_FIELD = SdkField
            .<PipeLogConfigurationParameters> builder(MarshallingType.SDK_POJO).memberName("LogConfiguration")
            .getter(getter(CreatePipeRequest::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(PipeLogConfigurationParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogConfiguration").build()).build();

    private static final SdkField<String> KMS_KEY_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyIdentifier").getter(getter(CreatePipeRequest::kmsKeyIdentifier))
            .setter(setter(Builder::kmsKeyIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyIdentifier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD, DESIRED_STATE_FIELD, SOURCE_FIELD,
                    SOURCE_PARAMETERS_FIELD, ENRICHMENT_FIELD, ENRICHMENT_PARAMETERS_FIELD, TARGET_FIELD,
                    TARGET_PARAMETERS_FIELD, ROLE_ARN_FIELD, TAGS_FIELD, LOG_CONFIGURATION_FIELD, KMS_KEY_IDENTIFIER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String name;

    private final String description;

    private final String desiredState;

    private final String source;

    private final PipeSourceParameters sourceParameters;

    private final String enrichment;

    private final PipeEnrichmentParameters enrichmentParameters;

    private final String target;

    private final PipeTargetParameters targetParameters;

    private final String roleArn;

    private final Map<String, String> tags;

    private final PipeLogConfigurationParameters logConfiguration;

    private final String kmsKeyIdentifier;

    private CreatePipeRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.description = builder.description;
        this.desiredState = builder.desiredState;
        this.source = builder.source;
        this.sourceParameters = builder.sourceParameters;
        this.enrichment = builder.enrichment;
        this.enrichmentParameters = builder.enrichmentParameters;
        this.target = builder.target;
        this.targetParameters = builder.targetParameters;
        this.roleArn = builder.roleArn;
        this.tags = builder.tags;
        this.logConfiguration = builder.logConfiguration;
        this.kmsKeyIdentifier = builder.kmsKeyIdentifier;
    }

    /**
     * <p>
     * The name of the pipe.
     * </p>
     * 
     * @return The name of the pipe.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A description of the pipe.
     * </p>
     * 
     * @return A description of the pipe.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The state the pipe should be in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #desiredState} will
     * return {@link RequestedPipeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #desiredStateAsString}.
     * </p>
     * 
     * @return The state the pipe should be in.
     * @see RequestedPipeState
     */
    public final RequestedPipeState desiredState() {
        return RequestedPipeState.fromValue(desiredState);
    }

    /**
     * <p>
     * The state the pipe should be in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #desiredState} will
     * return {@link RequestedPipeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #desiredStateAsString}.
     * </p>
     * 
     * @return The state the pipe should be in.
     * @see RequestedPipeState
     */
    public final String desiredStateAsString() {
        return desiredState;
    }

    /**
     * <p>
     * The ARN of the source resource.
     * </p>
     * 
     * @return The ARN of the source resource.
     */
    public final String source() {
        return source;
    }

    /**
     * <p>
     * The parameters required to set up a source for your pipe.
     * </p>
     * 
     * @return The parameters required to set up a source for your pipe.
     */
    public final PipeSourceParameters sourceParameters() {
        return sourceParameters;
    }

    /**
     * <p>
     * The ARN of the enrichment resource.
     * </p>
     * 
     * @return The ARN of the enrichment resource.
     */
    public final String enrichment() {
        return enrichment;
    }

    /**
     * <p>
     * The parameters required to set up enrichment on your pipe.
     * </p>
     * 
     * @return The parameters required to set up enrichment on your pipe.
     */
    public final PipeEnrichmentParameters enrichmentParameters() {
        return enrichmentParameters;
    }

    /**
     * <p>
     * The ARN of the target resource.
     * </p>
     * 
     * @return The ARN of the target resource.
     */
    public final String target() {
        return target;
    }

    /**
     * <p>
     * The parameters required to set up a target for your pipe.
     * </p>
     * <p>
     * For more information about pipe target parameters, including how to use dynamic path parameters, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-event-target.html">Target parameters</a>
     * in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * 
     * @return The parameters required to set up a target for your pipe.</p>
     *         <p>
     *         For more information about pipe target parameters, including how to use dynamic path parameters, see <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-event-target.html">Target
     *         parameters</a> in the <i>Amazon EventBridge User Guide</i>.
     */
    public final PipeTargetParameters targetParameters() {
        return targetParameters;
    }

    /**
     * <p>
     * The ARN of the role that allows the pipe to send data to the target.
     * </p>
     * 
     * @return The ARN of the role that allows the pipe to send data to the target.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of key-value pairs to associate with the pipe.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of key-value pairs to associate with the pipe.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The logging configuration settings for the pipe.
     * </p>
     * 
     * @return The logging configuration settings for the pipe.
     */
    public final PipeLogConfigurationParameters logConfiguration() {
        return logConfiguration;
    }

    /**
     * <p>
     * The identifier of the KMS customer managed key for EventBridge to use, if you choose to use a customer managed
     * key to encrypt pipe data. The identifier can be the key Amazon Resource Name (ARN), KeyId, key alias, or key
     * alias ARN.
     * </p>
     * <p>
     * If you do not specify a customer managed key identifier, EventBridge uses an Amazon Web Services owned key to
     * encrypt pipe data.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/getting-started.html">Managing keys</a> in the <i>Key
     * Management Service Developer Guide</i>.
     * </p>
     * 
     * @return The identifier of the KMS customer managed key for EventBridge to use, if you choose to use a customer
     *         managed key to encrypt pipe data. The identifier can be the key Amazon Resource Name (ARN), KeyId, key
     *         alias, or key alias ARN.</p>
     *         <p>
     *         If you do not specify a customer managed key identifier, EventBridge uses an Amazon Web Services owned
     *         key to encrypt pipe data.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/getting-started.html">Managing keys</a> in
     *         the <i>Key Management Service Developer Guide</i>.
     */
    public final String kmsKeyIdentifier() {
        return kmsKeyIdentifier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(desiredStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(sourceParameters());
        hashCode = 31 * hashCode + Objects.hashCode(enrichment());
        hashCode = 31 * hashCode + Objects.hashCode(enrichmentParameters());
        hashCode = 31 * hashCode + Objects.hashCode(target());
        hashCode = 31 * hashCode + Objects.hashCode(targetParameters());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyIdentifier());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreatePipeRequest)) {
            return false;
        }
        CreatePipeRequest other = (CreatePipeRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(desiredStateAsString(), other.desiredStateAsString())
                && Objects.equals(source(), other.source()) && Objects.equals(sourceParameters(), other.sourceParameters())
                && Objects.equals(enrichment(), other.enrichment())
                && Objects.equals(enrichmentParameters(), other.enrichmentParameters())
                && Objects.equals(target(), other.target()) && Objects.equals(targetParameters(), other.targetParameters())
                && Objects.equals(roleArn(), other.roleArn()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(logConfiguration(), other.logConfiguration())
                && Objects.equals(kmsKeyIdentifier(), other.kmsKeyIdentifier());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreatePipeRequest").add("Name", name())
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DesiredState", desiredStateAsString()).add("Source", source()).add("SourceParameters", sourceParameters())
                .add("Enrichment", enrichment()).add("EnrichmentParameters", enrichmentParameters()).add("Target", target())
                .add("TargetParameters", targetParameters()).add("RoleArn", roleArn())
                .add("Tags", tags() == null ? null : "*** Sensitive Data Redacted ***")
                .add("LogConfiguration", logConfiguration()).add("KmsKeyIdentifier", kmsKeyIdentifier()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DesiredState":
            return Optional.ofNullable(clazz.cast(desiredStateAsString()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "SourceParameters":
            return Optional.ofNullable(clazz.cast(sourceParameters()));
        case "Enrichment":
            return Optional.ofNullable(clazz.cast(enrichment()));
        case "EnrichmentParameters":
            return Optional.ofNullable(clazz.cast(enrichmentParameters()));
        case "Target":
            return Optional.ofNullable(clazz.cast(target()));
        case "TargetParameters":
            return Optional.ofNullable(clazz.cast(targetParameters()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "LogConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        case "KmsKeyIdentifier":
            return Optional.ofNullable(clazz.cast(kmsKeyIdentifier()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("DesiredState", DESIRED_STATE_FIELD);
        map.put("Source", SOURCE_FIELD);
        map.put("SourceParameters", SOURCE_PARAMETERS_FIELD);
        map.put("Enrichment", ENRICHMENT_FIELD);
        map.put("EnrichmentParameters", ENRICHMENT_PARAMETERS_FIELD);
        map.put("Target", TARGET_FIELD);
        map.put("TargetParameters", TARGET_PARAMETERS_FIELD);
        map.put("RoleArn", ROLE_ARN_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("LogConfiguration", LOG_CONFIGURATION_FIELD);
        map.put("KmsKeyIdentifier", KMS_KEY_IDENTIFIER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreatePipeRequest, T> g) {
        return obj -> g.apply((CreatePipeRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends PipesRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreatePipeRequest> {
        /**
         * <p>
         * The name of the pipe.
         * </p>
         * 
         * @param name
         *        The name of the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the pipe.
         * </p>
         * 
         * @param description
         *        A description of the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The state the pipe should be in.
         * </p>
         * 
         * @param desiredState
         *        The state the pipe should be in.
         * @see RequestedPipeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequestedPipeState
         */
        Builder desiredState(String desiredState);

        /**
         * <p>
         * The state the pipe should be in.
         * </p>
         * 
         * @param desiredState
         *        The state the pipe should be in.
         * @see RequestedPipeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequestedPipeState
         */
        Builder desiredState(RequestedPipeState desiredState);

        /**
         * <p>
         * The ARN of the source resource.
         * </p>
         * 
         * @param source
         *        The ARN of the source resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * The parameters required to set up a source for your pipe.
         * </p>
         * 
         * @param sourceParameters
         *        The parameters required to set up a source for your pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceParameters(PipeSourceParameters sourceParameters);

        /**
         * <p>
         * The parameters required to set up a source for your pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link PipeSourceParameters.Builder} avoiding
         * the need to create one manually via {@link PipeSourceParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PipeSourceParameters.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceParameters(PipeSourceParameters)}.
         * 
         * @param sourceParameters
         *        a consumer that will call methods on {@link PipeSourceParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceParameters(PipeSourceParameters)
         */
        default Builder sourceParameters(Consumer<PipeSourceParameters.Builder> sourceParameters) {
            return sourceParameters(PipeSourceParameters.builder().applyMutation(sourceParameters).build());
        }

        /**
         * <p>
         * The ARN of the enrichment resource.
         * </p>
         * 
         * @param enrichment
         *        The ARN of the enrichment resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enrichment(String enrichment);

        /**
         * <p>
         * The parameters required to set up enrichment on your pipe.
         * </p>
         * 
         * @param enrichmentParameters
         *        The parameters required to set up enrichment on your pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enrichmentParameters(PipeEnrichmentParameters enrichmentParameters);

        /**
         * <p>
         * The parameters required to set up enrichment on your pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link PipeEnrichmentParameters.Builder}
         * avoiding the need to create one manually via {@link PipeEnrichmentParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PipeEnrichmentParameters.Builder#build()} is called immediately
         * and its result is passed to {@link #enrichmentParameters(PipeEnrichmentParameters)}.
         * 
         * @param enrichmentParameters
         *        a consumer that will call methods on {@link PipeEnrichmentParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #enrichmentParameters(PipeEnrichmentParameters)
         */
        default Builder enrichmentParameters(Consumer<PipeEnrichmentParameters.Builder> enrichmentParameters) {
            return enrichmentParameters(PipeEnrichmentParameters.builder().applyMutation(enrichmentParameters).build());
        }

        /**
         * <p>
         * The ARN of the target resource.
         * </p>
         * 
         * @param target
         *        The ARN of the target resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder target(String target);

        /**
         * <p>
         * The parameters required to set up a target for your pipe.
         * </p>
         * <p>
         * For more information about pipe target parameters, including how to use dynamic path parameters, see <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-event-target.html">Target
         * parameters</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param targetParameters
         *        The parameters required to set up a target for your pipe.</p>
         *        <p>
         *        For more information about pipe target parameters, including how to use dynamic path parameters, see
         *        <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-event-target.html">Target
         *        parameters</a> in the <i>Amazon EventBridge User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetParameters(PipeTargetParameters targetParameters);

        /**
         * <p>
         * The parameters required to set up a target for your pipe.
         * </p>
         * <p>
         * For more information about pipe target parameters, including how to use dynamic path parameters, see <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes-event-target.html">Target
         * parameters</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link PipeTargetParameters.Builder} avoiding
         * the need to create one manually via {@link PipeTargetParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PipeTargetParameters.Builder#build()} is called immediately and
         * its result is passed to {@link #targetParameters(PipeTargetParameters)}.
         * 
         * @param targetParameters
         *        a consumer that will call methods on {@link PipeTargetParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetParameters(PipeTargetParameters)
         */
        default Builder targetParameters(Consumer<PipeTargetParameters.Builder> targetParameters) {
            return targetParameters(PipeTargetParameters.builder().applyMutation(targetParameters).build());
        }

        /**
         * <p>
         * The ARN of the role that allows the pipe to send data to the target.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the role that allows the pipe to send data to the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The list of key-value pairs to associate with the pipe.
         * </p>
         * 
         * @param tags
         *        The list of key-value pairs to associate with the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The logging configuration settings for the pipe.
         * </p>
         * 
         * @param logConfiguration
         *        The logging configuration settings for the pipe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(PipeLogConfigurationParameters logConfiguration);

        /**
         * <p>
         * The logging configuration settings for the pipe.
         * </p>
         * This is a convenience method that creates an instance of the {@link PipeLogConfigurationParameters.Builder}
         * avoiding the need to create one manually via {@link PipeLogConfigurationParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PipeLogConfigurationParameters.Builder#build()} is called
         * immediately and its result is passed to {@link #logConfiguration(PipeLogConfigurationParameters)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link PipeLogConfigurationParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(PipeLogConfigurationParameters)
         */
        default Builder logConfiguration(Consumer<PipeLogConfigurationParameters.Builder> logConfiguration) {
            return logConfiguration(PipeLogConfigurationParameters.builder().applyMutation(logConfiguration).build());
        }

        /**
         * <p>
         * The identifier of the KMS customer managed key for EventBridge to use, if you choose to use a customer
         * managed key to encrypt pipe data. The identifier can be the key Amazon Resource Name (ARN), KeyId, key alias,
         * or key alias ARN.
         * </p>
         * <p>
         * If you do not specify a customer managed key identifier, EventBridge uses an Amazon Web Services owned key to
         * encrypt pipe data.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/getting-started.html">Managing keys</a> in the
         * <i>Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param kmsKeyIdentifier
         *        The identifier of the KMS customer managed key for EventBridge to use, if you choose to use a customer
         *        managed key to encrypt pipe data. The identifier can be the key Amazon Resource Name (ARN), KeyId, key
         *        alias, or key alias ARN.</p>
         *        <p>
         *        If you do not specify a customer managed key identifier, EventBridge uses an Amazon Web Services owned
         *        key to encrypt pipe data.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/getting-started.html">Managing keys</a> in
         *        the <i>Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyIdentifier(String kmsKeyIdentifier);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends PipesRequest.BuilderImpl implements Builder {
        private String name;

        private String description;

        private String desiredState;

        private String source;

        private PipeSourceParameters sourceParameters;

        private String enrichment;

        private PipeEnrichmentParameters enrichmentParameters;

        private String target;

        private PipeTargetParameters targetParameters;

        private String roleArn;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private PipeLogConfigurationParameters logConfiguration;

        private String kmsKeyIdentifier;

        private BuilderImpl() {
        }

        private BuilderImpl(CreatePipeRequest model) {
            super(model);
            name(model.name);
            description(model.description);
            desiredState(model.desiredState);
            source(model.source);
            sourceParameters(model.sourceParameters);
            enrichment(model.enrichment);
            enrichmentParameters(model.enrichmentParameters);
            target(model.target);
            targetParameters(model.targetParameters);
            roleArn(model.roleArn);
            tags(model.tags);
            logConfiguration(model.logConfiguration);
            kmsKeyIdentifier(model.kmsKeyIdentifier);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDesiredState() {
            return desiredState;
        }

        public final void setDesiredState(String desiredState) {
            this.desiredState = desiredState;
        }

        @Override
        public final Builder desiredState(String desiredState) {
            this.desiredState = desiredState;
            return this;
        }

        @Override
        public final Builder desiredState(RequestedPipeState desiredState) {
            this.desiredState(desiredState == null ? null : desiredState.toString());
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final PipeSourceParameters.Builder getSourceParameters() {
            return sourceParameters != null ? sourceParameters.toBuilder() : null;
        }

        public final void setSourceParameters(PipeSourceParameters.BuilderImpl sourceParameters) {
            this.sourceParameters = sourceParameters != null ? sourceParameters.build() : null;
        }

        @Override
        public final Builder sourceParameters(PipeSourceParameters sourceParameters) {
            this.sourceParameters = sourceParameters;
            return this;
        }

        public final String getEnrichment() {
            return enrichment;
        }

        public final void setEnrichment(String enrichment) {
            this.enrichment = enrichment;
        }

        @Override
        public final Builder enrichment(String enrichment) {
            this.enrichment = enrichment;
            return this;
        }

        public final PipeEnrichmentParameters.Builder getEnrichmentParameters() {
            return enrichmentParameters != null ? enrichmentParameters.toBuilder() : null;
        }

        public final void setEnrichmentParameters(PipeEnrichmentParameters.BuilderImpl enrichmentParameters) {
            this.enrichmentParameters = enrichmentParameters != null ? enrichmentParameters.build() : null;
        }

        @Override
        public final Builder enrichmentParameters(PipeEnrichmentParameters enrichmentParameters) {
            this.enrichmentParameters = enrichmentParameters;
            return this;
        }

        public final String getTarget() {
            return target;
        }

        public final void setTarget(String target) {
            this.target = target;
        }

        @Override
        public final Builder target(String target) {
            this.target = target;
            return this;
        }

        public final PipeTargetParameters.Builder getTargetParameters() {
            return targetParameters != null ? targetParameters.toBuilder() : null;
        }

        public final void setTargetParameters(PipeTargetParameters.BuilderImpl targetParameters) {
            this.targetParameters = targetParameters != null ? targetParameters.build() : null;
        }

        @Override
        public final Builder targetParameters(PipeTargetParameters targetParameters) {
            this.targetParameters = targetParameters;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final PipeLogConfigurationParameters.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        public final void setLogConfiguration(PipeLogConfigurationParameters.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        @Override
        public final Builder logConfiguration(PipeLogConfigurationParameters logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        public final String getKmsKeyIdentifier() {
            return kmsKeyIdentifier;
        }

        public final void setKmsKeyIdentifier(String kmsKeyIdentifier) {
            this.kmsKeyIdentifier = kmsKeyIdentifier;
        }

        @Override
        public final Builder kmsKeyIdentifier(String kmsKeyIdentifier) {
            this.kmsKeyIdentifier = kmsKeyIdentifier;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreatePipeRequest build() {
            return new CreatePipeRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
