/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.pipes;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.pipes.model.CreatePipeRequest;
import software.amazon.awssdk.services.pipes.model.CreatePipeResponse;
import software.amazon.awssdk.services.pipes.model.DeletePipeRequest;
import software.amazon.awssdk.services.pipes.model.DeletePipeResponse;
import software.amazon.awssdk.services.pipes.model.DescribePipeRequest;
import software.amazon.awssdk.services.pipes.model.DescribePipeResponse;
import software.amazon.awssdk.services.pipes.model.ListPipesRequest;
import software.amazon.awssdk.services.pipes.model.ListPipesResponse;
import software.amazon.awssdk.services.pipes.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.pipes.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.pipes.model.StartPipeRequest;
import software.amazon.awssdk.services.pipes.model.StartPipeResponse;
import software.amazon.awssdk.services.pipes.model.StopPipeRequest;
import software.amazon.awssdk.services.pipes.model.StopPipeResponse;
import software.amazon.awssdk.services.pipes.model.TagResourceRequest;
import software.amazon.awssdk.services.pipes.model.TagResourceResponse;
import software.amazon.awssdk.services.pipes.model.UntagResourceRequest;
import software.amazon.awssdk.services.pipes.model.UntagResourceResponse;
import software.amazon.awssdk.services.pipes.model.UpdatePipeRequest;
import software.amazon.awssdk.services.pipes.model.UpdatePipeResponse;
import software.amazon.awssdk.services.pipes.paginators.ListPipesPublisher;

/**
 * Service client for accessing Amazon EventBridge Pipes asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * Amazon EventBridge Pipes connects event sources to targets. Pipes reduces the need for specialized knowledge and
 * integration code when developing event driven architectures. This helps ensures consistency across your company’s
 * applications. With Pipes, the target can be any available EventBridge target. To set up a pipe, you select the event
 * source, add optional event filtering, define optional enrichment, and select the target for the event data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface PipesAsyncClient extends AwsClient {
    String SERVICE_NAME = "pipes";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "pipes";

    /**
     * <p>
     * Create a pipe. Amazon EventBridge Pipes connect event sources to targets and reduces the need for specialized
     * knowledge and integration code.
     * </p>
     *
     * @param createPipeRequest
     * @return A Java Future containing the result of the CreatePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>ServiceQuotaExceededException A quota has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.CreatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/CreatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreatePipeResponse> createPipe(CreatePipeRequest createPipeRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Create a pipe. Amazon EventBridge Pipes connect event sources to targets and reduces the need for specialized
     * knowledge and integration code.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreatePipeRequest.Builder} avoiding the need to
     * create one manually via {@link CreatePipeRequest#builder()}
     * </p>
     *
     * @param createPipeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.CreatePipeRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreatePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>ServiceQuotaExceededException A quota has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.CreatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/CreatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreatePipeResponse> createPipe(Consumer<CreatePipeRequest.Builder> createPipeRequest) {
        return createPipe(CreatePipeRequest.builder().applyMutation(createPipeRequest).build());
    }

    /**
     * <p>
     * Delete an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param deletePipeRequest
     * @return A Java Future containing the result of the DeletePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.DeletePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DeletePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeletePipeResponse> deletePipe(DeletePipeRequest deletePipeRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Delete an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeletePipeRequest.Builder} avoiding the need to
     * create one manually via {@link DeletePipeRequest#builder()}
     * </p>
     *
     * @param deletePipeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.DeletePipeRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeletePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.DeletePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DeletePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeletePipeResponse> deletePipe(Consumer<DeletePipeRequest.Builder> deletePipeRequest) {
        return deletePipe(DeletePipeRequest.builder().applyMutation(deletePipeRequest).build());
    }

    /**
     * <p>
     * Get the information about an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param describePipeRequest
     * @return A Java Future containing the result of the DescribePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.DescribePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DescribePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribePipeResponse> describePipe(DescribePipeRequest describePipeRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the information about an existing pipe. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribePipeRequest.Builder} avoiding the need to
     * create one manually via {@link DescribePipeRequest#builder()}
     * </p>
     *
     * @param describePipeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.DescribePipeRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.DescribePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/DescribePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribePipeResponse> describePipe(Consumer<DescribePipeRequest.Builder> describePipeRequest) {
        return describePipe(DescribePipeRequest.builder().applyMutation(describePipeRequest).build());
    }

    /**
     * <p>
     * Get the pipes associated with this account. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     *
     * @param listPipesRequest
     * @return A Java Future containing the result of the ListPipes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListPipes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListPipes" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListPipesResponse> listPipes(ListPipesRequest listPipesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get the pipes associated with this account. For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html">Amazon EventBridge Pipes</a> in the
     * Amazon EventBridge User Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPipesRequest.Builder} avoiding the need to
     * create one manually via {@link ListPipesRequest#builder()}
     * </p>
     *
     * @param listPipesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.ListPipesRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListPipes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListPipes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListPipes" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListPipesResponse> listPipes(Consumer<ListPipesRequest.Builder> listPipesRequest) {
        return listPipes(ListPipesRequest.builder().applyMutation(listPipesRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listPipes(software.amazon.awssdk.services.pipes.model.ListPipesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pipes.paginators.ListPipesPublisher publisher = client.listPipesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pipes.paginators.ListPipesPublisher publisher = client.listPipesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.pipes.model.ListPipesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.pipes.model.ListPipesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPipes(software.amazon.awssdk.services.pipes.model.ListPipesRequest)} operation.</b>
     * </p>
     *
     * @param listPipesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListPipes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListPipes" target="_top">AWS API
     *      Documentation</a>
     */
    default ListPipesPublisher listPipesPaginator(ListPipesRequest listPipesRequest) {
        return new ListPipesPublisher(this, listPipesRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listPipes(software.amazon.awssdk.services.pipes.model.ListPipesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pipes.paginators.ListPipesPublisher publisher = client.listPipesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.pipes.paginators.ListPipesPublisher publisher = client.listPipesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.pipes.model.ListPipesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.pipes.model.ListPipesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPipes(software.amazon.awssdk.services.pipes.model.ListPipesRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPipesRequest.Builder} avoiding the need to
     * create one manually via {@link ListPipesRequest#builder()}
     * </p>
     *
     * @param listPipesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.ListPipesRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListPipes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListPipes" target="_top">AWS API
     *      Documentation</a>
     */
    default ListPipesPublisher listPipesPaginator(Consumer<ListPipesRequest.Builder> listPipesRequest) {
        return listPipesPaginator(ListPipesRequest.builder().applyMutation(listPipesRequest).build());
    }

    /**
     * <p>
     * Displays the tags associated with a pipe.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Displays the tags associated with a pipe.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Start an existing pipe.
     * </p>
     *
     * @param startPipeRequest
     * @return A Java Future containing the result of the StartPipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.StartPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StartPipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartPipeResponse> startPipe(StartPipeRequest startPipeRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Start an existing pipe.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartPipeRequest.Builder} avoiding the need to
     * create one manually via {@link StartPipeRequest#builder()}
     * </p>
     *
     * @param startPipeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.StartPipeRequest.Builder} to create a request.
     * @return A Java Future containing the result of the StartPipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.StartPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StartPipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StartPipeResponse> startPipe(Consumer<StartPipeRequest.Builder> startPipeRequest) {
        return startPipe(StartPipeRequest.builder().applyMutation(startPipeRequest).build());
    }

    /**
     * <p>
     * Stop an existing pipe.
     * </p>
     *
     * @param stopPipeRequest
     * @return A Java Future containing the result of the StopPipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.StopPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StopPipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StopPipeResponse> stopPipe(StopPipeRequest stopPipeRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Stop an existing pipe.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopPipeRequest.Builder} avoiding the need to
     * create one manually via {@link StopPipeRequest#builder()}
     * </p>
     *
     * @param stopPipeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.StopPipeRequest.Builder} to create a request.
     * @return A Java Future containing the result of the StopPipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.StopPipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/StopPipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<StopPipeResponse> stopPipe(Consumer<StopPipeRequest.Builder> stopPipeRequest) {
        return stopPipe(StopPipeRequest.builder().applyMutation(stopPipeRequest).build());
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified pipe. Tags can help you organize and categorize your
     * resources. You can also use them to scope user permissions by granting a user permission to access or change only
     * resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a pipe that already has tags. If you specify a new tag key,
     * this tag is appended to the list of tags associated with the pipe. If you specify a tag key that is already
     * associated with the pipe, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a pipe.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified pipe. Tags can help you organize and categorize your
     * resources. You can also use them to scope user permissions by granting a user permission to access or change only
     * resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a pipe that already has tags. If you specify a new tag key,
     * this tag is appended to the list of tags associated with the pipe. If you specify a tag key that is already
     * associated with the pipe, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a pipe.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes one or more tags from the specified pipes.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes one or more tags from the specified pipes.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Update an existing pipe. When you call <code>UpdatePipe</code>, EventBridge only the updates fields you have
     * specified in the request; the rest remain unchanged. The exception to this is if you modify any Amazon Web
     * Services-service specific fields in the <code>SourceParameters</code>, <code>EnrichmentParameters</code>, or
     * <code>TargetParameters</code> objects. For example, <code>DynamoDBStreamParameters</code> or
     * <code>EventBridgeEventBusParameters</code>. EventBridge updates the fields in these objects atomically as one and
     * overrides existing values. This is by design, and means that if you don't specify an optional field in one of
     * these <code>Parameters</code> objects, EventBridge sets that field to its system-default value during the update.
     * </p>
     * <p>
     * For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html"> Amazon EventBridge Pipes</a> in
     * the Amazon EventBridge User Guide.
     * </p>
     *
     * @param updatePipeRequest
     * @return A Java Future containing the result of the UpdatePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.UpdatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UpdatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdatePipeResponse> updatePipe(UpdatePipeRequest updatePipeRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Update an existing pipe. When you call <code>UpdatePipe</code>, EventBridge only the updates fields you have
     * specified in the request; the rest remain unchanged. The exception to this is if you modify any Amazon Web
     * Services-service specific fields in the <code>SourceParameters</code>, <code>EnrichmentParameters</code>, or
     * <code>TargetParameters</code> objects. For example, <code>DynamoDBStreamParameters</code> or
     * <code>EventBridgeEventBusParameters</code>. EventBridge updates the fields in these objects atomically as one and
     * overrides existing values. This is by design, and means that if you don't specify an optional field in one of
     * these <code>Parameters</code> objects, EventBridge sets that field to its system-default value during the update.
     * </p>
     * <p>
     * For more information about pipes, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-pipes.html"> Amazon EventBridge Pipes</a> in
     * the Amazon EventBridge User Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdatePipeRequest.Builder} avoiding the need to
     * create one manually via {@link UpdatePipeRequest#builder()}
     * </p>
     *
     * @param updatePipeRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.pipes.model.UpdatePipeRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdatePipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ValidationException Indicates that an error has occurred while performing a validate operation.</li>
     *         <li>ThrottlingException An action was throttled.</li>
     *         <li>NotFoundException An entity that you specified does not exist.</li>
     *         <li>ConflictException An action you attempted resulted in an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PipesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PipesAsyncClient.UpdatePipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/UpdatePipe" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdatePipeResponse> updatePipe(Consumer<UpdatePipeRequest.Builder> updatePipeRequest) {
        return updatePipe(UpdatePipeRequest.builder().applyMutation(updatePipeRequest).build());
    }

    @Override
    default PipesServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link PipesAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static PipesAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link PipesAsyncClient}.
     */
    static PipesAsyncClientBuilder builder() {
        return new DefaultPipesAsyncClientBuilder();
    }
}
