/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package software.amazon.awssdk.services.polly.internal.presigner;

import static java.util.stream.Collectors.toMap;
import static software.amazon.awssdk.auth.signer.AwsSignerExecutionAttribute.PRESIGNER_EXPIRATION;

import java.net.URI;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.CredentialUtils;
import software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider;
import software.amazon.awssdk.auth.signer.Aws4Signer;
import software.amazon.awssdk.auth.signer.AwsSignerExecutionAttribute;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.endpoint.DefaultServiceEndpointBuilder;
import software.amazon.awssdk.awscore.endpoint.DualstackEnabledProvider;
import software.amazon.awssdk.awscore.endpoint.FipsEnabledProvider;
import software.amazon.awssdk.awscore.presigner.PresignRequest;
import software.amazon.awssdk.awscore.presigner.PresignedRequest;
import software.amazon.awssdk.core.ClientType;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.signer.Presigner;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.http.SdkHttpMethod;
import software.amazon.awssdk.http.auth.aws.scheme.AwsV4AuthScheme;
import software.amazon.awssdk.http.auth.spi.scheme.AuthScheme;
import software.amazon.awssdk.identity.spi.AwsCredentialsIdentity;
import software.amazon.awssdk.identity.spi.IdentityProvider;
import software.amazon.awssdk.identity.spi.IdentityProviders;
import software.amazon.awssdk.profiles.ProfileFile;
import software.amazon.awssdk.profiles.ProfileFileSystemSetting;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain;
import software.amazon.awssdk.services.polly.auth.scheme.PollyAuthSchemeProvider;
import software.amazon.awssdk.services.polly.internal.presigner.model.transform.SynthesizeSpeechRequestMarshaller;
import software.amazon.awssdk.services.polly.model.PollyRequest;
import software.amazon.awssdk.services.polly.presigner.PollyPresigner;
import software.amazon.awssdk.services.polly.presigner.model.PresignedSynthesizeSpeechRequest;
import software.amazon.awssdk.services.polly.presigner.model.SynthesizeSpeechPresignRequest;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.IoUtils;
import software.amazon.awssdk.utils.Validate;

/**
 * Default implementation of {@link PollyPresigner}.
 */
@SdkInternalApi
public final class DefaultPollyPresigner implements PollyPresigner {
    private static final String SIGNING_NAME = "polly";
    private static final String SERVICE_NAME = "polly";
    private static final Aws4Signer DEFAULT_SIGNER = Aws4Signer.create();

    private final Supplier<ProfileFile> profileFile;
    private final String profileName;
    private final Region region;
    private final IdentityProvider<? extends AwsCredentialsIdentity> credentialsProvider;
    private final URI endpointOverride;
    private final Boolean dualstackEnabled;
    private final Boolean fipsEnabled;

    private DefaultPollyPresigner(BuilderImpl builder) {
        this.profileFile = ProfileFile::defaultProfileFile;
        this.profileName = ProfileFileSystemSetting.AWS_PROFILE.getStringValueOrThrow();
        this.region = builder.region != null ? builder.region
                                             : DefaultAwsRegionProviderChain.builder()
                                                                            .profileFile(profileFile)
                                                                            .profileName(profileName)
                                                                            .build()
                                                                            .getRegion();
        this.credentialsProvider = builder.credentialsProvider != null ? builder.credentialsProvider
                                                                       : DefaultCredentialsProvider.builder()
                                                                                                   .profileFile(profileFile)
                                                                                                   .profileName(profileName)
                                                                                                   .build();
        this.endpointOverride = builder.endpointOverride;
        this.dualstackEnabled = builder.dualstackEnabled != null ? builder.dualstackEnabled
                                                                 : DualstackEnabledProvider.builder()
                                                                                           .profileFile(profileFile)
                                                                                           .profileName(profileName)
                                                                                           .build()
                                                                                           .isDualstackEnabled()
                                                                                           .orElse(false);
        this.fipsEnabled = builder.fipsEnabled != null ? builder.fipsEnabled
                                                       : FipsEnabledProvider.builder()
                                                                            .profileFile(profileFile)
                                                                            .profileName(profileName)
                                                                            .build()
                                                                            .isFipsEnabled()
                                                                            .orElse(false);
    }

    IdentityProvider<? extends AwsCredentialsIdentity> credentialsProvider() {
        return credentialsProvider;
    }

    @Override
    public void close() {
        IoUtils.closeIfCloseable(credentialsProvider, null);
    }

    public static PollyPresigner.Builder builder() {
        return new BuilderImpl();
    }

    @Override
    public PresignedSynthesizeSpeechRequest presignSynthesizeSpeech(
            SynthesizeSpeechPresignRequest synthesizeSpeechPresignRequest) {
        return presign(PresignedSynthesizeSpeechRequest.builder(),
                       synthesizeSpeechPresignRequest,
                       synthesizeSpeechPresignRequest.synthesizeSpeechRequest(),
                       SynthesizeSpeechRequestMarshaller.getInstance()::marshall)
                .build();
    }

    private <T extends PollyRequest> SdkHttpFullRequest marshallRequest(
            T request, Function<T, SdkHttpFullRequest.Builder> marshalFn) {
        SdkHttpFullRequest.Builder requestBuilder = marshalFn.apply(request);
        applyOverrideHeadersAndQueryParams(requestBuilder, request);
        applyEndpoint(requestBuilder);
        return requestBuilder.build();
    }

    /**
     * Generate a {@link PresignedRequest} from a {@link PresignedRequest} and {@link PollyRequest}.
     */
    private <T extends PresignedRequest.Builder, U extends PollyRequest> T presign(T presignedRequest,
                                                              PresignRequest presignRequest,
                                                              U requestToPresign,
                                                              Function<U, SdkHttpFullRequest.Builder> requestMarshaller) {
        ExecutionAttributes execAttrs = createExecutionAttributes(presignRequest, requestToPresign);

        SdkHttpFullRequest marshalledRequest = marshallRequest(requestToPresign, requestMarshaller);
        SdkHttpFullRequest signedHttpRequest = presignRequest(requestToPresign, marshalledRequest, execAttrs);
        initializePresignedRequest(presignedRequest, execAttrs, signedHttpRequest);
        return presignedRequest;
    }

    private void initializePresignedRequest(PresignedRequest.Builder presignedRequest,
                                            ExecutionAttributes execAttrs,
                                            SdkHttpFullRequest signedHttpRequest) {
        List<String> signedHeadersQueryParam = signedHttpRequest.firstMatchingRawQueryParameters("X-Amz-SignedHeaders");

        Map<String, List<String>> signedHeaders =
                signedHeadersQueryParam.stream()
                        .flatMap(h -> Stream.of(h.split(";")))
                        .collect(toMap(h -> h, h -> signedHttpRequest.firstMatchingHeader(h)
                                .map(Collections::singletonList)
                                .orElseGet(ArrayList::new)));

        boolean isBrowserExecutable = signedHttpRequest.method() == SdkHttpMethod.GET &&
                (signedHeaders.isEmpty() ||
                        (signedHeaders.size() == 1 && signedHeaders.containsKey("host")));

        presignedRequest.expiration(execAttrs.getAttribute(PRESIGNER_EXPIRATION))
                .isBrowserExecutable(isBrowserExecutable)
                .httpRequest(signedHttpRequest)
                .signedHeaders(signedHeaders);
    }

    private SdkHttpFullRequest presignRequest(PollyRequest requestToPresign,
                                              SdkHttpFullRequest marshalledRequest,
                                              ExecutionAttributes executionAttributes) {
        Presigner presigner = resolvePresigner(requestToPresign);
        SdkHttpFullRequest presigned = presigner.presign(marshalledRequest, executionAttributes);
        List<String> signedHeadersQueryParam = presigned.firstMatchingRawQueryParameters("X-Amz-SignedHeaders");
        Validate.validState(!signedHeadersQueryParam.isEmpty(),
                "Only SigV4 presigners are supported at this time, but the configured "
                        + "presigner (%s) did not seem to generate a SigV4 signature.", presigner);
        return presigned;
    }

    private ExecutionAttributes createExecutionAttributes(PresignRequest presignRequest, PollyRequest requestToPresign) {
        Instant signatureExpiration = Instant.now().plus(presignRequest.signatureDuration());
        AwsCredentialsIdentity credentials = resolveCredentials(resolveCredentialsProvider(requestToPresign));
        Validate.validState(credentials != null, "Credential providers must never return null.");

        return new ExecutionAttributes()
                .putAttribute(AwsSignerExecutionAttribute.AWS_CREDENTIALS, CredentialUtils.toCredentials(credentials))
                .putAttribute(AwsSignerExecutionAttribute.SERVICE_SIGNING_NAME, SIGNING_NAME)
                .putAttribute(AwsExecutionAttribute.AWS_REGION, region)
                .putAttribute(AwsSignerExecutionAttribute.SIGNING_REGION, region)
                .putAttribute(SdkInternalExecutionAttribute.IS_FULL_DUPLEX, false)
                .putAttribute(SdkExecutionAttribute.CLIENT_TYPE, ClientType.SYNC)
                .putAttribute(SdkExecutionAttribute.SERVICE_NAME, SERVICE_NAME)
                .putAttribute(PRESIGNER_EXPIRATION, signatureExpiration)
                .putAttribute(SdkInternalExecutionAttribute.AUTH_SCHEME_RESOLVER, PollyAuthSchemeProvider.defaultProvider())
                .putAttribute(SdkInternalExecutionAttribute.AUTH_SCHEMES, authSchemes())
                .putAttribute(SdkInternalExecutionAttribute.IDENTITY_PROVIDERS,
                              IdentityProviders.builder()
                                               .putIdentityProvider(credentialsProvider())
                                               .build());
    }

    private Map<String, AuthScheme<?>> authSchemes() {
        AwsV4AuthScheme awsV4AuthScheme = AwsV4AuthScheme.create();
        return Collections.singletonMap(awsV4AuthScheme.schemeId(), awsV4AuthScheme);
    }

    private IdentityProvider<? extends AwsCredentialsIdentity> resolveCredentialsProvider(PollyRequest request) {
        return request.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                .orElse(credentialsProvider);
    }

    private AwsCredentialsIdentity resolveCredentials(IdentityProvider<? extends AwsCredentialsIdentity> credentialsProvider) {
        return CompletableFutureUtils.joinLikeSync(credentialsProvider.resolveIdentity());
    }

    private Presigner resolvePresigner(PollyRequest request) {
        Signer signer = request.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::signer)
                .orElse(DEFAULT_SIGNER);

        return Validate.isInstanceOf(Presigner.class, signer,
                "Signer of type %s given in request override is not a Presigner", signer.getClass().getSimpleName());
    }

    private void applyOverrideHeadersAndQueryParams(SdkHttpFullRequest.Builder httpRequestBuilder, PollyRequest request) {
        request.overrideConfiguration().ifPresent(o -> {
            o.headers().forEach(httpRequestBuilder::putHeader);
            o.rawQueryParameters().forEach(httpRequestBuilder::putRawQueryParameter);
        });
    }

    private void applyEndpoint(SdkHttpFullRequest.Builder httpRequestBuilder) {
        URI uri = resolveEndpoint();
        httpRequestBuilder.protocol(uri.getScheme())
                .host(uri.getHost())
                .port(uri.getPort());
    }

    private URI resolveEndpoint() {
        if (endpointOverride != null) {
            return endpointOverride;
        }

        return new DefaultServiceEndpointBuilder(SERVICE_NAME, "https")
                .withRegion(region)
                .withProfileFile(profileFile)
                .withProfileName(profileName)
                .withDualstackEnabled(dualstackEnabled)
                .withFipsEnabled(fipsEnabled)
                .getServiceEndpoint();
    }

    public static class BuilderImpl implements PollyPresigner.Builder {
        private Region region;
        private IdentityProvider<? extends AwsCredentialsIdentity> credentialsProvider;
        private URI endpointOverride;
        private Boolean dualstackEnabled;
        private Boolean fipsEnabled;

        @Override
        public Builder region(Region region) {
            this.region = region;
            return this;
        }

        @Override
        public Builder credentialsProvider(AwsCredentialsProvider credentialsProvider) {
            return credentialsProvider((IdentityProvider<? extends AwsCredentialsIdentity>) credentialsProvider);
        }

        @Override
        public Builder credentialsProvider(IdentityProvider<? extends AwsCredentialsIdentity> credentialsProvider) {
            this.credentialsProvider = credentialsProvider;
            return this;
        }

        @Override
        public Builder dualstackEnabled(Boolean dualstackEnabled) {
            this.dualstackEnabled = dualstackEnabled;
            return this;
        }

        @Override
        public Builder fipsEnabled(Boolean fipsEnabled) {
            this.fipsEnabled = fipsEnabled;
            return this;
        }

        @Override
        public Builder endpointOverride(URI endpointOverride) {
            this.endpointOverride = endpointOverride;
            return this;
        }

        @Override
        public PollyPresigner build() {
            return new DefaultPollyPresigner(this);
        }
    }
}
