/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.polly;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.polly.model.DeleteLexiconRequest;
import software.amazon.awssdk.services.polly.model.DeleteLexiconResponse;
import software.amazon.awssdk.services.polly.model.DescribeVoicesRequest;
import software.amazon.awssdk.services.polly.model.DescribeVoicesResponse;
import software.amazon.awssdk.services.polly.model.GetLexiconRequest;
import software.amazon.awssdk.services.polly.model.GetLexiconResponse;
import software.amazon.awssdk.services.polly.model.GetSpeechSynthesisTaskRequest;
import software.amazon.awssdk.services.polly.model.GetSpeechSynthesisTaskResponse;
import software.amazon.awssdk.services.polly.model.ListLexiconsRequest;
import software.amazon.awssdk.services.polly.model.ListLexiconsResponse;
import software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest;
import software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse;
import software.amazon.awssdk.services.polly.model.PutLexiconRequest;
import software.amazon.awssdk.services.polly.model.PutLexiconResponse;
import software.amazon.awssdk.services.polly.model.StartSpeechSynthesisTaskRequest;
import software.amazon.awssdk.services.polly.model.StartSpeechSynthesisTaskResponse;
import software.amazon.awssdk.services.polly.model.SynthesizeSpeechRequest;
import software.amazon.awssdk.services.polly.model.SynthesizeSpeechResponse;
import software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher;

/**
 * Service client for accessing Amazon Polly asynchronously. This can be created using the static {@link #builder()}
 * method.The asynchronous client performs non-blocking I/O when configured with any {@link SdkAsyncHttpClient}
 * supported in the SDK. However, full non-blocking is not guaranteed as the async client may perform blocking calls in
 * some cases such as credentials retrieval and endpoint discovery as part of the async API call.
 *
 * <p>
 * Amazon Polly is a web service that makes it easy to synthesize speech from text.
 * </p>
 * <p>
 * The Amazon Polly service provides API operations for synthesizing high-quality speech from plain text and Speech
 * Synthesis Markup Language (SSML), along with managing pronunciations lexicons that enable you to get the best results
 * for your application domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface PollyAsyncClient extends AwsClient {
    String SERVICE_NAME = "polly";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "polly";

    /**
     * <p>
     * Deletes the specified pronunciation lexicon stored in an Amazon Web Services Region. A lexicon which has been
     * deleted is not available for speech synthesis, nor is it possible to retrieve it using either the
     * <code>GetLexicon</code> or <code>ListLexicon</code> APIs.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param deleteLexiconRequest
     * @return A Java Future containing the result of the DeleteLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DeleteLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DeleteLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteLexiconResponse> deleteLexicon(DeleteLexiconRequest deleteLexiconRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified pronunciation lexicon stored in an Amazon Web Services Region. A lexicon which has been
     * deleted is not available for speech synthesis, nor is it possible to retrieve it using either the
     * <code>GetLexicon</code> or <code>ListLexicon</code> APIs.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteLexiconRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteLexiconRequest#builder()}
     * </p>
     *
     * @param deleteLexiconRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.DeleteLexiconRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DeleteLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DeleteLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteLexiconResponse> deleteLexicon(Consumer<DeleteLexiconRequest.Builder> deleteLexiconRequest) {
        return deleteLexicon(DeleteLexiconRequest.builder().applyMutation(deleteLexiconRequest).build());
    }

    /**
     * <p>
     * Returns the list of voices that are available for use when requesting speech synthesis. Each voice speaks a
     * specified language, is either male or female, and is identified by an ID, which is the ASCII version of the voice
     * name.
     * </p>
     * <p>
     * When synthesizing speech ( <code>SynthesizeSpeech</code> ), you provide the voice ID for the voice you want from
     * the list of voices returned by <code>DescribeVoices</code>.
     * </p>
     * <p>
     * For example, you want your news reader application to read news in a specific language, but giving a user the
     * option to choose the voice. Using the <code>DescribeVoices</code> operation you can provide the user with a list
     * of available voices to select from.
     * </p>
     * <p>
     * You can optionally specify a language code to filter the available voices. For example, if you specify
     * <code>en-US</code>, the operation returns a list of all available US English voices.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>polly:DescribeVoices</code> action.
     * </p>
     *
     * @param describeVoicesRequest
     * @return A Java Future containing the result of the DescribeVoices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DescribeVoices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DescribeVoices" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeVoicesResponse> describeVoices(DescribeVoicesRequest describeVoicesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the list of voices that are available for use when requesting speech synthesis. Each voice speaks a
     * specified language, is either male or female, and is identified by an ID, which is the ASCII version of the voice
     * name.
     * </p>
     * <p>
     * When synthesizing speech ( <code>SynthesizeSpeech</code> ), you provide the voice ID for the voice you want from
     * the list of voices returned by <code>DescribeVoices</code>.
     * </p>
     * <p>
     * For example, you want your news reader application to read news in a specific language, but giving a user the
     * option to choose the voice. Using the <code>DescribeVoices</code> operation you can provide the user with a list
     * of available voices to select from.
     * </p>
     * <p>
     * You can optionally specify a language code to filter the available voices. For example, if you specify
     * <code>en-US</code>, the operation returns a list of all available US English voices.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>polly:DescribeVoices</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeVoicesRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeVoicesRequest#builder()}
     * </p>
     *
     * @param describeVoicesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.DescribeVoicesRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeVoices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DescribeVoices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DescribeVoices" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeVoicesResponse> describeVoices(Consumer<DescribeVoicesRequest.Builder> describeVoicesRequest) {
        return describeVoices(DescribeVoicesRequest.builder().applyMutation(describeVoicesRequest).build());
    }

    /**
     * <p>
     * Returns the list of voices that are available for use when requesting speech synthesis. Each voice speaks a
     * specified language, is either male or female, and is identified by an ID, which is the ASCII version of the voice
     * name.
     * </p>
     * <p>
     * When synthesizing speech ( <code>SynthesizeSpeech</code> ), you provide the voice ID for the voice you want from
     * the list of voices returned by <code>DescribeVoices</code>.
     * </p>
     * <p>
     * For example, you want your news reader application to read news in a specific language, but giving a user the
     * option to choose the voice. Using the <code>DescribeVoices</code> operation you can provide the user with a list
     * of available voices to select from.
     * </p>
     * <p>
     * You can optionally specify a language code to filter the available voices. For example, if you specify
     * <code>en-US</code>, the operation returns a list of all available US English voices.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>polly:DescribeVoices</code> action.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeVoices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DescribeVoices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DescribeVoices" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeVoicesResponse> describeVoices() {
        return describeVoices(DescribeVoicesRequest.builder().build());
    }

    /**
     * <p>
     * Returns the content of the specified pronunciation lexicon stored in an Amazon Web Services Region. For more
     * information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param getLexiconRequest
     * @return A Java Future containing the result of the GetLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetLexiconResponse> getLexicon(GetLexiconRequest getLexiconRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the content of the specified pronunciation lexicon stored in an Amazon Web Services Region. For more
     * information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetLexiconRequest.Builder} avoiding the need to
     * create one manually via {@link GetLexiconRequest#builder()}
     * </p>
     *
     * @param getLexiconRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.GetLexiconRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetLexiconResponse> getLexicon(Consumer<GetLexiconRequest.Builder> getLexiconRequest) {
        return getLexicon(GetLexiconRequest.builder().applyMutation(getLexiconRequest).build());
    }

    /**
     * <p>
     * Retrieves a specific SpeechSynthesisTask object based on its TaskID. This object contains information about the
     * given speech synthesis task, including the status of the task, and a link to the S3 bucket containing the output
     * of the task.
     * </p>
     *
     * @param getSpeechSynthesisTaskRequest
     * @return A Java Future containing the result of the GetSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTaskIdException The provided Task ID is not valid. Please provide a valid Task ID and try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SynthesisTaskNotFoundException The Speech Synthesis task with requested Task ID cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetSpeechSynthesisTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetSpeechSynthesisTask" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetSpeechSynthesisTaskResponse> getSpeechSynthesisTask(
            GetSpeechSynthesisTaskRequest getSpeechSynthesisTaskRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves a specific SpeechSynthesisTask object based on its TaskID. This object contains information about the
     * given speech synthesis task, including the status of the task, and a link to the S3 bucket containing the output
     * of the task.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSpeechSynthesisTaskRequest.Builder} avoiding the
     * need to create one manually via {@link GetSpeechSynthesisTaskRequest#builder()}
     * </p>
     *
     * @param getSpeechSynthesisTaskRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.GetSpeechSynthesisTaskRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTaskIdException The provided Task ID is not valid. Please provide a valid Task ID and try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SynthesisTaskNotFoundException The Speech Synthesis task with requested Task ID cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetSpeechSynthesisTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetSpeechSynthesisTask" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetSpeechSynthesisTaskResponse> getSpeechSynthesisTask(
            Consumer<GetSpeechSynthesisTaskRequest.Builder> getSpeechSynthesisTaskRequest) {
        return getSpeechSynthesisTask(GetSpeechSynthesisTaskRequest.builder().applyMutation(getSpeechSynthesisTaskRequest)
                .build());
    }

    /**
     * <p>
     * Returns a list of pronunciation lexicons stored in an Amazon Web Services Region. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing Lexicons</a>.
     * </p>
     *
     * @param listLexiconsRequest
     * @return A Java Future containing the result of the ListLexicons operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListLexicons
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListLexicons" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListLexiconsResponse> listLexicons(ListLexiconsRequest listLexiconsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of pronunciation lexicons stored in an Amazon Web Services Region. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing Lexicons</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListLexiconsRequest.Builder} avoiding the need to
     * create one manually via {@link ListLexiconsRequest#builder()}
     * </p>
     *
     * @param listLexiconsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.ListLexiconsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListLexicons operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListLexicons
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListLexicons" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListLexiconsResponse> listLexicons(Consumer<ListLexiconsRequest.Builder> listLexiconsRequest) {
        return listLexicons(ListLexiconsRequest.builder().applyMutation(listLexiconsRequest).build());
    }

    /**
     * <p>
     * Returns a list of pronunciation lexicons stored in an Amazon Web Services Region. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing Lexicons</a>.
     * </p>
     *
     * @return A Java Future containing the result of the ListLexicons operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListLexicons
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListLexicons" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListLexiconsResponse> listLexicons() {
        return listLexicons(ListLexiconsRequest.builder().build());
    }

    /**
     * <p>
     * Returns a list of SpeechSynthesisTask objects ordered by their creation date. This operation can filter the tasks
     * by their status, for example, allowing users to list only tasks that are completed.
     * </p>
     *
     * @param listSpeechSynthesisTasksRequest
     * @return A Java Future containing the result of the ListSpeechSynthesisTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListSpeechSynthesisTasksResponse> listSpeechSynthesisTasks(
            ListSpeechSynthesisTasksRequest listSpeechSynthesisTasksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of SpeechSynthesisTask objects ordered by their creation date. This operation can filter the tasks
     * by their status, for example, allowing users to list only tasks that are completed.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSpeechSynthesisTasksRequest.Builder} avoiding
     * the need to create one manually via {@link ListSpeechSynthesisTasksRequest#builder()}
     * </p>
     *
     * @param listSpeechSynthesisTasksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListSpeechSynthesisTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListSpeechSynthesisTasksResponse> listSpeechSynthesisTasks(
            Consumer<ListSpeechSynthesisTasksRequest.Builder> listSpeechSynthesisTasksRequest) {
        return listSpeechSynthesisTasks(ListSpeechSynthesisTasksRequest.builder().applyMutation(listSpeechSynthesisTasksRequest)
                .build());
    }

    /**
     * <p>
     * Returns a list of SpeechSynthesisTask objects ordered by their creation date. This operation can filter the tasks
     * by their status, for example, allowing users to list only tasks that are completed.
     * </p>
     *
     * @return A Java Future containing the result of the ListSpeechSynthesisTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListSpeechSynthesisTasksResponse> listSpeechSynthesisTasks() {
        return listSpeechSynthesisTasks(ListSpeechSynthesisTasksRequest.builder().build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSpeechSynthesisTasksPublisher listSpeechSynthesisTasksPaginator() {
        return listSpeechSynthesisTasksPaginator(ListSpeechSynthesisTasksRequest.builder().build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSpeechSynthesisTasksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSpeechSynthesisTasksPublisher listSpeechSynthesisTasksPaginator(
            ListSpeechSynthesisTasksRequest listSpeechSynthesisTasksRequest) {
        return new ListSpeechSynthesisTasksPublisher(this, listSpeechSynthesisTasksRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.polly.paginators.ListSpeechSynthesisTasksPublisher publisher = client.listSpeechSynthesisTasksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSpeechSynthesisTasks(software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSpeechSynthesisTasksRequest.Builder} avoiding
     * the need to create one manually via {@link ListSpeechSynthesisTasksRequest#builder()}
     * </p>
     *
     * @param listSpeechSynthesisTasksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSpeechSynthesisTasksPublisher listSpeechSynthesisTasksPaginator(
            Consumer<ListSpeechSynthesisTasksRequest.Builder> listSpeechSynthesisTasksRequest) {
        return listSpeechSynthesisTasksPaginator(ListSpeechSynthesisTasksRequest.builder()
                .applyMutation(listSpeechSynthesisTasksRequest).build());
    }

    /**
     * <p>
     * Stores a pronunciation lexicon in an Amazon Web Services Region. If a lexicon with the same name already exists
     * in the region, it is overwritten by the new lexicon. Lexicon operations have eventual consistency, therefore, it
     * might take some time before the lexicon is available to the SynthesizeSpeech operation.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param putLexiconRequest
     * @return A Java Future containing the result of the PutLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidLexiconException Amazon Polly can't find the specified lexicon. Verify that the lexicon's name
     *         is spelled correctly, and then try again.</li>
     *         <li>UnsupportedPlsAlphabetException The alphabet specified by the lexicon is not a supported alphabet.
     *         Valid values are <code>x-sampa</code> and <code>ipa</code>.</li>
     *         <li>UnsupportedPlsLanguageException The language specified in the lexicon is unsupported. For a list of
     *         supported languages, see <a
     *         href="https://docs.aws.amazon.com/polly/latest/dg/API_LexiconAttributes.html">Lexicon Attributes</a>.</li>
     *         <li>LexiconSizeExceededException The maximum size of the specified lexicon would be exceeded by this
     *         operation.</li>
     *         <li>MaxLexemeLengthExceededException The maximum size of the lexeme would be exceeded by this operation.</li>
     *         <li>MaxLexiconsNumberExceededException The maximum number of lexicons would be exceeded by this
     *         operation.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.PutLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/PutLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutLexiconResponse> putLexicon(PutLexiconRequest putLexiconRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Stores a pronunciation lexicon in an Amazon Web Services Region. If a lexicon with the same name already exists
     * in the region, it is overwritten by the new lexicon. Lexicon operations have eventual consistency, therefore, it
     * might take some time before the lexicon is available to the SynthesizeSpeech operation.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutLexiconRequest.Builder} avoiding the need to
     * create one manually via {@link PutLexiconRequest#builder()}
     * </p>
     *
     * @param putLexiconRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.PutLexiconRequest.Builder} to create a request.
     * @return A Java Future containing the result of the PutLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidLexiconException Amazon Polly can't find the specified lexicon. Verify that the lexicon's name
     *         is spelled correctly, and then try again.</li>
     *         <li>UnsupportedPlsAlphabetException The alphabet specified by the lexicon is not a supported alphabet.
     *         Valid values are <code>x-sampa</code> and <code>ipa</code>.</li>
     *         <li>UnsupportedPlsLanguageException The language specified in the lexicon is unsupported. For a list of
     *         supported languages, see <a
     *         href="https://docs.aws.amazon.com/polly/latest/dg/API_LexiconAttributes.html">Lexicon Attributes</a>.</li>
     *         <li>LexiconSizeExceededException The maximum size of the specified lexicon would be exceeded by this
     *         operation.</li>
     *         <li>MaxLexemeLengthExceededException The maximum size of the lexeme would be exceeded by this operation.</li>
     *         <li>MaxLexiconsNumberExceededException The maximum number of lexicons would be exceeded by this
     *         operation.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.PutLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/PutLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<PutLexiconResponse> putLexicon(Consumer<PutLexiconRequest.Builder> putLexiconRequest) {
        return putLexicon(PutLexiconRequest.builder().applyMutation(putLexiconRequest).build());
    }

    /**
     * <p>
     * Allows the creation of an asynchronous synthesis task, by starting a new <code>SpeechSynthesisTask</code>. This
     * operation requires all the standard information needed for speech synthesis, plus the name of an Amazon S3 bucket
     * for the service to store the output of the synthesis task and two optional parameters (
     * <code>OutputS3KeyPrefix</code> and <code>SnsTopicArn</code>). Once the synthesis task is created, this operation
     * will return a <code>SpeechSynthesisTask</code> object, which will include an identifier of this task as well as
     * the current status. The <code>SpeechSynthesisTask</code> object is available for 72 hours after starting the
     * asynchronous synthesis task.
     * </p>
     *
     * @param startSpeechSynthesisTaskRequest
     * @return A Java Future containing the result of the StartSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidS3BucketException The provided Amazon S3 bucket name is invalid. Please check your input with
     *         S3 bucket naming requirements and try again.</li>
     *         <li>InvalidS3KeyException The provided Amazon S3 key prefix is invalid. Please provide a valid S3 object
     *         key name.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSnsTopicArnException The provided SNS topic ARN is invalid. Please provide a valid SNS topic
     *         ARN and try again.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.StartSpeechSynthesisTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/StartSpeechSynthesisTask"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartSpeechSynthesisTaskResponse> startSpeechSynthesisTask(
            StartSpeechSynthesisTaskRequest startSpeechSynthesisTaskRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Allows the creation of an asynchronous synthesis task, by starting a new <code>SpeechSynthesisTask</code>. This
     * operation requires all the standard information needed for speech synthesis, plus the name of an Amazon S3 bucket
     * for the service to store the output of the synthesis task and two optional parameters (
     * <code>OutputS3KeyPrefix</code> and <code>SnsTopicArn</code>). Once the synthesis task is created, this operation
     * will return a <code>SpeechSynthesisTask</code> object, which will include an identifier of this task as well as
     * the current status. The <code>SpeechSynthesisTask</code> object is available for 72 hours after starting the
     * asynchronous synthesis task.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartSpeechSynthesisTaskRequest.Builder} avoiding
     * the need to create one manually via {@link StartSpeechSynthesisTaskRequest#builder()}
     * </p>
     *
     * @param startSpeechSynthesisTaskRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.StartSpeechSynthesisTaskRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the StartSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidS3BucketException The provided Amazon S3 bucket name is invalid. Please check your input with
     *         S3 bucket naming requirements and try again.</li>
     *         <li>InvalidS3KeyException The provided Amazon S3 key prefix is invalid. Please provide a valid S3 object
     *         key name.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSnsTopicArnException The provided SNS topic ARN is invalid. Please provide a valid SNS topic
     *         ARN and try again.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.StartSpeechSynthesisTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/StartSpeechSynthesisTask"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<StartSpeechSynthesisTaskResponse> startSpeechSynthesisTask(
            Consumer<StartSpeechSynthesisTaskRequest.Builder> startSpeechSynthesisTaskRequest) {
        return startSpeechSynthesisTask(StartSpeechSynthesisTaskRequest.builder().applyMutation(startSpeechSynthesisTaskRequest)
                .build());
    }

    /**
     * <p>
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes. SSML input must be valid, well-formed SSML.
     * Some alphabets might not be available with all the voices (for example, Cyrillic might not be read at all by
     * English voices) unless phoneme mapping is used. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/how-text-to-speech-works.html">How it Works</a>.
     * </p>
     *
     * @param synthesizeSpeechRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        Stream containing the synthesized speech.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.SynthesizeSpeech
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/SynthesizeSpeech" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> synthesizeSpeech(SynthesizeSpeechRequest synthesizeSpeechRequest,
            AsyncResponseTransformer<SynthesizeSpeechResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes. SSML input must be valid, well-formed SSML.
     * Some alphabets might not be available with all the voices (for example, Cyrillic might not be read at all by
     * English voices) unless phoneme mapping is used. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/how-text-to-speech-works.html">How it Works</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SynthesizeSpeechRequest.Builder} avoiding the need
     * to create one manually via {@link SynthesizeSpeechRequest#builder()}
     * </p>
     *
     * @param synthesizeSpeechRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.SynthesizeSpeechRequest.Builder} to create a request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        Stream containing the synthesized speech.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.SynthesizeSpeech
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/SynthesizeSpeech" target="_top">AWS API
     *      Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> synthesizeSpeech(
            Consumer<SynthesizeSpeechRequest.Builder> synthesizeSpeechRequest,
            AsyncResponseTransformer<SynthesizeSpeechResponse, ReturnT> asyncResponseTransformer) {
        return synthesizeSpeech(SynthesizeSpeechRequest.builder().applyMutation(synthesizeSpeechRequest).build(),
                asyncResponseTransformer);
    }

    /**
     * <p>
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes. SSML input must be valid, well-formed SSML.
     * Some alphabets might not be available with all the voices (for example, Cyrillic might not be read at all by
     * English voices) unless phoneme mapping is used. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/how-text-to-speech-works.html">How it Works</a>.
     * </p>
     *
     * @param synthesizeSpeechRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        Stream containing the synthesized speech.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.SynthesizeSpeech
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/SynthesizeSpeech" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<SynthesizeSpeechResponse> synthesizeSpeech(SynthesizeSpeechRequest synthesizeSpeechRequest,
            Path destinationPath) {
        return synthesizeSpeech(synthesizeSpeechRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes. SSML input must be valid, well-formed SSML.
     * Some alphabets might not be available with all the voices (for example, Cyrillic might not be read at all by
     * English voices) unless phoneme mapping is used. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/how-text-to-speech-works.html">How it Works</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SynthesizeSpeechRequest.Builder} avoiding the need
     * to create one manually via {@link SynthesizeSpeechRequest#builder()}
     * </p>
     *
     * @param synthesizeSpeechRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.polly.model.SynthesizeSpeechRequest.Builder} to create a request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        Stream containing the synthesized speech.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.SynthesizeSpeech
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/SynthesizeSpeech" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<SynthesizeSpeechResponse> synthesizeSpeech(
            Consumer<SynthesizeSpeechRequest.Builder> synthesizeSpeechRequest, Path destinationPath) {
        return synthesizeSpeech(SynthesizeSpeechRequest.builder().applyMutation(synthesizeSpeechRequest).build(), destinationPath);
    }

    @Override
    default PollyServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link PollyAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static PollyAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link PollyAsyncClient}.
     */
    static PollyAsyncClientBuilder builder() {
        return new DefaultPollyAsyncClientBuilder();
    }
}
