/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.polly;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.polly.internal.PollyServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.polly.model.DeleteLexiconRequest;
import software.amazon.awssdk.services.polly.model.DeleteLexiconResponse;
import software.amazon.awssdk.services.polly.model.DescribeVoicesRequest;
import software.amazon.awssdk.services.polly.model.DescribeVoicesResponse;
import software.amazon.awssdk.services.polly.model.EngineNotSupportedException;
import software.amazon.awssdk.services.polly.model.GetLexiconRequest;
import software.amazon.awssdk.services.polly.model.GetLexiconResponse;
import software.amazon.awssdk.services.polly.model.GetSpeechSynthesisTaskRequest;
import software.amazon.awssdk.services.polly.model.GetSpeechSynthesisTaskResponse;
import software.amazon.awssdk.services.polly.model.InvalidLexiconException;
import software.amazon.awssdk.services.polly.model.InvalidNextTokenException;
import software.amazon.awssdk.services.polly.model.InvalidS3BucketException;
import software.amazon.awssdk.services.polly.model.InvalidS3KeyException;
import software.amazon.awssdk.services.polly.model.InvalidSampleRateException;
import software.amazon.awssdk.services.polly.model.InvalidSnsTopicArnException;
import software.amazon.awssdk.services.polly.model.InvalidSsmlException;
import software.amazon.awssdk.services.polly.model.InvalidTaskIdException;
import software.amazon.awssdk.services.polly.model.LanguageNotSupportedException;
import software.amazon.awssdk.services.polly.model.LexiconNotFoundException;
import software.amazon.awssdk.services.polly.model.LexiconSizeExceededException;
import software.amazon.awssdk.services.polly.model.ListLexiconsRequest;
import software.amazon.awssdk.services.polly.model.ListLexiconsResponse;
import software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksRequest;
import software.amazon.awssdk.services.polly.model.ListSpeechSynthesisTasksResponse;
import software.amazon.awssdk.services.polly.model.MarksNotSupportedForFormatException;
import software.amazon.awssdk.services.polly.model.MaxLexemeLengthExceededException;
import software.amazon.awssdk.services.polly.model.MaxLexiconsNumberExceededException;
import software.amazon.awssdk.services.polly.model.PollyException;
import software.amazon.awssdk.services.polly.model.PutLexiconRequest;
import software.amazon.awssdk.services.polly.model.PutLexiconResponse;
import software.amazon.awssdk.services.polly.model.ServiceFailureException;
import software.amazon.awssdk.services.polly.model.SsmlMarksNotSupportedForTextTypeException;
import software.amazon.awssdk.services.polly.model.StartSpeechSynthesisTaskRequest;
import software.amazon.awssdk.services.polly.model.StartSpeechSynthesisTaskResponse;
import software.amazon.awssdk.services.polly.model.SynthesisTaskNotFoundException;
import software.amazon.awssdk.services.polly.model.SynthesizeSpeechRequest;
import software.amazon.awssdk.services.polly.model.SynthesizeSpeechResponse;
import software.amazon.awssdk.services.polly.model.TextLengthExceededException;
import software.amazon.awssdk.services.polly.model.UnsupportedPlsAlphabetException;
import software.amazon.awssdk.services.polly.model.UnsupportedPlsLanguageException;
import software.amazon.awssdk.services.polly.transform.DeleteLexiconRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.DescribeVoicesRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.GetLexiconRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.GetSpeechSynthesisTaskRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.ListLexiconsRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.ListSpeechSynthesisTasksRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.PutLexiconRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.StartSpeechSynthesisTaskRequestMarshaller;
import software.amazon.awssdk.services.polly.transform.SynthesizeSpeechRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link PollyAsyncClient}.
 *
 * @see PollyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPollyAsyncClient implements PollyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPollyAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPollyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the specified pronunciation lexicon stored in an Amazon Web Services Region. A lexicon which has been
     * deleted is not available for speech synthesis, nor is it possible to retrieve it using either the
     * <code>GetLexicon</code> or <code>ListLexicon</code> APIs.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param deleteLexiconRequest
     * @return A Java Future containing the result of the DeleteLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DeleteLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DeleteLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLexiconResponse> deleteLexicon(DeleteLexiconRequest deleteLexiconRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLexiconRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLexiconRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLexicon");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLexiconResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLexiconResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLexiconResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLexiconRequest, DeleteLexiconResponse>()
                            .withOperationName("DeleteLexicon").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteLexiconRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLexiconRequest));
            CompletableFuture<DeleteLexiconResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of voices that are available for use when requesting speech synthesis. Each voice speaks a
     * specified language, is either male or female, and is identified by an ID, which is the ASCII version of the voice
     * name.
     * </p>
     * <p>
     * When synthesizing speech ( <code>SynthesizeSpeech</code> ), you provide the voice ID for the voice you want from
     * the list of voices returned by <code>DescribeVoices</code>.
     * </p>
     * <p>
     * For example, you want your news reader application to read news in a specific language, but giving a user the
     * option to choose the voice. Using the <code>DescribeVoices</code> operation you can provide the user with a list
     * of available voices to select from.
     * </p>
     * <p>
     * You can optionally specify a language code to filter the available voices. For example, if you specify
     * <code>en-US</code>, the operation returns a list of all available US English voices.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>polly:DescribeVoices</code> action.
     * </p>
     *
     * @param describeVoicesRequest
     * @return A Java Future containing the result of the DescribeVoices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.DescribeVoices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/DescribeVoices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVoicesResponse> describeVoices(DescribeVoicesRequest describeVoicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVoicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVoicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVoices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVoicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVoicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVoicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVoicesRequest, DescribeVoicesResponse>()
                            .withOperationName("DescribeVoices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVoicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVoicesRequest));
            CompletableFuture<DescribeVoicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the content of the specified pronunciation lexicon stored in an Amazon Web Services Region. For more
     * information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param getLexiconRequest
     * @return A Java Future containing the result of the GetLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLexiconResponse> getLexicon(GetLexiconRequest getLexiconRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLexiconRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLexiconRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLexicon");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLexiconResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLexiconResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLexiconResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLexiconRequest, GetLexiconResponse>().withOperationName("GetLexicon")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLexiconRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLexiconRequest));
            CompletableFuture<GetLexiconResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a specific SpeechSynthesisTask object based on its TaskID. This object contains information about the
     * given speech synthesis task, including the status of the task, and a link to the S3 bucket containing the output
     * of the task.
     * </p>
     *
     * @param getSpeechSynthesisTaskRequest
     * @return A Java Future containing the result of the GetSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidTaskIdException The provided Task ID is not valid. Please provide a valid Task ID and try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SynthesisTaskNotFoundException The Speech Synthesis task with requested Task ID cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.GetSpeechSynthesisTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/GetSpeechSynthesisTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSpeechSynthesisTaskResponse> getSpeechSynthesisTask(
            GetSpeechSynthesisTaskRequest getSpeechSynthesisTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSpeechSynthesisTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSpeechSynthesisTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSpeechSynthesisTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSpeechSynthesisTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSpeechSynthesisTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSpeechSynthesisTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSpeechSynthesisTaskRequest, GetSpeechSynthesisTaskResponse>()
                            .withOperationName("GetSpeechSynthesisTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSpeechSynthesisTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSpeechSynthesisTaskRequest));
            CompletableFuture<GetSpeechSynthesisTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of pronunciation lexicons stored in an Amazon Web Services Region. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing Lexicons</a>.
     * </p>
     *
     * @param listLexiconsRequest
     * @return A Java Future containing the result of the ListLexicons operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListLexicons
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListLexicons" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLexiconsResponse> listLexicons(ListLexiconsRequest listLexiconsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLexiconsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLexiconsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLexicons");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLexiconsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLexiconsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLexiconsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLexiconsRequest, ListLexiconsResponse>()
                            .withOperationName("ListLexicons").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLexiconsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLexiconsRequest));
            CompletableFuture<ListLexiconsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of SpeechSynthesisTask objects ordered by their creation date. This operation can filter the tasks
     * by their status, for example, allowing users to list only tasks that are completed.
     * </p>
     *
     * @param listSpeechSynthesisTasksRequest
     * @return A Java Future containing the result of the ListSpeechSynthesisTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidNextTokenException The NextToken is invalid. Verify that it's spelled correctly, and then try
     *         again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.ListSpeechSynthesisTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/ListSpeechSynthesisTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSpeechSynthesisTasksResponse> listSpeechSynthesisTasks(
            ListSpeechSynthesisTasksRequest listSpeechSynthesisTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSpeechSynthesisTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSpeechSynthesisTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSpeechSynthesisTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSpeechSynthesisTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSpeechSynthesisTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSpeechSynthesisTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSpeechSynthesisTasksRequest, ListSpeechSynthesisTasksResponse>()
                            .withOperationName("ListSpeechSynthesisTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSpeechSynthesisTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSpeechSynthesisTasksRequest));
            CompletableFuture<ListSpeechSynthesisTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stores a pronunciation lexicon in an Amazon Web Services Region. If a lexicon with the same name already exists
     * in the region, it is overwritten by the new lexicon. Lexicon operations have eventual consistency, therefore, it
     * might take some time before the lexicon is available to the SynthesizeSpeech operation.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/polly/latest/dg/managing-lexicons.html">Managing
     * Lexicons</a>.
     * </p>
     *
     * @param putLexiconRequest
     * @return A Java Future containing the result of the PutLexicon operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidLexiconException Amazon Polly can't find the specified lexicon. Verify that the lexicon's name
     *         is spelled correctly, and then try again.</li>
     *         <li>UnsupportedPlsAlphabetException The alphabet specified by the lexicon is not a supported alphabet.
     *         Valid values are <code>x-sampa</code> and <code>ipa</code>.</li>
     *         <li>UnsupportedPlsLanguageException The language specified in the lexicon is unsupported. For a list of
     *         supported languages, see <a
     *         href="https://docs.aws.amazon.com/polly/latest/dg/API_LexiconAttributes.html">Lexicon Attributes</a>.</li>
     *         <li>LexiconSizeExceededException The maximum size of the specified lexicon would be exceeded by this
     *         operation.</li>
     *         <li>MaxLexemeLengthExceededException The maximum size of the lexeme would be exceeded by this operation.</li>
     *         <li>MaxLexiconsNumberExceededException The maximum number of lexicons would be exceeded by this
     *         operation.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.PutLexicon
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/PutLexicon" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutLexiconResponse> putLexicon(PutLexiconRequest putLexiconRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLexiconRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLexiconRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLexicon");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLexiconResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutLexiconResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLexiconResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLexiconRequest, PutLexiconResponse>().withOperationName("PutLexicon")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutLexiconRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putLexiconRequest));
            CompletableFuture<PutLexiconResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows the creation of an asynchronous synthesis task, by starting a new <code>SpeechSynthesisTask</code>. This
     * operation requires all the standard information needed for speech synthesis, plus the name of an Amazon S3 bucket
     * for the service to store the output of the synthesis task and two optional parameters (
     * <code>OutputS3KeyPrefix</code> and <code>SnsTopicArn</code>). Once the synthesis task is created, this operation
     * will return a <code>SpeechSynthesisTask</code> object, which will include an identifier of this task as well as
     * the current status. The <code>SpeechSynthesisTask</code> object is available for 72 hours after starting the
     * asynchronous synthesis task.
     * </p>
     *
     * @param startSpeechSynthesisTaskRequest
     * @return A Java Future containing the result of the StartSpeechSynthesisTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidS3BucketException The provided Amazon S3 bucket name is invalid. Please check your input with
     *         S3 bucket naming requirements and try again.</li>
     *         <li>InvalidS3KeyException The provided Amazon S3 key prefix is invalid. Please provide a valid S3 object
     *         key name.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSnsTopicArnException The provided SNS topic ARN is invalid. Please provide a valid SNS topic
     *         ARN and try again.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.StartSpeechSynthesisTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/StartSpeechSynthesisTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSpeechSynthesisTaskResponse> startSpeechSynthesisTask(
            StartSpeechSynthesisTaskRequest startSpeechSynthesisTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSpeechSynthesisTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSpeechSynthesisTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSpeechSynthesisTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSpeechSynthesisTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSpeechSynthesisTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSpeechSynthesisTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSpeechSynthesisTaskRequest, StartSpeechSynthesisTaskResponse>()
                            .withOperationName("StartSpeechSynthesisTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSpeechSynthesisTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startSpeechSynthesisTaskRequest));
            CompletableFuture<StartSpeechSynthesisTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Synthesizes UTF-8 input, plain text or SSML, to a stream of bytes. SSML input must be valid, well-formed SSML.
     * Some alphabets might not be available with all the voices (for example, Cyrillic might not be read at all by
     * English voices) unless phoneme mapping is used. For more information, see <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/how-text-to-speech-works.html">How it Works</a>.
     * </p>
     *
     * @param synthesizeSpeechRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        Stream containing the synthesized speech.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>TextLengthExceededException The value of the "Text" parameter is longer than the accepted limits. For
     *         the <code>SynthesizeSpeech</code> API, the limit for input text is a maximum of 6000 characters total, of
     *         which no more than 3000 can be billed characters. For the <code>StartSpeechSynthesisTask</code> API, the
     *         maximum is 200,000 characters, of which no more than 100,000 can be billed characters. SSML tags are not
     *         counted as billed characters.</li>
     *         <li>InvalidSampleRateException The specified sample rate is not valid.</li>
     *         <li>InvalidSsmlException The SSML you provided is invalid. Verify the SSML syntax, spelling of tags and
     *         values, and then try again.</li>
     *         <li>LexiconNotFoundException Amazon Polly can't find the specified lexicon. This could be caused by a
     *         lexicon that is missing, its name is misspelled or specifying a lexicon that is in a different
     *         region.</p>
     *         <p>
     *         Verify that the lexicon exists, is in the region (see <a>ListLexicons</a>) and that you spelled its name
     *         is spelled correctly. Then try again.</li>
     *         <li>ServiceFailureException An unknown condition has caused a service failure.</li>
     *         <li>MarksNotSupportedForFormatException Speech marks are not supported for the <code>OutputFormat</code>
     *         selected. Speech marks are only available for content in <code>json</code> format.</li>
     *         <li>SsmlMarksNotSupportedForTextTypeException SSML speech marks are not supported for plain text-type
     *         input.</li>
     *         <li>LanguageNotSupportedException The language specified is not currently supported by Amazon Polly in
     *         this capacity.</li>
     *         <li>EngineNotSupportedException This engine is not compatible with the voice that you have designated.
     *         Choose a new voice that is compatible with the engine or change the engine and restart the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PollyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample PollyAsyncClient.SynthesizeSpeech
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/polly-2016-06-10/SynthesizeSpeech" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> synthesizeSpeech(SynthesizeSpeechRequest synthesizeSpeechRequest,
            AsyncResponseTransformer<SynthesizeSpeechResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(synthesizeSpeechRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, synthesizeSpeechRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Polly");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SynthesizeSpeech");
            Pair<AsyncResponseTransformer<SynthesizeSpeechResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<SynthesizeSpeechResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SynthesizeSpeechResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<SynthesizeSpeechRequest, SynthesizeSpeechResponse>()
                            .withOperationName("SynthesizeSpeech").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SynthesizeSpeechRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(synthesizeSpeechRequest), asyncResponseTransformer);
            AsyncResponseTransformer<SynthesizeSpeechResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<SynthesizeSpeechResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final PollyServiceClientConfiguration serviceClientConfiguration() {
        return new PollyServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PollyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3BucketException")
                                .exceptionBuilderSupplier(InvalidS3BucketException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSampleRateException")
                                .exceptionBuilderSupplier(InvalidSampleRateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SynthesisTaskNotFoundException")
                                .exceptionBuilderSupplier(SynthesisTaskNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaxLexiconsNumberExceededException")
                                .exceptionBuilderSupplier(MaxLexiconsNumberExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaxLexemeLengthExceededException")
                                .exceptionBuilderSupplier(MaxLexemeLengthExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedPlsAlphabetException")
                                .exceptionBuilderSupplier(UnsupportedPlsAlphabetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LanguageNotSupportedException")
                                .exceptionBuilderSupplier(LanguageNotSupportedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextLengthExceededException")
                                .exceptionBuilderSupplier(TextLengthExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLexiconException")
                                .exceptionBuilderSupplier(InvalidLexiconException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SsmlMarksNotSupportedForTextTypeException")
                                .exceptionBuilderSupplier(SsmlMarksNotSupportedForTextTypeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3KeyException")
                                .exceptionBuilderSupplier(InvalidS3KeyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSnsTopicArnException")
                                .exceptionBuilderSupplier(InvalidSnsTopicArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MarksNotSupportedForFormatException")
                                .exceptionBuilderSupplier(MarksNotSupportedForFormatException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTaskIdException")
                                .exceptionBuilderSupplier(InvalidTaskIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedPlsLanguageException")
                                .exceptionBuilderSupplier(UnsupportedPlsLanguageException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LexiconNotFoundException")
                                .exceptionBuilderSupplier(LexiconNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LexiconSizeExceededException")
                                .exceptionBuilderSupplier(LexiconSizeExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSsmlException")
                                .exceptionBuilderSupplier(InvalidSsmlException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EngineNotSupportedException")
                                .exceptionBuilderSupplier(EngineNotSupportedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        PollyServiceClientConfigurationBuilder serviceConfigBuilder = new PollyServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
