/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.polly.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * SynthesisTask object that provides information about a speech synthesis task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SynthesisTask implements SdkPojo, Serializable, ToCopyableBuilder<SynthesisTask.Builder, SynthesisTask> {
    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(SynthesisTask::engineAsString)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> TASK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("TaskId")
            .getter(getter(SynthesisTask::taskId)).setter(setter(Builder::taskId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskId").build()).build();

    private static final SdkField<String> TASK_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskStatus").getter(getter(SynthesisTask::taskStatusAsString)).setter(setter(Builder::taskStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskStatus").build()).build();

    private static final SdkField<String> TASK_STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskStatusReason").getter(getter(SynthesisTask::taskStatusReason))
            .setter(setter(Builder::taskStatusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskStatusReason").build()).build();

    private static final SdkField<String> OUTPUT_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputUri").getter(getter(SynthesisTask::outputUri)).setter(setter(Builder::outputUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputUri").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(SynthesisTask::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Integer> REQUEST_CHARACTERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RequestCharacters").getter(getter(SynthesisTask::requestCharacters))
            .setter(setter(Builder::requestCharacters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestCharacters").build()).build();

    private static final SdkField<String> SNS_TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnsTopicArn").getter(getter(SynthesisTask::snsTopicArn)).setter(setter(Builder::snsTopicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsTopicArn").build()).build();

    private static final SdkField<List<String>> LEXICON_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LexiconNames")
            .getter(getter(SynthesisTask::lexiconNames))
            .setter(setter(Builder::lexiconNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LexiconNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OUTPUT_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputFormat").getter(getter(SynthesisTask::outputFormatAsString)).setter(setter(Builder::outputFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputFormat").build()).build();

    private static final SdkField<String> SAMPLE_RATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SampleRate").getter(getter(SynthesisTask::sampleRate)).setter(setter(Builder::sampleRate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleRate").build()).build();

    private static final SdkField<List<String>> SPEECH_MARK_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SpeechMarkTypes")
            .getter(getter(SynthesisTask::speechMarkTypesAsStrings))
            .setter(setter(Builder::speechMarkTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpeechMarkTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TEXT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TextType").getter(getter(SynthesisTask::textTypeAsString)).setter(setter(Builder::textType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextType").build()).build();

    private static final SdkField<String> VOICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VoiceId").getter(getter(SynthesisTask::voiceIdAsString)).setter(setter(Builder::voiceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VoiceId").build()).build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(SynthesisTask::languageCodeAsString)).setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENGINE_FIELD, TASK_ID_FIELD,
            TASK_STATUS_FIELD, TASK_STATUS_REASON_FIELD, OUTPUT_URI_FIELD, CREATION_TIME_FIELD, REQUEST_CHARACTERS_FIELD,
            SNS_TOPIC_ARN_FIELD, LEXICON_NAMES_FIELD, OUTPUT_FORMAT_FIELD, SAMPLE_RATE_FIELD, SPEECH_MARK_TYPES_FIELD,
            TEXT_TYPE_FIELD, VOICE_ID_FIELD, LANGUAGE_CODE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String engine;

    private final String taskId;

    private final String taskStatus;

    private final String taskStatusReason;

    private final String outputUri;

    private final Instant creationTime;

    private final Integer requestCharacters;

    private final String snsTopicArn;

    private final List<String> lexiconNames;

    private final String outputFormat;

    private final String sampleRate;

    private final List<String> speechMarkTypes;

    private final String textType;

    private final String voiceId;

    private final String languageCode;

    private SynthesisTask(BuilderImpl builder) {
        this.engine = builder.engine;
        this.taskId = builder.taskId;
        this.taskStatus = builder.taskStatus;
        this.taskStatusReason = builder.taskStatusReason;
        this.outputUri = builder.outputUri;
        this.creationTime = builder.creationTime;
        this.requestCharacters = builder.requestCharacters;
        this.snsTopicArn = builder.snsTopicArn;
        this.lexiconNames = builder.lexiconNames;
        this.outputFormat = builder.outputFormat;
        this.sampleRate = builder.sampleRate;
        this.speechMarkTypes = builder.speechMarkTypes;
        this.textType = builder.textType;
        this.voiceId = builder.voiceId;
        this.languageCode = builder.languageCode;
    }

    /**
     * <p>
     * Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
     * <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis. Using a voice
     * that is not supported for the engine selected will result in an error.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #engine} will
     * return {@link Engine#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #engineAsString}.
     * </p>
     * 
     * @return Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
     *         <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis. Using a
     *         voice that is not supported for the engine selected will result in an error.
     * @see Engine
     */
    public final Engine engine() {
        return Engine.fromValue(engine);
    }

    /**
     * <p>
     * Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
     * <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis. Using a voice
     * that is not supported for the engine selected will result in an error.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #engine} will
     * return {@link Engine#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #engineAsString}.
     * </p>
     * 
     * @return Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
     *         <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis. Using a
     *         voice that is not supported for the engine selected will result in an error.
     * @see Engine
     */
    public final String engineAsString() {
        return engine;
    }

    /**
     * <p>
     * The Amazon Polly generated identifier for a speech synthesis task.
     * </p>
     * 
     * @return The Amazon Polly generated identifier for a speech synthesis task.
     */
    public final String taskId() {
        return taskId;
    }

    /**
     * <p>
     * Current status of the individual speech synthesis task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskStatus} will
     * return {@link TaskStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #taskStatusAsString}.
     * </p>
     * 
     * @return Current status of the individual speech synthesis task.
     * @see TaskStatus
     */
    public final TaskStatus taskStatus() {
        return TaskStatus.fromValue(taskStatus);
    }

    /**
     * <p>
     * Current status of the individual speech synthesis task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskStatus} will
     * return {@link TaskStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #taskStatusAsString}.
     * </p>
     * 
     * @return Current status of the individual speech synthesis task.
     * @see TaskStatus
     */
    public final String taskStatusAsString() {
        return taskStatus;
    }

    /**
     * <p>
     * Reason for the current status of a specific speech synthesis task, including errors if the task has failed.
     * </p>
     * 
     * @return Reason for the current status of a specific speech synthesis task, including errors if the task has
     *         failed.
     */
    public final String taskStatusReason() {
        return taskStatusReason;
    }

    /**
     * <p>
     * Pathway for the output speech file.
     * </p>
     * 
     * @return Pathway for the output speech file.
     */
    public final String outputUri() {
        return outputUri;
    }

    /**
     * <p>
     * Timestamp for the time the synthesis task was started.
     * </p>
     * 
     * @return Timestamp for the time the synthesis task was started.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * Number of billable characters synthesized.
     * </p>
     * 
     * @return Number of billable characters synthesized.
     */
    public final Integer requestCharacters() {
        return requestCharacters;
    }

    /**
     * <p>
     * ARN for the SNS topic optionally used for providing status notification for a speech synthesis task.
     * </p>
     * 
     * @return ARN for the SNS topic optionally used for providing status notification for a speech synthesis task.
     */
    public final String snsTopicArn() {
        return snsTopicArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the LexiconNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLexiconNames() {
        return lexiconNames != null && !(lexiconNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of one or more pronunciation lexicon names you want the service to apply during synthesis. Lexicons are
     * applied only if the language of the lexicon is the same as the language of the voice.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLexiconNames} method.
     * </p>
     * 
     * @return List of one or more pronunciation lexicon names you want the service to apply during synthesis. Lexicons
     *         are applied only if the language of the lexicon is the same as the language of the voice.
     */
    public final List<String> lexiconNames() {
        return lexiconNames;
    }

    /**
     * <p>
     * The format in which the returned output will be encoded. For audio stream, this will be mp3, ogg_vorbis, or pcm.
     * For speech marks, this will be json.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputFormat} will
     * return {@link OutputFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #outputFormatAsString}.
     * </p>
     * 
     * @return The format in which the returned output will be encoded. For audio stream, this will be mp3, ogg_vorbis,
     *         or pcm. For speech marks, this will be json.
     * @see OutputFormat
     */
    public final OutputFormat outputFormat() {
        return OutputFormat.fromValue(outputFormat);
    }

    /**
     * <p>
     * The format in which the returned output will be encoded. For audio stream, this will be mp3, ogg_vorbis, or pcm.
     * For speech marks, this will be json.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputFormat} will
     * return {@link OutputFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #outputFormatAsString}.
     * </p>
     * 
     * @return The format in which the returned output will be encoded. For audio stream, this will be mp3, ogg_vorbis,
     *         or pcm. For speech marks, this will be json.
     * @see OutputFormat
     */
    public final String outputFormatAsString() {
        return outputFormat;
    }

    /**
     * <p>
     * The audio frequency specified in Hz.
     * </p>
     * <p>
     * The valid values for mp3 and ogg_vorbis are "8000", "16000", "22050", and "24000". The default value for standard
     * voices is "22050". The default value for neural voices is "24000". The default value for long-form voices is
     * "24000". The default value for generative voices is "24000".
     * </p>
     * <p>
     * Valid values for pcm are "8000" and "16000" The default value is "16000".
     * </p>
     * 
     * @return The audio frequency specified in Hz.</p>
     *         <p>
     *         The valid values for mp3 and ogg_vorbis are "8000", "16000", "22050", and "24000". The default value for
     *         standard voices is "22050". The default value for neural voices is "24000". The default value for
     *         long-form voices is "24000". The default value for generative voices is "24000".
     *         </p>
     *         <p>
     *         Valid values for pcm are "8000" and "16000" The default value is "16000".
     */
    public final String sampleRate() {
        return sampleRate;
    }

    /**
     * <p>
     * The type of speech marks returned for the input text.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSpeechMarkTypes} method.
     * </p>
     * 
     * @return The type of speech marks returned for the input text.
     */
    public final List<SpeechMarkType> speechMarkTypes() {
        return SpeechMarkTypeListCopier.copyStringToEnum(speechMarkTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the SpeechMarkTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSpeechMarkTypes() {
        return speechMarkTypes != null && !(speechMarkTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of speech marks returned for the input text.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSpeechMarkTypes} method.
     * </p>
     * 
     * @return The type of speech marks returned for the input text.
     */
    public final List<String> speechMarkTypesAsStrings() {
        return speechMarkTypes;
    }

    /**
     * <p>
     * Specifies whether the input text is plain text or SSML. The default value is plain text.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #textType} will
     * return {@link TextType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #textTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the input text is plain text or SSML. The default value is plain text.
     * @see TextType
     */
    public final TextType textType() {
        return TextType.fromValue(textType);
    }

    /**
     * <p>
     * Specifies whether the input text is plain text or SSML. The default value is plain text.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #textType} will
     * return {@link TextType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #textTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the input text is plain text or SSML. The default value is plain text.
     * @see TextType
     */
    public final String textTypeAsString() {
        return textType;
    }

    /**
     * <p>
     * Voice ID to use for the synthesis.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #voiceId} will
     * return {@link VoiceId#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #voiceIdAsString}.
     * </p>
     * 
     * @return Voice ID to use for the synthesis.
     * @see VoiceId
     */
    public final VoiceId voiceId() {
        return VoiceId.fromValue(voiceId);
    }

    /**
     * <p>
     * Voice ID to use for the synthesis.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #voiceId} will
     * return {@link VoiceId#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #voiceIdAsString}.
     * </p>
     * 
     * @return Voice ID to use for the synthesis.
     * @see VoiceId
     */
    public final String voiceIdAsString() {
        return voiceId;
    }

    /**
     * <p>
     * Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such as Aditi,
     * which can be used for either Indian English (en-IN) or Hindi (hi-IN).
     * </p>
     * <p>
     * If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language of the
     * bilingual voice. The default language for any voice is the one returned by the <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a> operation for the
     * <code>LanguageCode</code> parameter. For example, if no language code is specified, Aditi will use Indian English
     * rather than Hindi.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such as
     *         Aditi, which can be used for either Indian English (en-IN) or Hindi (hi-IN). </p>
     *         <p>
     *         If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language of
     *         the bilingual voice. The default language for any voice is the one returned by the <a
     *         href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a> operation
     *         for the <code>LanguageCode</code> parameter. For example, if no language code is specified, Aditi will
     *         use Indian English rather than Hindi.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such as Aditi,
     * which can be used for either Indian English (en-IN) or Hindi (hi-IN).
     * </p>
     * <p>
     * If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language of the
     * bilingual voice. The default language for any voice is the one returned by the <a
     * href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a> operation for the
     * <code>LanguageCode</code> parameter. For example, if no language code is specified, Aditi will use Indian English
     * rather than Hindi.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such as
     *         Aditi, which can be used for either Indian English (en-IN) or Hindi (hi-IN). </p>
     *         <p>
     *         If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language of
     *         the bilingual voice. The default language for any voice is the one returned by the <a
     *         href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a> operation
     *         for the <code>LanguageCode</code> parameter. For example, if no language code is specified, Aditi will
     *         use Indian English rather than Hindi.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(engineAsString());
        hashCode = 31 * hashCode + Objects.hashCode(taskId());
        hashCode = 31 * hashCode + Objects.hashCode(taskStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(taskStatusReason());
        hashCode = 31 * hashCode + Objects.hashCode(outputUri());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(requestCharacters());
        hashCode = 31 * hashCode + Objects.hashCode(snsTopicArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasLexiconNames() ? lexiconNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(outputFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sampleRate());
        hashCode = 31 * hashCode + Objects.hashCode(hasSpeechMarkTypes() ? speechMarkTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(textTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(voiceIdAsString());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SynthesisTask)) {
            return false;
        }
        SynthesisTask other = (SynthesisTask) obj;
        return Objects.equals(engineAsString(), other.engineAsString()) && Objects.equals(taskId(), other.taskId())
                && Objects.equals(taskStatusAsString(), other.taskStatusAsString())
                && Objects.equals(taskStatusReason(), other.taskStatusReason()) && Objects.equals(outputUri(), other.outputUri())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(requestCharacters(), other.requestCharacters())
                && Objects.equals(snsTopicArn(), other.snsTopicArn()) && hasLexiconNames() == other.hasLexiconNames()
                && Objects.equals(lexiconNames(), other.lexiconNames())
                && Objects.equals(outputFormatAsString(), other.outputFormatAsString())
                && Objects.equals(sampleRate(), other.sampleRate()) && hasSpeechMarkTypes() == other.hasSpeechMarkTypes()
                && Objects.equals(speechMarkTypesAsStrings(), other.speechMarkTypesAsStrings())
                && Objects.equals(textTypeAsString(), other.textTypeAsString())
                && Objects.equals(voiceIdAsString(), other.voiceIdAsString())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SynthesisTask").add("Engine", engineAsString()).add("TaskId", taskId())
                .add("TaskStatus", taskStatusAsString()).add("TaskStatusReason", taskStatusReason())
                .add("OutputUri", outputUri()).add("CreationTime", creationTime()).add("RequestCharacters", requestCharacters())
                .add("SnsTopicArn", snsTopicArn()).add("LexiconNames", hasLexiconNames() ? lexiconNames() : null)
                .add("OutputFormat", outputFormatAsString()).add("SampleRate", sampleRate())
                .add("SpeechMarkTypes", hasSpeechMarkTypes() ? speechMarkTypesAsStrings() : null)
                .add("TextType", textTypeAsString()).add("VoiceId", voiceIdAsString())
                .add("LanguageCode", languageCodeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Engine":
            return Optional.ofNullable(clazz.cast(engineAsString()));
        case "TaskId":
            return Optional.ofNullable(clazz.cast(taskId()));
        case "TaskStatus":
            return Optional.ofNullable(clazz.cast(taskStatusAsString()));
        case "TaskStatusReason":
            return Optional.ofNullable(clazz.cast(taskStatusReason()));
        case "OutputUri":
            return Optional.ofNullable(clazz.cast(outputUri()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "RequestCharacters":
            return Optional.ofNullable(clazz.cast(requestCharacters()));
        case "SnsTopicArn":
            return Optional.ofNullable(clazz.cast(snsTopicArn()));
        case "LexiconNames":
            return Optional.ofNullable(clazz.cast(lexiconNames()));
        case "OutputFormat":
            return Optional.ofNullable(clazz.cast(outputFormatAsString()));
        case "SampleRate":
            return Optional.ofNullable(clazz.cast(sampleRate()));
        case "SpeechMarkTypes":
            return Optional.ofNullable(clazz.cast(speechMarkTypesAsStrings()));
        case "TextType":
            return Optional.ofNullable(clazz.cast(textTypeAsString()));
        case "VoiceId":
            return Optional.ofNullable(clazz.cast(voiceIdAsString()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Engine", ENGINE_FIELD);
        map.put("TaskId", TASK_ID_FIELD);
        map.put("TaskStatus", TASK_STATUS_FIELD);
        map.put("TaskStatusReason", TASK_STATUS_REASON_FIELD);
        map.put("OutputUri", OUTPUT_URI_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("RequestCharacters", REQUEST_CHARACTERS_FIELD);
        map.put("SnsTopicArn", SNS_TOPIC_ARN_FIELD);
        map.put("LexiconNames", LEXICON_NAMES_FIELD);
        map.put("OutputFormat", OUTPUT_FORMAT_FIELD);
        map.put("SampleRate", SAMPLE_RATE_FIELD);
        map.put("SpeechMarkTypes", SPEECH_MARK_TYPES_FIELD);
        map.put("TextType", TEXT_TYPE_FIELD);
        map.put("VoiceId", VOICE_ID_FIELD);
        map.put("LanguageCode", LANGUAGE_CODE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SynthesisTask, T> g) {
        return obj -> g.apply((SynthesisTask) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SynthesisTask> {
        /**
         * <p>
         * Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
         * <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis. Using a
         * voice that is not supported for the engine selected will result in an error.
         * </p>
         * 
         * @param engine
         *        Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
         *        <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis.
         *        Using a voice that is not supported for the engine selected will result in an error.
         * @see Engine
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Engine
         */
        Builder engine(String engine);

        /**
         * <p>
         * Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
         * <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis. Using a
         * voice that is not supported for the engine selected will result in an error.
         * </p>
         * 
         * @param engine
         *        Specifies the engine (<code>standard</code>, <code>neural</code>, <code>long-form</code> or
         *        <code>generative</code>) for Amazon Polly to use when processing input text for speech synthesis.
         *        Using a voice that is not supported for the engine selected will result in an error.
         * @see Engine
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Engine
         */
        Builder engine(Engine engine);

        /**
         * <p>
         * The Amazon Polly generated identifier for a speech synthesis task.
         * </p>
         * 
         * @param taskId
         *        The Amazon Polly generated identifier for a speech synthesis task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskId(String taskId);

        /**
         * <p>
         * Current status of the individual speech synthesis task.
         * </p>
         * 
         * @param taskStatus
         *        Current status of the individual speech synthesis task.
         * @see TaskStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskStatus
         */
        Builder taskStatus(String taskStatus);

        /**
         * <p>
         * Current status of the individual speech synthesis task.
         * </p>
         * 
         * @param taskStatus
         *        Current status of the individual speech synthesis task.
         * @see TaskStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskStatus
         */
        Builder taskStatus(TaskStatus taskStatus);

        /**
         * <p>
         * Reason for the current status of a specific speech synthesis task, including errors if the task has failed.
         * </p>
         * 
         * @param taskStatusReason
         *        Reason for the current status of a specific speech synthesis task, including errors if the task has
         *        failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskStatusReason(String taskStatusReason);

        /**
         * <p>
         * Pathway for the output speech file.
         * </p>
         * 
         * @param outputUri
         *        Pathway for the output speech file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputUri(String outputUri);

        /**
         * <p>
         * Timestamp for the time the synthesis task was started.
         * </p>
         * 
         * @param creationTime
         *        Timestamp for the time the synthesis task was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * Number of billable characters synthesized.
         * </p>
         * 
         * @param requestCharacters
         *        Number of billable characters synthesized.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestCharacters(Integer requestCharacters);

        /**
         * <p>
         * ARN for the SNS topic optionally used for providing status notification for a speech synthesis task.
         * </p>
         * 
         * @param snsTopicArn
         *        ARN for the SNS topic optionally used for providing status notification for a speech synthesis task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicArn(String snsTopicArn);

        /**
         * <p>
         * List of one or more pronunciation lexicon names you want the service to apply during synthesis. Lexicons are
         * applied only if the language of the lexicon is the same as the language of the voice.
         * </p>
         * 
         * @param lexiconNames
         *        List of one or more pronunciation lexicon names you want the service to apply during synthesis.
         *        Lexicons are applied only if the language of the lexicon is the same as the language of the voice.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lexiconNames(Collection<String> lexiconNames);

        /**
         * <p>
         * List of one or more pronunciation lexicon names you want the service to apply during synthesis. Lexicons are
         * applied only if the language of the lexicon is the same as the language of the voice.
         * </p>
         * 
         * @param lexiconNames
         *        List of one or more pronunciation lexicon names you want the service to apply during synthesis.
         *        Lexicons are applied only if the language of the lexicon is the same as the language of the voice.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lexiconNames(String... lexiconNames);

        /**
         * <p>
         * The format in which the returned output will be encoded. For audio stream, this will be mp3, ogg_vorbis, or
         * pcm. For speech marks, this will be json.
         * </p>
         * 
         * @param outputFormat
         *        The format in which the returned output will be encoded. For audio stream, this will be mp3,
         *        ogg_vorbis, or pcm. For speech marks, this will be json.
         * @see OutputFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputFormat
         */
        Builder outputFormat(String outputFormat);

        /**
         * <p>
         * The format in which the returned output will be encoded. For audio stream, this will be mp3, ogg_vorbis, or
         * pcm. For speech marks, this will be json.
         * </p>
         * 
         * @param outputFormat
         *        The format in which the returned output will be encoded. For audio stream, this will be mp3,
         *        ogg_vorbis, or pcm. For speech marks, this will be json.
         * @see OutputFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputFormat
         */
        Builder outputFormat(OutputFormat outputFormat);

        /**
         * <p>
         * The audio frequency specified in Hz.
         * </p>
         * <p>
         * The valid values for mp3 and ogg_vorbis are "8000", "16000", "22050", and "24000". The default value for
         * standard voices is "22050". The default value for neural voices is "24000". The default value for long-form
         * voices is "24000". The default value for generative voices is "24000".
         * </p>
         * <p>
         * Valid values for pcm are "8000" and "16000" The default value is "16000".
         * </p>
         * 
         * @param sampleRate
         *        The audio frequency specified in Hz.</p>
         *        <p>
         *        The valid values for mp3 and ogg_vorbis are "8000", "16000", "22050", and "24000". The default value
         *        for standard voices is "22050". The default value for neural voices is "24000". The default value for
         *        long-form voices is "24000". The default value for generative voices is "24000".
         *        </p>
         *        <p>
         *        Valid values for pcm are "8000" and "16000" The default value is "16000".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleRate(String sampleRate);

        /**
         * <p>
         * The type of speech marks returned for the input text.
         * </p>
         * 
         * @param speechMarkTypes
         *        The type of speech marks returned for the input text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder speechMarkTypesWithStrings(Collection<String> speechMarkTypes);

        /**
         * <p>
         * The type of speech marks returned for the input text.
         * </p>
         * 
         * @param speechMarkTypes
         *        The type of speech marks returned for the input text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder speechMarkTypesWithStrings(String... speechMarkTypes);

        /**
         * <p>
         * The type of speech marks returned for the input text.
         * </p>
         * 
         * @param speechMarkTypes
         *        The type of speech marks returned for the input text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder speechMarkTypes(Collection<SpeechMarkType> speechMarkTypes);

        /**
         * <p>
         * The type of speech marks returned for the input text.
         * </p>
         * 
         * @param speechMarkTypes
         *        The type of speech marks returned for the input text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder speechMarkTypes(SpeechMarkType... speechMarkTypes);

        /**
         * <p>
         * Specifies whether the input text is plain text or SSML. The default value is plain text.
         * </p>
         * 
         * @param textType
         *        Specifies whether the input text is plain text or SSML. The default value is plain text.
         * @see TextType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextType
         */
        Builder textType(String textType);

        /**
         * <p>
         * Specifies whether the input text is plain text or SSML. The default value is plain text.
         * </p>
         * 
         * @param textType
         *        Specifies whether the input text is plain text or SSML. The default value is plain text.
         * @see TextType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextType
         */
        Builder textType(TextType textType);

        /**
         * <p>
         * Voice ID to use for the synthesis.
         * </p>
         * 
         * @param voiceId
         *        Voice ID to use for the synthesis.
         * @see VoiceId
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VoiceId
         */
        Builder voiceId(String voiceId);

        /**
         * <p>
         * Voice ID to use for the synthesis.
         * </p>
         * 
         * @param voiceId
         *        Voice ID to use for the synthesis.
         * @see VoiceId
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VoiceId
         */
        Builder voiceId(VoiceId voiceId);

        /**
         * <p>
         * Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such as
         * Aditi, which can be used for either Indian English (en-IN) or Hindi (hi-IN).
         * </p>
         * <p>
         * If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language of the
         * bilingual voice. The default language for any voice is the one returned by the <a
         * href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a> operation for
         * the <code>LanguageCode</code> parameter. For example, if no language code is specified, Aditi will use Indian
         * English rather than Hindi.
         * </p>
         * 
         * @param languageCode
         *        Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such
         *        as Aditi, which can be used for either Indian English (en-IN) or Hindi (hi-IN). </p>
         *        <p>
         *        If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language
         *        of the bilingual voice. The default language for any voice is the one returned by the <a
         *        href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a>
         *        operation for the <code>LanguageCode</code> parameter. For example, if no language code is specified,
         *        Aditi will use Indian English rather than Hindi.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such as
         * Aditi, which can be used for either Indian English (en-IN) or Hindi (hi-IN).
         * </p>
         * <p>
         * If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language of the
         * bilingual voice. The default language for any voice is the one returned by the <a
         * href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a> operation for
         * the <code>LanguageCode</code> parameter. For example, if no language code is specified, Aditi will use Indian
         * English rather than Hindi.
         * </p>
         * 
         * @param languageCode
         *        Optional language code for a synthesis task. This is only necessary if using a bilingual voice, such
         *        as Aditi, which can be used for either Indian English (en-IN) or Hindi (hi-IN). </p>
         *        <p>
         *        If a bilingual voice is used and no language code is specified, Amazon Polly uses the default language
         *        of the bilingual voice. The default language for any voice is the one returned by the <a
         *        href="https://docs.aws.amazon.com/polly/latest/dg/API_DescribeVoices.html">DescribeVoices</a>
         *        operation for the <code>LanguageCode</code> parameter. For example, if no language code is specified,
         *        Aditi will use Indian English rather than Hindi.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);
    }

    static final class BuilderImpl implements Builder {
        private String engine;

        private String taskId;

        private String taskStatus;

        private String taskStatusReason;

        private String outputUri;

        private Instant creationTime;

        private Integer requestCharacters;

        private String snsTopicArn;

        private List<String> lexiconNames = DefaultSdkAutoConstructList.getInstance();

        private String outputFormat;

        private String sampleRate;

        private List<String> speechMarkTypes = DefaultSdkAutoConstructList.getInstance();

        private String textType;

        private String voiceId;

        private String languageCode;

        private BuilderImpl() {
        }

        private BuilderImpl(SynthesisTask model) {
            engine(model.engine);
            taskId(model.taskId);
            taskStatus(model.taskStatus);
            taskStatusReason(model.taskStatusReason);
            outputUri(model.outputUri);
            creationTime(model.creationTime);
            requestCharacters(model.requestCharacters);
            snsTopicArn(model.snsTopicArn);
            lexiconNames(model.lexiconNames);
            outputFormat(model.outputFormat);
            sampleRate(model.sampleRate);
            speechMarkTypesWithStrings(model.speechMarkTypes);
            textType(model.textType);
            voiceId(model.voiceId);
            languageCode(model.languageCode);
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        @Override
        public final Builder engine(Engine engine) {
            this.engine(engine == null ? null : engine.toString());
            return this;
        }

        public final String getTaskId() {
            return taskId;
        }

        public final void setTaskId(String taskId) {
            this.taskId = taskId;
        }

        @Override
        public final Builder taskId(String taskId) {
            this.taskId = taskId;
            return this;
        }

        public final String getTaskStatus() {
            return taskStatus;
        }

        public final void setTaskStatus(String taskStatus) {
            this.taskStatus = taskStatus;
        }

        @Override
        public final Builder taskStatus(String taskStatus) {
            this.taskStatus = taskStatus;
            return this;
        }

        @Override
        public final Builder taskStatus(TaskStatus taskStatus) {
            this.taskStatus(taskStatus == null ? null : taskStatus.toString());
            return this;
        }

        public final String getTaskStatusReason() {
            return taskStatusReason;
        }

        public final void setTaskStatusReason(String taskStatusReason) {
            this.taskStatusReason = taskStatusReason;
        }

        @Override
        public final Builder taskStatusReason(String taskStatusReason) {
            this.taskStatusReason = taskStatusReason;
            return this;
        }

        public final String getOutputUri() {
            return outputUri;
        }

        public final void setOutputUri(String outputUri) {
            this.outputUri = outputUri;
        }

        @Override
        public final Builder outputUri(String outputUri) {
            this.outputUri = outputUri;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Integer getRequestCharacters() {
            return requestCharacters;
        }

        public final void setRequestCharacters(Integer requestCharacters) {
            this.requestCharacters = requestCharacters;
        }

        @Override
        public final Builder requestCharacters(Integer requestCharacters) {
            this.requestCharacters = requestCharacters;
            return this;
        }

        public final String getSnsTopicArn() {
            return snsTopicArn;
        }

        public final void setSnsTopicArn(String snsTopicArn) {
            this.snsTopicArn = snsTopicArn;
        }

        @Override
        public final Builder snsTopicArn(String snsTopicArn) {
            this.snsTopicArn = snsTopicArn;
            return this;
        }

        public final Collection<String> getLexiconNames() {
            if (lexiconNames instanceof SdkAutoConstructList) {
                return null;
            }
            return lexiconNames;
        }

        public final void setLexiconNames(Collection<String> lexiconNames) {
            this.lexiconNames = LexiconNameListCopier.copy(lexiconNames);
        }

        @Override
        public final Builder lexiconNames(Collection<String> lexiconNames) {
            this.lexiconNames = LexiconNameListCopier.copy(lexiconNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lexiconNames(String... lexiconNames) {
            lexiconNames(Arrays.asList(lexiconNames));
            return this;
        }

        public final String getOutputFormat() {
            return outputFormat;
        }

        public final void setOutputFormat(String outputFormat) {
            this.outputFormat = outputFormat;
        }

        @Override
        public final Builder outputFormat(String outputFormat) {
            this.outputFormat = outputFormat;
            return this;
        }

        @Override
        public final Builder outputFormat(OutputFormat outputFormat) {
            this.outputFormat(outputFormat == null ? null : outputFormat.toString());
            return this;
        }

        public final String getSampleRate() {
            return sampleRate;
        }

        public final void setSampleRate(String sampleRate) {
            this.sampleRate = sampleRate;
        }

        @Override
        public final Builder sampleRate(String sampleRate) {
            this.sampleRate = sampleRate;
            return this;
        }

        public final Collection<String> getSpeechMarkTypes() {
            if (speechMarkTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return speechMarkTypes;
        }

        public final void setSpeechMarkTypes(Collection<String> speechMarkTypes) {
            this.speechMarkTypes = SpeechMarkTypeListCopier.copy(speechMarkTypes);
        }

        @Override
        public final Builder speechMarkTypesWithStrings(Collection<String> speechMarkTypes) {
            this.speechMarkTypes = SpeechMarkTypeListCopier.copy(speechMarkTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder speechMarkTypesWithStrings(String... speechMarkTypes) {
            speechMarkTypesWithStrings(Arrays.asList(speechMarkTypes));
            return this;
        }

        @Override
        public final Builder speechMarkTypes(Collection<SpeechMarkType> speechMarkTypes) {
            this.speechMarkTypes = SpeechMarkTypeListCopier.copyEnumToString(speechMarkTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder speechMarkTypes(SpeechMarkType... speechMarkTypes) {
            speechMarkTypes(Arrays.asList(speechMarkTypes));
            return this;
        }

        public final String getTextType() {
            return textType;
        }

        public final void setTextType(String textType) {
            this.textType = textType;
        }

        @Override
        public final Builder textType(String textType) {
            this.textType = textType;
            return this;
        }

        @Override
        public final Builder textType(TextType textType) {
            this.textType(textType == null ? null : textType.toString());
            return this;
        }

        public final String getVoiceId() {
            return voiceId;
        }

        public final void setVoiceId(String voiceId) {
            this.voiceId = voiceId;
        }

        @Override
        public final Builder voiceId(String voiceId) {
            this.voiceId = voiceId;
            return this;
        }

        @Override
        public final Builder voiceId(VoiceId voiceId) {
            this.voiceId(voiceId == null ? null : voiceId.toString());
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        @Override
        public SynthesisTask build() {
            return new SynthesisTask(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
