/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.privatenetworks;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.privatenetworks.model.AccessDeniedException;
import software.amazon.awssdk.services.privatenetworks.model.AcknowledgeOrderReceiptRequest;
import software.amazon.awssdk.services.privatenetworks.model.AcknowledgeOrderReceiptResponse;
import software.amazon.awssdk.services.privatenetworks.model.ActivateDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.ActivateDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.ActivateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.ActivateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.ConfigureAccessPointRequest;
import software.amazon.awssdk.services.privatenetworks.model.ConfigureAccessPointResponse;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeactivateDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeactivateDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetOrderRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetOrderResponse;
import software.amazon.awssdk.services.privatenetworks.model.InternalServerException;
import software.amazon.awssdk.services.privatenetworks.model.LimitExceededException;
import software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworksRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworksResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListOrdersRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListOrdersResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.PingRequest;
import software.amazon.awssdk.services.privatenetworks.model.PingResponse;
import software.amazon.awssdk.services.privatenetworks.model.PrivateNetworksException;
import software.amazon.awssdk.services.privatenetworks.model.PrivateNetworksRequest;
import software.amazon.awssdk.services.privatenetworks.model.ResourceNotFoundException;
import software.amazon.awssdk.services.privatenetworks.model.TagResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.TagResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.ThrottlingException;
import software.amazon.awssdk.services.privatenetworks.model.UntagResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.UntagResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSitePlanRequest;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSitePlanResponse;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.ValidationException;
import software.amazon.awssdk.services.privatenetworks.paginators.ListDeviceIdentifiersPublisher;
import software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkResourcesPublisher;
import software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkSitesPublisher;
import software.amazon.awssdk.services.privatenetworks.paginators.ListNetworksPublisher;
import software.amazon.awssdk.services.privatenetworks.paginators.ListOrdersPublisher;
import software.amazon.awssdk.services.privatenetworks.transform.AcknowledgeOrderReceiptRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ActivateDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ActivateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ConfigureAccessPointRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.CreateNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.CreateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeactivateDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeleteNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeleteNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetOrderRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListDeviceIdentifiersRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworkResourcesRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworkSitesRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworksRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListOrdersRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.PingRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UpdateNetworkSitePlanRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UpdateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link PrivateNetworksAsyncClient}.
 *
 * @see PrivateNetworksAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPrivateNetworksAsyncClient implements PrivateNetworksAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultPrivateNetworksAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultPrivateNetworksAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Acknowledges that the specified network order was received.
     * </p>
     *
     * @param acknowledgeOrderReceiptRequest
     * @return A Java Future containing the result of the AcknowledgeOrderReceipt operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.AcknowledgeOrderReceipt
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/AcknowledgeOrderReceipt"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcknowledgeOrderReceiptResponse> acknowledgeOrderReceipt(
            AcknowledgeOrderReceiptRequest acknowledgeOrderReceiptRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acknowledgeOrderReceiptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcknowledgeOrderReceipt");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcknowledgeOrderReceiptResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcknowledgeOrderReceiptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcknowledgeOrderReceiptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcknowledgeOrderReceiptRequest, AcknowledgeOrderReceiptResponse>()
                            .withOperationName("AcknowledgeOrderReceipt")
                            .withMarshaller(new AcknowledgeOrderReceiptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acknowledgeOrderReceiptRequest));
            CompletableFuture<AcknowledgeOrderReceiptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates the specified device identifier.
     * </p>
     *
     * @param activateDeviceIdentifierRequest
     * @return A Java Future containing the result of the ActivateDeviceIdentifier operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ActivateDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ActivateDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateDeviceIdentifierResponse> activateDeviceIdentifier(
            ActivateDeviceIdentifierRequest activateDeviceIdentifierRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateDeviceIdentifier");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateDeviceIdentifierResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateDeviceIdentifierResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateDeviceIdentifierRequest, ActivateDeviceIdentifierResponse>()
                            .withOperationName("ActivateDeviceIdentifier")
                            .withMarshaller(new ActivateDeviceIdentifierRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(activateDeviceIdentifierRequest));
            CompletableFuture<ActivateDeviceIdentifierResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates the specified network site.
     * </p>
     *
     * @param activateNetworkSiteRequest
     * @return A Java Future containing the result of the ActivateNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ActivateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ActivateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateNetworkSiteResponse> activateNetworkSite(
            ActivateNetworkSiteRequest activateNetworkSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateNetworkSiteRequest, ActivateNetworkSiteResponse>()
                            .withOperationName("ActivateNetworkSite")
                            .withMarshaller(new ActivateNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(activateNetworkSiteRequest));
            CompletableFuture<ActivateNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures the specified network resource.
     * </p>
     * <p>
     * Use this action to specify the geographic position of the hardware. You must provide Certified Professional
     * Installer (CPI) credentials in the request so that we can obtain spectrum grants. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/radio-units.html">Radio units</a> in the
     * <i>Amazon Web Services Private 5G User Guide</i>.
     * </p>
     *
     * @param configureAccessPointRequest
     * @return A Java Future containing the result of the ConfigureAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ConfigureAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ConfigureAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ConfigureAccessPointResponse> configureAccessPoint(
            ConfigureAccessPointRequest configureAccessPointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, configureAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureAccessPoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConfigureAccessPointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ConfigureAccessPointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConfigureAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfigureAccessPointRequest, ConfigureAccessPointResponse>()
                            .withOperationName("ConfigureAccessPoint")
                            .withMarshaller(new ConfigureAccessPointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(configureAccessPointRequest));
            CompletableFuture<ConfigureAccessPointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a network.
     * </p>
     *
     * @param createNetworkRequest
     * @return A Java Future containing the result of the CreateNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException The limit was exceeded.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.CreateNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/CreateNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNetworkResponse> createNetwork(CreateNetworkRequest createNetworkRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNetworkRequest, CreateNetworkResponse>()
                            .withOperationName("CreateNetwork")
                            .withMarshaller(new CreateNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createNetworkRequest));
            CompletableFuture<CreateNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a network site.
     * </p>
     *
     * @param createNetworkSiteRequest
     * @return A Java Future containing the result of the CreateNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.CreateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/CreateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNetworkSiteResponse> createNetworkSite(CreateNetworkSiteRequest createNetworkSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNetworkSiteRequest, CreateNetworkSiteResponse>()
                            .withOperationName("CreateNetworkSite")
                            .withMarshaller(new CreateNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createNetworkSiteRequest));
            CompletableFuture<CreateNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates the specified device identifier.
     * </p>
     *
     * @param deactivateDeviceIdentifierRequest
     * @return A Java Future containing the result of the DeactivateDeviceIdentifier operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.DeactivateDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeactivateDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateDeviceIdentifierResponse> deactivateDeviceIdentifier(
            DeactivateDeviceIdentifierRequest deactivateDeviceIdentifierRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateDeviceIdentifier");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateDeviceIdentifierResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateDeviceIdentifierResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateDeviceIdentifierRequest, DeactivateDeviceIdentifierResponse>()
                            .withOperationName("DeactivateDeviceIdentifier")
                            .withMarshaller(new DeactivateDeviceIdentifierRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deactivateDeviceIdentifierRequest));
            CompletableFuture<DeactivateDeviceIdentifierResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified network. You must delete network sites before you delete the network. For more information,
     * see <a href="https://docs.aws.amazon.com/private-networks/latest/APIReference/API_DeleteNetworkSite.html">
     * DeleteNetworkSite</a> in the <i>API Reference for Amazon Web Services Private 5G</i>.
     * </p>
     *
     * @param deleteNetworkRequest
     * @return A Java Future containing the result of the DeleteNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.DeleteNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeleteNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNetworkResponse> deleteNetwork(DeleteNetworkRequest deleteNetworkRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNetworkRequest, DeleteNetworkResponse>()
                            .withOperationName("DeleteNetwork")
                            .withMarshaller(new DeleteNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteNetworkRequest));
            CompletableFuture<DeleteNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified network site. Return the hardware after you delete the network site. You are responsible
     * for minimum charges. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/hardware-maintenance.html">Hardware
     * returns</a> in the <i>Amazon Web Services Private 5G User Guide</i>.
     * </p>
     *
     * @param deleteNetworkSiteRequest
     * @return A Java Future containing the result of the DeleteNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.DeleteNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeleteNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNetworkSiteResponse> deleteNetworkSite(DeleteNetworkSiteRequest deleteNetworkSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNetworkSiteRequest, DeleteNetworkSiteResponse>()
                            .withOperationName("DeleteNetworkSite")
                            .withMarshaller(new DeleteNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteNetworkSiteRequest));
            CompletableFuture<DeleteNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified device identifier.
     * </p>
     *
     * @param getDeviceIdentifierRequest
     * @return A Java Future containing the result of the GetDeviceIdentifier operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeviceIdentifierResponse> getDeviceIdentifier(
            GetDeviceIdentifierRequest getDeviceIdentifierRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeviceIdentifier");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDeviceIdentifierResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeviceIdentifierResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeviceIdentifierRequest, GetDeviceIdentifierResponse>()
                            .withOperationName("GetDeviceIdentifier")
                            .withMarshaller(new GetDeviceIdentifierRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDeviceIdentifierRequest));
            CompletableFuture<GetDeviceIdentifierResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified network.
     * </p>
     *
     * @param getNetworkRequest
     * @return A Java Future containing the result of the GetNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNetworkResponse> getNetwork(GetNetworkRequest getNetworkRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNetworkRequest, GetNetworkResponse>().withOperationName("GetNetwork")
                            .withMarshaller(new GetNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getNetworkRequest));
            CompletableFuture<GetNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified network resource.
     * </p>
     *
     * @param getNetworkResourceRequest
     * @return A Java Future containing the result of the GetNetworkResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetNetworkResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetworkResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNetworkResourceResponse> getNetworkResource(GetNetworkResourceRequest getNetworkResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetworkResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNetworkResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetNetworkResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNetworkResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNetworkResourceRequest, GetNetworkResourceResponse>()
                            .withOperationName("GetNetworkResource")
                            .withMarshaller(new GetNetworkResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getNetworkResourceRequest));
            CompletableFuture<GetNetworkResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified network site.
     * </p>
     *
     * @param getNetworkSiteRequest
     * @return A Java Future containing the result of the GetNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNetworkSiteResponse> getNetworkSite(GetNetworkSiteRequest getNetworkSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNetworkSiteRequest, GetNetworkSiteResponse>()
                            .withOperationName("GetNetworkSite")
                            .withMarshaller(new GetNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getNetworkSiteRequest));
            CompletableFuture<GetNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the specified order.
     * </p>
     *
     * @param getOrderRequest
     * @return A Java Future containing the result of the GetOrder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.GetOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetOrderResponse> getOrder(GetOrderRequest getOrderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOrder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOrderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOrderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOrderRequest, GetOrderResponse>().withOperationName("GetOrder")
                            .withMarshaller(new GetOrderRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getOrderRequest));
            CompletableFuture<GetOrderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists device identifiers. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order, the status of device identifiers, or the ARN of the traffic
     * group.
     * </p>
     * 
     * <pre>
     * <code> &lt;p&gt;If you specify multiple filters, filters are joined with an OR, and the request </code>
     * </pre>
     * <p>
     * returns results that match all of the specified filters.
     * </p>
     * </p>
     *
     * @param listDeviceIdentifiersRequest
     * @return A Java Future containing the result of the ListDeviceIdentifiers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListDeviceIdentifiers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListDeviceIdentifiers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeviceIdentifiersResponse> listDeviceIdentifiers(
            ListDeviceIdentifiersRequest listDeviceIdentifiersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeviceIdentifiersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeviceIdentifiers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeviceIdentifiersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeviceIdentifiersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeviceIdentifiersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeviceIdentifiersRequest, ListDeviceIdentifiersResponse>()
                            .withOperationName("ListDeviceIdentifiers")
                            .withMarshaller(new ListDeviceIdentifiersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDeviceIdentifiersRequest));
            CompletableFuture<ListDeviceIdentifiersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists device identifiers. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order, the status of device identifiers, or the ARN of the traffic
     * group.
     * </p>
     * 
     * <pre>
     * <code> &lt;p&gt;If you specify multiple filters, filters are joined with an OR, and the request </code>
     * </pre>
     * <p>
     * returns results that match all of the specified filters.
     * </p>
     * </p><br/>
     * <p>
     * This is a variant of
     * {@link #listDeviceIdentifiers(software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListDeviceIdentifiersPublisher publisher = client.listDeviceIdentifiersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListDeviceIdentifiersPublisher publisher = client.listDeviceIdentifiersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeviceIdentifiers(software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDeviceIdentifiersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListDeviceIdentifiers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListDeviceIdentifiers"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDeviceIdentifiersPublisher listDeviceIdentifiersPaginator(ListDeviceIdentifiersRequest listDeviceIdentifiersRequest) {
        return new ListDeviceIdentifiersPublisher(this, applyPaginatorUserAgent(listDeviceIdentifiersRequest));
    }

    /**
     * <p>
     * Lists network resources. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order or the status of network resources.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listNetworkResourcesRequest
     * @return A Java Future containing the result of the ListNetworkResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworkResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNetworkResourcesResponse> listNetworkResources(
            ListNetworkResourcesRequest listNetworkResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworkResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworkResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNetworkResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNetworkResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNetworkResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNetworkResourcesRequest, ListNetworkResourcesResponse>()
                            .withOperationName("ListNetworkResources")
                            .withMarshaller(new ListNetworkResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listNetworkResourcesRequest));
            CompletableFuture<ListNetworkResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists network resources. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order or the status of network resources.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworkResources(software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkResourcesPublisher publisher = client.listNetworkResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkResourcesPublisher publisher = client.listNetworkResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworkResources(software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNetworkResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworkResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkResources"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNetworkResourcesPublisher listNetworkResourcesPaginator(ListNetworkResourcesRequest listNetworkResourcesRequest) {
        return new ListNetworkResourcesPublisher(this, applyPaginatorUserAgent(listNetworkResourcesRequest));
    }

    /**
     * <p>
     * Lists network sites. Add filters to your request to return a more specific list of results. Use filters to match
     * the status of the network site.
     * </p>
     *
     * @param listNetworkSitesRequest
     * @return A Java Future containing the result of the ListNetworkSites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworkSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkSites"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNetworkSitesResponse> listNetworkSites(ListNetworkSitesRequest listNetworkSitesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworkSitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworkSites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNetworkSitesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNetworkSitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNetworkSitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNetworkSitesRequest, ListNetworkSitesResponse>()
                            .withOperationName("ListNetworkSites")
                            .withMarshaller(new ListNetworkSitesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listNetworkSitesRequest));
            CompletableFuture<ListNetworkSitesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists network sites. Add filters to your request to return a more specific list of results. Use filters to match
     * the status of the network site.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworkSites(software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkSitesPublisher publisher = client.listNetworkSitesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkSitesPublisher publisher = client.listNetworkSitesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworkSites(software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNetworkSitesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworkSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkSites"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNetworkSitesPublisher listNetworkSitesPaginator(ListNetworkSitesRequest listNetworkSitesRequest) {
        return new ListNetworkSitesPublisher(this, applyPaginatorUserAgent(listNetworkSitesRequest));
    }

    /**
     * <p>
     * Lists networks. Add filters to your request to return a more specific list of results. Use filters to match the
     * status of the network.
     * </p>
     *
     * @param listNetworksRequest
     * @return A Java Future containing the result of the ListNetworks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNetworksResponse> listNetworks(ListNetworksRequest listNetworksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNetworksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListNetworksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNetworksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNetworksRequest, ListNetworksResponse>()
                            .withOperationName("ListNetworks").withMarshaller(new ListNetworksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listNetworksRequest));
            CompletableFuture<ListNetworksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists networks. Add filters to your request to return a more specific list of results. Use filters to match the
     * status of the network.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworks(software.amazon.awssdk.services.privatenetworks.model.ListNetworksRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworksPublisher publisher = client.listNetworksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworksPublisher publisher = client.listNetworksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.privatenetworks.model.ListNetworksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.privatenetworks.model.ListNetworksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworks(software.amazon.awssdk.services.privatenetworks.model.ListNetworksRequest)} operation.</b>
     * </p>
     *
     * @param listNetworksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListNetworks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworks" target="_top">AWS
     *      API Documentation</a>
     */
    public ListNetworksPublisher listNetworksPaginator(ListNetworksRequest listNetworksRequest) {
        return new ListNetworksPublisher(this, applyPaginatorUserAgent(listNetworksRequest));
    }

    /**
     * <p>
     * Lists orders. Add filters to your request to return a more specific list of results. Use filters to match the
     * Amazon Resource Name (ARN) of the network site or the status of the order.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listOrdersRequest
     * @return A Java Future containing the result of the ListOrders operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListOrders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrdersResponse> listOrders(ListOrdersRequest listOrdersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOrdersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrdersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListOrdersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrdersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrdersRequest, ListOrdersResponse>().withOperationName("ListOrders")
                            .withMarshaller(new ListOrdersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOrdersRequest));
            CompletableFuture<ListOrdersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists orders. Add filters to your request to return a more specific list of results. Use filters to match the
     * Amazon Resource Name (ARN) of the network site or the status of the order.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOrders(software.amazon.awssdk.services.privatenetworks.model.ListOrdersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListOrdersPublisher publisher = client.listOrdersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListOrdersPublisher publisher = client.listOrdersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.privatenetworks.model.ListOrdersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.privatenetworks.model.ListOrdersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOrders(software.amazon.awssdk.services.privatenetworks.model.ListOrdersRequest)} operation.</b>
     * </p>
     *
     * @param listOrdersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListOrders" target="_top">AWS
     *      API Documentation</a>
     */
    public ListOrdersPublisher listOrdersPaginator(ListOrdersRequest listOrdersRequest) {
        return new ListOrdersPublisher(this, applyPaginatorUserAgent(listOrdersRequest));
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks the health of the service.
     * </p>
     *
     * @param pingRequest
     * @return A Java Future containing the result of the Ping operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.Ping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/Ping" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PingResponse> ping(PingRequest pingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pingRequest.overrideConfiguration()
                .orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Ping");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PingRequest, PingResponse>().withOperationName("Ping")
                            .withMarshaller(new PingRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(pingRequest));
            CompletableFuture<PingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException You do not have permission to perform this operation.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified network site.
     * </p>
     *
     * @param updateNetworkSiteRequest
     * @return A Java Future containing the result of the UpdateNetworkSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.UpdateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UpdateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNetworkSiteResponse> updateNetworkSite(UpdateNetworkSiteRequest updateNetworkSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNetworkSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNetworkSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNetworkSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNetworkSiteRequest, UpdateNetworkSiteResponse>()
                            .withOperationName("UpdateNetworkSite")
                            .withMarshaller(new UpdateNetworkSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateNetworkSiteRequest));
            CompletableFuture<UpdateNetworkSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified network site plan.
     * </p>
     *
     * @param updateNetworkSitePlanRequest
     * @return A Java Future containing the result of the UpdateNetworkSitePlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The request failed validation.</li>
     *         <li>InternalServerException Information about an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>PrivateNetworksException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample PrivateNetworksAsyncClient.UpdateNetworkSitePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UpdateNetworkSitePlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNetworkSitePlanResponse> updateNetworkSitePlan(
            UpdateNetworkSitePlanRequest updateNetworkSitePlanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNetworkSitePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNetworkSitePlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNetworkSitePlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNetworkSitePlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNetworkSitePlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNetworkSitePlanRequest, UpdateNetworkSitePlanResponse>()
                            .withOperationName("UpdateNetworkSitePlan")
                            .withMarshaller(new UpdateNetworkSitePlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateNetworkSitePlanRequest));
            CompletableFuture<UpdateNetworkSitePlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PrivateNetworksException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends PrivateNetworksRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
