/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.proton.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.proton.ProtonClient;
import software.amazon.awssdk.services.proton.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.proton.model.GetComponentRequest;
import software.amazon.awssdk.services.proton.model.GetComponentResponse;
import software.amazon.awssdk.services.proton.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.proton.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.proton.model.GetEnvironmentTemplateVersionRequest;
import software.amazon.awssdk.services.proton.model.GetEnvironmentTemplateVersionResponse;
import software.amazon.awssdk.services.proton.model.GetServiceInstanceRequest;
import software.amazon.awssdk.services.proton.model.GetServiceInstanceResponse;
import software.amazon.awssdk.services.proton.model.GetServiceRequest;
import software.amazon.awssdk.services.proton.model.GetServiceResponse;
import software.amazon.awssdk.services.proton.model.GetServiceTemplateVersionRequest;
import software.amazon.awssdk.services.proton.model.GetServiceTemplateVersionResponse;
import software.amazon.awssdk.services.proton.model.ProtonRequest;
import software.amazon.awssdk.services.proton.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultProtonWaiter implements ProtonWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final ProtonClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetComponentResponse> componentDeletedWaiter;

    private final Waiter<GetComponentResponse> componentDeployedWaiter;

    private final Waiter<GetEnvironmentResponse> environmentDeployedWaiter;

    private final Waiter<GetEnvironmentTemplateVersionResponse> environmentTemplateVersionRegisteredWaiter;

    private final Waiter<GetServiceResponse> serviceCreatedWaiter;

    private final Waiter<GetServiceResponse> serviceDeletedWaiter;

    private final Waiter<GetServiceInstanceResponse> serviceInstanceDeployedWaiter;

    private final Waiter<GetServiceResponse> servicePipelineDeployedWaiter;

    private final Waiter<GetServiceTemplateVersionResponse> serviceTemplateVersionRegisteredWaiter;

    private final Waiter<GetServiceResponse> serviceUpdatedWaiter;

    private DefaultProtonWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ProtonClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.componentDeletedWaiter = Waiter.builder(GetComponentResponse.class).acceptors(componentDeletedWaiterAcceptors())
                .overrideConfiguration(componentDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.componentDeployedWaiter = Waiter.builder(GetComponentResponse.class).acceptors(componentDeployedWaiterAcceptors())
                .overrideConfiguration(componentDeployedWaiterConfig(builder.overrideConfiguration)).build();
        this.environmentDeployedWaiter = Waiter.builder(GetEnvironmentResponse.class)
                .acceptors(environmentDeployedWaiterAcceptors())
                .overrideConfiguration(environmentDeployedWaiterConfig(builder.overrideConfiguration)).build();
        this.environmentTemplateVersionRegisteredWaiter = Waiter.builder(GetEnvironmentTemplateVersionResponse.class)
                .acceptors(environmentTemplateVersionRegisteredWaiterAcceptors())
                .overrideConfiguration(environmentTemplateVersionRegisteredWaiterConfig(builder.overrideConfiguration)).build();
        this.serviceCreatedWaiter = Waiter.builder(GetServiceResponse.class).acceptors(serviceCreatedWaiterAcceptors())
                .overrideConfiguration(serviceCreatedWaiterConfig(builder.overrideConfiguration)).build();
        this.serviceDeletedWaiter = Waiter.builder(GetServiceResponse.class).acceptors(serviceDeletedWaiterAcceptors())
                .overrideConfiguration(serviceDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.serviceInstanceDeployedWaiter = Waiter.builder(GetServiceInstanceResponse.class)
                .acceptors(serviceInstanceDeployedWaiterAcceptors())
                .overrideConfiguration(serviceInstanceDeployedWaiterConfig(builder.overrideConfiguration)).build();
        this.servicePipelineDeployedWaiter = Waiter.builder(GetServiceResponse.class)
                .acceptors(servicePipelineDeployedWaiterAcceptors())
                .overrideConfiguration(servicePipelineDeployedWaiterConfig(builder.overrideConfiguration)).build();
        this.serviceTemplateVersionRegisteredWaiter = Waiter.builder(GetServiceTemplateVersionResponse.class)
                .acceptors(serviceTemplateVersionRegisteredWaiterAcceptors())
                .overrideConfiguration(serviceTemplateVersionRegisteredWaiterConfig(builder.overrideConfiguration)).build();
        this.serviceUpdatedWaiter = Waiter.builder(GetServiceResponse.class).acceptors(serviceUpdatedWaiterAcceptors())
                .overrideConfiguration(serviceUpdatedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetComponentResponse> waitUntilComponentDeleted(GetComponentRequest getComponentRequest) {
        return componentDeletedWaiter.run(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)));
    }

    @Override
    public WaiterResponse<GetComponentResponse> waitUntilComponentDeleted(GetComponentRequest getComponentRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return componentDeletedWaiter.run(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)),
                componentDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetComponentResponse> waitUntilComponentDeployed(GetComponentRequest getComponentRequest) {
        return componentDeployedWaiter.run(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)));
    }

    @Override
    public WaiterResponse<GetComponentResponse> waitUntilComponentDeployed(GetComponentRequest getComponentRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return componentDeployedWaiter.run(() -> client.getComponent(applyWaitersUserAgent(getComponentRequest)),
                componentDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentDeployed(GetEnvironmentRequest getEnvironmentRequest) {
        return environmentDeployedWaiter.run(() -> client.getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)));
    }

    @Override
    public WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentDeployed(GetEnvironmentRequest getEnvironmentRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return environmentDeployedWaiter.run(() -> client.getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)),
                environmentDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetEnvironmentTemplateVersionResponse> waitUntilEnvironmentTemplateVersionRegistered(
            GetEnvironmentTemplateVersionRequest getEnvironmentTemplateVersionRequest) {
        return environmentTemplateVersionRegisteredWaiter.run(() -> client
                .getEnvironmentTemplateVersion(applyWaitersUserAgent(getEnvironmentTemplateVersionRequest)));
    }

    @Override
    public WaiterResponse<GetEnvironmentTemplateVersionResponse> waitUntilEnvironmentTemplateVersionRegistered(
            GetEnvironmentTemplateVersionRequest getEnvironmentTemplateVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentTemplateVersionRegisteredWaiter.run(
                () -> client.getEnvironmentTemplateVersion(applyWaitersUserAgent(getEnvironmentTemplateVersionRequest)),
                environmentTemplateVersionRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServiceCreated(GetServiceRequest getServiceRequest) {
        return serviceCreatedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServiceCreated(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return serviceCreatedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                serviceCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServiceDeleted(GetServiceRequest getServiceRequest) {
        return serviceDeletedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServiceDeleted(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return serviceDeletedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                serviceDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetServiceInstanceResponse> waitUntilServiceInstanceDeployed(
            GetServiceInstanceRequest getServiceInstanceRequest) {
        return serviceInstanceDeployedWaiter.run(() -> client
                .getServiceInstance(applyWaitersUserAgent(getServiceInstanceRequest)));
    }

    @Override
    public WaiterResponse<GetServiceInstanceResponse> waitUntilServiceInstanceDeployed(
            GetServiceInstanceRequest getServiceInstanceRequest, WaiterOverrideConfiguration overrideConfig) {
        return serviceInstanceDeployedWaiter.run(
                () -> client.getServiceInstance(applyWaitersUserAgent(getServiceInstanceRequest)),
                serviceInstanceDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServicePipelineDeployed(GetServiceRequest getServiceRequest) {
        return servicePipelineDeployedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServicePipelineDeployed(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return servicePipelineDeployedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                servicePipelineDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetServiceTemplateVersionResponse> waitUntilServiceTemplateVersionRegistered(
            GetServiceTemplateVersionRequest getServiceTemplateVersionRequest) {
        return serviceTemplateVersionRegisteredWaiter.run(() -> client
                .getServiceTemplateVersion(applyWaitersUserAgent(getServiceTemplateVersionRequest)));
    }

    @Override
    public WaiterResponse<GetServiceTemplateVersionResponse> waitUntilServiceTemplateVersionRegistered(
            GetServiceTemplateVersionRequest getServiceTemplateVersionRequest, WaiterOverrideConfiguration overrideConfig) {
        return serviceTemplateVersionRegisteredWaiter.run(
                () -> client.getServiceTemplateVersion(applyWaitersUserAgent(getServiceTemplateVersionRequest)),
                serviceTemplateVersionRegisteredWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServiceUpdated(GetServiceRequest getServiceRequest) {
        return serviceUpdatedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)));
    }

    @Override
    public WaiterResponse<GetServiceResponse> waitUntilServiceUpdated(GetServiceRequest getServiceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return serviceUpdatedWaiter.run(() -> client.getService(applyWaitersUserAgent(getServiceRequest)),
                serviceUpdatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetComponentResponse>> componentDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetComponentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("component").field("deploymentStatus").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (component.deploymentStatus=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetComponentResponse>> componentDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetComponentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("component").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("component").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (component.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetEnvironmentResponse>> environmentDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetEnvironmentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("environment").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("environment").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (environment.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetEnvironmentTemplateVersionResponse>> environmentTemplateVersionRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super GetEnvironmentTemplateVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("environmentTemplateVersion").field("status").value(), "DRAFT");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("environmentTemplateVersion").field("status").value(), "PUBLISHED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("environmentTemplateVersion").field("status").value(),
                            "REGISTRATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (environmentTemplateVersion.status=REGISTRATION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> serviceCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("service").field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "CREATE_FAILED_CLEANUP_COMPLETE");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=CREATE_FAILED_CLEANUP_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "CREATE_FAILED_CLEANUP_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=CREATE_FAILED_CLEANUP_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> serviceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "DELETE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceInstanceResponse>> serviceInstanceDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceInstanceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("serviceInstance").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("serviceInstance").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (serviceInstance.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> servicePipelineDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("service").field("pipeline").field("deploymentStatus").value(), "SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("pipeline").field("deploymentStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.pipeline.deploymentStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceTemplateVersionResponse>> serviceTemplateVersionRegisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceTemplateVersionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("serviceTemplateVersion").field("status").value(), "DRAFT");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("serviceTemplateVersion").field("status").value(), "PUBLISHED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("serviceTemplateVersion").field("status").value(), "REGISTRATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (serviceTemplateVersion.status=REGISTRATION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetServiceResponse>> serviceUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetServiceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("service").field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_FAILED_CLEANUP_COMPLETE");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_FAILED_CLEANUP_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_FAILED_CLEANUP_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_FAILED_CLEANUP_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("service").field("status").value(), "UPDATE_COMPLETE_CLEANUP_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (service.status=UPDATE_COMPLETE_CLEANUP_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration componentDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration componentDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentTemplateVersionRegisteredWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(150);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceInstanceDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration servicePipelineDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(360);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceTemplateVersionRegisteredWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(150);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration serviceUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ProtonWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ProtonRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ProtonWaiter.Builder {
        private ProtonClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public ProtonWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ProtonWaiter.Builder client(ProtonClient client) {
            this.client = client;
            return this;
        }

        public ProtonWaiter build() {
            return new DefaultProtonWaiter(this);
        }
    }
}
