/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.proton.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed data of an Proton service template version resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceTemplateVersion implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceTemplateVersion.Builder, ServiceTemplateVersion> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(ServiceTemplateVersion::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<List<CompatibleEnvironmentTemplate>> COMPATIBLE_ENVIRONMENT_TEMPLATES_FIELD = SdkField
            .<List<CompatibleEnvironmentTemplate>> builder(MarshallingType.LIST)
            .memberName("compatibleEnvironmentTemplates")
            .getter(getter(ServiceTemplateVersion::compatibleEnvironmentTemplates))
            .setter(setter(Builder::compatibleEnvironmentTemplates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("compatibleEnvironmentTemplates")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CompatibleEnvironmentTemplate> builder(MarshallingType.SDK_POJO)
                                            .constructor(CompatibleEnvironmentTemplate::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(ServiceTemplateVersion::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(ServiceTemplateVersion::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastModifiedAt").getter(getter(ServiceTemplateVersion::lastModifiedAt))
            .setter(setter(Builder::lastModifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModifiedAt").build()).build();

    private static final SdkField<String> MAJOR_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("majorVersion").getter(getter(ServiceTemplateVersion::majorVersion))
            .setter(setter(Builder::majorVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("majorVersion").build()).build();

    private static final SdkField<String> MINOR_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("minorVersion").getter(getter(ServiceTemplateVersion::minorVersion))
            .setter(setter(Builder::minorVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minorVersion").build()).build();

    private static final SdkField<String> RECOMMENDED_MINOR_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recommendedMinorVersion").getter(getter(ServiceTemplateVersion::recommendedMinorVersion))
            .setter(setter(Builder::recommendedMinorVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendedMinorVersion").build())
            .build();

    private static final SdkField<String> SCHEMA_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("schema")
            .getter(getter(ServiceTemplateVersion::schema)).setter(setter(Builder::schema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schema").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ServiceTemplateVersion::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusMessage").getter(getter(ServiceTemplateVersion::statusMessage))
            .setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusMessage").build()).build();

    private static final SdkField<List<String>> SUPPORTED_COMPONENT_SOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("supportedComponentSources")
            .getter(getter(ServiceTemplateVersion::supportedComponentSourcesAsStrings))
            .setter(setter(Builder::supportedComponentSourcesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportedComponentSources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("templateName").getter(getter(ServiceTemplateVersion::templateName))
            .setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("templateName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            COMPATIBLE_ENVIRONMENT_TEMPLATES_FIELD, CREATED_AT_FIELD, DESCRIPTION_FIELD, LAST_MODIFIED_AT_FIELD,
            MAJOR_VERSION_FIELD, MINOR_VERSION_FIELD, RECOMMENDED_MINOR_VERSION_FIELD, SCHEMA_FIELD, STATUS_FIELD,
            STATUS_MESSAGE_FIELD, SUPPORTED_COMPONENT_SOURCES_FIELD, TEMPLATE_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final List<CompatibleEnvironmentTemplate> compatibleEnvironmentTemplates;

    private final Instant createdAt;

    private final String description;

    private final Instant lastModifiedAt;

    private final String majorVersion;

    private final String minorVersion;

    private final String recommendedMinorVersion;

    private final String schema;

    private final String status;

    private final String statusMessage;

    private final List<String> supportedComponentSources;

    private final String templateName;

    private ServiceTemplateVersion(BuilderImpl builder) {
        this.arn = builder.arn;
        this.compatibleEnvironmentTemplates = builder.compatibleEnvironmentTemplates;
        this.createdAt = builder.createdAt;
        this.description = builder.description;
        this.lastModifiedAt = builder.lastModifiedAt;
        this.majorVersion = builder.majorVersion;
        this.minorVersion = builder.minorVersion;
        this.recommendedMinorVersion = builder.recommendedMinorVersion;
        this.schema = builder.schema;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
        this.supportedComponentSources = builder.supportedComponentSources;
        this.templateName = builder.templateName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the version of a service template.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the version of a service template.
     */
    public final String arn() {
        return arn;
    }

    /**
     * For responses, this returns true if the service returned a value for the CompatibleEnvironmentTemplates property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCompatibleEnvironmentTemplates() {
        return compatibleEnvironmentTemplates != null && !(compatibleEnvironmentTemplates instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of compatible environment template names for the major version of a service template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCompatibleEnvironmentTemplates} method.
     * </p>
     * 
     * @return An array of compatible environment template names for the major version of a service template.
     */
    public final List<CompatibleEnvironmentTemplate> compatibleEnvironmentTemplates() {
        return compatibleEnvironmentTemplates;
    }

    /**
     * <p>
     * The time when the version of a service template was created.
     * </p>
     * 
     * @return The time when the version of a service template was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * A description of the version of a service template.
     * </p>
     * 
     * @return A description of the version of a service template.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The time when the version of a service template was last modified.
     * </p>
     * 
     * @return The time when the version of a service template was last modified.
     */
    public final Instant lastModifiedAt() {
        return lastModifiedAt;
    }

    /**
     * <p>
     * The latest major version that's associated with the version of a service template.
     * </p>
     * 
     * @return The latest major version that's associated with the version of a service template.
     */
    public final String majorVersion() {
        return majorVersion;
    }

    /**
     * <p>
     * The minor version of a service template.
     * </p>
     * 
     * @return The minor version of a service template.
     */
    public final String minorVersion() {
        return minorVersion;
    }

    /**
     * <p>
     * The recommended minor version of the service template.
     * </p>
     * 
     * @return The recommended minor version of the service template.
     */
    public final String recommendedMinorVersion() {
        return recommendedMinorVersion;
    }

    /**
     * <p>
     * The schema of the version of a service template.
     * </p>
     * 
     * @return The schema of the version of a service template.
     */
    public final String schema() {
        return schema;
    }

    /**
     * <p>
     * The service template version status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TemplateVersionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The service template version status.
     * @see TemplateVersionStatus
     */
    public final TemplateVersionStatus status() {
        return TemplateVersionStatus.fromValue(status);
    }

    /**
     * <p>
     * The service template version status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TemplateVersionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The service template version status.
     * @see TemplateVersionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A service template version status message.
     * </p>
     * 
     * @return A service template version status message.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * An array of supported component sources. Components with supported sources can be attached to service instances
     * based on this service template version.
     * </p>
     * <p>
     * For more information about components, see <a
     * href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in the
     * <i>Proton User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedComponentSources} method.
     * </p>
     * 
     * @return An array of supported component sources. Components with supported sources can be attached to service
     *         instances based on this service template version.</p>
     *         <p>
     *         For more information about components, see <a
     *         href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in
     *         the <i>Proton User Guide</i>.
     */
    public final List<ServiceTemplateSupportedComponentSourceType> supportedComponentSources() {
        return ServiceTemplateSupportedComponentSourceInputListCopier.copyStringToEnum(supportedComponentSources);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedComponentSources property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedComponentSources() {
        return supportedComponentSources != null && !(supportedComponentSources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of supported component sources. Components with supported sources can be attached to service instances
     * based on this service template version.
     * </p>
     * <p>
     * For more information about components, see <a
     * href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in the
     * <i>Proton User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedComponentSources} method.
     * </p>
     * 
     * @return An array of supported component sources. Components with supported sources can be attached to service
     *         instances based on this service template version.</p>
     *         <p>
     *         For more information about components, see <a
     *         href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in
     *         the <i>Proton User Guide</i>.
     */
    public final List<String> supportedComponentSourcesAsStrings() {
        return supportedComponentSources;
    }

    /**
     * <p>
     * The name of the version of a service template.
     * </p>
     * 
     * @return The name of the version of a service template.
     */
    public final String templateName() {
        return templateName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasCompatibleEnvironmentTemplates() ? compatibleEnvironmentTemplates() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(majorVersion());
        hashCode = 31 * hashCode + Objects.hashCode(minorVersion());
        hashCode = 31 * hashCode + Objects.hashCode(recommendedMinorVersion());
        hashCode = 31 * hashCode + Objects.hashCode(schema());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedComponentSources() ? supportedComponentSourcesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceTemplateVersion)) {
            return false;
        }
        ServiceTemplateVersion other = (ServiceTemplateVersion) obj;
        return Objects.equals(arn(), other.arn())
                && hasCompatibleEnvironmentTemplates() == other.hasCompatibleEnvironmentTemplates()
                && Objects.equals(compatibleEnvironmentTemplates(), other.compatibleEnvironmentTemplates())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(description(), other.description())
                && Objects.equals(lastModifiedAt(), other.lastModifiedAt())
                && Objects.equals(majorVersion(), other.majorVersion()) && Objects.equals(minorVersion(), other.minorVersion())
                && Objects.equals(recommendedMinorVersion(), other.recommendedMinorVersion())
                && Objects.equals(schema(), other.schema()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage())
                && hasSupportedComponentSources() == other.hasSupportedComponentSources()
                && Objects.equals(supportedComponentSourcesAsStrings(), other.supportedComponentSourcesAsStrings())
                && Objects.equals(templateName(), other.templateName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("ServiceTemplateVersion")
                .add("Arn", arn())
                .add("CompatibleEnvironmentTemplates",
                        hasCompatibleEnvironmentTemplates() ? compatibleEnvironmentTemplates() : null)
                .add("CreatedAt", createdAt())
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("LastModifiedAt", lastModifiedAt()).add("MajorVersion", majorVersion()).add("MinorVersion", minorVersion())
                .add("RecommendedMinorVersion", recommendedMinorVersion())
                .add("Schema", schema() == null ? null : "*** Sensitive Data Redacted ***").add("Status", statusAsString())
                .add("StatusMessage", statusMessage() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SupportedComponentSources", hasSupportedComponentSources() ? supportedComponentSourcesAsStrings() : null)
                .add("TemplateName", templateName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "compatibleEnvironmentTemplates":
            return Optional.ofNullable(clazz.cast(compatibleEnvironmentTemplates()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "lastModifiedAt":
            return Optional.ofNullable(clazz.cast(lastModifiedAt()));
        case "majorVersion":
            return Optional.ofNullable(clazz.cast(majorVersion()));
        case "minorVersion":
            return Optional.ofNullable(clazz.cast(minorVersion()));
        case "recommendedMinorVersion":
            return Optional.ofNullable(clazz.cast(recommendedMinorVersion()));
        case "schema":
            return Optional.ofNullable(clazz.cast(schema()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "supportedComponentSources":
            return Optional.ofNullable(clazz.cast(supportedComponentSourcesAsStrings()));
        case "templateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("compatibleEnvironmentTemplates", COMPATIBLE_ENVIRONMENT_TEMPLATES_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("lastModifiedAt", LAST_MODIFIED_AT_FIELD);
        map.put("majorVersion", MAJOR_VERSION_FIELD);
        map.put("minorVersion", MINOR_VERSION_FIELD);
        map.put("recommendedMinorVersion", RECOMMENDED_MINOR_VERSION_FIELD);
        map.put("schema", SCHEMA_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusMessage", STATUS_MESSAGE_FIELD);
        map.put("supportedComponentSources", SUPPORTED_COMPONENT_SOURCES_FIELD);
        map.put("templateName", TEMPLATE_NAME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ServiceTemplateVersion, T> g) {
        return obj -> g.apply((ServiceTemplateVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceTemplateVersion> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the version of a service template.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * An array of compatible environment template names for the major version of a service template.
         * </p>
         * 
         * @param compatibleEnvironmentTemplates
         *        An array of compatible environment template names for the major version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibleEnvironmentTemplates(Collection<CompatibleEnvironmentTemplate> compatibleEnvironmentTemplates);

        /**
         * <p>
         * An array of compatible environment template names for the major version of a service template.
         * </p>
         * 
         * @param compatibleEnvironmentTemplates
         *        An array of compatible environment template names for the major version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibleEnvironmentTemplates(CompatibleEnvironmentTemplate... compatibleEnvironmentTemplates);

        /**
         * <p>
         * An array of compatible environment template names for the major version of a service template.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.proton.model.CompatibleEnvironmentTemplate.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.proton.model.CompatibleEnvironmentTemplate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.proton.model.CompatibleEnvironmentTemplate.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #compatibleEnvironmentTemplates(List<CompatibleEnvironmentTemplate>)}.
         * 
         * @param compatibleEnvironmentTemplates
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.proton.model.CompatibleEnvironmentTemplate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #compatibleEnvironmentTemplates(java.util.Collection<CompatibleEnvironmentTemplate>)
         */
        Builder compatibleEnvironmentTemplates(Consumer<CompatibleEnvironmentTemplate.Builder>... compatibleEnvironmentTemplates);

        /**
         * <p>
         * The time when the version of a service template was created.
         * </p>
         * 
         * @param createdAt
         *        The time when the version of a service template was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * A description of the version of a service template.
         * </p>
         * 
         * @param description
         *        A description of the version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The time when the version of a service template was last modified.
         * </p>
         * 
         * @param lastModifiedAt
         *        The time when the version of a service template was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedAt(Instant lastModifiedAt);

        /**
         * <p>
         * The latest major version that's associated with the version of a service template.
         * </p>
         * 
         * @param majorVersion
         *        The latest major version that's associated with the version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder majorVersion(String majorVersion);

        /**
         * <p>
         * The minor version of a service template.
         * </p>
         * 
         * @param minorVersion
         *        The minor version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minorVersion(String minorVersion);

        /**
         * <p>
         * The recommended minor version of the service template.
         * </p>
         * 
         * @param recommendedMinorVersion
         *        The recommended minor version of the service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendedMinorVersion(String recommendedMinorVersion);

        /**
         * <p>
         * The schema of the version of a service template.
         * </p>
         * 
         * @param schema
         *        The schema of the version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(String schema);

        /**
         * <p>
         * The service template version status.
         * </p>
         * 
         * @param status
         *        The service template version status.
         * @see TemplateVersionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TemplateVersionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The service template version status.
         * </p>
         * 
         * @param status
         *        The service template version status.
         * @see TemplateVersionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TemplateVersionStatus
         */
        Builder status(TemplateVersionStatus status);

        /**
         * <p>
         * A service template version status message.
         * </p>
         * 
         * @param statusMessage
         *        A service template version status message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * An array of supported component sources. Components with supported sources can be attached to service
         * instances based on this service template version.
         * </p>
         * <p>
         * For more information about components, see <a
         * href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in the
         * <i>Proton User Guide</i>.
         * </p>
         * 
         * @param supportedComponentSources
         *        An array of supported component sources. Components with supported sources can be attached to service
         *        instances based on this service template version.</p>
         *        <p>
         *        For more information about components, see <a
         *        href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in
         *        the <i>Proton User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedComponentSourcesWithStrings(Collection<String> supportedComponentSources);

        /**
         * <p>
         * An array of supported component sources. Components with supported sources can be attached to service
         * instances based on this service template version.
         * </p>
         * <p>
         * For more information about components, see <a
         * href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in the
         * <i>Proton User Guide</i>.
         * </p>
         * 
         * @param supportedComponentSources
         *        An array of supported component sources. Components with supported sources can be attached to service
         *        instances based on this service template version.</p>
         *        <p>
         *        For more information about components, see <a
         *        href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in
         *        the <i>Proton User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedComponentSourcesWithStrings(String... supportedComponentSources);

        /**
         * <p>
         * An array of supported component sources. Components with supported sources can be attached to service
         * instances based on this service template version.
         * </p>
         * <p>
         * For more information about components, see <a
         * href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in the
         * <i>Proton User Guide</i>.
         * </p>
         * 
         * @param supportedComponentSources
         *        An array of supported component sources. Components with supported sources can be attached to service
         *        instances based on this service template version.</p>
         *        <p>
         *        For more information about components, see <a
         *        href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in
         *        the <i>Proton User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedComponentSources(Collection<ServiceTemplateSupportedComponentSourceType> supportedComponentSources);

        /**
         * <p>
         * An array of supported component sources. Components with supported sources can be attached to service
         * instances based on this service template version.
         * </p>
         * <p>
         * For more information about components, see <a
         * href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in the
         * <i>Proton User Guide</i>.
         * </p>
         * 
         * @param supportedComponentSources
         *        An array of supported component sources. Components with supported sources can be attached to service
         *        instances based on this service template version.</p>
         *        <p>
         *        For more information about components, see <a
         *        href="https://docs.aws.amazon.com/proton/latest/userguide/ag-components.html">Proton components</a> in
         *        the <i>Proton User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedComponentSources(ServiceTemplateSupportedComponentSourceType... supportedComponentSources);

        /**
         * <p>
         * The name of the version of a service template.
         * </p>
         * 
         * @param templateName
         *        The name of the version of a service template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private List<CompatibleEnvironmentTemplate> compatibleEnvironmentTemplates = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private String description;

        private Instant lastModifiedAt;

        private String majorVersion;

        private String minorVersion;

        private String recommendedMinorVersion;

        private String schema;

        private String status;

        private String statusMessage;

        private List<String> supportedComponentSources = DefaultSdkAutoConstructList.getInstance();

        private String templateName;

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceTemplateVersion model) {
            arn(model.arn);
            compatibleEnvironmentTemplates(model.compatibleEnvironmentTemplates);
            createdAt(model.createdAt);
            description(model.description);
            lastModifiedAt(model.lastModifiedAt);
            majorVersion(model.majorVersion);
            minorVersion(model.minorVersion);
            recommendedMinorVersion(model.recommendedMinorVersion);
            schema(model.schema);
            status(model.status);
            statusMessage(model.statusMessage);
            supportedComponentSourcesWithStrings(model.supportedComponentSources);
            templateName(model.templateName);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final List<CompatibleEnvironmentTemplate.Builder> getCompatibleEnvironmentTemplates() {
            List<CompatibleEnvironmentTemplate.Builder> result = CompatibleEnvironmentTemplateListCopier
                    .copyToBuilder(this.compatibleEnvironmentTemplates);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCompatibleEnvironmentTemplates(
                Collection<CompatibleEnvironmentTemplate.BuilderImpl> compatibleEnvironmentTemplates) {
            this.compatibleEnvironmentTemplates = CompatibleEnvironmentTemplateListCopier
                    .copyFromBuilder(compatibleEnvironmentTemplates);
        }

        @Override
        public final Builder compatibleEnvironmentTemplates(
                Collection<CompatibleEnvironmentTemplate> compatibleEnvironmentTemplates) {
            this.compatibleEnvironmentTemplates = CompatibleEnvironmentTemplateListCopier.copy(compatibleEnvironmentTemplates);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder compatibleEnvironmentTemplates(CompatibleEnvironmentTemplate... compatibleEnvironmentTemplates) {
            compatibleEnvironmentTemplates(Arrays.asList(compatibleEnvironmentTemplates));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder compatibleEnvironmentTemplates(
                Consumer<CompatibleEnvironmentTemplate.Builder>... compatibleEnvironmentTemplates) {
            compatibleEnvironmentTemplates(Stream.of(compatibleEnvironmentTemplates)
                    .map(c -> CompatibleEnvironmentTemplate.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getLastModifiedAt() {
            return lastModifiedAt;
        }

        public final void setLastModifiedAt(Instant lastModifiedAt) {
            this.lastModifiedAt = lastModifiedAt;
        }

        @Override
        public final Builder lastModifiedAt(Instant lastModifiedAt) {
            this.lastModifiedAt = lastModifiedAt;
            return this;
        }

        public final String getMajorVersion() {
            return majorVersion;
        }

        public final void setMajorVersion(String majorVersion) {
            this.majorVersion = majorVersion;
        }

        @Override
        public final Builder majorVersion(String majorVersion) {
            this.majorVersion = majorVersion;
            return this;
        }

        public final String getMinorVersion() {
            return minorVersion;
        }

        public final void setMinorVersion(String minorVersion) {
            this.minorVersion = minorVersion;
        }

        @Override
        public final Builder minorVersion(String minorVersion) {
            this.minorVersion = minorVersion;
            return this;
        }

        public final String getRecommendedMinorVersion() {
            return recommendedMinorVersion;
        }

        public final void setRecommendedMinorVersion(String recommendedMinorVersion) {
            this.recommendedMinorVersion = recommendedMinorVersion;
        }

        @Override
        public final Builder recommendedMinorVersion(String recommendedMinorVersion) {
            this.recommendedMinorVersion = recommendedMinorVersion;
            return this;
        }

        public final String getSchema() {
            return schema;
        }

        public final void setSchema(String schema) {
            this.schema = schema;
        }

        @Override
        public final Builder schema(String schema) {
            this.schema = schema;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TemplateVersionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final Collection<String> getSupportedComponentSources() {
            if (supportedComponentSources instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedComponentSources;
        }

        public final void setSupportedComponentSources(Collection<String> supportedComponentSources) {
            this.supportedComponentSources = ServiceTemplateSupportedComponentSourceInputListCopier
                    .copy(supportedComponentSources);
        }

        @Override
        public final Builder supportedComponentSourcesWithStrings(Collection<String> supportedComponentSources) {
            this.supportedComponentSources = ServiceTemplateSupportedComponentSourceInputListCopier
                    .copy(supportedComponentSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedComponentSourcesWithStrings(String... supportedComponentSources) {
            supportedComponentSourcesWithStrings(Arrays.asList(supportedComponentSources));
            return this;
        }

        @Override
        public final Builder supportedComponentSources(
                Collection<ServiceTemplateSupportedComponentSourceType> supportedComponentSources) {
            this.supportedComponentSources = ServiceTemplateSupportedComponentSourceInputListCopier
                    .copyEnumToString(supportedComponentSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedComponentSources(ServiceTemplateSupportedComponentSourceType... supportedComponentSources) {
            supportedComponentSources(Arrays.asList(supportedComponentSources));
            return this;
        }

        public final String getTemplateName() {
            return templateName;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        @Override
        public ServiceTemplateVersion build() {
            return new ServiceTemplateVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
