/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.qapps;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.qapps.internal.QAppsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.qapps.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.qapps.model.AccessDeniedException;
import software.amazon.awssdk.services.qapps.model.AssociateLibraryItemReviewRequest;
import software.amazon.awssdk.services.qapps.model.AssociateLibraryItemReviewResponse;
import software.amazon.awssdk.services.qapps.model.AssociateQAppWithUserRequest;
import software.amazon.awssdk.services.qapps.model.AssociateQAppWithUserResponse;
import software.amazon.awssdk.services.qapps.model.BatchCreateCategoryRequest;
import software.amazon.awssdk.services.qapps.model.BatchCreateCategoryResponse;
import software.amazon.awssdk.services.qapps.model.BatchDeleteCategoryRequest;
import software.amazon.awssdk.services.qapps.model.BatchDeleteCategoryResponse;
import software.amazon.awssdk.services.qapps.model.BatchUpdateCategoryRequest;
import software.amazon.awssdk.services.qapps.model.BatchUpdateCategoryResponse;
import software.amazon.awssdk.services.qapps.model.ConflictException;
import software.amazon.awssdk.services.qapps.model.ContentTooLargeException;
import software.amazon.awssdk.services.qapps.model.CreateLibraryItemRequest;
import software.amazon.awssdk.services.qapps.model.CreateLibraryItemResponse;
import software.amazon.awssdk.services.qapps.model.CreatePresignedUrlRequest;
import software.amazon.awssdk.services.qapps.model.CreatePresignedUrlResponse;
import software.amazon.awssdk.services.qapps.model.CreateQAppRequest;
import software.amazon.awssdk.services.qapps.model.CreateQAppResponse;
import software.amazon.awssdk.services.qapps.model.DeleteLibraryItemRequest;
import software.amazon.awssdk.services.qapps.model.DeleteLibraryItemResponse;
import software.amazon.awssdk.services.qapps.model.DeleteQAppRequest;
import software.amazon.awssdk.services.qapps.model.DeleteQAppResponse;
import software.amazon.awssdk.services.qapps.model.DescribeQAppPermissionsRequest;
import software.amazon.awssdk.services.qapps.model.DescribeQAppPermissionsResponse;
import software.amazon.awssdk.services.qapps.model.DisassociateLibraryItemReviewRequest;
import software.amazon.awssdk.services.qapps.model.DisassociateLibraryItemReviewResponse;
import software.amazon.awssdk.services.qapps.model.DisassociateQAppFromUserRequest;
import software.amazon.awssdk.services.qapps.model.DisassociateQAppFromUserResponse;
import software.amazon.awssdk.services.qapps.model.ExportQAppSessionDataRequest;
import software.amazon.awssdk.services.qapps.model.ExportQAppSessionDataResponse;
import software.amazon.awssdk.services.qapps.model.GetLibraryItemRequest;
import software.amazon.awssdk.services.qapps.model.GetLibraryItemResponse;
import software.amazon.awssdk.services.qapps.model.GetQAppRequest;
import software.amazon.awssdk.services.qapps.model.GetQAppResponse;
import software.amazon.awssdk.services.qapps.model.GetQAppSessionMetadataRequest;
import software.amazon.awssdk.services.qapps.model.GetQAppSessionMetadataResponse;
import software.amazon.awssdk.services.qapps.model.GetQAppSessionRequest;
import software.amazon.awssdk.services.qapps.model.GetQAppSessionResponse;
import software.amazon.awssdk.services.qapps.model.ImportDocumentRequest;
import software.amazon.awssdk.services.qapps.model.ImportDocumentResponse;
import software.amazon.awssdk.services.qapps.model.InternalServerException;
import software.amazon.awssdk.services.qapps.model.ListCategoriesRequest;
import software.amazon.awssdk.services.qapps.model.ListCategoriesResponse;
import software.amazon.awssdk.services.qapps.model.ListLibraryItemsRequest;
import software.amazon.awssdk.services.qapps.model.ListLibraryItemsResponse;
import software.amazon.awssdk.services.qapps.model.ListQAppSessionDataRequest;
import software.amazon.awssdk.services.qapps.model.ListQAppSessionDataResponse;
import software.amazon.awssdk.services.qapps.model.ListQAppsRequest;
import software.amazon.awssdk.services.qapps.model.ListQAppsResponse;
import software.amazon.awssdk.services.qapps.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.qapps.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.qapps.model.PredictQAppRequest;
import software.amazon.awssdk.services.qapps.model.PredictQAppResponse;
import software.amazon.awssdk.services.qapps.model.QAppsException;
import software.amazon.awssdk.services.qapps.model.ResourceNotFoundException;
import software.amazon.awssdk.services.qapps.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.qapps.model.StartQAppSessionRequest;
import software.amazon.awssdk.services.qapps.model.StartQAppSessionResponse;
import software.amazon.awssdk.services.qapps.model.StopQAppSessionRequest;
import software.amazon.awssdk.services.qapps.model.StopQAppSessionResponse;
import software.amazon.awssdk.services.qapps.model.TagResourceRequest;
import software.amazon.awssdk.services.qapps.model.TagResourceResponse;
import software.amazon.awssdk.services.qapps.model.ThrottlingException;
import software.amazon.awssdk.services.qapps.model.UnauthorizedException;
import software.amazon.awssdk.services.qapps.model.UntagResourceRequest;
import software.amazon.awssdk.services.qapps.model.UntagResourceResponse;
import software.amazon.awssdk.services.qapps.model.UpdateLibraryItemMetadataRequest;
import software.amazon.awssdk.services.qapps.model.UpdateLibraryItemMetadataResponse;
import software.amazon.awssdk.services.qapps.model.UpdateLibraryItemRequest;
import software.amazon.awssdk.services.qapps.model.UpdateLibraryItemResponse;
import software.amazon.awssdk.services.qapps.model.UpdateQAppPermissionsRequest;
import software.amazon.awssdk.services.qapps.model.UpdateQAppPermissionsResponse;
import software.amazon.awssdk.services.qapps.model.UpdateQAppRequest;
import software.amazon.awssdk.services.qapps.model.UpdateQAppResponse;
import software.amazon.awssdk.services.qapps.model.UpdateQAppSessionMetadataRequest;
import software.amazon.awssdk.services.qapps.model.UpdateQAppSessionMetadataResponse;
import software.amazon.awssdk.services.qapps.model.UpdateQAppSessionRequest;
import software.amazon.awssdk.services.qapps.model.UpdateQAppSessionResponse;
import software.amazon.awssdk.services.qapps.model.ValidationException;
import software.amazon.awssdk.services.qapps.transform.AssociateLibraryItemReviewRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.AssociateQAppWithUserRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.BatchCreateCategoryRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.BatchDeleteCategoryRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.BatchUpdateCategoryRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.CreateLibraryItemRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.CreatePresignedUrlRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.CreateQAppRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.DeleteLibraryItemRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.DeleteQAppRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.DescribeQAppPermissionsRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.DisassociateLibraryItemReviewRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.DisassociateQAppFromUserRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.ExportQAppSessionDataRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.GetLibraryItemRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.GetQAppRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.GetQAppSessionMetadataRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.GetQAppSessionRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.ImportDocumentRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.ListCategoriesRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.ListLibraryItemsRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.ListQAppSessionDataRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.ListQAppsRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.PredictQAppRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.StartQAppSessionRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.StopQAppSessionRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.UpdateLibraryItemMetadataRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.UpdateLibraryItemRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.UpdateQAppPermissionsRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.UpdateQAppRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.UpdateQAppSessionMetadataRequestMarshaller;
import software.amazon.awssdk.services.qapps.transform.UpdateQAppSessionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link QAppsAsyncClient}.
 *
 * @see QAppsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultQAppsAsyncClient implements QAppsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultQAppsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultQAppsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "QApps" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a rating or review for a library item with the user submitting the request. This increments the rating
     * count for the specified library item.
     * </p>
     *
     * @param associateLibraryItemReviewRequest
     * @return A Java Future containing the result of the AssociateLibraryItemReview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.AssociateLibraryItemReview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/AssociateLibraryItemReview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateLibraryItemReviewResponse> associateLibraryItemReview(
            AssociateLibraryItemReviewRequest associateLibraryItemReviewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateLibraryItemReviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateLibraryItemReviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateLibraryItemReview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateLibraryItemReviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateLibraryItemReviewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateLibraryItemReviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateLibraryItemReviewRequest, AssociateLibraryItemReviewResponse>()
                            .withOperationName("AssociateLibraryItemReview").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateLibraryItemReviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateLibraryItemReviewRequest));
            CompletableFuture<AssociateLibraryItemReviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation creates a link between the user's identity calling the operation and a specific Q App. This is
     * useful to mark the Q App as a <i>favorite</i> for the user if the user doesn't own the Amazon Q App so they can
     * still run it and see it in their inventory of Q Apps.
     * </p>
     *
     * @param associateQAppWithUserRequest
     * @return A Java Future containing the result of the AssociateQAppWithUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.AssociateQAppWithUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/AssociateQAppWithUser" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateQAppWithUserResponse> associateQAppWithUser(
            AssociateQAppWithUserRequest associateQAppWithUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateQAppWithUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateQAppWithUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateQAppWithUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateQAppWithUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateQAppWithUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateQAppWithUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateQAppWithUserRequest, AssociateQAppWithUserResponse>()
                            .withOperationName("AssociateQAppWithUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateQAppWithUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateQAppWithUserRequest));
            CompletableFuture<AssociateQAppWithUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates Categories for the Amazon Q Business application environment instance. Web experience users use
     * Categories to tag and filter library items. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/qapps-custom-labels.html">Custom labels for Amazon
     * Q Apps</a>.
     * </p>
     *
     * @param batchCreateCategoryRequest
     * @return A Java Future containing the result of the BatchCreateCategory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.BatchCreateCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/BatchCreateCategory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCreateCategoryResponse> batchCreateCategory(
            BatchCreateCategoryRequest batchCreateCategoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCreateCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCreateCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateCategory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCreateCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCreateCategoryResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchCreateCategoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCreateCategoryRequest, BatchCreateCategoryResponse>()
                            .withOperationName("BatchCreateCategory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchCreateCategoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchCreateCategoryRequest));
            CompletableFuture<BatchCreateCategoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes Categories for the Amazon Q Business application environment instance. Web experience users use
     * Categories to tag and filter library items. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/qapps-custom-labels.html">Custom labels for Amazon
     * Q Apps</a>.
     * </p>
     *
     * @param batchDeleteCategoryRequest
     * @return A Java Future containing the result of the BatchDeleteCategory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.BatchDeleteCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/BatchDeleteCategory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteCategoryResponse> batchDeleteCategory(
            BatchDeleteCategoryRequest batchDeleteCategoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteCategory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteCategoryResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchDeleteCategoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteCategoryRequest, BatchDeleteCategoryResponse>()
                            .withOperationName("BatchDeleteCategory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDeleteCategoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDeleteCategoryRequest));
            CompletableFuture<BatchDeleteCategoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates Categories for the Amazon Q Business application environment instance. Web experience users use
     * Categories to tag and filter library items. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/qapps-custom-labels.html">Custom labels for Amazon
     * Q Apps</a>.
     * </p>
     *
     * @param batchUpdateCategoryRequest
     * @return A Java Future containing the result of the BatchUpdateCategory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.BatchUpdateCategory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/BatchUpdateCategory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateCategoryResponse> batchUpdateCategory(
            BatchUpdateCategoryRequest batchUpdateCategoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUpdateCategoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateCategoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateCategory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateCategoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateCategoryResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchUpdateCategoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateCategoryRequest, BatchUpdateCategoryResponse>()
                            .withOperationName("BatchUpdateCategory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchUpdateCategoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchUpdateCategoryRequest));
            CompletableFuture<BatchUpdateCategoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new library item for an Amazon Q App, allowing it to be discovered and used by other allowed users.
     * </p>
     *
     * @param createLibraryItemRequest
     * @return A Java Future containing the result of the CreateLibraryItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.CreateLibraryItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/CreateLibraryItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLibraryItemResponse> createLibraryItem(CreateLibraryItemRequest createLibraryItemRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLibraryItemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLibraryItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLibraryItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLibraryItemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLibraryItemResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateLibraryItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLibraryItemRequest, CreateLibraryItemResponse>()
                            .withOperationName("CreateLibraryItem").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateLibraryItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLibraryItemRequest));
            CompletableFuture<CreateLibraryItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a presigned URL for an S3 POST operation to upload a file. You can use this URL to set a default file for
     * a <code>FileUploadCard</code> in a Q App definition or to provide a file for a single Q App run. The
     * <code>scope</code> parameter determines how the file will be used, either at the app definition level or the app
     * session level.
     * </p>
     * <note>
     * <p>
     * The IAM permissions are derived from the <code>qapps:ImportDocument</code> action. For more information on the
     * IAM policy for Amazon Q Apps, see <a
     * href="https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/deploy-q-apps-iam-permissions.html">IAM permissions
     * for using Amazon Q Apps</a>.
     * </p>
     * </note>
     *
     * @param createPresignedUrlRequest
     * @return A Java Future containing the result of the CreatePresignedUrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.CreatePresignedUrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/CreatePresignedUrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePresignedUrlResponse> createPresignedUrl(CreatePresignedUrlRequest createPresignedUrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPresignedUrlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPresignedUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePresignedUrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePresignedUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePresignedUrlResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreatePresignedUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePresignedUrlRequest, CreatePresignedUrlResponse>()
                            .withOperationName("CreatePresignedUrl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePresignedUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPresignedUrlRequest));
            CompletableFuture<CreatePresignedUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon Q App based on the provided definition. The Q App definition specifies the cards and flow of
     * the Q App. This operation also calculates the dependencies between the cards by inspecting the references in the
     * prompts.
     * </p>
     *
     * @param createQAppRequest
     * @return A Java Future containing the result of the CreateQApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ContentTooLargeException The requested operation could not be completed because the content exceeds
     *         the maximum allowed size.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.CreateQApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/CreateQApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateQAppResponse> createQApp(CreateQAppRequest createQAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createQAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateQAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateQAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateQAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateQAppRequest, CreateQAppResponse>().withOperationName("CreateQApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateQAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createQAppRequest));
            CompletableFuture<CreateQAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a library item for an Amazon Q App, removing it from the library so it can no longer be discovered or
     * used by other users.
     * </p>
     *
     * @param deleteLibraryItemRequest
     * @return A Java Future containing the result of the DeleteLibraryItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.DeleteLibraryItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/DeleteLibraryItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLibraryItemResponse> deleteLibraryItem(DeleteLibraryItemRequest deleteLibraryItemRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLibraryItemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLibraryItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLibraryItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLibraryItemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLibraryItemResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteLibraryItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLibraryItemRequest, DeleteLibraryItemResponse>()
                            .withOperationName("DeleteLibraryItem").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteLibraryItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLibraryItemRequest));
            CompletableFuture<DeleteLibraryItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Q App owned by the user. If the Q App was previously published to the library, it is also
     * removed from the library.
     * </p>
     *
     * @param deleteQAppRequest
     * @return A Java Future containing the result of the DeleteQApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.DeleteQApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/DeleteQApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteQAppResponse> deleteQApp(DeleteQAppRequest deleteQAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteQAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteQAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteQApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteQAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteQAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteQAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteQAppRequest, DeleteQAppResponse>().withOperationName("DeleteQApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteQAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteQAppRequest));
            CompletableFuture<DeleteQAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes read permissions for a Amazon Q App in Amazon Q Business application environment instance.
     * </p>
     *
     * @param describeQAppPermissionsRequest
     * @return A Java Future containing the result of the DescribeQAppPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.DescribeQAppPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/DescribeQAppPermissions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeQAppPermissionsResponse> describeQAppPermissions(
            DescribeQAppPermissionsRequest describeQAppPermissionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeQAppPermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeQAppPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeQAppPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeQAppPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeQAppPermissionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeQAppPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeQAppPermissionsRequest, DescribeQAppPermissionsResponse>()
                            .withOperationName("DescribeQAppPermissions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeQAppPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeQAppPermissionsRequest));
            CompletableFuture<DescribeQAppPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a rating or review previously submitted by the user for a library item.
     * </p>
     *
     * @param disassociateLibraryItemReviewRequest
     * @return A Java Future containing the result of the DisassociateLibraryItemReview operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.DisassociateLibraryItemReview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/DisassociateLibraryItemReview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateLibraryItemReviewResponse> disassociateLibraryItemReview(
            DisassociateLibraryItemReviewRequest disassociateLibraryItemReviewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateLibraryItemReviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateLibraryItemReviewRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateLibraryItemReview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateLibraryItemReviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateLibraryItemReviewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateLibraryItemReviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateLibraryItemReviewRequest, DisassociateLibraryItemReviewResponse>()
                            .withOperationName("DisassociateLibraryItemReview").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateLibraryItemReviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateLibraryItemReviewRequest));
            CompletableFuture<DisassociateLibraryItemReviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a Q App from a user removing the user's access to run the Q App.
     * </p>
     *
     * @param disassociateQAppFromUserRequest
     * @return A Java Future containing the result of the DisassociateQAppFromUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.DisassociateQAppFromUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/DisassociateQAppFromUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateQAppFromUserResponse> disassociateQAppFromUser(
            DisassociateQAppFromUserRequest disassociateQAppFromUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateQAppFromUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateQAppFromUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateQAppFromUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateQAppFromUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateQAppFromUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateQAppFromUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateQAppFromUserRequest, DisassociateQAppFromUserResponse>()
                            .withOperationName("DisassociateQAppFromUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateQAppFromUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateQAppFromUserRequest));
            CompletableFuture<DisassociateQAppFromUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports the collected data of a Q App data collection session.
     * </p>
     *
     * @param exportQAppSessionDataRequest
     * @return A Java Future containing the result of the ExportQAppSessionData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.ExportQAppSessionData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/ExportQAppSessionData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportQAppSessionDataResponse> exportQAppSessionData(
            ExportQAppSessionDataRequest exportQAppSessionDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportQAppSessionDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportQAppSessionDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportQAppSessionData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportQAppSessionDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportQAppSessionDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ExportQAppSessionDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportQAppSessionDataRequest, ExportQAppSessionDataResponse>()
                            .withOperationName("ExportQAppSessionData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportQAppSessionDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportQAppSessionDataRequest));
            CompletableFuture<ExportQAppSessionDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about a library item for an Amazon Q App, including its metadata, categories, ratings, and
     * usage statistics.
     * </p>
     *
     * @param getLibraryItemRequest
     * @return A Java Future containing the result of the GetLibraryItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.GetLibraryItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/GetLibraryItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLibraryItemResponse> getLibraryItem(GetLibraryItemRequest getLibraryItemRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLibraryItemRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLibraryItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLibraryItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLibraryItemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLibraryItemResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetLibraryItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLibraryItemRequest, GetLibraryItemResponse>()
                            .withOperationName("GetLibraryItem").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLibraryItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLibraryItemRequest));
            CompletableFuture<GetLibraryItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the full details of an Q App, including its definition specifying the cards and flow.
     * </p>
     *
     * @param getQAppRequest
     * @return A Java Future containing the result of the GetQApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.GetQApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/GetQApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQAppResponse> getQApp(GetQAppRequest getQAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetQAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQAppRequest, GetQAppResponse>().withOperationName("GetQApp")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetQAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQAppRequest));
            CompletableFuture<GetQAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current state and results for an active session of an Amazon Q App.
     * </p>
     *
     * @param getQAppSessionRequest
     * @return A Java Future containing the result of the GetQAppSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.GetQAppSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/GetQAppSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQAppSessionResponse> getQAppSession(GetQAppSessionRequest getQAppSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQAppSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQAppSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQAppSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQAppSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQAppSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQAppSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQAppSessionRequest, GetQAppSessionResponse>()
                            .withOperationName("GetQAppSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQAppSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQAppSessionRequest));
            CompletableFuture<GetQAppSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current configuration of a Q App session.
     * </p>
     *
     * @param getQAppSessionMetadataRequest
     * @return A Java Future containing the result of the GetQAppSessionMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.GetQAppSessionMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/GetQAppSessionMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQAppSessionMetadataResponse> getQAppSessionMetadata(
            GetQAppSessionMetadataRequest getQAppSessionMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQAppSessionMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQAppSessionMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQAppSessionMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQAppSessionMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQAppSessionMetadataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetQAppSessionMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQAppSessionMetadataRequest, GetQAppSessionMetadataResponse>()
                            .withOperationName("GetQAppSessionMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQAppSessionMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQAppSessionMetadataRequest));
            CompletableFuture<GetQAppSessionMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads a file that can then be used either as a default in a <code>FileUploadCard</code> from Q App definition
     * or as a file that is used inside a single Q App run. The purpose of the document is determined by a scope
     * parameter that indicates whether it is at the app definition level or at the app session level.
     * </p>
     *
     * @param importDocumentRequest
     * @return A Java Future containing the result of the ImportDocument operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ContentTooLargeException The requested operation could not be completed because the content exceeds
     *         the maximum allowed size.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.ImportDocument
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/ImportDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ImportDocumentResponse> importDocument(ImportDocumentRequest importDocumentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importDocumentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importDocumentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportDocument");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportDocumentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportDocumentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ImportDocumentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportDocumentRequest, ImportDocumentResponse>()
                            .withOperationName("ImportDocument").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ImportDocumentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(importDocumentRequest));
            CompletableFuture<ImportDocumentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the categories of a Amazon Q Business application environment instance. For more information, see <a
     * href="https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/qapps-custom-labels.html">Custom labels for Amazon
     * Q Apps</a>.
     * </p>
     *
     * @param listCategoriesRequest
     * @return A Java Future containing the result of the ListCategories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.ListCategories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/ListCategories" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListCategoriesResponse> listCategories(ListCategoriesRequest listCategoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCategoriesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCategoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCategories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCategoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCategoriesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListCategoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCategoriesRequest, ListCategoriesResponse>()
                            .withOperationName("ListCategories").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCategoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCategoriesRequest));
            CompletableFuture<ListCategoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the library items for Amazon Q Apps that are published and available for users in your Amazon Web Services
     * account.
     * </p>
     *
     * @param listLibraryItemsRequest
     * @return A Java Future containing the result of the ListLibraryItems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.ListLibraryItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/ListLibraryItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLibraryItemsResponse> listLibraryItems(ListLibraryItemsRequest listLibraryItemsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLibraryItemsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLibraryItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLibraryItems");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLibraryItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLibraryItemsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListLibraryItemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLibraryItemsRequest, ListLibraryItemsResponse>()
                            .withOperationName("ListLibraryItems").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLibraryItemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLibraryItemsRequest));
            CompletableFuture<ListLibraryItemsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the collected data of a Q App data collection session.
     * </p>
     *
     * @param listQAppSessionDataRequest
     * @return A Java Future containing the result of the ListQAppSessionData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.ListQAppSessionData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/ListQAppSessionData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListQAppSessionDataResponse> listQAppSessionData(
            ListQAppSessionDataRequest listQAppSessionDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQAppSessionDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQAppSessionDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQAppSessionData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQAppSessionDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListQAppSessionDataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListQAppSessionDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQAppSessionDataRequest, ListQAppSessionDataResponse>()
                            .withOperationName("ListQAppSessionData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQAppSessionDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQAppSessionDataRequest));
            CompletableFuture<ListQAppSessionDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon Q Apps owned by or associated with the user either because they created it or because they used
     * it from the library in the past. The user identity is extracted from the credentials used to invoke this
     * operation..
     * </p>
     *
     * @param listQAppsRequest
     * @return A Java Future containing the result of the ListQApps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.ListQApps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/ListQApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListQAppsResponse> listQApps(ListQAppsRequest listQAppsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQAppsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQAppsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQApps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListQAppsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListQAppsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQAppsRequest, ListQAppsResponse>().withOperationName("ListQApps")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListQAppsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listQAppsRequest));
            CompletableFuture<ListQAppsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags associated with an Amazon Q Apps resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates an Amazon Q App definition based on either a conversation or a problem statement provided as input.The
     * resulting app definition can be used to call <code>CreateQApp</code>. This API doesn't create Amazon Q Apps
     * directly.
     * </p>
     *
     * @param predictQAppRequest
     * @return A Java Future containing the result of the PredictQApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.PredictQApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/PredictQApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PredictQAppResponse> predictQApp(PredictQAppRequest predictQAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(predictQAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, predictQAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PredictQApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PredictQAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PredictQAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PredictQAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PredictQAppRequest, PredictQAppResponse>()
                            .withOperationName("PredictQApp").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PredictQAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(predictQAppRequest));
            CompletableFuture<PredictQAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new session for an Amazon Q App, allowing inputs to be provided and the app to be run.
     * </p>
     * <note>
     * <p>
     * Each Q App session will be condensed into a single conversation in the web experience.
     * </p>
     * </note>
     *
     * @param startQAppSessionRequest
     * @return A Java Future containing the result of the StartQAppSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.StartQAppSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/StartQAppSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartQAppSessionResponse> startQAppSession(StartQAppSessionRequest startQAppSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startQAppSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startQAppSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQAppSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQAppSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartQAppSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartQAppSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQAppSessionRequest, StartQAppSessionResponse>()
                            .withOperationName("StartQAppSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartQAppSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startQAppSessionRequest));
            CompletableFuture<StartQAppSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an active session for an Amazon Q App.This deletes all data related to the session and makes it invalid for
     * future uses. The results of the session will be persisted as part of the conversation.
     * </p>
     *
     * @param stopQAppSessionRequest
     * @return A Java Future containing the result of the StopQAppSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.StopQAppSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/StopQAppSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopQAppSessionResponse> stopQAppSession(StopQAppSessionRequest stopQAppSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopQAppSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopQAppSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQAppSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopQAppSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopQAppSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopQAppSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopQAppSessionRequest, StopQAppSessionResponse>()
                            .withOperationName("StopQAppSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopQAppSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopQAppSessionRequest));
            CompletableFuture<StopQAppSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates tags with an Amazon Q Apps resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates tags from an Amazon Q Apps resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the library item for an Amazon Q App.
     * </p>
     *
     * @param updateLibraryItemRequest
     * @return A Java Future containing the result of the UpdateLibraryItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.UpdateLibraryItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/UpdateLibraryItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLibraryItemResponse> updateLibraryItem(UpdateLibraryItemRequest updateLibraryItemRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLibraryItemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLibraryItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLibraryItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLibraryItemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLibraryItemResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateLibraryItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLibraryItemRequest, UpdateLibraryItemResponse>()
                            .withOperationName("UpdateLibraryItem").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateLibraryItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLibraryItemRequest));
            CompletableFuture<UpdateLibraryItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the verification status of a library item for an Amazon Q App.
     * </p>
     *
     * @param updateLibraryItemMetadataRequest
     * @return A Java Future containing the result of the UpdateLibraryItemMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ConflictException The requested operation could not be completed due to a conflict with the current
     *         state of the resource.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.UpdateLibraryItemMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/UpdateLibraryItemMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLibraryItemMetadataResponse> updateLibraryItemMetadata(
            UpdateLibraryItemMetadataRequest updateLibraryItemMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLibraryItemMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLibraryItemMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLibraryItemMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLibraryItemMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLibraryItemMetadataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateLibraryItemMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLibraryItemMetadataRequest, UpdateLibraryItemMetadataResponse>()
                            .withOperationName("UpdateLibraryItemMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateLibraryItemMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLibraryItemMetadataRequest));
            CompletableFuture<UpdateLibraryItemMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing Amazon Q App, allowing modifications to its title, description, and definition.
     * </p>
     *
     * @param updateQAppRequest
     * @return A Java Future containing the result of the UpdateQApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ContentTooLargeException The requested operation could not be completed because the content exceeds
     *         the maximum allowed size.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.UpdateQApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/UpdateQApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateQAppResponse> updateQApp(UpdateQAppRequest updateQAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateQAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateQAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateQAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateQAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateQAppRequest, UpdateQAppResponse>().withOperationName("UpdateQApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateQAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateQAppRequest));
            CompletableFuture<UpdateQAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates read permissions for a Amazon Q App in Amazon Q Business application environment instance.
     * </p>
     *
     * @param updateQAppPermissionsRequest
     * @return A Java Future containing the result of the UpdateQAppPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.UpdateQAppPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/UpdateQAppPermissions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateQAppPermissionsResponse> updateQAppPermissions(
            UpdateQAppPermissionsRequest updateQAppPermissionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateQAppPermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQAppPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQAppPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateQAppPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateQAppPermissionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateQAppPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateQAppPermissionsRequest, UpdateQAppPermissionsResponse>()
                            .withOperationName("UpdateQAppPermissions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateQAppPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateQAppPermissionsRequest));
            CompletableFuture<UpdateQAppPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the session for a given Q App <code>sessionId</code>. This is only valid when at least one card of the
     * session is in the <code>WAITING</code> state. Data for each <code>WAITING</code> card can be provided as input.
     * If inputs are not provided, the call will be accepted but session will not move forward. Inputs for cards that
     * are not in the <code>WAITING</code> status will be ignored.
     * </p>
     *
     * @param updateQAppSessionRequest
     * @return A Java Future containing the result of the UpdateQAppSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.UpdateQAppSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/UpdateQAppSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateQAppSessionResponse> updateQAppSession(UpdateQAppSessionRequest updateQAppSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateQAppSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQAppSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQAppSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateQAppSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateQAppSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateQAppSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateQAppSessionRequest, UpdateQAppSessionResponse>()
                            .withOperationName("UpdateQAppSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateQAppSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateQAppSessionRequest));
            CompletableFuture<UpdateQAppSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration metadata of a session for a given Q App <code>sessionId</code>.
     * </p>
     *
     * @param updateQAppSessionMetadataRequest
     * @return A Java Future containing the result of the UpdateQAppSessionMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>AccessDeniedException The client is not authorized to perform the requested operation.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by the service.</li>
     *         <li>InternalServerException An internal service error occurred while processing the request.</li>
     *         <li>UnauthorizedException The client is not authenticated or authorized to perform the requested
     *         operation.</li>
     *         <li>ServiceQuotaExceededException The requested operation could not be completed because it would exceed
     *         the service's quota or limit.</li>
     *         <li>ThrottlingException The requested operation could not be completed because too many requests were
     *         sent at once. Wait a bit and try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>QAppsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample QAppsAsyncClient.UpdateQAppSessionMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/qapps-2023-11-27/UpdateQAppSessionMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateQAppSessionMetadataResponse> updateQAppSessionMetadata(
            UpdateQAppSessionMetadataRequest updateQAppSessionMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateQAppSessionMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQAppSessionMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "QApps");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQAppSessionMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateQAppSessionMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateQAppSessionMetadataResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ContentTooLargeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ContentTooLargeException").httpStatusCode(413)
                            .exceptionBuilderSupplier(ContentTooLargeException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateQAppSessionMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateQAppSessionMetadataRequest, UpdateQAppSessionMetadataResponse>()
                            .withOperationName("UpdateQAppSessionMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateQAppSessionMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateQAppSessionMetadataRequest));
            CompletableFuture<UpdateQAppSessionMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final QAppsServiceClientConfiguration serviceClientConfiguration() {
        return new QAppsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(QAppsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        QAppsServiceClientConfigurationBuilder serviceConfigBuilder = new QAppsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
